"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Rule = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const constructs_1 = require("constructs");
const events_generated_1 = require("./events.generated");
const schedule_1 = require("./schedule");
const util_1 = require("./util");
/**
 * Defines an EventBridge Rule in this stack.
 *
 * @stability stable
 * @resource AWS::Events::Rule
 */
class Rule extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.ruleName,
        });
        this.targets = new Array();
        this.eventPattern = {};
        this.accountEventBusTargets = {};
        if (props.eventBus && props.schedule) {
            throw new Error('Cannot associate rule with \'eventBus\' when using \'schedule\'');
        }
        this.description = props.description;
        this.scheduleExpression = props.schedule && props.schedule.expressionString;
        const resource = new events_generated_1.CfnRule(this, 'Resource', {
            name: this.physicalName,
            description: this.description,
            state: props.enabled == null ? 'ENABLED' : (props.enabled ? 'ENABLED' : 'DISABLED'),
            scheduleExpression: this.scheduleExpression,
            eventPattern: core_1.Lazy.any({ produce: () => this._renderEventPattern() }),
            targets: core_1.Lazy.any({ produce: () => this.renderTargets() }),
            eventBusName: props.eventBus && props.eventBus.eventBusName,
        });
        this.ruleArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'events',
            resource: 'rule',
            resourceName: this.physicalName,
        });
        this.ruleName = this.getResourceNameAttribute(resource.ref);
        this.addEventPattern(props.eventPattern);
        for (const target of props.targets || []) {
            this.addTarget(target);
        }
    }
    /**
     * Import an existing EventBridge Rule provided an ARN.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param eventRuleArn Event Rule ARN (i.e. arn:aws:events:<region>:<account-id>:rule/MyScheduledRule).
     * @stability stable
     */
    static fromEventRuleArn(scope, id, eventRuleArn) {
        const parts = core_1.Stack.of(scope).parseArn(eventRuleArn);
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.ruleArn = eventRuleArn;
                this.ruleName = parts.resourceName || '';
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds a target to the rule. The abstract class RuleTarget can be extended to define new targets.
     *
     * No-op if target is undefined.
     *
     * @stability stable
     */
    addTarget(target) {
        var _b;
        if (!target) {
            return;
        }
        // Simply increment id for each `addTarget` call. This is guaranteed to be unique.
        const autoGeneratedId = `Target${this.targets.length}`;
        const targetProps = target.bind(this, autoGeneratedId);
        const inputProps = targetProps.input && targetProps.input.bind(this);
        const roleArn = (_b = targetProps.role) === null || _b === void 0 ? void 0 : _b.roleArn;
        const id = targetProps.id || autoGeneratedId;
        if (targetProps.targetResource) {
            const targetStack = core_1.Stack.of(targetProps.targetResource);
            const targetAccount = targetStack.account;
            const targetRegion = targetStack.region;
            const sourceStack = core_1.Stack.of(this);
            const sourceAccount = sourceStack.account;
            const sourceRegion = sourceStack.region;
            if (targetRegion !== sourceRegion) {
                throw new Error('Rule and target must be in the same region');
            }
            if (targetAccount !== sourceAccount) {
                // cross-account event - strap in, this works differently than regular events!
                // based on:
                // https://docs.aws.amazon.com/eventbridge/latest/userguide/eventbridge-cross-account-event-delivery.html
                // for cross-account events, we require concrete accounts
                if (core_1.Token.isUnresolved(targetAccount)) {
                    throw new Error('You need to provide a concrete account for the target stack when using cross-account events');
                }
                if (core_1.Token.isUnresolved(sourceAccount)) {
                    throw new Error('You need to provide a concrete account for the source stack when using cross-account events');
                }
                // and the target region has to be concrete as well
                if (core_1.Token.isUnresolved(targetRegion)) {
                    throw new Error('You need to provide a concrete region for the target stack when using cross-account events');
                }
                // the _actual_ target is just the event bus of the target's account
                // make sure we only add it once per account
                const exists = this.accountEventBusTargets[targetAccount];
                if (!exists) {
                    this.accountEventBusTargets[targetAccount] = true;
                    this.targets.push({
                        id,
                        arn: targetStack.formatArn({
                            service: 'events',
                            resource: 'event-bus',
                            resourceName: 'default',
                            region: targetRegion,
                            account: targetAccount,
                        }),
                    });
                }
                // Grant the source account permissions to publish events to the event bus of the target account.
                // Do it in a separate stack instead of the target stack (which seems like the obvious place to put it),
                // because it needs to be deployed before the rule containing the above event-bus target in the source stack
                // (EventBridge verifies whether you have permissions to the targets on rule creation),
                // but it's common for the target stack to depend on the source stack
                // (that's the case with CodePipeline, for example)
                const sourceApp = this.node.root;
                if (!sourceApp || !core_1.App.isApp(sourceApp)) {
                    throw new Error('Event stack which uses cross-account targets must be part of a CDK app');
                }
                const targetApp = constructs_1.Node.of(targetProps.targetResource).root;
                if (!targetApp || !core_1.App.isApp(targetApp)) {
                    throw new Error('Target stack which uses cross-account event targets must be part of a CDK app');
                }
                if (sourceApp !== targetApp) {
                    throw new Error('Event stack and target stack must belong to the same CDK app');
                }
                const stackId = `EventBusPolicy-${sourceAccount}-${targetRegion}-${targetAccount}`;
                let eventBusPolicyStack = sourceApp.node.tryFindChild(stackId);
                if (!eventBusPolicyStack) {
                    eventBusPolicyStack = new core_1.Stack(sourceApp, stackId, {
                        env: {
                            account: targetAccount,
                            region: targetRegion,
                        },
                        stackName: `${targetStack.stackName}-EventBusPolicy-support-${targetRegion}-${sourceAccount}`,
                    });
                    new events_generated_1.CfnEventBusPolicy(eventBusPolicyStack, 'GivePermToOtherAccount', {
                        action: 'events:PutEvents',
                        statementId: `Allow-account-${sourceAccount}`,
                        principal: sourceAccount,
                    });
                }
                // deploy the event bus permissions before the source stack
                sourceStack.addDependency(eventBusPolicyStack);
                // The actual rule lives in the target stack.
                // Other than the account, it's identical to this one
                // eventPattern is mutable through addEventPattern(), so we need to lazy evaluate it
                // but only Tokens can be lazy in the framework, so make a subclass instead
                const self = this;
                class CopyRule extends Rule {
                    _renderEventPattern() {
                        return self._renderEventPattern();
                    }
                    // we need to override validate(), as it uses the
                    // value of the eventPattern field,
                    // which might be empty in the case of the copied rule
                    // (as the patterns in the original might be added through addEventPattern(),
                    // not passed through the constructor).
                    // Anyway, even if the original rule is invalid,
                    // we would get duplicate errors if we didn't override this,
                    // which is probably a bad idea in and of itself
                    validate() {
                        return [];
                    }
                }
                new CopyRule(targetStack, `${core_1.Names.uniqueId(this)}-${id}`, {
                    targets: [target],
                    eventPattern: this.eventPattern,
                    schedule: this.scheduleExpression ? schedule_1.Schedule.expression(this.scheduleExpression) : undefined,
                    description: this.description,
                });
                return;
            }
        }
        this.targets.push({
            id,
            arn: targetProps.arn,
            roleArn,
            ecsParameters: targetProps.ecsParameters,
            kinesisParameters: targetProps.kinesisParameters,
            runCommandParameters: targetProps.runCommandParameters,
            batchParameters: targetProps.batchParameters,
            deadLetterConfig: targetProps.deadLetterConfig,
            retryPolicy: targetProps.retryPolicy,
            sqsParameters: targetProps.sqsParameters,
            httpParameters: targetProps.httpParameters,
            input: inputProps && inputProps.input,
            inputPath: inputProps && inputProps.inputPath,
            inputTransformer: (inputProps === null || inputProps === void 0 ? void 0 : inputProps.inputTemplate) !== undefined ? {
                inputTemplate: inputProps.inputTemplate,
                inputPathsMap: inputProps.inputPathsMap,
            } : undefined,
        });
    }
    /**
     * Adds an event pattern filter to this rule.
     *
     * If a pattern was already specified,
     * these values are merged into the existing pattern.
     *
     * For example, if the rule already contains the pattern:
     *
     *     {
     *       "resources": [ "r1" ],
     *       "detail": {
     *         "hello": [ 1 ]
     *       }
     *     }
     *
     * And `addEventPattern` is called with the pattern:
     *
     *     {
     *       "resources": [ "r2" ],
     *       "detail": {
     *         "foo": [ "bar" ]
     *       }
     *     }
     *
     * The resulting event pattern will be:
     *
     *     {
     *       "resources": [ "r1", "r2" ],
     *       "detail": {
     *         "hello": [ 1 ],
     *         "foo": [ "bar" ]
     *       }
     *     }
     *
     * @stability stable
     */
    addEventPattern(eventPattern) {
        if (!eventPattern) {
            return;
        }
        util_1.mergeEventPattern(this.eventPattern, eventPattern);
    }
    /**
     * Not private only to be overrideen in CopyRule.
     *
     * @internal
     */
    _renderEventPattern() {
        const eventPattern = this.eventPattern;
        if (Object.keys(eventPattern).length === 0) {
            return undefined;
        }
        // rename 'detailType' to 'detail-type'
        const out = {};
        for (let key of Object.keys(eventPattern)) {
            const value = eventPattern[key];
            if (key === 'detailType') {
                key = 'detail-type';
            }
            out[key] = value;
        }
        return out;
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    validate() {
        if (Object.keys(this.eventPattern).length === 0 && !this.scheduleExpression) {
            return ['Either \'eventPattern\' or \'schedule\' must be defined'];
        }
        return [];
    }
    renderTargets() {
        if (this.targets.length === 0) {
            return undefined;
        }
        return this.targets;
    }
}
exports.Rule = Rule;
_a = JSII_RTTI_SYMBOL_1;
Rule[_a] = { fqn: "@aws-cdk/aws-events.Rule", version: "1.104.0" };
//# sourceMappingURL=data:application/json;base64,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