"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service execute-api
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonapigateway.html
 */
class ExecuteApi extends shared_1.PolicyStatement {
    /**
     * Action provider for service execute-api
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonapigateway.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'execute-api';
        this.actions = {
            "InvalidateCache": {
                "url": "https://docs.aws.amazon.com/apigateway/api-reference/api-gateway-caching.html",
                "description": "Used to invalidate API cache upon a client request",
                "accessLevel": "Write",
                "resourceTypes": {
                    "execute-api-general": {
                        "required": true
                    }
                }
            },
            "Invoke": {
                "url": "https://docs.aws.amazon.com/apigateway/api-reference/how-to-call-api.html",
                "description": "Used to invoke an API upon a client request",
                "accessLevel": "Write",
                "resourceTypes": {
                    "execute-api-general": {
                        "required": true
                    }
                }
            },
            "ManageConnections": {
                "url": "https://docs.aws.amazon.com/apigateway/api-reference/apigateway-websocket-control-access-iam.html",
                "description": "ManageConnections controls access to the @connections API",
                "accessLevel": "Write",
                "resourceTypes": {
                    "execute-api-general": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "execute-api-general": {
                "name": "execute-api-general",
                "url": "",
                "arn": "arn:${Partition}:execute-api:${Region}:${Account}:${ApiId}/${Stage}/${Method}/${ApiSpecificResourcePath}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Used to invalidate API cache upon a client request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/apigateway/api-reference/api-gateway-caching.html
     */
    invalidateCache() {
        this.add('execute-api:InvalidateCache');
        return this;
    }
    /**
     * Used to invoke an API upon a client request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/apigateway/api-reference/how-to-call-api.html
     */
    invoke() {
        this.add('execute-api:Invoke');
        return this;
    }
    /**
     * ManageConnections controls access to the @connections API
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/apigateway/api-reference/apigateway-websocket-control-access-iam.html
     */
    manageConnections() {
        this.add('execute-api:ManageConnections');
        return this;
    }
    /**
     * Adds a resource of type execute-api-general to the statement
     *
     * @param apiId - Identifier for the apiId.
     * @param stage - Identifier for the stage.
     * @param method - Identifier for the method.
     * @param apiSpecificResourcePath - Identifier for the apiSpecificResourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onExecuteApiGeneral(apiId, stage, method, apiSpecificResourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:execute-api:${Region}:${Account}:${ApiId}/${Stage}/${Method}/${ApiSpecificResourcePath}';
        arn = arn.replace('${ApiId}', apiId);
        arn = arn.replace('${Stage}', stage);
        arn = arn.replace('${Method}', method);
        arn = arn.replace('${ApiSpecificResourcePath}', apiSpecificResourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.ExecuteApi = ExecuteApi;
//# sourceMappingURL=data:application/json;base64,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