"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service chime
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonchime.html
 */
class Chime extends shared_1.PolicyStatement {
    /**
     * Action provider for service chime
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonchime.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'chime';
        this.actions = {
            "AcceptDelegate": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to accept the delegate invitation to share management of an Amazon Chime account with another AWS Account",
                "accessLevel": "Write"
            },
            "ActivateUsers": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/manage-access.html",
                "description": "Grants permission to activate users in an Amazon Chime Enterprise account",
                "accessLevel": "Write"
            },
            "AddDomain": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/claim-domain.html",
                "description": "Grants permission to add a domain to your Amazon Chime account",
                "accessLevel": "Write"
            },
            "AddOrUpdateGroups": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/manage-chime-account.html",
                "description": "Grants permission to add new or update existing Active Directory or Okta user groups associated with your Amazon Chime Enterprise account",
                "accessLevel": "Write"
            },
            "AssociatePhoneNumberWithUser": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_AssociatePhoneNumberWithUser.html",
                "description": "Grants permission to associate a phone number with an Amazon Chime user",
                "accessLevel": "Write"
            },
            "AssociatePhoneNumbersWithVoiceConnector": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_AssociatePhoneNumbersWithVoiceConnector.html",
                "description": "Grants permission to associate multiple phone numbers with an Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "AssociatePhoneNumbersWithVoiceConnectorGroup": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_AssociatePhoneNumbersWithVoiceConnectorGroup.html",
                "description": "Grants permission to associate multiple phone numbers with an Amazon Chime Voice Connector Group",
                "accessLevel": "Write"
            },
            "AssociateSigninDelegateGroupsWithAccount": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_AssociateSigninDelegateGroupsWithAccount.html",
                "description": "Grants permission to associate the specified sign-in delegate groups with the specified Amazon Chime account.",
                "accessLevel": "Write"
            },
            "AuthorizeDirectory": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to authorize an Active Directory for your Amazon Chime Enterprise account",
                "accessLevel": "Write"
            },
            "BatchCreateAttendee": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchCreateAttendee.html",
                "description": "Grants permission to create new attendees for an active Amazon Chime SDK meeting",
                "accessLevel": "Write",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "BatchCreateRoomMembership": {
                "url": "",
                "description": "Grants permission to batch add room members",
                "accessLevel": "Write"
            },
            "BatchDeletePhoneNumber": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchDeletePhoneNumber.html",
                "description": "Grants permission to move up to 50 phone numbers to the deletion queue",
                "accessLevel": "Write"
            },
            "BatchSuspendUser": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchSuspendUser.html",
                "description": "Grants permission to suspend up to 50 users from a Team or EnterpriseLWA Amazon Chime account",
                "accessLevel": "Write"
            },
            "BatchUnsuspendUser": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchUnsuspendUser.html",
                "description": "Grants permission to remove the suspension from up to 50 previously suspended users for the specified Amazon Chime EnterpriseLWA account",
                "accessLevel": "Write"
            },
            "BatchUpdatePhoneNumber": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchUpdatePhoneNumber.html",
                "description": "Grants permission to update phone number details within the UpdatePhoneNumberRequestItem object for up to 50 phone numbers",
                "accessLevel": "Write"
            },
            "BatchUpdateUser": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchUpdateUser.html",
                "description": "Grants permission to update user details within the UpdateUserRequestItem object for up to 20 users for the specified Amazon Chime account",
                "accessLevel": "Write"
            },
            "ConnectDirectory": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/active_directory.html",
                "description": "Grants permission to connect an Active Directory to your Amazon Chime Enterprise account",
                "accessLevel": "Write"
            },
            "CreateAccount": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateAccount.html",
                "description": "Grants permission to create an Amazon Chime account under the administrator's AWS account",
                "accessLevel": "Write"
            },
            "CreateApiKey": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html",
                "description": "Grants permission to create a new SCIM access key for your Amazon Chime account and Okta configuration",
                "accessLevel": "Write"
            },
            "CreateAttendee": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateAttendee.html",
                "description": "Grants permission to create a new attendee for an active Amazon Chime SDK meeting",
                "accessLevel": "Write",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "CreateBot": {
                "url": "",
                "description": "Grants permission to create a bot for an Amazon Chime Enterprise account",
                "accessLevel": "Write"
            },
            "CreateBotMembership": {
                "url": "",
                "description": "Grants permission to add a bot to a chat room in your Amazon Chime Enterprise account",
                "accessLevel": "Write"
            },
            "CreateCDRBucket": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/manage-access.html",
                "description": "Grants permission to create a new Call Detail Record S3 bucket",
                "accessLevel": "Write"
            },
            "CreateMeeting": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateMeeting.html",
                "description": "Grants permission to create a new Amazon Chime SDK meeting in the specified media Region, with no initial attendees",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreatePhoneNumberOrder": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_CreatePhoneNumberOrder.html",
                "description": "Grants permission to create a phone number order with the Carriers",
                "accessLevel": "Write"
            },
            "CreateProxySession": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateProxySession.html",
                "description": "Grants permission to create a proxy session for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "CreateRoom": {
                "url": "",
                "description": "Grants permission to create a room",
                "accessLevel": "Write"
            },
            "CreateRoomMembership": {
                "url": "",
                "description": "Grants permission to add a room member",
                "accessLevel": "Write"
            },
            "CreateUser": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateUser.html",
                "description": "Grants permission to create a user under the specified Amazon Chime account.",
                "accessLevel": "Write"
            },
            "CreateVoiceConnector": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateVoiceConnector.html",
                "description": "Grants permission to create a Amazon Chime Voice Connector under the administrator's AWS account",
                "accessLevel": "Write"
            },
            "CreateVoiceConnectorGroup": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateVoiceConnectorGroup.html",
                "description": "Grants permission to create a Amazon Chime Voice Connector Group under the administrator's AWS account",
                "accessLevel": "Write"
            },
            "DeleteAccount": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteAccount.html",
                "description": "Grants permission to delete the specified Amazon Chime account",
                "accessLevel": "Write"
            },
            "DeleteAccountOpenIdConfig": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html",
                "description": "Grants permission to delete the OpenIdConfig attributes from your Amazon Chime account",
                "accessLevel": "Write"
            },
            "DeleteApiKey": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html",
                "description": "Grants permission to delete the specified SCIM access key associated with your Amazon Chime account and Okta configuration",
                "accessLevel": "Write"
            },
            "DeleteAttendee": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteAttendee.html",
                "description": "Grants permission to delete the specified attendee from an Amazon Chime SDK meeting",
                "accessLevel": "Write",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "DeleteCDRBucket": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to delete a Call Detail Record S3 bucket from your Amazon Chime account",
                "accessLevel": "Write"
            },
            "DeleteDelegate": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to delete delegated AWS account management from your Amazon Chime account",
                "accessLevel": "Write"
            },
            "DeleteDomain": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/claim-domain.html",
                "description": "Grants permission to delete a domain from your Amazon Chime account",
                "accessLevel": "Write"
            },
            "DeleteEventsConfiguration": {
                "url": "",
                "description": "Grants permission to delete an events configuration for a bot to receive outgoing events",
                "accessLevel": "Write"
            },
            "DeleteGroups": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to delete Active Directory or Okta user groups from your Amazon Chime Enterprise account",
                "accessLevel": "Write"
            },
            "DeleteMeeting": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteMeeting.html",
                "description": "Grants permission to delete the specified Amazon Chime SDK meeting",
                "accessLevel": "Write",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "DeletePhoneNumber": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeletePhoneNumber.html",
                "description": "Grants permission to move a phone number to the deletion queue",
                "accessLevel": "Write"
            },
            "DeleteProxySession": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteProxySession.html",
                "description": "Grants permission to delete a proxy session for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "DeleteRoom": {
                "url": "",
                "description": "Grants permission to delete a room",
                "accessLevel": "Write"
            },
            "DeleteRoomMembership": {
                "url": "",
                "description": "Grants permission to remove a room member",
                "accessLevel": "Write"
            },
            "DeleteVoiceConnector": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnector.html",
                "description": "Grants permission to delete the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "DeleteVoiceConnectorEmergencyCallingConfiguration": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorEmergencyCallingConfiguration.html",
                "description": "Grants permission to delete emergency calling configuration for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "DeleteVoiceConnectorGroup": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorGroup.html",
                "description": "Grants permission to delete the specified Amazon Chime Voice Connector Group",
                "accessLevel": "Write"
            },
            "DeleteVoiceConnectorOrigination": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorOrigination.html",
                "description": "Grants permission to delete the origination settings for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "DeleteVoiceConnectorProxy": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorProxy.html",
                "description": "Grants permission to delete proxy configuration for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "DeleteVoiceConnectorStreamingConfiguration": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorStreamingConfiguration.html",
                "description": "Grants permission to delete streaming configuration for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "DeleteVoiceConnectorTermination": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorTermination.html",
                "description": "Grants permission to delete the termination settings for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "DeleteVoiceConnectorTerminationCredentials": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorTerminationCredentials.html",
                "description": "Grants permission to delete SIP termination credentials for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "DisassociatePhoneNumberFromUser": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DisassociatePhoneNumberFromUser.html",
                "description": "Grants permission to disassociate the primary provisioned number from the specified Amazon Chime user",
                "accessLevel": "Write"
            },
            "DisassociatePhoneNumbersFromVoiceConnector": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DisassociatePhoneNumbersFromVoiceConnector.html",
                "description": "Grants permission to disassociate multiple phone numbers from the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "DisassociatePhoneNumbersFromVoiceConnectorGroup": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DisassociatePhoneNumbersFromVoiceConnectorGroup.html",
                "description": "Grants permission to disassociate multiple phone numbers from the specified Amazon Chime Voice Connector Group",
                "accessLevel": "Write"
            },
            "DisassociateSigninDelegateGroupsFromAccount": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DisassociateSigninDelegateGroupsFromAccount.html",
                "description": "Grants permission to disassociate the specified sign-in delegate groups from the specified Amazon Chime account.",
                "accessLevel": "Write"
            },
            "DisconnectDirectory": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to disconnect the Active Directory from your Amazon Chime Enterprise account",
                "accessLevel": "Write"
            },
            "GetAccount": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetAccount.html",
                "description": "Grants permission to get details for the specified Amazon Chime account",
                "accessLevel": "Read"
            },
            "GetAccountResource": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to get details for the account resource associated with your Amazon Chime account",
                "accessLevel": "Read"
            },
            "GetAccountSettings": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetAccountSettings.html",
                "description": "Grants permission to get account settings for the specified Amazon Chime account ID",
                "accessLevel": "Read"
            },
            "GetAccountWithOpenIdConfig": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html",
                "description": "Grants permission to get the account details and OpenIdConfig attributes for your Amazon Chime account",
                "accessLevel": "Read"
            },
            "GetAttendee": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetAttendee.html",
                "description": "Grants permission to get attendee details for a specified meeting ID and attendee ID",
                "accessLevel": "Read",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "GetBot": {
                "url": "",
                "description": "Grants permission to retrieve details for the specified bot",
                "accessLevel": "Read"
            },
            "GetCDRBucket": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to get details of a Call Detail Record S3 bucket associated with your Amazon Chime account",
                "accessLevel": "Read"
            },
            "GetDomain": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/claim-domain.html",
                "description": "Grants permission to get domain details for a domain associated with your Amazon Chime account",
                "accessLevel": "Read"
            },
            "GetEventsConfiguration": {
                "url": "",
                "description": "Grants permission to retrieve details for an events configuration for a bot to receive outgoing events",
                "accessLevel": "Read"
            },
            "GetGlobalSettings": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetGlobalSettings.html",
                "description": "Grants permission to get global settings related to Amazon Chime for the AWS account",
                "accessLevel": "Read"
            },
            "GetMeeting": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetMeeting.html",
                "description": "Grants permission to get the meeting record for a specified meeting ID",
                "accessLevel": "Read",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "GetMeetingDetail": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to get attendee, connection, and other details for a meeting",
                "accessLevel": "Read"
            },
            "GetPhoneNumber": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetPhoneNumber.html",
                "description": "Grants permission to get details for the specified phone number",
                "accessLevel": "Read"
            },
            "GetPhoneNumberOrder": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetPhoneNumberOrder.html",
                "description": "Grants permission to get details for the specified phone number order",
                "accessLevel": "Read"
            },
            "GetPhoneNumberSettings": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetPhoneNumberSettings.html",
                "description": "Grants permission to get phone number settings related to Amazon Chime for the AWS account",
                "accessLevel": "Read"
            },
            "GetProxySession": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetProxySession.html",
                "description": "Grants permission to get details of the specified proxy session for the specified Amazon Chime Voice Connector",
                "accessLevel": "Read"
            },
            "GetRetentionSettings": {
                "url": "",
                "description": "Gets the retention settings for the specified Amazon Chime account.",
                "accessLevel": "Read"
            },
            "GetRoom": {
                "url": "",
                "description": "Grants permission to retrieve a room",
                "accessLevel": "Read"
            },
            "GetTelephonyLimits": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/phone-numbers.html",
                "description": "Grants permission to get telephony limits for the AWS account",
                "accessLevel": "Read"
            },
            "GetUser": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetUser.html",
                "description": "Grants permission to get details for the specified user ID",
                "accessLevel": "Read"
            },
            "GetUserActivityReportData": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/user-details.html",
                "description": "Grants permission to get a summary of user activity on the user details page",
                "accessLevel": "Read"
            },
            "GetUserByEmail": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/user-details.html",
                "description": "Grants permission to get user details for an Amazon Chime user based on the email address in an Amazon Chime Enterprise or Team account",
                "accessLevel": "Read"
            },
            "GetUserSettings": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetUserSettings.html",
                "description": "Grants permission to get user settings related to the specified Amazon Chime user",
                "accessLevel": "Read"
            },
            "GetVoiceConnector": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnector.html",
                "description": "Grants permission to get details for the specified Amazon Chime Voice Connector",
                "accessLevel": "Read"
            },
            "GetVoiceConnectorEmergencyCallingConfiguration": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorEmergencyCallingConfiguration.html",
                "description": "Grants permission to get details of the emergency calling configuration for the specified Amazon Chime Voice Connector",
                "accessLevel": "Read"
            },
            "GetVoiceConnectorGroup": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorGroup.html",
                "description": "Grants permission to get details for the specified Amazon Chime Voice Connector Group",
                "accessLevel": "Read"
            },
            "GetVoiceConnectorLoggingConfiguration": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorLoggingConfiguration.html",
                "description": "Grants permission to get details of the logging configuration for the specified Amazon Chime Voice Connector",
                "accessLevel": "Read"
            },
            "GetVoiceConnectorOrigination": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorOrigination.html",
                "description": "Grants permission to get details of the origination settings for the specified Amazon Chime Voice Connector",
                "accessLevel": "Read"
            },
            "GetVoiceConnectorProxy": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorProxy.html",
                "description": "Grants permission to get details of the proxy configuration for the specified Amazon Chime Voice Connector",
                "accessLevel": "Read"
            },
            "GetVoiceConnectorStreamingConfiguration": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorStreamingConfiguration.html",
                "description": "Grants permission to get details of the streaming configuration for the specified Amazon Chime Voice Connector",
                "accessLevel": "Read"
            },
            "GetVoiceConnectorTermination": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorTermination.html",
                "description": "Grants permission to get details of the termination settings for the specified Amazon Chime Voice Connector",
                "accessLevel": "Read"
            },
            "GetVoiceConnectorTerminationHealth": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorTerminationHealth.html",
                "description": "Grants permission to get details of the termination health for the specified Amazon Chime Voice Connector",
                "accessLevel": "Read"
            },
            "InviteDelegate": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to send an invitation to accept a request for AWS account delegation for an Amazon Chime account",
                "accessLevel": "Write"
            },
            "InviteUsers": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_InviteUsers.html",
                "description": "Grants permission to invite as many as 50 users to the specified Amazon Chime account",
                "accessLevel": "Write"
            },
            "InviteUsersFromProvider": {
                "url": "",
                "description": "Grants permission to invite users from a third party provider to your Amazon Chime account",
                "accessLevel": "Write"
            },
            "ListAccountUsageReportData": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/view-reports.html",
                "description": "Grants permission to list Amazon Chime account usage reporting data",
                "accessLevel": "List"
            },
            "ListAccounts": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListAccounts.html",
                "description": "Grants permission to list the Amazon Chime accounts under the administrator's AWS account",
                "accessLevel": "List"
            },
            "ListApiKeys": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html",
                "description": "Grants permission to list the SCIM access keys defined for your Amazon Chime account and Okta configuration",
                "accessLevel": "List"
            },
            "ListAttendeeTags": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListAttendeeTags.html",
                "description": "Grants permission to list the tags applied to an Amazon Chime SDK attendee resource",
                "accessLevel": "Read",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "ListAttendees": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListAttendees.html",
                "description": "Grants permission to list up to 100 attendees for a specified Amazon Chime SDK meeting",
                "accessLevel": "Read",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "ListBots": {
                "url": "",
                "description": "Grants permission to list the bots associated with the administrator's Amazon Chime Enterprise account",
                "accessLevel": "List"
            },
            "ListCDRBucket": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to list Call Detail Record S3 buckets",
                "accessLevel": "List"
            },
            "ListCallingRegions": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/phone-numbers.html",
                "description": "Grants permission to list the calling regions available for the administrator's AWS account",
                "accessLevel": "List"
            },
            "ListDelegates": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to list account delegate information associated with your Amazon Chime account",
                "accessLevel": "List"
            },
            "ListDirectories": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to list active Active Directories hosted in the Directory Service of your AWS account",
                "accessLevel": "List"
            },
            "ListDomains": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/claim-domain.html",
                "description": "Grants permission to list domains associated with your Amazon Chime account",
                "accessLevel": "List"
            },
            "ListGroups": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to list Active Directory or Okta user groups associated with your Amazon Chime Enterprise account",
                "accessLevel": "List"
            },
            "ListMeetingEvents": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/view-reports.html",
                "description": "Grants permission to list all events that occurred for a specified meeting",
                "accessLevel": "List"
            },
            "ListMeetingTags": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListMeetingTags.html",
                "description": "Grants permission to list the tags applied to an Amazon Chime SDK meeting resource.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "ListMeetings": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListMeetings.html",
                "description": "Grants permission to list up to 100 active Amazon Chime SDK meetings",
                "accessLevel": "Read"
            },
            "ListMeetingsReportData": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/view-reports.html",
                "description": "Grants permission to list meetings ended during the specified date range",
                "accessLevel": "List"
            },
            "ListPhoneNumberOrders": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListPhoneNumberOrders.html",
                "description": "Grants permission to list the phone number orders under the administrator's AWS account",
                "accessLevel": "List"
            },
            "ListPhoneNumbers": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListPhoneNumbers.html",
                "description": "Grants permission to list the phone numbers under the administrator's AWS account",
                "accessLevel": "List"
            },
            "ListProxySessions": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListProxySessions.html",
                "description": "Grants permission to list proxy sessions for the specified Amazon Chime Voice Connector",
                "accessLevel": "List"
            },
            "ListRoomMemberships": {
                "url": "",
                "description": "Grants permission to list all room members",
                "accessLevel": "Read"
            },
            "ListRooms": {
                "url": "",
                "description": "Grants permission to list rooms",
                "accessLevel": "Read"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListTagsForResource.html",
                "description": "Grants permission to list the tags applied to an Amazon Chime SDK meeting resource.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "meeting": {
                        "required": false
                    }
                }
            },
            "ListUsers": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListUsers.html",
                "description": "Grants permission to list the users that belong to the specified Amazon Chime account",
                "accessLevel": "List"
            },
            "ListVoiceConnectorGroups": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListVoiceConnectorGroups.html",
                "description": "Grants permission to list the Amazon Chime Voice Connector Groups under the administrator's AWS account",
                "accessLevel": "List"
            },
            "ListVoiceConnectorTerminationCredentials": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListVoiceConnectorTerminationCredentials.html",
                "description": "Grants permission to list the SIP termination credentials for the specified Amazon Chime Voice Connector",
                "accessLevel": "List"
            },
            "ListVoiceConnectors": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListVoiceConnectors.html",
                "description": "Grants permission to list the Amazon Chime Voice Connectors under the administrator's AWS account",
                "accessLevel": "List"
            },
            "LogoutUser": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_LogoutUser.html",
                "description": "Grants permission to log out the specified user from all of the devices they are currently logged into",
                "accessLevel": "Write"
            },
            "PutEventsConfiguration": {
                "url": "",
                "description": "Grants permission to update details for an events configuration for a bot to receive outgoing events",
                "accessLevel": "Write"
            },
            "PutRetentionSettings": {
                "url": "",
                "description": "Puts retention settings for the specified Amazon Chime account",
                "accessLevel": "Write"
            },
            "PutVoiceConnectorEmergencyCallingConfiguration": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorEmergencyCallingConfiguration.html",
                "description": "Grants permission to add emergency calling configuration for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "PutVoiceConnectorLoggingConfiguration": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorLoggingConfiguration.html",
                "description": "Grants permission to add logging configuration for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "PutVoiceConnectorOrigination": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorOrigination.html",
                "description": "Grants permission to update the origination settings for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "PutVoiceConnectorProxy": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorProxy.html",
                "description": "Grants permission to add proxy configuration for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "PutVoiceConnectorStreamingConfiguration": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorStreamingConfiguration.html",
                "description": "Grants permission to add streaming configuration for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "PutVoiceConnectorTermination": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorTermination.html",
                "description": "Grants permission to update the termination settings for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "PutVoiceConnectorTerminationCredentials": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorTerminationCredentials.html",
                "description": "Grants permission to add SIP termination credentials for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "RedactConversationMessage": {
                "url": "",
                "description": "Redacts the specified Chime conversation Message",
                "accessLevel": "Write"
            },
            "RedactRoomMessage": {
                "url": "",
                "description": "Redacts the specified Chime room Message",
                "accessLevel": "Write"
            },
            "RegenerateSecurityToken": {
                "url": "",
                "description": "Grants permission to regenerate the security token for the specified bot",
                "accessLevel": "Write"
            },
            "RenameAccount": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/rename-account.html",
                "description": "Grants permission to modify the account name for your Amazon Chime Enterprise or Team account",
                "accessLevel": "Write"
            },
            "RenewDelegate": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to renew the delegation request associated with an Amazon Chime account",
                "accessLevel": "Write"
            },
            "ResetAccountResource": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to reset the account resource in your Amazon Chime account",
                "accessLevel": "Write"
            },
            "ResetPersonalPIN": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ResetPersonalPIN.html",
                "description": "Grants permission to reset the personal meeting PIN for the specified user on an Amazon Chime account",
                "accessLevel": "Write"
            },
            "RestorePhoneNumber": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_RestorePhoneNumber.html",
                "description": "Grants permission to restore the specified phone number from the deltion queue back to the phone number inventory",
                "accessLevel": "Write"
            },
            "RetrieveDataExports": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/request-attachments.html",
                "description": "Grants permission to download the file containing links to all user attachments returned as part of the \"Request attachments\" action",
                "accessLevel": "List"
            },
            "SearchAvailablePhoneNumbers": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_SearchAvailablePhoneNumbers.html",
                "description": "Grants permission to search phone numbers that can be ordered from the carrier",
                "accessLevel": "Read"
            },
            "StartDataExport": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/request-attachments.html",
                "description": "Grants permission to submit the \"Request attachments\" request",
                "accessLevel": "Write"
            },
            "SubmitSupportRequest": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/chime-getting-admin-support.html",
                "description": "Grants permission to submit a customer service support request",
                "accessLevel": "Write"
            },
            "SuspendUsers": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/manage-access.html",
                "description": "Grants permission to suspend users from an Amazon Chime Enterprise account",
                "accessLevel": "Write"
            },
            "TagAttendee": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_TagAttendee.html",
                "description": "Grants permission to apply the specified tags to the specified Amazon Chime SDK attendee",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "TagMeeting": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_TagMeeting.html",
                "description": "Grants permission to apply the specified tags to the specified Amazon Chime SDK meeting.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_TagResource.html",
                "description": "Grants permission to apply the specified tags to the specified Amazon Chime SDK meeting resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "meeting": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "UnauthorizeDirectory": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to unauthorize an Active Directory from your Amazon Chime Enterprise account",
                "accessLevel": "Write"
            },
            "UntagAttendee": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UntagAttendee.html",
                "description": "Grants permission to untag the specified tags from the specified Amazon Chime SDK attendee.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "UntagMeeting": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UntagMeeting.html",
                "description": "Grants permission to untag the specified tags from the specified Amazon Chime SDK meeting.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UntagResource.html",
                "description": "Grants permission to untag the specified tags from the specified Amazon Chime SDK meeting resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "meeting": {
                        "required": false
                    }
                }
            },
            "UpdateAccount": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateAccount.html",
                "description": "Grants permission to update account details for the specified Amazon Chime account",
                "accessLevel": "Write"
            },
            "UpdateAccountOpenIdConfig": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html",
                "description": "Grants permission to update the OpenIdConfig attributes for your Amazon Chime account",
                "accessLevel": "Write"
            },
            "UpdateAccountResource": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to update the account resource in your Amazon Chime account",
                "accessLevel": "Write"
            },
            "UpdateAccountSettings": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateAccountSettings.html",
                "description": "Grants permission to update the settings for the specified Amazon Chime account",
                "accessLevel": "Write"
            },
            "UpdateBot": {
                "url": "",
                "description": "Grants permission to update the status of the specified bot",
                "accessLevel": "Write"
            },
            "UpdateCDRSettings": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to update your Call Detail Record S3 bucket",
                "accessLevel": "Write"
            },
            "UpdateGlobalSettings": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateGlobalSettings.html",
                "description": "Grants permission to update the global settings related to Amazon Chime for the AWS account",
                "accessLevel": "Write"
            },
            "UpdatePhoneNumber": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdatePhoneNumber.html",
                "description": "Grants permission to update phone number details for the specified phone number",
                "accessLevel": "Write"
            },
            "UpdatePhoneNumberSettings": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdatePhoneNumberSettings.html",
                "description": "Grants permission to update phone number settings related to Amazon Chime for the AWS account",
                "accessLevel": "Write"
            },
            "UpdateProxySession": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateProxySession.html",
                "description": "Grants permission to update a proxy session for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "UpdateRoom": {
                "url": "",
                "description": "Grants permission to update a room",
                "accessLevel": "Write"
            },
            "UpdateRoomMembership": {
                "url": "",
                "description": "Grants permission to update room membership role",
                "accessLevel": "Write"
            },
            "UpdateSupportedLicenses": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/manage-access.html",
                "description": "Grants permission to update the supported license tiers available for users in your Amazon Chime account",
                "accessLevel": "Write"
            },
            "UpdateUser": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateUser.html",
                "description": "Grants permission to update user details for a specified user ID",
                "accessLevel": "Write"
            },
            "UpdateUserLicenses": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/manage-access.html",
                "description": "Grants permission to update the licenses for your Amazon Chime users",
                "accessLevel": "Write"
            },
            "UpdateUserSettings": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateUserSettings.html",
                "description": "Grants permission to update user settings related to the specified Amazon Chime user",
                "accessLevel": "Write"
            },
            "UpdateVoiceConnector": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateVoiceConnector.html",
                "description": "Grants permission to update Amazon Chime Voice Connector details for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "UpdateVoiceConnectorGroup": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateVoiceConnectorGroup.html",
                "description": "Grants permission to update Amazon Chime Voice Connector Group details for the specified Amazon Chime Voice Connector Group",
                "accessLevel": "Write"
            },
            "ValidateAccountResource": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to validate the account resource in your Amazon Chime account",
                "accessLevel": "Read"
            }
        };
        this.resourceTypes = {
            "meeting": {
                "name": "meeting",
                "url": "",
                "arn": "arn:${Partition}:chime::${AccountId}:meeting/${MeetingId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Grants permission to accept the delegate invitation to share management of an Amazon Chime account with another AWS Account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    acceptDelegate() {
        this.add('chime:AcceptDelegate');
        return this;
    }
    /**
     * Grants permission to activate users in an Amazon Chime Enterprise account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/manage-access.html
     */
    activateUsers() {
        this.add('chime:ActivateUsers');
        return this;
    }
    /**
     * Grants permission to add a domain to your Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/claim-domain.html
     */
    addDomain() {
        this.add('chime:AddDomain');
        return this;
    }
    /**
     * Grants permission to add new or update existing Active Directory or Okta user groups associated with your Amazon Chime Enterprise account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/manage-chime-account.html
     */
    addOrUpdateGroups() {
        this.add('chime:AddOrUpdateGroups');
        return this;
    }
    /**
     * Grants permission to associate a phone number with an Amazon Chime user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_AssociatePhoneNumberWithUser.html
     */
    associatePhoneNumberWithUser() {
        this.add('chime:AssociatePhoneNumberWithUser');
        return this;
    }
    /**
     * Grants permission to associate multiple phone numbers with an Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_AssociatePhoneNumbersWithVoiceConnector.html
     */
    associatePhoneNumbersWithVoiceConnector() {
        this.add('chime:AssociatePhoneNumbersWithVoiceConnector');
        return this;
    }
    /**
     * Grants permission to associate multiple phone numbers with an Amazon Chime Voice Connector Group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_AssociatePhoneNumbersWithVoiceConnectorGroup.html
     */
    associatePhoneNumbersWithVoiceConnectorGroup() {
        this.add('chime:AssociatePhoneNumbersWithVoiceConnectorGroup');
        return this;
    }
    /**
     * Grants permission to associate the specified sign-in delegate groups with the specified Amazon Chime account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_AssociateSigninDelegateGroupsWithAccount.html
     */
    associateSigninDelegateGroupsWithAccount() {
        this.add('chime:AssociateSigninDelegateGroupsWithAccount');
        return this;
    }
    /**
     * Grants permission to authorize an Active Directory for your Amazon Chime Enterprise account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    authorizeDirectory() {
        this.add('chime:AuthorizeDirectory');
        return this;
    }
    /**
     * Grants permission to create new attendees for an active Amazon Chime SDK meeting
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchCreateAttendee.html
     */
    batchCreateAttendee() {
        this.add('chime:BatchCreateAttendee');
        return this;
    }
    /**
     * Grants permission to batch add room members
     *
     * Access Level: Write
     */
    batchCreateRoomMembership() {
        this.add('chime:BatchCreateRoomMembership');
        return this;
    }
    /**
     * Grants permission to move up to 50 phone numbers to the deletion queue
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchDeletePhoneNumber.html
     */
    batchDeletePhoneNumber() {
        this.add('chime:BatchDeletePhoneNumber');
        return this;
    }
    /**
     * Grants permission to suspend up to 50 users from a Team or EnterpriseLWA Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchSuspendUser.html
     */
    batchSuspendUser() {
        this.add('chime:BatchSuspendUser');
        return this;
    }
    /**
     * Grants permission to remove the suspension from up to 50 previously suspended users for the specified Amazon Chime EnterpriseLWA account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchUnsuspendUser.html
     */
    batchUnsuspendUser() {
        this.add('chime:BatchUnsuspendUser');
        return this;
    }
    /**
     * Grants permission to update phone number details within the UpdatePhoneNumberRequestItem object for up to 50 phone numbers
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchUpdatePhoneNumber.html
     */
    batchUpdatePhoneNumber() {
        this.add('chime:BatchUpdatePhoneNumber');
        return this;
    }
    /**
     * Grants permission to update user details within the UpdateUserRequestItem object for up to 20 users for the specified Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchUpdateUser.html
     */
    batchUpdateUser() {
        this.add('chime:BatchUpdateUser');
        return this;
    }
    /**
     * Grants permission to connect an Active Directory to your Amazon Chime Enterprise account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/active_directory.html
     */
    connectDirectory() {
        this.add('chime:ConnectDirectory');
        return this;
    }
    /**
     * Grants permission to create an Amazon Chime account under the administrator's AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateAccount.html
     */
    createAccount() {
        this.add('chime:CreateAccount');
        return this;
    }
    /**
     * Grants permission to create a new SCIM access key for your Amazon Chime account and Okta configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html
     */
    createApiKey() {
        this.add('chime:CreateApiKey');
        return this;
    }
    /**
     * Grants permission to create a new attendee for an active Amazon Chime SDK meeting
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateAttendee.html
     */
    createAttendee() {
        this.add('chime:CreateAttendee');
        return this;
    }
    /**
     * Grants permission to create a bot for an Amazon Chime Enterprise account
     *
     * Access Level: Write
     */
    createBot() {
        this.add('chime:CreateBot');
        return this;
    }
    /**
     * Grants permission to add a bot to a chat room in your Amazon Chime Enterprise account
     *
     * Access Level: Write
     */
    createBotMembership() {
        this.add('chime:CreateBotMembership');
        return this;
    }
    /**
     * Grants permission to create a new Call Detail Record S3 bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/manage-access.html
     */
    createCDRBucket() {
        this.add('chime:CreateCDRBucket');
        return this;
    }
    /**
     * Grants permission to create a new Amazon Chime SDK meeting in the specified media Region, with no initial attendees
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateMeeting.html
     */
    createMeeting() {
        this.add('chime:CreateMeeting');
        return this;
    }
    /**
     * Grants permission to create a phone number order with the Carriers
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreatePhoneNumberOrder.html
     */
    createPhoneNumberOrder() {
        this.add('chime:CreatePhoneNumberOrder');
        return this;
    }
    /**
     * Grants permission to create a proxy session for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateProxySession.html
     */
    createProxySession() {
        this.add('chime:CreateProxySession');
        return this;
    }
    /**
     * Grants permission to create a room
     *
     * Access Level: Write
     */
    createRoom() {
        this.add('chime:CreateRoom');
        return this;
    }
    /**
     * Grants permission to add a room member
     *
     * Access Level: Write
     */
    createRoomMembership() {
        this.add('chime:CreateRoomMembership');
        return this;
    }
    /**
     * Grants permission to create a user under the specified Amazon Chime account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateUser.html
     */
    createUser() {
        this.add('chime:CreateUser');
        return this;
    }
    /**
     * Grants permission to create a Amazon Chime Voice Connector under the administrator's AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateVoiceConnector.html
     */
    createVoiceConnector() {
        this.add('chime:CreateVoiceConnector');
        return this;
    }
    /**
     * Grants permission to create a Amazon Chime Voice Connector Group under the administrator's AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateVoiceConnectorGroup.html
     */
    createVoiceConnectorGroup() {
        this.add('chime:CreateVoiceConnectorGroup');
        return this;
    }
    /**
     * Grants permission to delete the specified Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteAccount.html
     */
    deleteAccount() {
        this.add('chime:DeleteAccount');
        return this;
    }
    /**
     * Grants permission to delete the OpenIdConfig attributes from your Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html
     */
    deleteAccountOpenIdConfig() {
        this.add('chime:DeleteAccountOpenIdConfig');
        return this;
    }
    /**
     * Grants permission to delete the specified SCIM access key associated with your Amazon Chime account and Okta configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html
     */
    deleteApiKey() {
        this.add('chime:DeleteApiKey');
        return this;
    }
    /**
     * Grants permission to delete the specified attendee from an Amazon Chime SDK meeting
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteAttendee.html
     */
    deleteAttendee() {
        this.add('chime:DeleteAttendee');
        return this;
    }
    /**
     * Grants permission to delete a Call Detail Record S3 bucket from your Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    deleteCDRBucket() {
        this.add('chime:DeleteCDRBucket');
        return this;
    }
    /**
     * Grants permission to delete delegated AWS account management from your Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    deleteDelegate() {
        this.add('chime:DeleteDelegate');
        return this;
    }
    /**
     * Grants permission to delete a domain from your Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/claim-domain.html
     */
    deleteDomain() {
        this.add('chime:DeleteDomain');
        return this;
    }
    /**
     * Grants permission to delete an events configuration for a bot to receive outgoing events
     *
     * Access Level: Write
     */
    deleteEventsConfiguration() {
        this.add('chime:DeleteEventsConfiguration');
        return this;
    }
    /**
     * Grants permission to delete Active Directory or Okta user groups from your Amazon Chime Enterprise account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    deleteGroups() {
        this.add('chime:DeleteGroups');
        return this;
    }
    /**
     * Grants permission to delete the specified Amazon Chime SDK meeting
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteMeeting.html
     */
    deleteMeeting() {
        this.add('chime:DeleteMeeting');
        return this;
    }
    /**
     * Grants permission to move a phone number to the deletion queue
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeletePhoneNumber.html
     */
    deletePhoneNumber() {
        this.add('chime:DeletePhoneNumber');
        return this;
    }
    /**
     * Grants permission to delete a proxy session for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteProxySession.html
     */
    deleteProxySession() {
        this.add('chime:DeleteProxySession');
        return this;
    }
    /**
     * Grants permission to delete a room
     *
     * Access Level: Write
     */
    deleteRoom() {
        this.add('chime:DeleteRoom');
        return this;
    }
    /**
     * Grants permission to remove a room member
     *
     * Access Level: Write
     */
    deleteRoomMembership() {
        this.add('chime:DeleteRoomMembership');
        return this;
    }
    /**
     * Grants permission to delete the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnector.html
     */
    deleteVoiceConnector() {
        this.add('chime:DeleteVoiceConnector');
        return this;
    }
    /**
     * Grants permission to delete emergency calling configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorEmergencyCallingConfiguration.html
     */
    deleteVoiceConnectorEmergencyCallingConfiguration() {
        this.add('chime:DeleteVoiceConnectorEmergencyCallingConfiguration');
        return this;
    }
    /**
     * Grants permission to delete the specified Amazon Chime Voice Connector Group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorGroup.html
     */
    deleteVoiceConnectorGroup() {
        this.add('chime:DeleteVoiceConnectorGroup');
        return this;
    }
    /**
     * Grants permission to delete the origination settings for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorOrigination.html
     */
    deleteVoiceConnectorOrigination() {
        this.add('chime:DeleteVoiceConnectorOrigination');
        return this;
    }
    /**
     * Grants permission to delete proxy configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorProxy.html
     */
    deleteVoiceConnectorProxy() {
        this.add('chime:DeleteVoiceConnectorProxy');
        return this;
    }
    /**
     * Grants permission to delete streaming configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorStreamingConfiguration.html
     */
    deleteVoiceConnectorStreamingConfiguration() {
        this.add('chime:DeleteVoiceConnectorStreamingConfiguration');
        return this;
    }
    /**
     * Grants permission to delete the termination settings for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorTermination.html
     */
    deleteVoiceConnectorTermination() {
        this.add('chime:DeleteVoiceConnectorTermination');
        return this;
    }
    /**
     * Grants permission to delete SIP termination credentials for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorTerminationCredentials.html
     */
    deleteVoiceConnectorTerminationCredentials() {
        this.add('chime:DeleteVoiceConnectorTerminationCredentials');
        return this;
    }
    /**
     * Grants permission to disassociate the primary provisioned number from the specified Amazon Chime user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DisassociatePhoneNumberFromUser.html
     */
    disassociatePhoneNumberFromUser() {
        this.add('chime:DisassociatePhoneNumberFromUser');
        return this;
    }
    /**
     * Grants permission to disassociate multiple phone numbers from the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DisassociatePhoneNumbersFromVoiceConnector.html
     */
    disassociatePhoneNumbersFromVoiceConnector() {
        this.add('chime:DisassociatePhoneNumbersFromVoiceConnector');
        return this;
    }
    /**
     * Grants permission to disassociate multiple phone numbers from the specified Amazon Chime Voice Connector Group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DisassociatePhoneNumbersFromVoiceConnectorGroup.html
     */
    disassociatePhoneNumbersFromVoiceConnectorGroup() {
        this.add('chime:DisassociatePhoneNumbersFromVoiceConnectorGroup');
        return this;
    }
    /**
     * Grants permission to disassociate the specified sign-in delegate groups from the specified Amazon Chime account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DisassociateSigninDelegateGroupsFromAccount.html
     */
    disassociateSigninDelegateGroupsFromAccount() {
        this.add('chime:DisassociateSigninDelegateGroupsFromAccount');
        return this;
    }
    /**
     * Grants permission to disconnect the Active Directory from your Amazon Chime Enterprise account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    disconnectDirectory() {
        this.add('chime:DisconnectDirectory');
        return this;
    }
    /**
     * Grants permission to get details for the specified Amazon Chime account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetAccount.html
     */
    getAccount() {
        this.add('chime:GetAccount');
        return this;
    }
    /**
     * Grants permission to get details for the account resource associated with your Amazon Chime account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    getAccountResource() {
        this.add('chime:GetAccountResource');
        return this;
    }
    /**
     * Grants permission to get account settings for the specified Amazon Chime account ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetAccountSettings.html
     */
    getAccountSettings() {
        this.add('chime:GetAccountSettings');
        return this;
    }
    /**
     * Grants permission to get the account details and OpenIdConfig attributes for your Amazon Chime account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html
     */
    getAccountWithOpenIdConfig() {
        this.add('chime:GetAccountWithOpenIdConfig');
        return this;
    }
    /**
     * Grants permission to get attendee details for a specified meeting ID and attendee ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetAttendee.html
     */
    getAttendee() {
        this.add('chime:GetAttendee');
        return this;
    }
    /**
     * Grants permission to retrieve details for the specified bot
     *
     * Access Level: Read
     */
    getBot() {
        this.add('chime:GetBot');
        return this;
    }
    /**
     * Grants permission to get details of a Call Detail Record S3 bucket associated with your Amazon Chime account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    getCDRBucket() {
        this.add('chime:GetCDRBucket');
        return this;
    }
    /**
     * Grants permission to get domain details for a domain associated with your Amazon Chime account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/ag/claim-domain.html
     */
    getDomain() {
        this.add('chime:GetDomain');
        return this;
    }
    /**
     * Grants permission to retrieve details for an events configuration for a bot to receive outgoing events
     *
     * Access Level: Read
     */
    getEventsConfiguration() {
        this.add('chime:GetEventsConfiguration');
        return this;
    }
    /**
     * Grants permission to get global settings related to Amazon Chime for the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetGlobalSettings.html
     */
    getGlobalSettings() {
        this.add('chime:GetGlobalSettings');
        return this;
    }
    /**
     * Grants permission to get the meeting record for a specified meeting ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetMeeting.html
     */
    getMeeting() {
        this.add('chime:GetMeeting');
        return this;
    }
    /**
     * Grants permission to get attendee, connection, and other details for a meeting
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    getMeetingDetail() {
        this.add('chime:GetMeetingDetail');
        return this;
    }
    /**
     * Grants permission to get details for the specified phone number
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetPhoneNumber.html
     */
    getPhoneNumber() {
        this.add('chime:GetPhoneNumber');
        return this;
    }
    /**
     * Grants permission to get details for the specified phone number order
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetPhoneNumberOrder.html
     */
    getPhoneNumberOrder() {
        this.add('chime:GetPhoneNumberOrder');
        return this;
    }
    /**
     * Grants permission to get phone number settings related to Amazon Chime for the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetPhoneNumberSettings.html
     */
    getPhoneNumberSettings() {
        this.add('chime:GetPhoneNumberSettings');
        return this;
    }
    /**
     * Grants permission to get details of the specified proxy session for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetProxySession.html
     */
    getProxySession() {
        this.add('chime:GetProxySession');
        return this;
    }
    /**
     * Gets the retention settings for the specified Amazon Chime account.
     *
     * Access Level: Read
     */
    getRetentionSettings() {
        this.add('chime:GetRetentionSettings');
        return this;
    }
    /**
     * Grants permission to retrieve a room
     *
     * Access Level: Read
     */
    getRoom() {
        this.add('chime:GetRoom');
        return this;
    }
    /**
     * Grants permission to get telephony limits for the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/ag/phone-numbers.html
     */
    getTelephonyLimits() {
        this.add('chime:GetTelephonyLimits');
        return this;
    }
    /**
     * Grants permission to get details for the specified user ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetUser.html
     */
    getUser() {
        this.add('chime:GetUser');
        return this;
    }
    /**
     * Grants permission to get a summary of user activity on the user details page
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/ag/user-details.html
     */
    getUserActivityReportData() {
        this.add('chime:GetUserActivityReportData');
        return this;
    }
    /**
     * Grants permission to get user details for an Amazon Chime user based on the email address in an Amazon Chime Enterprise or Team account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/ag/user-details.html
     */
    getUserByEmail() {
        this.add('chime:GetUserByEmail');
        return this;
    }
    /**
     * Grants permission to get user settings related to the specified Amazon Chime user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetUserSettings.html
     */
    getUserSettings() {
        this.add('chime:GetUserSettings');
        return this;
    }
    /**
     * Grants permission to get details for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnector.html
     */
    getVoiceConnector() {
        this.add('chime:GetVoiceConnector');
        return this;
    }
    /**
     * Grants permission to get details of the emergency calling configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorEmergencyCallingConfiguration.html
     */
    getVoiceConnectorEmergencyCallingConfiguration() {
        this.add('chime:GetVoiceConnectorEmergencyCallingConfiguration');
        return this;
    }
    /**
     * Grants permission to get details for the specified Amazon Chime Voice Connector Group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorGroup.html
     */
    getVoiceConnectorGroup() {
        this.add('chime:GetVoiceConnectorGroup');
        return this;
    }
    /**
     * Grants permission to get details of the logging configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorLoggingConfiguration.html
     */
    getVoiceConnectorLoggingConfiguration() {
        this.add('chime:GetVoiceConnectorLoggingConfiguration');
        return this;
    }
    /**
     * Grants permission to get details of the origination settings for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorOrigination.html
     */
    getVoiceConnectorOrigination() {
        this.add('chime:GetVoiceConnectorOrigination');
        return this;
    }
    /**
     * Grants permission to get details of the proxy configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorProxy.html
     */
    getVoiceConnectorProxy() {
        this.add('chime:GetVoiceConnectorProxy');
        return this;
    }
    /**
     * Grants permission to get details of the streaming configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorStreamingConfiguration.html
     */
    getVoiceConnectorStreamingConfiguration() {
        this.add('chime:GetVoiceConnectorStreamingConfiguration');
        return this;
    }
    /**
     * Grants permission to get details of the termination settings for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorTermination.html
     */
    getVoiceConnectorTermination() {
        this.add('chime:GetVoiceConnectorTermination');
        return this;
    }
    /**
     * Grants permission to get details of the termination health for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorTerminationHealth.html
     */
    getVoiceConnectorTerminationHealth() {
        this.add('chime:GetVoiceConnectorTerminationHealth');
        return this;
    }
    /**
     * Grants permission to send an invitation to accept a request for AWS account delegation for an Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    inviteDelegate() {
        this.add('chime:InviteDelegate');
        return this;
    }
    /**
     * Grants permission to invite as many as 50 users to the specified Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_InviteUsers.html
     */
    inviteUsers() {
        this.add('chime:InviteUsers');
        return this;
    }
    /**
     * Grants permission to invite users from a third party provider to your Amazon Chime account
     *
     * Access Level: Write
     */
    inviteUsersFromProvider() {
        this.add('chime:InviteUsersFromProvider');
        return this;
    }
    /**
     * Grants permission to list Amazon Chime account usage reporting data
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/view-reports.html
     */
    listAccountUsageReportData() {
        this.add('chime:ListAccountUsageReportData');
        return this;
    }
    /**
     * Grants permission to list the Amazon Chime accounts under the administrator's AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListAccounts.html
     */
    listAccounts() {
        this.add('chime:ListAccounts');
        return this;
    }
    /**
     * Grants permission to list the SCIM access keys defined for your Amazon Chime account and Okta configuration
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html
     */
    listApiKeys() {
        this.add('chime:ListApiKeys');
        return this;
    }
    /**
     * Grants permission to list the tags applied to an Amazon Chime SDK attendee resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListAttendeeTags.html
     */
    listAttendeeTags() {
        this.add('chime:ListAttendeeTags');
        return this;
    }
    /**
     * Grants permission to list up to 100 attendees for a specified Amazon Chime SDK meeting
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListAttendees.html
     */
    listAttendees() {
        this.add('chime:ListAttendees');
        return this;
    }
    /**
     * Grants permission to list the bots associated with the administrator's Amazon Chime Enterprise account
     *
     * Access Level: List
     */
    listBots() {
        this.add('chime:ListBots');
        return this;
    }
    /**
     * Grants permission to list Call Detail Record S3 buckets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    listCDRBucket() {
        this.add('chime:ListCDRBucket');
        return this;
    }
    /**
     * Grants permission to list the calling regions available for the administrator's AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/phone-numbers.html
     */
    listCallingRegions() {
        this.add('chime:ListCallingRegions');
        return this;
    }
    /**
     * Grants permission to list account delegate information associated with your Amazon Chime account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    listDelegates() {
        this.add('chime:ListDelegates');
        return this;
    }
    /**
     * Grants permission to list active Active Directories hosted in the Directory Service of your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    listDirectories() {
        this.add('chime:ListDirectories');
        return this;
    }
    /**
     * Grants permission to list domains associated with your Amazon Chime account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/claim-domain.html
     */
    listDomains() {
        this.add('chime:ListDomains');
        return this;
    }
    /**
     * Grants permission to list Active Directory or Okta user groups associated with your Amazon Chime Enterprise account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    listGroups() {
        this.add('chime:ListGroups');
        return this;
    }
    /**
     * Grants permission to list all events that occurred for a specified meeting
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/view-reports.html
     */
    listMeetingEvents() {
        this.add('chime:ListMeetingEvents');
        return this;
    }
    /**
     * Grants permission to list the tags applied to an Amazon Chime SDK meeting resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListMeetingTags.html
     */
    listMeetingTags() {
        this.add('chime:ListMeetingTags');
        return this;
    }
    /**
     * Grants permission to list up to 100 active Amazon Chime SDK meetings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListMeetings.html
     */
    listMeetings() {
        this.add('chime:ListMeetings');
        return this;
    }
    /**
     * Grants permission to list meetings ended during the specified date range
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/view-reports.html
     */
    listMeetingsReportData() {
        this.add('chime:ListMeetingsReportData');
        return this;
    }
    /**
     * Grants permission to list the phone number orders under the administrator's AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListPhoneNumberOrders.html
     */
    listPhoneNumberOrders() {
        this.add('chime:ListPhoneNumberOrders');
        return this;
    }
    /**
     * Grants permission to list the phone numbers under the administrator's AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListPhoneNumbers.html
     */
    listPhoneNumbers() {
        this.add('chime:ListPhoneNumbers');
        return this;
    }
    /**
     * Grants permission to list proxy sessions for the specified Amazon Chime Voice Connector
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListProxySessions.html
     */
    listProxySessions() {
        this.add('chime:ListProxySessions');
        return this;
    }
    /**
     * Grants permission to list all room members
     *
     * Access Level: Read
     */
    listRoomMemberships() {
        this.add('chime:ListRoomMemberships');
        return this;
    }
    /**
     * Grants permission to list rooms
     *
     * Access Level: Read
     */
    listRooms() {
        this.add('chime:ListRooms');
        return this;
    }
    /**
     * Grants permission to list the tags applied to an Amazon Chime SDK meeting resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('chime:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to list the users that belong to the specified Amazon Chime account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListUsers.html
     */
    listUsers() {
        this.add('chime:ListUsers');
        return this;
    }
    /**
     * Grants permission to list the Amazon Chime Voice Connector Groups under the administrator's AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListVoiceConnectorGroups.html
     */
    listVoiceConnectorGroups() {
        this.add('chime:ListVoiceConnectorGroups');
        return this;
    }
    /**
     * Grants permission to list the SIP termination credentials for the specified Amazon Chime Voice Connector
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListVoiceConnectorTerminationCredentials.html
     */
    listVoiceConnectorTerminationCredentials() {
        this.add('chime:ListVoiceConnectorTerminationCredentials');
        return this;
    }
    /**
     * Grants permission to list the Amazon Chime Voice Connectors under the administrator's AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListVoiceConnectors.html
     */
    listVoiceConnectors() {
        this.add('chime:ListVoiceConnectors');
        return this;
    }
    /**
     * Grants permission to log out the specified user from all of the devices they are currently logged into
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_LogoutUser.html
     */
    logoutUser() {
        this.add('chime:LogoutUser');
        return this;
    }
    /**
     * Grants permission to update details for an events configuration for a bot to receive outgoing events
     *
     * Access Level: Write
     */
    putEventsConfiguration() {
        this.add('chime:PutEventsConfiguration');
        return this;
    }
    /**
     * Puts retention settings for the specified Amazon Chime account
     *
     * Access Level: Write
     */
    putRetentionSettings() {
        this.add('chime:PutRetentionSettings');
        return this;
    }
    /**
     * Grants permission to add emergency calling configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorEmergencyCallingConfiguration.html
     */
    putVoiceConnectorEmergencyCallingConfiguration() {
        this.add('chime:PutVoiceConnectorEmergencyCallingConfiguration');
        return this;
    }
    /**
     * Grants permission to add logging configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorLoggingConfiguration.html
     */
    putVoiceConnectorLoggingConfiguration() {
        this.add('chime:PutVoiceConnectorLoggingConfiguration');
        return this;
    }
    /**
     * Grants permission to update the origination settings for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorOrigination.html
     */
    putVoiceConnectorOrigination() {
        this.add('chime:PutVoiceConnectorOrigination');
        return this;
    }
    /**
     * Grants permission to add proxy configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorProxy.html
     */
    putVoiceConnectorProxy() {
        this.add('chime:PutVoiceConnectorProxy');
        return this;
    }
    /**
     * Grants permission to add streaming configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorStreamingConfiguration.html
     */
    putVoiceConnectorStreamingConfiguration() {
        this.add('chime:PutVoiceConnectorStreamingConfiguration');
        return this;
    }
    /**
     * Grants permission to update the termination settings for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorTermination.html
     */
    putVoiceConnectorTermination() {
        this.add('chime:PutVoiceConnectorTermination');
        return this;
    }
    /**
     * Grants permission to add SIP termination credentials for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorTerminationCredentials.html
     */
    putVoiceConnectorTerminationCredentials() {
        this.add('chime:PutVoiceConnectorTerminationCredentials');
        return this;
    }
    /**
     * Redacts the specified Chime conversation Message
     *
     * Access Level: Write
     */
    redactConversationMessage() {
        this.add('chime:RedactConversationMessage');
        return this;
    }
    /**
     * Redacts the specified Chime room Message
     *
     * Access Level: Write
     */
    redactRoomMessage() {
        this.add('chime:RedactRoomMessage');
        return this;
    }
    /**
     * Grants permission to regenerate the security token for the specified bot
     *
     * Access Level: Write
     */
    regenerateSecurityToken() {
        this.add('chime:RegenerateSecurityToken');
        return this;
    }
    /**
     * Grants permission to modify the account name for your Amazon Chime Enterprise or Team account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/rename-account.html
     */
    renameAccount() {
        this.add('chime:RenameAccount');
        return this;
    }
    /**
     * Grants permission to renew the delegation request associated with an Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    renewDelegate() {
        this.add('chime:RenewDelegate');
        return this;
    }
    /**
     * Grants permission to reset the account resource in your Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    resetAccountResource() {
        this.add('chime:ResetAccountResource');
        return this;
    }
    /**
     * Grants permission to reset the personal meeting PIN for the specified user on an Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ResetPersonalPIN.html
     */
    resetPersonalPIN() {
        this.add('chime:ResetPersonalPIN');
        return this;
    }
    /**
     * Grants permission to restore the specified phone number from the deltion queue back to the phone number inventory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_RestorePhoneNumber.html
     */
    restorePhoneNumber() {
        this.add('chime:RestorePhoneNumber');
        return this;
    }
    /**
     * Grants permission to download the file containing links to all user attachments returned as part of the "Request attachments" action
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/request-attachments.html
     */
    retrieveDataExports() {
        this.add('chime:RetrieveDataExports');
        return this;
    }
    /**
     * Grants permission to search phone numbers that can be ordered from the carrier
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_SearchAvailablePhoneNumbers.html
     */
    searchAvailablePhoneNumbers() {
        this.add('chime:SearchAvailablePhoneNumbers');
        return this;
    }
    /**
     * Grants permission to submit the "Request attachments" request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/request-attachments.html
     */
    startDataExport() {
        this.add('chime:StartDataExport');
        return this;
    }
    /**
     * Grants permission to submit a customer service support request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/chime-getting-admin-support.html
     */
    submitSupportRequest() {
        this.add('chime:SubmitSupportRequest');
        return this;
    }
    /**
     * Grants permission to suspend users from an Amazon Chime Enterprise account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/manage-access.html
     */
    suspendUsers() {
        this.add('chime:SuspendUsers');
        return this;
    }
    /**
     * Grants permission to apply the specified tags to the specified Amazon Chime SDK attendee
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_TagAttendee.html
     */
    tagAttendee() {
        this.add('chime:TagAttendee');
        return this;
    }
    /**
     * Grants permission to apply the specified tags to the specified Amazon Chime SDK meeting.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_TagMeeting.html
     */
    tagMeeting() {
        this.add('chime:TagMeeting');
        return this;
    }
    /**
     * Grants permission to apply the specified tags to the specified Amazon Chime SDK meeting resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_TagResource.html
     */
    tagResource() {
        this.add('chime:TagResource');
        return this;
    }
    /**
     * Grants permission to unauthorize an Active Directory from your Amazon Chime Enterprise account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    unauthorizeDirectory() {
        this.add('chime:UnauthorizeDirectory');
        return this;
    }
    /**
     * Grants permission to untag the specified tags from the specified Amazon Chime SDK attendee.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UntagAttendee.html
     */
    untagAttendee() {
        this.add('chime:UntagAttendee');
        return this;
    }
    /**
     * Grants permission to untag the specified tags from the specified Amazon Chime SDK meeting.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UntagMeeting.html
     */
    untagMeeting() {
        this.add('chime:UntagMeeting');
        return this;
    }
    /**
     * Grants permission to untag the specified tags from the specified Amazon Chime SDK meeting resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UntagResource.html
     */
    untagResource() {
        this.add('chime:UntagResource');
        return this;
    }
    /**
     * Grants permission to update account details for the specified Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateAccount.html
     */
    updateAccount() {
        this.add('chime:UpdateAccount');
        return this;
    }
    /**
     * Grants permission to update the OpenIdConfig attributes for your Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html
     */
    updateAccountOpenIdConfig() {
        this.add('chime:UpdateAccountOpenIdConfig');
        return this;
    }
    /**
     * Grants permission to update the account resource in your Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    updateAccountResource() {
        this.add('chime:UpdateAccountResource');
        return this;
    }
    /**
     * Grants permission to update the settings for the specified Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateAccountSettings.html
     */
    updateAccountSettings() {
        this.add('chime:UpdateAccountSettings');
        return this;
    }
    /**
     * Grants permission to update the status of the specified bot
     *
     * Access Level: Write
     */
    updateBot() {
        this.add('chime:UpdateBot');
        return this;
    }
    /**
     * Grants permission to update your Call Detail Record S3 bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    updateCDRSettings() {
        this.add('chime:UpdateCDRSettings');
        return this;
    }
    /**
     * Grants permission to update the global settings related to Amazon Chime for the AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateGlobalSettings.html
     */
    updateGlobalSettings() {
        this.add('chime:UpdateGlobalSettings');
        return this;
    }
    /**
     * Grants permission to update phone number details for the specified phone number
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdatePhoneNumber.html
     */
    updatePhoneNumber() {
        this.add('chime:UpdatePhoneNumber');
        return this;
    }
    /**
     * Grants permission to update phone number settings related to Amazon Chime for the AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdatePhoneNumberSettings.html
     */
    updatePhoneNumberSettings() {
        this.add('chime:UpdatePhoneNumberSettings');
        return this;
    }
    /**
     * Grants permission to update a proxy session for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateProxySession.html
     */
    updateProxySession() {
        this.add('chime:UpdateProxySession');
        return this;
    }
    /**
     * Grants permission to update a room
     *
     * Access Level: Write
     */
    updateRoom() {
        this.add('chime:UpdateRoom');
        return this;
    }
    /**
     * Grants permission to update room membership role
     *
     * Access Level: Write
     */
    updateRoomMembership() {
        this.add('chime:UpdateRoomMembership');
        return this;
    }
    /**
     * Grants permission to update the supported license tiers available for users in your Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/manage-access.html
     */
    updateSupportedLicenses() {
        this.add('chime:UpdateSupportedLicenses');
        return this;
    }
    /**
     * Grants permission to update user details for a specified user ID
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateUser.html
     */
    updateUser() {
        this.add('chime:UpdateUser');
        return this;
    }
    /**
     * Grants permission to update the licenses for your Amazon Chime users
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/manage-access.html
     */
    updateUserLicenses() {
        this.add('chime:UpdateUserLicenses');
        return this;
    }
    /**
     * Grants permission to update user settings related to the specified Amazon Chime user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateUserSettings.html
     */
    updateUserSettings() {
        this.add('chime:UpdateUserSettings');
        return this;
    }
    /**
     * Grants permission to update Amazon Chime Voice Connector details for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateVoiceConnector.html
     */
    updateVoiceConnector() {
        this.add('chime:UpdateVoiceConnector');
        return this;
    }
    /**
     * Grants permission to update Amazon Chime Voice Connector Group details for the specified Amazon Chime Voice Connector Group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateVoiceConnectorGroup.html
     */
    updateVoiceConnectorGroup() {
        this.add('chime:UpdateVoiceConnectorGroup');
        return this;
    }
    /**
     * Grants permission to validate the account resource in your Amazon Chime account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    validateAccountResource() {
        this.add('chime:ValidateAccountResource');
        return this;
    }
    /**
     * Adds a resource of type meeting to the statement
     *
     * @param meetingId - Identifier for the meetingId.
     * @param accountId - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onMeeting(meetingId, accountId, partition) {
        var arn = 'arn:${Partition}:chime::${AccountId}:meeting/${MeetingId}';
        arn = arn.replace('${MeetingId}', meetingId);
        arn = arn.replace('${AccountId}', accountId || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Chime = Chime;
//# sourceMappingURL=data:application/json;base64,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