"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service clouddirectory
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonclouddirectory.html
 */
class Clouddirectory extends shared_1.PolicyStatement {
    /**
     * Action provider for service clouddirectory
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonclouddirectory.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'clouddirectory';
        this.actions = {
            "AddFacetToObject": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_AddFacetToObject.html",
                "description": "Adds a new Facet to an object.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "ApplySchema": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ApplySchema.html",
                "description": "Copies input published schema into Directory with same name and version as that of published schema.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    },
                    "publishedSchema": {
                        "required": true
                    }
                }
            },
            "AttachObject": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_AttachObject.html",
                "description": "Attaches an existing object to another existing object.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "AttachPolicy": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_AttachPolicy.html",
                "description": "Attaches a policy object to any other object.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "AttachToIndex": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_AttachToIndex.html",
                "description": "Attaches the specified object to the specified index.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "AttachTypedLink": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_AttachTypedLink.html",
                "description": "Attaches a typed link b/w a source & target object reference.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "BatchRead": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_BatchRead.html",
                "description": "Performs all the read operations in a batch. Each individual operation inside BatchRead needs to be granted permissions explicitly.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "BatchWrite": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_BatchWrite.html",
                "description": "Performs all the write operations in a batch. Each individual operation inside BatchWrite needs to be granted permissions explicitly.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "CreateDirectory": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_CreateDirectory.html",
                "description": "Creates a Directory by copying the published schema into the directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "publishedSchema": {
                        "required": true
                    }
                }
            },
            "CreateFacet": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_CreateFacet.html",
                "description": "Creates a new Facet in a schema.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "appliedSchema": {
                        "required": true
                    },
                    "developmentSchema": {
                        "required": true
                    }
                }
            },
            "CreateIndex": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_CreateIndex.html",
                "description": "Creates an index object.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "CreateObject": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_CreateObject.html",
                "description": "Creates an object in a Directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "CreateSchema": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_CreateSchema.html",
                "description": "Creates a new schema in a development state.",
                "accessLevel": "Write"
            },
            "CreateTypedLinkFacet": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_CreateTypedLinkFacet.html",
                "description": "Creates a new Typed Link facet in a schema.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "appliedSchema": {
                        "required": true
                    },
                    "developmentSchema": {
                        "required": true
                    }
                }
            },
            "DeleteDirectory": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DeleteDirectory.html",
                "description": "Deletes a directory. Only disabled directories can be deleted.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DeleteFacet": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DeleteFacet.html",
                "description": "Deletes a given Facet. All attributes and Rules associated with the facet will be deleted.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "developmentSchema": {
                        "required": true
                    }
                }
            },
            "DeleteObject": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DeleteObject.html",
                "description": "Deletes an object and its associated attributes.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DeleteSchema": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DeleteSchema.html",
                "description": "Deletes a given schema.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "developmentSchema": {
                        "required": true
                    },
                    "publishedSchema": {
                        "required": true
                    }
                }
            },
            "DeleteTypedLinkFacet": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DeleteTypedLinkFacet.html",
                "description": "Deletes a given TypedLink Facet. All attributes and Rules associated with the facet will be deleted.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "developmentSchema": {
                        "required": true
                    }
                }
            },
            "DetachFromIndex": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DetachFromIndex.html",
                "description": "Detaches the specified object from the specified index.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DetachObject": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DetachObject.html",
                "description": "Detaches a given object from the parent object.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DetachPolicy": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DetachPolicy.html",
                "description": "Detaches a policy from an object.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DetachTypedLink": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DetachTypedLink.html",
                "description": "Detaches a given typed link b/w given source and target object reference.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DisableDirectory": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DisableDirectory.html",
                "description": "Disables the specified directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "EnableDirectory": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_EnableDirectory.html",
                "description": "Enables the specified directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "GetDirectory": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_GetDirectory.html",
                "description": "Retrieves metadata about a directory.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "GetFacet": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_GetFacet.html",
                "description": "Gets details of the Facet, such as Facet Name, Attributes, Rules, or ObjectType.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "appliedSchema": {
                        "required": true
                    },
                    "developmentSchema": {
                        "required": true
                    },
                    "publishedSchema": {
                        "required": true
                    }
                }
            },
            "GetLinkAttributes": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_GetLinkAttributes.html",
                "description": "Retrieves attributes that are associated with a typed link.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "GetObjectAttributes": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_GetObjectAttributes.html",
                "description": "Retrieves attributes within a facet that are associated with an object.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "GetObjectInformation": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_GetObjectInformation.html",
                "description": "Retrieves metadata about an object.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "GetSchemaAsJson": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_GetSchemaAsJson.html",
                "description": "Retrieves a JSON representation of the schema.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "appliedSchema": {
                        "required": true
                    },
                    "developmentSchema": {
                        "required": true
                    },
                    "publishedSchema": {
                        "required": true
                    }
                }
            },
            "GetTypedLinkFacetInformation": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_GetTypedLinkFacetInformation.html",
                "description": "Returns identity attributes order information associated with a given typed link facet.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "appliedSchema": {
                        "required": true
                    },
                    "developmentSchema": {
                        "required": true
                    },
                    "publishedSchema": {
                        "required": true
                    }
                }
            },
            "ListAppliedSchemaArns": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListAppliedSchemaArns.html",
                "description": "Lists schemas applied to a directory.",
                "accessLevel": "List",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "ListAttachedIndices": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListAttachedIndices.html",
                "description": "Lists indices attached to an object.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "ListDevelopmentSchemaArns": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListDevelopmentSchemaArns.html",
                "description": "Retrieves the ARNs of schemas in the development state.",
                "accessLevel": "List"
            },
            "ListDirectories": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListDirectories.html",
                "description": "Lists directories created within an account.",
                "accessLevel": "List"
            },
            "ListFacetAttributes": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListFacetAttributes.html",
                "description": "Retrieves attributes attached to the facet.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "appliedSchema": {
                        "required": true
                    },
                    "developmentSchema": {
                        "required": true
                    },
                    "publishedSchema": {
                        "required": true
                    }
                }
            },
            "ListFacetNames": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListFacetNames.html",
                "description": "Retrieves the names of facets that exist in a schema.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "appliedSchema": {
                        "required": true
                    },
                    "developmentSchema": {
                        "required": true
                    },
                    "publishedSchema": {
                        "required": true
                    }
                }
            },
            "ListIncomingTypedLinks": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListIncomingTypedLinks.html",
                "description": "Returns a paginated list of all incoming TypedLinks for a given object.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "ListIndex": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListIndex.html",
                "description": "Lists objects attached to the specified index.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "ListObjectAttributes": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListObjectAttributes.html",
                "description": "Lists all attributes associated with an object.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "ListObjectChildren": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListObjectChildren.html",
                "description": "Returns a paginated list of child objects associated with a given object.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "ListObjectParentPaths": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListObjectParentPaths.html",
                "description": "Retrieves all available parent paths for any object type such as node, leaf node, policy node, and index node objects.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "ListObjectParents": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListObjectParents.html",
                "description": "Lists parent objects associated with a given object in pagination fashion.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "ListObjectPolicies": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListObjectPolicies.html",
                "description": "Returns policies attached to an object in pagination fashion.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "ListOutgoingTypedLinks": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListOutgoingTypedLinks.html",
                "description": "Returns a paginated list of all outgoing TypedLinks for a given object.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "ListPolicyAttachments": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListPolicyAttachments.html",
                "description": "Returns all of the ObjectIdentifiers to which a given policy is attached.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "ListPublishedSchemaArns": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListPublishedSchemaArns.html",
                "description": "Retrieves published schema ARNs.",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListTagsForResource.html",
                "description": "Returns tags for a resource.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "ListTypedLinkFacetAttributes": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListTypedLinkFacetAttributes.html",
                "description": "Returns a paginated list of attributes associated with typed link facet.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "appliedSchema": {
                        "required": true
                    },
                    "developmentSchema": {
                        "required": true
                    },
                    "publishedSchema": {
                        "required": true
                    }
                }
            },
            "ListTypedLinkFacetNames": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListTypedLinkFacetNames.html",
                "description": "Returns a paginated list of typed link facet names that exist in a schema.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "appliedSchema": {
                        "required": true
                    },
                    "developmentSchema": {
                        "required": true
                    },
                    "publishedSchema": {
                        "required": true
                    }
                }
            },
            "LookupPolicy": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_LookupPolicy.html",
                "description": "Lists all policies from the root of the Directory to the object specified.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "PublishSchema": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_PublishSchema.html",
                "description": "Publishes a development schema with a version.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "developmentSchema": {
                        "required": true
                    }
                }
            },
            "PutSchemaFromJson": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_PutSchemaFromJson.html",
                "description": "Allows a schema to be updated using JSON upload. Only available for development schemas.",
                "accessLevel": "Write"
            },
            "RemoveFacetFromObject": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_RemoveFacetFromObject.html",
                "description": "Removes the specified facet from the specified object.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_TagResource.html",
                "description": "Adds tags to a resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_UntagResource.html",
                "description": "Removes tags from a resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "UpdateFacet": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_UpdateFacet.html",
                "description": "Adds/Updates/Deletes existing Attributes, Rules, or ObjectType of a Facet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "appliedSchema": {
                        "required": true
                    },
                    "developmentSchema": {
                        "required": true
                    }
                }
            },
            "UpdateLinkAttributes": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_UpdateLinkAttributes.html",
                "description": "Updates a given typed link’s attributes. Attributes to be updated must not contribute to the typed link’s identity, as defined by its IdentityAttributeOrder.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "UpdateObjectAttributes": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_UpdateObjectAttributes.html",
                "description": "Updates a given object's attributes.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "UpdateSchema": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_UpdateSchema.html",
                "description": "Updates the schema name with a new name.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "developmentSchema": {
                        "required": true
                    }
                }
            },
            "UpdateTypedLinkFacet": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_UpdateTypedLinkFacet.html",
                "description": "Adds/Updates/Deletes existing Attributes, Rules, identity attribute order of a TypedLink Facet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "developmentSchema": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "appliedSchema": {
                "name": "appliedSchema",
                "url": "https://docs.aws.amazon.com/directoryservice/latest/admin-guide/cd_key_concepts.html#whatisdirectory",
                "arn": "arn:${Partition}:clouddirectory:${Region}:${Account}:directory/${DirectoryId}/schema/${SchemaName}/${Version}",
                "conditionKeys": []
            },
            "developmentSchema": {
                "name": "developmentSchema",
                "url": "https://docs.aws.amazon.com/directoryservice/latest/admin-guide/cd_key_concepts.html#whatisdirectory",
                "arn": "arn:${Partition}:clouddirectory:${Region}:${Account}:schema/development/${SchemaName}",
                "conditionKeys": []
            },
            "directory": {
                "name": "directory",
                "url": "https://docs.aws.amazon.com/directoryservice/latest/admin-guide/cd_key_concepts.html#whatisdirectory",
                "arn": "arn:${Partition}:clouddirectory:${Region}:${Account}:directory/${DirectoryId}",
                "conditionKeys": []
            },
            "publishedSchema": {
                "name": "publishedSchema",
                "url": "https://docs.aws.amazon.com/directoryservice/latest/admin-guide/cd_key_concepts.html#whatisdirectory",
                "arn": "arn:${Partition}:clouddirectory:${Region}:${Account}:schema/published/${SchemaName}/${Version}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Adds a new Facet to an object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_AddFacetToObject.html
     */
    addFacetToObject() {
        this.add('clouddirectory:AddFacetToObject');
        return this;
    }
    /**
     * Copies input published schema into Directory with same name and version as that of published schema.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ApplySchema.html
     */
    applySchema() {
        this.add('clouddirectory:ApplySchema');
        return this;
    }
    /**
     * Attaches an existing object to another existing object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_AttachObject.html
     */
    attachObject() {
        this.add('clouddirectory:AttachObject');
        return this;
    }
    /**
     * Attaches a policy object to any other object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_AttachPolicy.html
     */
    attachPolicy() {
        this.add('clouddirectory:AttachPolicy');
        return this;
    }
    /**
     * Attaches the specified object to the specified index.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_AttachToIndex.html
     */
    attachToIndex() {
        this.add('clouddirectory:AttachToIndex');
        return this;
    }
    /**
     * Attaches a typed link b/w a source & target object reference.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_AttachTypedLink.html
     */
    attachTypedLink() {
        this.add('clouddirectory:AttachTypedLink');
        return this;
    }
    /**
     * Performs all the read operations in a batch. Each individual operation inside BatchRead needs to be granted permissions explicitly.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_BatchRead.html
     */
    batchRead() {
        this.add('clouddirectory:BatchRead');
        return this;
    }
    /**
     * Performs all the write operations in a batch. Each individual operation inside BatchWrite needs to be granted permissions explicitly.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_BatchWrite.html
     */
    batchWrite() {
        this.add('clouddirectory:BatchWrite');
        return this;
    }
    /**
     * Creates a Directory by copying the published schema into the directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_CreateDirectory.html
     */
    createDirectory() {
        this.add('clouddirectory:CreateDirectory');
        return this;
    }
    /**
     * Creates a new Facet in a schema.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_CreateFacet.html
     */
    createFacet() {
        this.add('clouddirectory:CreateFacet');
        return this;
    }
    /**
     * Creates an index object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_CreateIndex.html
     */
    createIndex() {
        this.add('clouddirectory:CreateIndex');
        return this;
    }
    /**
     * Creates an object in a Directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_CreateObject.html
     */
    createObject() {
        this.add('clouddirectory:CreateObject');
        return this;
    }
    /**
     * Creates a new schema in a development state.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_CreateSchema.html
     */
    createSchema() {
        this.add('clouddirectory:CreateSchema');
        return this;
    }
    /**
     * Creates a new Typed Link facet in a schema.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_CreateTypedLinkFacet.html
     */
    createTypedLinkFacet() {
        this.add('clouddirectory:CreateTypedLinkFacet');
        return this;
    }
    /**
     * Deletes a directory. Only disabled directories can be deleted.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DeleteDirectory.html
     */
    deleteDirectory() {
        this.add('clouddirectory:DeleteDirectory');
        return this;
    }
    /**
     * Deletes a given Facet. All attributes and Rules associated with the facet will be deleted.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DeleteFacet.html
     */
    deleteFacet() {
        this.add('clouddirectory:DeleteFacet');
        return this;
    }
    /**
     * Deletes an object and its associated attributes.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DeleteObject.html
     */
    deleteObject() {
        this.add('clouddirectory:DeleteObject');
        return this;
    }
    /**
     * Deletes a given schema.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DeleteSchema.html
     */
    deleteSchema() {
        this.add('clouddirectory:DeleteSchema');
        return this;
    }
    /**
     * Deletes a given TypedLink Facet. All attributes and Rules associated with the facet will be deleted.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DeleteTypedLinkFacet.html
     */
    deleteTypedLinkFacet() {
        this.add('clouddirectory:DeleteTypedLinkFacet');
        return this;
    }
    /**
     * Detaches the specified object from the specified index.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DetachFromIndex.html
     */
    detachFromIndex() {
        this.add('clouddirectory:DetachFromIndex');
        return this;
    }
    /**
     * Detaches a given object from the parent object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DetachObject.html
     */
    detachObject() {
        this.add('clouddirectory:DetachObject');
        return this;
    }
    /**
     * Detaches a policy from an object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DetachPolicy.html
     */
    detachPolicy() {
        this.add('clouddirectory:DetachPolicy');
        return this;
    }
    /**
     * Detaches a given typed link b/w given source and target object reference.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DetachTypedLink.html
     */
    detachTypedLink() {
        this.add('clouddirectory:DetachTypedLink');
        return this;
    }
    /**
     * Disables the specified directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_DisableDirectory.html
     */
    disableDirectory() {
        this.add('clouddirectory:DisableDirectory');
        return this;
    }
    /**
     * Enables the specified directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_EnableDirectory.html
     */
    enableDirectory() {
        this.add('clouddirectory:EnableDirectory');
        return this;
    }
    /**
     * Retrieves metadata about a directory.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_GetDirectory.html
     */
    getDirectory() {
        this.add('clouddirectory:GetDirectory');
        return this;
    }
    /**
     * Gets details of the Facet, such as Facet Name, Attributes, Rules, or ObjectType.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_GetFacet.html
     */
    getFacet() {
        this.add('clouddirectory:GetFacet');
        return this;
    }
    /**
     * Retrieves attributes that are associated with a typed link.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_GetLinkAttributes.html
     */
    getLinkAttributes() {
        this.add('clouddirectory:GetLinkAttributes');
        return this;
    }
    /**
     * Retrieves attributes within a facet that are associated with an object.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_GetObjectAttributes.html
     */
    getObjectAttributes() {
        this.add('clouddirectory:GetObjectAttributes');
        return this;
    }
    /**
     * Retrieves metadata about an object.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_GetObjectInformation.html
     */
    getObjectInformation() {
        this.add('clouddirectory:GetObjectInformation');
        return this;
    }
    /**
     * Retrieves a JSON representation of the schema.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_GetSchemaAsJson.html
     */
    getSchemaAsJson() {
        this.add('clouddirectory:GetSchemaAsJson');
        return this;
    }
    /**
     * Returns identity attributes order information associated with a given typed link facet.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_GetTypedLinkFacetInformation.html
     */
    getTypedLinkFacetInformation() {
        this.add('clouddirectory:GetTypedLinkFacetInformation');
        return this;
    }
    /**
     * Lists schemas applied to a directory.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListAppliedSchemaArns.html
     */
    listAppliedSchemaArns() {
        this.add('clouddirectory:ListAppliedSchemaArns');
        return this;
    }
    /**
     * Lists indices attached to an object.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListAttachedIndices.html
     */
    listAttachedIndices() {
        this.add('clouddirectory:ListAttachedIndices');
        return this;
    }
    /**
     * Retrieves the ARNs of schemas in the development state.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListDevelopmentSchemaArns.html
     */
    listDevelopmentSchemaArns() {
        this.add('clouddirectory:ListDevelopmentSchemaArns');
        return this;
    }
    /**
     * Lists directories created within an account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListDirectories.html
     */
    listDirectories() {
        this.add('clouddirectory:ListDirectories');
        return this;
    }
    /**
     * Retrieves attributes attached to the facet.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListFacetAttributes.html
     */
    listFacetAttributes() {
        this.add('clouddirectory:ListFacetAttributes');
        return this;
    }
    /**
     * Retrieves the names of facets that exist in a schema.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListFacetNames.html
     */
    listFacetNames() {
        this.add('clouddirectory:ListFacetNames');
        return this;
    }
    /**
     * Returns a paginated list of all incoming TypedLinks for a given object.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListIncomingTypedLinks.html
     */
    listIncomingTypedLinks() {
        this.add('clouddirectory:ListIncomingTypedLinks');
        return this;
    }
    /**
     * Lists objects attached to the specified index.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListIndex.html
     */
    listIndex() {
        this.add('clouddirectory:ListIndex');
        return this;
    }
    /**
     * Lists all attributes associated with an object.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListObjectAttributes.html
     */
    listObjectAttributes() {
        this.add('clouddirectory:ListObjectAttributes');
        return this;
    }
    /**
     * Returns a paginated list of child objects associated with a given object.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListObjectChildren.html
     */
    listObjectChildren() {
        this.add('clouddirectory:ListObjectChildren');
        return this;
    }
    /**
     * Retrieves all available parent paths for any object type such as node, leaf node, policy node, and index node objects.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListObjectParentPaths.html
     */
    listObjectParentPaths() {
        this.add('clouddirectory:ListObjectParentPaths');
        return this;
    }
    /**
     * Lists parent objects associated with a given object in pagination fashion.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListObjectParents.html
     */
    listObjectParents() {
        this.add('clouddirectory:ListObjectParents');
        return this;
    }
    /**
     * Returns policies attached to an object in pagination fashion.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListObjectPolicies.html
     */
    listObjectPolicies() {
        this.add('clouddirectory:ListObjectPolicies');
        return this;
    }
    /**
     * Returns a paginated list of all outgoing TypedLinks for a given object.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListOutgoingTypedLinks.html
     */
    listOutgoingTypedLinks() {
        this.add('clouddirectory:ListOutgoingTypedLinks');
        return this;
    }
    /**
     * Returns all of the ObjectIdentifiers to which a given policy is attached.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListPolicyAttachments.html
     */
    listPolicyAttachments() {
        this.add('clouddirectory:ListPolicyAttachments');
        return this;
    }
    /**
     * Retrieves published schema ARNs.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListPublishedSchemaArns.html
     */
    listPublishedSchemaArns() {
        this.add('clouddirectory:ListPublishedSchemaArns');
        return this;
    }
    /**
     * Returns tags for a resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('clouddirectory:ListTagsForResource');
        return this;
    }
    /**
     * Returns a paginated list of attributes associated with typed link facet.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListTypedLinkFacetAttributes.html
     */
    listTypedLinkFacetAttributes() {
        this.add('clouddirectory:ListTypedLinkFacetAttributes');
        return this;
    }
    /**
     * Returns a paginated list of typed link facet names that exist in a schema.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_ListTypedLinkFacetNames.html
     */
    listTypedLinkFacetNames() {
        this.add('clouddirectory:ListTypedLinkFacetNames');
        return this;
    }
    /**
     * Lists all policies from the root of the Directory to the object specified.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_LookupPolicy.html
     */
    lookupPolicy() {
        this.add('clouddirectory:LookupPolicy');
        return this;
    }
    /**
     * Publishes a development schema with a version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_PublishSchema.html
     */
    publishSchema() {
        this.add('clouddirectory:PublishSchema');
        return this;
    }
    /**
     * Allows a schema to be updated using JSON upload. Only available for development schemas.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_PutSchemaFromJson.html
     */
    putSchemaFromJson() {
        this.add('clouddirectory:PutSchemaFromJson');
        return this;
    }
    /**
     * Removes the specified facet from the specified object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_RemoveFacetFromObject.html
     */
    removeFacetFromObject() {
        this.add('clouddirectory:RemoveFacetFromObject');
        return this;
    }
    /**
     * Adds tags to a resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_TagResource.html
     */
    tagResource() {
        this.add('clouddirectory:TagResource');
        return this;
    }
    /**
     * Removes tags from a resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_UntagResource.html
     */
    untagResource() {
        this.add('clouddirectory:UntagResource');
        return this;
    }
    /**
     * Adds/Updates/Deletes existing Attributes, Rules, or ObjectType of a Facet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_UpdateFacet.html
     */
    updateFacet() {
        this.add('clouddirectory:UpdateFacet');
        return this;
    }
    /**
     * Updates a given typed link’s attributes. Attributes to be updated must not contribute to the typed link’s identity, as defined by its IdentityAttributeOrder.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_UpdateLinkAttributes.html
     */
    updateLinkAttributes() {
        this.add('clouddirectory:UpdateLinkAttributes');
        return this;
    }
    /**
     * Updates a given object's attributes.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_UpdateObjectAttributes.html
     */
    updateObjectAttributes() {
        this.add('clouddirectory:UpdateObjectAttributes');
        return this;
    }
    /**
     * Updates the schema name with a new name.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_UpdateSchema.html
     */
    updateSchema() {
        this.add('clouddirectory:UpdateSchema');
        return this;
    }
    /**
     * Adds/Updates/Deletes existing Attributes, Rules, identity attribute order of a TypedLink Facet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/APIReference/API_UpdateTypedLinkFacet.html
     */
    updateTypedLinkFacet() {
        this.add('clouddirectory:UpdateTypedLinkFacet');
        return this;
    }
    /**
     * Adds a resource of type appliedSchema to the statement
     *
     * https://docs.aws.amazon.com/directoryservice/latest/admin-guide/cd_key_concepts.html#whatisdirectory
     *
     * @param directoryId - Identifier for the directoryId.
     * @param schemaName - Identifier for the schemaName.
     * @param version - Identifier for the version.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAppliedSchema(directoryId, schemaName, version, account, region, partition) {
        var arn = 'arn:${Partition}:clouddirectory:${Region}:${Account}:directory/${DirectoryId}/schema/${SchemaName}/${Version}';
        arn = arn.replace('${DirectoryId}', directoryId);
        arn = arn.replace('${SchemaName}', schemaName);
        arn = arn.replace('${Version}', version);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type developmentSchema to the statement
     *
     * https://docs.aws.amazon.com/directoryservice/latest/admin-guide/cd_key_concepts.html#whatisdirectory
     *
     * @param schemaName - Identifier for the schemaName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDevelopmentSchema(schemaName, account, region, partition) {
        var arn = 'arn:${Partition}:clouddirectory:${Region}:${Account}:schema/development/${SchemaName}';
        arn = arn.replace('${SchemaName}', schemaName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type directory to the statement
     *
     * https://docs.aws.amazon.com/directoryservice/latest/admin-guide/cd_key_concepts.html#whatisdirectory
     *
     * @param directoryId - Identifier for the directoryId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDirectory(directoryId, account, region, partition) {
        var arn = 'arn:${Partition}:clouddirectory:${Region}:${Account}:directory/${DirectoryId}';
        arn = arn.replace('${DirectoryId}', directoryId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type publishedSchema to the statement
     *
     * https://docs.aws.amazon.com/directoryservice/latest/admin-guide/cd_key_concepts.html#whatisdirectory
     *
     * @param schemaName - Identifier for the schemaName.
     * @param version - Identifier for the version.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onPublishedSchema(schemaName, version, account, region, partition) {
        var arn = 'arn:${Partition}:clouddirectory:${Region}:${Account}:schema/published/${SchemaName}/${Version}';
        arn = arn.replace('${SchemaName}', schemaName);
        arn = arn.replace('${Version}', version);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Clouddirectory = Clouddirectory;
//# sourceMappingURL=data:application/json;base64,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