"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service cognito-sync
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncognitosync.html
 */
class CognitoSync extends shared_1.PolicyStatement {
    /**
     * Action provider for service cognito-sync
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncognitosync.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'cognito-sync';
        this.actions = {
            "BulkPublish": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_BulkPublish.html",
                "description": "Initiates a bulk publish of all existing datasets for an Identity Pool to the configured stream.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "identitypool": {
                        "required": true
                    }
                }
            },
            "DeleteDataset": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_DeleteDataset.html",
                "description": "Deletes the specific dataset.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dataset": {
                        "required": true
                    }
                }
            },
            "DescribeDataset": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_DescribeDataset.html",
                "description": "Gets meta data about a dataset by identity and dataset name.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dataset": {
                        "required": true
                    }
                }
            },
            "DescribeIdentityPoolUsage": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_DescribeIdentityPoolUsage.html",
                "description": "Gets usage details (for example, data storage) about a particular identity pool.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "identitypool": {
                        "required": true
                    }
                }
            },
            "DescribeIdentityUsage": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_DescribeIdentityUsage.html",
                "description": "Gets usage information for an identity, including number of datasets and data usage.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "identity": {
                        "required": true
                    }
                }
            },
            "GetBulkPublishDetails": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_GetBulkPublishDetails.html",
                "description": "Get the status of the last BulkPublish operation for an identity pool.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "identitypool": {
                        "required": true
                    }
                }
            },
            "GetCognitoEvents": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_GetCognitoEvents.html",
                "description": "Gets the events and the corresponding Lambda functions associated with an identity pool.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "identitypool": {
                        "required": true
                    }
                }
            },
            "GetIdentityPoolConfiguration": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_GetIdentityPoolConfiguration.html",
                "description": "Gets the configuration settings of an identity pool.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "identitypool": {
                        "required": true
                    }
                }
            },
            "ListDatasets": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_ListDatasets.html",
                "description": "Lists datasets for an identity.",
                "accessLevel": "List",
                "resourceTypes": {
                    "dataset": {
                        "required": true
                    }
                }
            },
            "ListIdentityPoolUsage": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_ListIdentityPoolUsage.html",
                "description": "Gets a list of identity pools registered with Cognito.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "identitypool": {
                        "required": true
                    }
                }
            },
            "ListRecords": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_ListRecords.html",
                "description": "Gets paginated records, optionally changed after a particular sync count for a dataset and identity.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dataset": {
                        "required": true
                    }
                }
            },
            "QueryRecords": {
                "url": "",
                "description": "A permission that grants the ability to query records.",
                "accessLevel": "Read"
            },
            "RegisterDevice": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_RegisterDevice.html",
                "description": "Registers a device to receive push sync notifications.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "identity": {
                        "required": true
                    }
                }
            },
            "SetCognitoEvents": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_SetCognitoEvents.html",
                "description": "Sets the AWS Lambda function for a given event type for an identity pool.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "identitypool": {
                        "required": true
                    }
                }
            },
            "SetDatasetConfiguration": {
                "url": "",
                "description": "A permission that grants ability to configure datasets.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dataset": {
                        "required": true
                    }
                }
            },
            "SetIdentityPoolConfiguration": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_SetIdentityPoolConfiguration.html",
                "description": "Sets the necessary configuration for push sync.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "identitypool": {
                        "required": true
                    }
                }
            },
            "SubscribeToDataset": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_SubscribeToDataset.html",
                "description": "Subscribes to receive notifications when a dataset is modified by another device.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dataset": {
                        "required": true
                    }
                }
            },
            "UnsubscribeFromDataset": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_UnsubscribeFromDataset.html",
                "description": "Unsubscribes from receiving notifications when a dataset is modified by another device.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dataset": {
                        "required": true
                    }
                }
            },
            "UpdateRecords": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_UpdateRecords.html",
                "description": "Posts updates to records and adds and deletes records for a dataset and user.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dataset": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "dataset": {
                "name": "dataset",
                "url": "https://docs.aws.amazon.com/cognito/latest/developerguide/synchronizing-data.html#understanding-datasets",
                "arn": "arn:${Partition}:cognito-sync:${Region}:${Account}:identitypool/${IdentityPoolId}/identity/${IdentityId}/dataset/${DatasetName}",
                "conditionKeys": []
            },
            "identity": {
                "name": "identity",
                "url": "https://docs.aws.amazon.com/cognito/latest/developerguide/identity-pools.html#authenticated-and-unauthenticated-identities",
                "arn": "arn:${Partition}:cognito-sync:${Region}:${Account}:identitypool/${IdentityPoolId}/identity/${IdentityId}",
                "conditionKeys": []
            },
            "identitypool": {
                "name": "identitypool",
                "url": "https://docs.aws.amazon.com/cognito/latest/developerguide/identity-pools.html",
                "arn": "arn:${Partition}:cognito-sync:${Region}:${Account}:identitypool/${IdentityPoolId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Initiates a bulk publish of all existing datasets for an Identity Pool to the configured stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_BulkPublish.html
     */
    bulkPublish() {
        this.add('cognito-sync:BulkPublish');
        return this;
    }
    /**
     * Deletes the specific dataset.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_DeleteDataset.html
     */
    deleteDataset() {
        this.add('cognito-sync:DeleteDataset');
        return this;
    }
    /**
     * Gets meta data about a dataset by identity and dataset name.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_DescribeDataset.html
     */
    describeDataset() {
        this.add('cognito-sync:DescribeDataset');
        return this;
    }
    /**
     * Gets usage details (for example, data storage) about a particular identity pool.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_DescribeIdentityPoolUsage.html
     */
    describeIdentityPoolUsage() {
        this.add('cognito-sync:DescribeIdentityPoolUsage');
        return this;
    }
    /**
     * Gets usage information for an identity, including number of datasets and data usage.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_DescribeIdentityUsage.html
     */
    describeIdentityUsage() {
        this.add('cognito-sync:DescribeIdentityUsage');
        return this;
    }
    /**
     * Get the status of the last BulkPublish operation for an identity pool.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_GetBulkPublishDetails.html
     */
    getBulkPublishDetails() {
        this.add('cognito-sync:GetBulkPublishDetails');
        return this;
    }
    /**
     * Gets the events and the corresponding Lambda functions associated with an identity pool.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_GetCognitoEvents.html
     */
    getCognitoEvents() {
        this.add('cognito-sync:GetCognitoEvents');
        return this;
    }
    /**
     * Gets the configuration settings of an identity pool.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_GetIdentityPoolConfiguration.html
     */
    getIdentityPoolConfiguration() {
        this.add('cognito-sync:GetIdentityPoolConfiguration');
        return this;
    }
    /**
     * Lists datasets for an identity.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_ListDatasets.html
     */
    listDatasets() {
        this.add('cognito-sync:ListDatasets');
        return this;
    }
    /**
     * Gets a list of identity pools registered with Cognito.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_ListIdentityPoolUsage.html
     */
    listIdentityPoolUsage() {
        this.add('cognito-sync:ListIdentityPoolUsage');
        return this;
    }
    /**
     * Gets paginated records, optionally changed after a particular sync count for a dataset and identity.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_ListRecords.html
     */
    listRecords() {
        this.add('cognito-sync:ListRecords');
        return this;
    }
    /**
     * A permission that grants the ability to query records.
     *
     * Access Level: Read
     */
    queryRecords() {
        this.add('cognito-sync:QueryRecords');
        return this;
    }
    /**
     * Registers a device to receive push sync notifications.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_RegisterDevice.html
     */
    registerDevice() {
        this.add('cognito-sync:RegisterDevice');
        return this;
    }
    /**
     * Sets the AWS Lambda function for a given event type for an identity pool.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_SetCognitoEvents.html
     */
    setCognitoEvents() {
        this.add('cognito-sync:SetCognitoEvents');
        return this;
    }
    /**
     * A permission that grants ability to configure datasets.
     *
     * Access Level: Write
     */
    setDatasetConfiguration() {
        this.add('cognito-sync:SetDatasetConfiguration');
        return this;
    }
    /**
     * Sets the necessary configuration for push sync.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_SetIdentityPoolConfiguration.html
     */
    setIdentityPoolConfiguration() {
        this.add('cognito-sync:SetIdentityPoolConfiguration');
        return this;
    }
    /**
     * Subscribes to receive notifications when a dataset is modified by another device.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_SubscribeToDataset.html
     */
    subscribeToDataset() {
        this.add('cognito-sync:SubscribeToDataset');
        return this;
    }
    /**
     * Unsubscribes from receiving notifications when a dataset is modified by another device.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_UnsubscribeFromDataset.html
     */
    unsubscribeFromDataset() {
        this.add('cognito-sync:UnsubscribeFromDataset');
        return this;
    }
    /**
     * Posts updates to records and adds and deletes records for a dataset and user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_UpdateRecords.html
     */
    updateRecords() {
        this.add('cognito-sync:UpdateRecords');
        return this;
    }
    /**
     * Adds a resource of type dataset to the statement
     *
     * https://docs.aws.amazon.com/cognito/latest/developerguide/synchronizing-data.html#understanding-datasets
     *
     * @param identityPoolId - Identifier for the identityPoolId.
     * @param identityId - Identifier for the identityId.
     * @param datasetName - Identifier for the datasetName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDataset(identityPoolId, identityId, datasetName, account, region, partition) {
        var arn = 'arn:${Partition}:cognito-sync:${Region}:${Account}:identitypool/${IdentityPoolId}/identity/${IdentityId}/dataset/${DatasetName}';
        arn = arn.replace('${IdentityPoolId}', identityPoolId);
        arn = arn.replace('${IdentityId}', identityId);
        arn = arn.replace('${DatasetName}', datasetName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type identity to the statement
     *
     * https://docs.aws.amazon.com/cognito/latest/developerguide/identity-pools.html#authenticated-and-unauthenticated-identities
     *
     * @param identityPoolId - Identifier for the identityPoolId.
     * @param identityId - Identifier for the identityId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onIdentity(identityPoolId, identityId, account, region, partition) {
        var arn = 'arn:${Partition}:cognito-sync:${Region}:${Account}:identitypool/${IdentityPoolId}/identity/${IdentityId}';
        arn = arn.replace('${IdentityPoolId}', identityPoolId);
        arn = arn.replace('${IdentityId}', identityId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type identitypool to the statement
     *
     * https://docs.aws.amazon.com/cognito/latest/developerguide/identity-pools.html
     *
     * @param identityPoolId - Identifier for the identityPoolId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onIdentitypool(identityPoolId, account, region, partition) {
        var arn = 'arn:${Partition}:cognito-sync:${Region}:${Account}:identitypool/${IdentityPoolId}';
        arn = arn.replace('${IdentityPoolId}', identityPoolId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.CognitoSync = CognitoSync;
//# sourceMappingURL=data:application/json;base64,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