"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service imagebuilder
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonec2imagebuilder.html
 */
class Imagebuilder extends shared_1.PolicyStatement {
    /**
     * Action provider for service imagebuilder
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonec2imagebuilder.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'imagebuilder';
        this.actions = {
            "CancelImageCreation": {
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_CancelImageCreation.html",
                "description": "Cancel an image creation",
                "accessLevel": "Write",
                "resourceTypes": {
                    "image": {
                        "required": true
                    }
                }
            },
            "CreateComponent": {
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_CreateComponent.html",
                "description": "Create a new component",
                "accessLevel": "Write",
                "resourceTypes": {
                    "component": {
                        "required": true
                    },
                    "kmsKey": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateDistributionConfiguration": {
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_CreateDistributionConfiguration.html",
                "description": "Create a new distribution configuration",
                "accessLevel": "Write",
                "resourceTypes": {
                    "distributionConfiguration": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateImage": {
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_CreateImage.html",
                "description": "Create a new image",
                "accessLevel": "Write",
                "resourceTypes": {
                    "image": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateImagePipeline": {
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_CreateImagePipeline.html",
                "description": "Create a new image pipeline",
                "accessLevel": "Write",
                "resourceTypes": {
                    "imagePipeline": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateImageRecipe": {
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_CreateImageRecipe.html",
                "description": "Create a new Image Recipe",
                "accessLevel": "Write",
                "resourceTypes": {
                    "imageRecipe": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateInfrastructureConfiguration": {
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_CreateInfrastructureConfiguration.html",
                "description": "Create a new infrastructure configuration",
                "accessLevel": "Write",
                "resourceTypes": {
                    "infrastructureConfiguration": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "imagebuilder:CreatedResourceTagKeys",
                    "imagebuilder:CreatedResourceTag/<key>"
                ]
            },
            "DeleteComponent": {
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_DeleteComponent.html",
                "description": "Delete a component",
                "accessLevel": "Write",
                "resourceTypes": {
                    "component": {
                        "required": true
                    }
                }
            },
            "DeleteDistributionConfiguration": {
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_DeleteDistributionConfiguration.html",
                "description": "Delete a distribution configuration",
                "accessLevel": "Write",
                "resourceTypes": {
                    "distributionConfiguration": {
                        "required": true
                    }
                }
            },
            "DeleteImage": {
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_DeleteImage.html",
                "description": "Delete an image",
                "accessLevel": "Write",
                "resourceTypes": {
                    "image": {
                        "required": true
                    }
                }
            },
            "DeleteImagePipeline": {
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_DeleteImagePipeline.html",
                "description": "Delete an image pipeline",
                "accessLevel": "Write",
                "resourceTypes": {
                    "imagePipeline": {
                        "required": true
                    }
                }
            },
            "DeleteImageRecipe": {
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_DeleteImageRecipe.html",
                "description": "Grants permission to delete an image recipe",
                "accessLevel": "Write",
                "resourceTypes": {
                    "imageRecipe": {
                        "required": true
                    }
                }
            },
            "DeleteInfrastructureConfiguration": {
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_DeleteInfrastructureConfiguration.html",
                "description": "Delete an infrastructure configuration",
                "accessLevel": "Write",
                "resourceTypes": {
                    "infrastructureConfiguration": {
                        "required": true
                    }
                }
            },
            "GetComponent": {
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_GetComponent.html",
                "description": "View details about a component",
                "accessLevel": "Read",
                "resourceTypes": {
                    "component": {
                        "required": true
                    }
                }
            },
            "GetComponentPolicy": {
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_GetComponentPolicy.html",
                "description": "View the resource policy associated with a component",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "component": {
                        "required": true
                    }
                }
            },
            "GetDistributionConfiguration": {
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_GetDistributionConfiguration.html",
                "description": "View details about a distribution configuration",
                "accessLevel": "Read",
                "resourceTypes": {
                    "distributionConfiguration": {
                        "required": true
                    }
                }
            },
            "GetImage": {
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_GetImage.html",
                "description": "View details about an image",
                "accessLevel": "Read",
                "resourceTypes": {
                    "image": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "GetImagePipeline": {
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_GetImagePipeline.html",
                "description": "View details about an image pipeline",
                "accessLevel": "Read",
                "resourceTypes": {
                    "imagePipeline": {
                        "required": true
                    }
                }
            },
            "GetImagePolicy": {
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_GetImagePolicy.html",
                "description": "View the resource policy associated with an image",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "image": {
                        "required": true
                    }
                }
            },
            "GetImageRecipe": {
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_GetImageRecipe.html",
                "description": "View details about an image recipe",
                "accessLevel": "Read",
                "resourceTypes": {
                    "imageRecipe": {
                        "required": true
                    }
                }
            },
            "GetImageRecipePolicy": {
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_GetImageRecipePolicy.html",
                "description": "View the resource policy associated with an image recipe",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "imageRecipe": {
                        "required": true
                    }
                }
            },
            "GetInfrastructureConfiguration": {
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_GetInfrastructureConfiguration.html",
                "description": "View details about an infrastructure configuration",
                "accessLevel": "Read",
                "resourceTypes": {
                    "infrastructureConfiguration": {
                        "required": true
                    }
                }
            },
            "ListComponentBuildVersions": {
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ListComponentBuildVersions.html",
                "description": "List the component build versions in your account",
                "accessLevel": "List",
                "resourceTypes": {
                    "componentVersion": {
                        "required": true
                    }
                }
            },
            "ListComponents": {
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ListComponents.html",
                "description": "List the component versions owned by or shared with your account",
                "accessLevel": "List"
            },
            "ListDistributionConfigurations": {
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ListDistributionConfigurations.html",
                "description": "List the distribution configurations in your account",
                "accessLevel": "List"
            },
            "ListImageBuildVersions": {
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ListImageBuildVersions.html",
                "description": "List the image build versions in your account",
                "accessLevel": "List",
                "resourceTypes": {
                    "imageVersion": {
                        "required": true
                    }
                }
            },
            "ListImagePipelines": {
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ListImagePipelines.html",
                "description": "List the image pipelines in your account",
                "accessLevel": "List"
            },
            "ListImageRecipes": {
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ListImageRecipes.html",
                "description": "List the image recipes owned by or shared with your account",
                "accessLevel": "List"
            },
            "ListImages": {
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ListImages.html",
                "description": "List the image versions owned by or shared with your account",
                "accessLevel": "List"
            },
            "ListInfrastructureConfigurations": {
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ListInfrastructureConfigurations.html",
                "description": "List the infrastructure configurations in your account",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ListTagsForResource.html",
                "description": "List tag for an Image Builder resource",
                "accessLevel": "Read",
                "resourceTypes": {
                    "component": {
                        "required": false
                    },
                    "distributionConfiguration": {
                        "required": false
                    },
                    "image": {
                        "required": false
                    },
                    "imagePipeline": {
                        "required": false
                    },
                    "imageRecipe": {
                        "required": false
                    },
                    "infrastructureConfiguration": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "PutComponentPolicy": {
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_PutComponentPolicy.html",
                "description": "Set the resource policy associated with a component",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "component": {
                        "required": true
                    }
                }
            },
            "PutImagePolicy": {
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_PutImagePolicy.html",
                "description": "Set the resource policy associated with an image",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "image": {
                        "required": true
                    }
                }
            },
            "PutImageRecipePolicy": {
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_PutImageRecipePolicy.html",
                "description": "Set the resource policy associated with an image recipe",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "imageRecipe": {
                        "required": true
                    }
                }
            },
            "StartImagePipelineExecution": {
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_StartImagePipelineExecution.html",
                "description": "Create a new image from a pipeline",
                "accessLevel": "Write",
                "resourceTypes": {
                    "imagePipeline": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_TagResource.html",
                "description": "Tag an Image Builder resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "component": {
                        "required": false
                    },
                    "distributionConfiguration": {
                        "required": false
                    },
                    "image": {
                        "required": false
                    },
                    "imagePipeline": {
                        "required": false
                    },
                    "imageRecipe": {
                        "required": false
                    },
                    "infrastructureConfiguration": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_UntagResource.html",
                "description": "Untag an Image Builder resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "component": {
                        "required": false
                    },
                    "distributionConfiguration": {
                        "required": false
                    },
                    "image": {
                        "required": false
                    },
                    "imagePipeline": {
                        "required": false
                    },
                    "imageRecipe": {
                        "required": false
                    },
                    "infrastructureConfiguration": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UpdateDistributionConfiguration": {
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_UpdateDistributionConfiguration.html",
                "description": "Update an existing distribution configuration",
                "accessLevel": "Write",
                "resourceTypes": {
                    "distributionConfiguration": {
                        "required": true
                    }
                }
            },
            "UpdateImagePipeline": {
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_UpdateImagePipeline.html",
                "description": "Update an existing image pipeline",
                "accessLevel": "Write",
                "resourceTypes": {
                    "imagePipeline": {
                        "required": true
                    }
                }
            },
            "UpdateInfrastructureConfiguration": {
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_UpdateInfrastructureConfiguration.html",
                "description": "Update an existing infrastructure configuration",
                "accessLevel": "Write",
                "resourceTypes": {
                    "infrastructureConfiguration": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}",
                    "imagebuilder:CreatedResourceTagKeys",
                    "imagebuilder:CreatedResourceTag/<key>"
                ]
            }
        };
        this.resourceTypes = {
            "component": {
                "name": "component",
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_Component.html",
                "arn": "arn:${Partition}:imagebuilder:${Region}:${Account}:component/${ComponentName}/${ComponentVersion}/${ComponentBuildVersion}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "componentVersion": {
                "name": "componentVersion",
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ComponentVersion",
                "arn": "arn:${Partition}:imagebuilder:${Region}:${Account}:component/${ComponentName}/${ComponentVersion}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "distributionConfiguration": {
                "name": "distributionConfiguration",
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_DistributionConfiguration.html",
                "arn": "arn:${Partition}:imagebuilder:${Region}:${Account}:distribution-configuration/${DistributionConfigurationName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "image": {
                "name": "image",
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_Image.html",
                "arn": "arn:${Partition}:imagebuilder:${Region}:${Account}:image/${ImageName}/${ImageVersion}/${ImageBuildVersion}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "imageVersion": {
                "name": "imageVersion",
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ImageVersion.html",
                "arn": "arn:${Partition}:imagebuilder:${Region}:${Account}:image/${ImageName}/${ImageVersion}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "imageRecipe": {
                "name": "imageRecipe",
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ImageRecipe.html",
                "arn": "arn:${Partition}:imagebuilder:${Region}:${Account}:image-recipe/${ImageRecipeName}/${ImageRecipeVersion}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "imagePipeline": {
                "name": "imagePipeline",
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ImagePipeline.html",
                "arn": "arn:${Partition}:imagebuilder:${Region}:${Account}:image-pipeline/${ImagePipelineName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "infrastructureConfiguration": {
                "name": "infrastructureConfiguration",
                "url": "https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_InfrastructureConfiguration.html",
                "arn": "arn:${Partition}:imagebuilder:${Region}:${Account}:infrastructure-configuration/${ResourceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "kmsKey": {
                "name": "kmsKey",
                "url": "https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#master_keys",
                "arn": "arn:${Partition}:kms:${Region}:${Account}:key/${KeyId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Cancel an image creation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_CancelImageCreation.html
     */
    cancelImageCreation() {
        this.add('imagebuilder:CancelImageCreation');
        return this;
    }
    /**
     * Create a new component
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_CreateComponent.html
     */
    createComponent() {
        this.add('imagebuilder:CreateComponent');
        return this;
    }
    /**
     * Create a new distribution configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_CreateDistributionConfiguration.html
     */
    createDistributionConfiguration() {
        this.add('imagebuilder:CreateDistributionConfiguration');
        return this;
    }
    /**
     * Create a new image
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_CreateImage.html
     */
    createImage() {
        this.add('imagebuilder:CreateImage');
        return this;
    }
    /**
     * Create a new image pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_CreateImagePipeline.html
     */
    createImagePipeline() {
        this.add('imagebuilder:CreateImagePipeline');
        return this;
    }
    /**
     * Create a new Image Recipe
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_CreateImageRecipe.html
     */
    createImageRecipe() {
        this.add('imagebuilder:CreateImageRecipe');
        return this;
    }
    /**
     * Create a new infrastructure configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_CreateInfrastructureConfiguration.html
     */
    createInfrastructureConfiguration() {
        this.add('imagebuilder:CreateInfrastructureConfiguration');
        return this;
    }
    /**
     * Delete a component
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_DeleteComponent.html
     */
    deleteComponent() {
        this.add('imagebuilder:DeleteComponent');
        return this;
    }
    /**
     * Delete a distribution configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_DeleteDistributionConfiguration.html
     */
    deleteDistributionConfiguration() {
        this.add('imagebuilder:DeleteDistributionConfiguration');
        return this;
    }
    /**
     * Delete an image
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_DeleteImage.html
     */
    deleteImage() {
        this.add('imagebuilder:DeleteImage');
        return this;
    }
    /**
     * Delete an image pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_DeleteImagePipeline.html
     */
    deleteImagePipeline() {
        this.add('imagebuilder:DeleteImagePipeline');
        return this;
    }
    /**
     * Grants permission to delete an image recipe
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_DeleteImageRecipe.html
     */
    deleteImageRecipe() {
        this.add('imagebuilder:DeleteImageRecipe');
        return this;
    }
    /**
     * Delete an infrastructure configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_DeleteInfrastructureConfiguration.html
     */
    deleteInfrastructureConfiguration() {
        this.add('imagebuilder:DeleteInfrastructureConfiguration');
        return this;
    }
    /**
     * View details about a component
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_GetComponent.html
     */
    getComponent() {
        this.add('imagebuilder:GetComponent');
        return this;
    }
    /**
     * View the resource policy associated with a component
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_GetComponentPolicy.html
     */
    getComponentPolicy() {
        this.add('imagebuilder:GetComponentPolicy');
        return this;
    }
    /**
     * View details about a distribution configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_GetDistributionConfiguration.html
     */
    getDistributionConfiguration() {
        this.add('imagebuilder:GetDistributionConfiguration');
        return this;
    }
    /**
     * View details about an image
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_GetImage.html
     */
    getImage() {
        this.add('imagebuilder:GetImage');
        return this;
    }
    /**
     * View details about an image pipeline
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_GetImagePipeline.html
     */
    getImagePipeline() {
        this.add('imagebuilder:GetImagePipeline');
        return this;
    }
    /**
     * View the resource policy associated with an image
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_GetImagePolicy.html
     */
    getImagePolicy() {
        this.add('imagebuilder:GetImagePolicy');
        return this;
    }
    /**
     * View details about an image recipe
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_GetImageRecipe.html
     */
    getImageRecipe() {
        this.add('imagebuilder:GetImageRecipe');
        return this;
    }
    /**
     * View the resource policy associated with an image recipe
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_GetImageRecipePolicy.html
     */
    getImageRecipePolicy() {
        this.add('imagebuilder:GetImageRecipePolicy');
        return this;
    }
    /**
     * View details about an infrastructure configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_GetInfrastructureConfiguration.html
     */
    getInfrastructureConfiguration() {
        this.add('imagebuilder:GetInfrastructureConfiguration');
        return this;
    }
    /**
     * List the component build versions in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ListComponentBuildVersions.html
     */
    listComponentBuildVersions() {
        this.add('imagebuilder:ListComponentBuildVersions');
        return this;
    }
    /**
     * List the component versions owned by or shared with your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ListComponents.html
     */
    listComponents() {
        this.add('imagebuilder:ListComponents');
        return this;
    }
    /**
     * List the distribution configurations in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ListDistributionConfigurations.html
     */
    listDistributionConfigurations() {
        this.add('imagebuilder:ListDistributionConfigurations');
        return this;
    }
    /**
     * List the image build versions in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ListImageBuildVersions.html
     */
    listImageBuildVersions() {
        this.add('imagebuilder:ListImageBuildVersions');
        return this;
    }
    /**
     * List the image pipelines in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ListImagePipelines.html
     */
    listImagePipelines() {
        this.add('imagebuilder:ListImagePipelines');
        return this;
    }
    /**
     * List the image recipes owned by or shared with your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ListImageRecipes.html
     */
    listImageRecipes() {
        this.add('imagebuilder:ListImageRecipes');
        return this;
    }
    /**
     * List the image versions owned by or shared with your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ListImages.html
     */
    listImages() {
        this.add('imagebuilder:ListImages');
        return this;
    }
    /**
     * List the infrastructure configurations in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ListInfrastructureConfigurations.html
     */
    listInfrastructureConfigurations() {
        this.add('imagebuilder:ListInfrastructureConfigurations');
        return this;
    }
    /**
     * List tag for an Image Builder resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('imagebuilder:ListTagsForResource');
        return this;
    }
    /**
     * Set the resource policy associated with a component
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_PutComponentPolicy.html
     */
    putComponentPolicy() {
        this.add('imagebuilder:PutComponentPolicy');
        return this;
    }
    /**
     * Set the resource policy associated with an image
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_PutImagePolicy.html
     */
    putImagePolicy() {
        this.add('imagebuilder:PutImagePolicy');
        return this;
    }
    /**
     * Set the resource policy associated with an image recipe
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_PutImageRecipePolicy.html
     */
    putImageRecipePolicy() {
        this.add('imagebuilder:PutImageRecipePolicy');
        return this;
    }
    /**
     * Create a new image from a pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_StartImagePipelineExecution.html
     */
    startImagePipelineExecution() {
        this.add('imagebuilder:StartImagePipelineExecution');
        return this;
    }
    /**
     * Tag an Image Builder resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_TagResource.html
     */
    tagResource() {
        this.add('imagebuilder:TagResource');
        return this;
    }
    /**
     * Untag an Image Builder resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_UntagResource.html
     */
    untagResource() {
        this.add('imagebuilder:UntagResource');
        return this;
    }
    /**
     * Update an existing distribution configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_UpdateDistributionConfiguration.html
     */
    updateDistributionConfiguration() {
        this.add('imagebuilder:UpdateDistributionConfiguration');
        return this;
    }
    /**
     * Update an existing image pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_UpdateImagePipeline.html
     */
    updateImagePipeline() {
        this.add('imagebuilder:UpdateImagePipeline');
        return this;
    }
    /**
     * Update an existing infrastructure configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_UpdateInfrastructureConfiguration.html
     */
    updateInfrastructureConfiguration() {
        this.add('imagebuilder:UpdateInfrastructureConfiguration');
        return this;
    }
    /**
     * Adds a resource of type component to the statement
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_Component.html
     *
     * @param componentName - Identifier for the componentName.
     * @param componentVersion - Identifier for the componentVersion.
     * @param componentBuildVersion - Identifier for the componentBuildVersion.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onComponent(componentName, componentVersion, componentBuildVersion, account, region, partition) {
        var arn = 'arn:${Partition}:imagebuilder:${Region}:${Account}:component/${ComponentName}/${ComponentVersion}/${ComponentBuildVersion}';
        arn = arn.replace('${ComponentName}', componentName);
        arn = arn.replace('${ComponentVersion}', componentVersion);
        arn = arn.replace('${ComponentBuildVersion}', componentBuildVersion);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type componentVersion to the statement
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ComponentVersion
     *
     * @param componentName - Identifier for the componentName.
     * @param componentVersion - Identifier for the componentVersion.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onComponentVersion(componentName, componentVersion, account, region, partition) {
        var arn = 'arn:${Partition}:imagebuilder:${Region}:${Account}:component/${ComponentName}/${ComponentVersion}';
        arn = arn.replace('${ComponentName}', componentName);
        arn = arn.replace('${ComponentVersion}', componentVersion);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type distributionConfiguration to the statement
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_DistributionConfiguration.html
     *
     * @param distributionConfigurationName - Identifier for the distributionConfigurationName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onDistributionConfiguration(distributionConfigurationName, account, region, partition) {
        var arn = 'arn:${Partition}:imagebuilder:${Region}:${Account}:distribution-configuration/${DistributionConfigurationName}';
        arn = arn.replace('${DistributionConfigurationName}', distributionConfigurationName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type image to the statement
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_Image.html
     *
     * @param imageName - Identifier for the imageName.
     * @param imageVersion - Identifier for the imageVersion.
     * @param imageBuildVersion - Identifier for the imageBuildVersion.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onImage(imageName, imageVersion, imageBuildVersion, account, region, partition) {
        var arn = 'arn:${Partition}:imagebuilder:${Region}:${Account}:image/${ImageName}/${ImageVersion}/${ImageBuildVersion}';
        arn = arn.replace('${ImageName}', imageName);
        arn = arn.replace('${ImageVersion}', imageVersion);
        arn = arn.replace('${ImageBuildVersion}', imageBuildVersion);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type imageVersion to the statement
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ImageVersion.html
     *
     * @param imageName - Identifier for the imageName.
     * @param imageVersion - Identifier for the imageVersion.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onImageVersion(imageName, imageVersion, account, region, partition) {
        var arn = 'arn:${Partition}:imagebuilder:${Region}:${Account}:image/${ImageName}/${ImageVersion}';
        arn = arn.replace('${ImageName}', imageName);
        arn = arn.replace('${ImageVersion}', imageVersion);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type imageRecipe to the statement
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ImageRecipe.html
     *
     * @param imageRecipeName - Identifier for the imageRecipeName.
     * @param imageRecipeVersion - Identifier for the imageRecipeVersion.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onImageRecipe(imageRecipeName, imageRecipeVersion, account, region, partition) {
        var arn = 'arn:${Partition}:imagebuilder:${Region}:${Account}:image-recipe/${ImageRecipeName}/${ImageRecipeVersion}';
        arn = arn.replace('${ImageRecipeName}', imageRecipeName);
        arn = arn.replace('${ImageRecipeVersion}', imageRecipeVersion);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type imagePipeline to the statement
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ImagePipeline.html
     *
     * @param imagePipelineName - Identifier for the imagePipelineName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onImagePipeline(imagePipelineName, account, region, partition) {
        var arn = 'arn:${Partition}:imagebuilder:${Region}:${Account}:image-pipeline/${ImagePipelineName}';
        arn = arn.replace('${ImagePipelineName}', imagePipelineName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type infrastructureConfiguration to the statement
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_InfrastructureConfiguration.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onInfrastructureConfiguration(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:imagebuilder:${Region}:${Account}:infrastructure-configuration/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type kmsKey to the statement
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#master_keys
     *
     * @param keyId - Identifier for the keyId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onKmsKey(keyId, account, region, partition) {
        var arn = 'arn:${Partition}:kms:${Region}:${Account}:key/${KeyId}';
        arn = arn.replace('${KeyId}', keyId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters access by the tag key-value pairs attached to the resource created by Image Builder
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/userguide/security_iam_service-with-iam.html#image-builder-security-createdresourcetag
     *
     * @param key The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCreatedResourceTag(key, value, operator) {
        return this.if(`imagebuilder:CreatedResourceTag/${key}`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/userguide/security_iam_service-with-iam.html#image-builder-security-createdresourcetagkeys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCreatedResourceTagKeys(value, operator) {
        return this.if(`imagebuilder:CreatedResourceTagKeys`, value, operator || 'StringLike');
    }
}
exports.Imagebuilder = Imagebuilder;
//# sourceMappingURL=data:application/json;base64,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