"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service ebs
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelasticblockstore.html
 */
class Ebs extends shared_1.PolicyStatement {
    /**
     * Action provider for service ebs
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelasticblockstore.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'ebs';
        this.actions = {
            "GetSnapshotBlock": {
                "url": "https://docs.aws.amazon.com/ebs/latest/APIReference/API_GetSnapshotBlock.html",
                "description": "Grants permission to return the data of a block in an Amazon Elastic Block Store (EBS) snapshot",
                "accessLevel": "Read",
                "resourceTypes": {
                    "snapshot": {
                        "required": true
                    }
                }
            },
            "ListChangedBlocks": {
                "url": "https://docs.aws.amazon.com/ebs/latest/APIReference/API_ListChangedBlocks.html",
                "description": "Grants permission to list the block indexes and block tokens for blocks that are different between two Amazon Elastic Block Store (EBS) snapshots of the same volume/snapshot lineage",
                "accessLevel": "Read",
                "resourceTypes": {
                    "snapshot": {
                        "required": true
                    }
                }
            },
            "ListSnapshotBlocks": {
                "url": "https://docs.aws.amazon.com/ebs/latest/APIReference/API_ListSnapshotBlocks.html",
                "description": "Grants permission to list the block indexes and block tokens for blocks in an Amazon Elastic Block Store (EBS) snapshot.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "snapshot": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "snapshot": {
                "name": "snapshot",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#EC2_ARN_Format",
                "arn": "arn:${Partition}:ec2:${Region}::snapshot/${SnapshotId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Grants permission to return the data of a block in an Amazon Elastic Block Store (EBS) snapshot
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ebs/latest/APIReference/API_GetSnapshotBlock.html
     */
    getSnapshotBlock() {
        this.add('ebs:GetSnapshotBlock');
        return this;
    }
    /**
     * Grants permission to list the block indexes and block tokens for blocks that are different between two Amazon Elastic Block Store (EBS) snapshots of the same volume/snapshot lineage
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ebs/latest/APIReference/API_ListChangedBlocks.html
     */
    listChangedBlocks() {
        this.add('ebs:ListChangedBlocks');
        return this;
    }
    /**
     * Grants permission to list the block indexes and block tokens for blocks in an Amazon Elastic Block Store (EBS) snapshot.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ebs/latest/APIReference/API_ListSnapshotBlocks.html
     */
    listSnapshotBlocks() {
        this.add('ebs:ListSnapshotBlocks');
        return this;
    }
    /**
     * Adds a resource of type snapshot to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#EC2_ARN_Format
     *
     * @param snapshotId - Identifier for the snapshotId.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onSnapshot(snapshotId, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}::snapshot/${SnapshotId}';
        arn = arn.replace('${SnapshotId}', snapshotId);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Ebs = Ebs;
//# sourceMappingURL=data:application/json;base64,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