"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service ecs
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelasticcontainerservice.html
 */
class Ecs extends shared_1.PolicyStatement {
    /**
     * Action provider for service ecs
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelasticcontainerservice.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'ecs';
        this.actions = {
            "CreateCluster": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_CreateCluster.html",
                "description": "Creates a new Amazon ECS cluster.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateService": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_CreateService.html",
                "description": "Runs and maintains a desired number of tasks from a specified task definition.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "service": {
                        "required": true
                    }
                },
                "conditions": [
                    "ecs:cluster",
                    "ecs:task-definition",
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateTaskSet": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_CreateTaskSet.html",
                "description": "Creates a new Amazon ECS task set.",
                "accessLevel": "Write",
                "conditions": [
                    "ecs:cluster",
                    "ecs:service",
                    "ecs:task-definition"
                ]
            },
            "DeleteAccountSetting": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DeleteAccountSetting.html",
                "description": "Modifies the ARN and resource ID format of a resource for a specified IAM user, IAM role, or the root user for an account. You can specify whether the new ARN and resource ID format are disabled for new resources that are created.",
                "accessLevel": "Write"
            },
            "DeleteAttributes": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DeleteAttributes.html",
                "description": "Deletes one or more custom attributes from an Amazon ECS resource.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "container-instance": {
                        "required": true
                    }
                },
                "conditions": [
                    "ecs:cluster"
                ]
            },
            "DeleteCluster": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DeleteCluster.html",
                "description": "Deletes the specified cluster.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "DeleteService": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DeleteService.html",
                "description": "Deletes a specified service within a cluster.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "service": {
                        "required": true
                    }
                },
                "conditions": [
                    "ecs:cluster"
                ]
            },
            "DeleteTaskSet": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DeleteTaskSet.html",
                "description": "Deletes the specified task set.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "task-set": {
                        "required": true
                    }
                },
                "conditions": [
                    "ecs:cluster",
                    "ecs:service"
                ]
            },
            "DeregisterContainerInstance": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DeregisterContainerInstance.html",
                "description": "Deregisters an Amazon ECS container instance from the specified cluster.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "DeregisterTaskDefinition": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DeregisterTaskDefinition.html",
                "description": "Deregisters the specified task definition by family and revision.",
                "accessLevel": "Write"
            },
            "DescribeClusters": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeClusters.html",
                "description": "Describes one or more of your clusters.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "DescribeContainerInstances": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeContainerInstances.html",
                "description": "Describes Amazon ECS container instances.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "container-instance": {
                        "required": true
                    }
                },
                "conditions": [
                    "ecs:cluster"
                ]
            },
            "DescribeServices": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeServices.html",
                "description": "Describes the specified services running in your cluster.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "service": {
                        "required": true
                    }
                },
                "conditions": [
                    "ecs:cluster"
                ]
            },
            "DescribeTaskDefinition": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeTaskDefinition.html",
                "description": "Describes a task definition. You can specify a family and revision to find information about a specific task definition, or you can simply specify the family to find the latest ACTIVE revision in that family.",
                "accessLevel": "Read"
            },
            "DescribeTaskSets": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeTaskSets.html",
                "description": "Describes Amazon ECS task sets.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "task-set": {
                        "required": true
                    }
                },
                "conditions": [
                    "ecs:cluster",
                    "ecs:service"
                ]
            },
            "DescribeTasks": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeTasks.html",
                "description": "Describes a specified task or tasks.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "task": {
                        "required": true
                    }
                },
                "conditions": [
                    "ecs:cluster"
                ]
            },
            "DiscoverPollEndpoint": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DiscoverPollEndpoint.html",
                "description": "Returns an endpoint for the Amazon ECS agent to poll for updates.",
                "accessLevel": "Write"
            },
            "ListAccountSettings": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListAccountSettings.html",
                "description": "Lists the account settings for an Amazon ECS resource for a specified principal.",
                "accessLevel": "List"
            },
            "ListAttributes": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListAttributes.html",
                "description": "Lists the attributes for Amazon ECS resources within a specified target type and cluster.",
                "accessLevel": "List",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "ListClusters": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListClusters.html",
                "description": "Returns a list of existing clusters.",
                "accessLevel": "List"
            },
            "ListContainerInstances": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListContainerInstances.html",
                "description": "Returns a list of container instances in a specified cluster.",
                "accessLevel": "List",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "ListServices": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListServices.html",
                "description": "Lists the services that are running in a specified cluster.",
                "accessLevel": "List",
                "conditions": [
                    "ecs:cluster"
                ]
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListTagsForResource.html",
                "description": "List tags for the specified resource.",
                "accessLevel": "List",
                "resourceTypes": {
                    "cluster": {
                        "required": false
                    },
                    "container-instance": {
                        "required": false
                    },
                    "task": {
                        "required": false
                    },
                    "task-definition": {
                        "required": false
                    }
                }
            },
            "ListTaskDefinitionFamilies": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListServices.html",
                "description": "Returns a list of task definition families that are registered to your account (which may include task definition families that no longer have any ACTIVE task definitions).",
                "accessLevel": "List"
            },
            "ListTaskDefinitions": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListTaskDefinitions.html",
                "description": "Returns a list of task definitions that are registered to your account.",
                "accessLevel": "List"
            },
            "ListTasks": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListTasks.html",
                "description": "Returns a list of tasks for a specified cluster.",
                "accessLevel": "List",
                "resourceTypes": {
                    "container-instance": {
                        "required": true
                    }
                },
                "conditions": [
                    "ecs:cluster"
                ]
            },
            "Poll": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/instance_IAM_role.html",
                "description": "Grants permission to an agent to connect with the Amazon ECS service to report status and get commands.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "container-instance": {
                        "required": true
                    }
                },
                "conditions": [
                    "ecs:cluster"
                ]
            },
            "PutAccountSetting": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_PutAccountSetting.html",
                "description": "Modifies the ARN and resource ID format of a resource for a specified IAM user, IAM role, or the root user for an account. You can specify whether the new ARN and resource ID format are enabled for new resources that are created. Enabling this setting is required to use new Amazon ECS features such as resource tagging.",
                "accessLevel": "Write"
            },
            "PutAccountSettingDefault": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_PutAccountSettingDefault.html",
                "description": "Modifies the ARN and resource ID format of a resource type for all IAM users on an account for which no individual account setting has been set. Enabling this setting is required to use new Amazon ECS features such as resource tagging.",
                "accessLevel": "Write"
            },
            "PutAttributes": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_PutAttributes.html",
                "description": "Create or update an attribute on an Amazon ECS resource.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "container-instance": {
                        "required": true
                    }
                },
                "conditions": [
                    "ecs:cluster"
                ]
            },
            "RegisterContainerInstance": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_RegisterContainerInstance.html",
                "description": "Registers an EC2 instance into the specified cluster.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "RegisterTaskDefinition": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_RegisterTaskDefinition.html",
                "description": "Registers a new task definition from the supplied family and containerDefinitions.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "RunTask": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_RunTask.html",
                "description": "Start a task using random placement and the default Amazon ECS scheduler.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "task-definition": {
                        "required": true
                    }
                },
                "conditions": [
                    "ecs:cluster",
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "StartTask": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_StartTask.html",
                "description": "Starts a new task from the specified task definition on the specified container instance or instances.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "task-definition": {
                        "required": true
                    }
                },
                "conditions": [
                    "ecs:cluster",
                    "ecs:container-instances",
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "StartTelemetrySession": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/cloudwatch-metrics.html#enable_cloudwatch",
                "description": "Grants permission to start a telemetry session.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "container-instance": {
                        "required": true
                    }
                },
                "conditions": [
                    "ecs:cluster"
                ]
            },
            "StopTask": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_StopTask.html",
                "description": "Stops a running task.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "task": {
                        "required": true
                    }
                },
                "conditions": [
                    "ecs:cluster"
                ]
            },
            "SubmitAttachmentStateChanges": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_SubmitAttachmentStateChanges.html",
                "description": "Sent to acknowledge that attachments changed states.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "SubmitContainerStateChange": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_SubmitContainerStateChange.html",
                "description": "Sent to acknowledge that a container changed states.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "SubmitTaskStateChange": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_SubmitTaskStateChange.html",
                "description": "Sent to acknowledge that a task changed states.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_TagResource.html",
                "description": "Tags the specified resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "cluster": {
                        "required": false
                    },
                    "container-instance": {
                        "required": false
                    },
                    "service": {
                        "required": false
                    },
                    "task": {
                        "required": false
                    },
                    "task-definition": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_UntagResource.html",
                "description": "Untags the specified resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "cluster": {
                        "required": false
                    },
                    "container-instance": {
                        "required": false
                    },
                    "service": {
                        "required": false
                    },
                    "task": {
                        "required": false
                    },
                    "task-definition": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateContainerAgent": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_UpdateContainerAgent.html",
                "description": "Updates the Amazon ECS container agent on a specified container instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "container-instance": {
                        "required": true
                    }
                },
                "conditions": [
                    "ecs:cluster"
                ]
            },
            "UpdateContainerInstancesState": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_UpdateContainerInstancesState.html",
                "description": "Enables the user to modify the status of an Amazon ECS container instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "container-instance": {
                        "required": true
                    }
                },
                "conditions": [
                    "ecs:cluster"
                ]
            },
            "UpdateService": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_UpdateService.html",
                "description": "Modifies the desired count, deployment configuration, or task definition used in a service.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "service": {
                        "required": true
                    }
                },
                "conditions": [
                    "ecs:cluster",
                    "ecs:task-definition"
                ]
            },
            "UpdateServicePrimaryTaskSet": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_UpdateServicePrimaryTaskSet.html",
                "description": "Modifies the primary task set used in a service.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "service": {
                        "required": true
                    }
                },
                "conditions": [
                    "ecs:cluster"
                ]
            },
            "UpdateTaskSet": {
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_UpdateTaskSet.html",
                "description": "Updates the specified task set.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "task-set": {
                        "required": true
                    }
                },
                "conditions": [
                    "ecs:cluster",
                    "ecs:service"
                ]
            }
        };
        this.resourceTypes = {
            "cluster": {
                "name": "cluster",
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ECS_clusters.html",
                "arn": "arn:${Partition}:ecs:${Region}:${Account}:cluster/${ClusterName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "ecs:ResourceTag/${TagKey}"
                ]
            },
            "container-instance": {
                "name": "container-instance",
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ECS_instances.html",
                "arn": "arn:${Partition}:ecs:${Region}:${Account}:container-instance/${ContainerInstanceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "ecs:ResourceTag/${TagKey}"
                ]
            },
            "service": {
                "name": "service",
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs_services.html",
                "arn": "arn:${Partition}:ecs:${Region}:${Account}:service/${ServiceName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "ecs:ResourceTag/${TagKey}"
                ]
            },
            "task": {
                "name": "task",
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/scheduling_tasks.html",
                "arn": "arn:${Partition}:ecs:${Region}:${Account}:task/${TaskId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "ecs:ResourceTag/${TagKey}"
                ]
            },
            "task-definition": {
                "name": "task-definition",
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_definitions.html",
                "arn": "arn:${Partition}:ecs:${Region}:${Account}:task-definition/${TaskDefinitionFamilyName}:${TaskDefinitionRevisionNumber}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "ecs:ResourceTag/${TagKey}"
                ]
            },
            "task-set": {
                "name": "task-set",
                "url": "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_sets.html",
                "arn": "arn:${Partition}:ecs:${Region}:${Account}:task-set/${ClusterName}/${ServiceName}/${TaskSetId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "ecs:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Creates a new Amazon ECS cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_CreateCluster.html
     */
    createCluster() {
        this.add('ecs:CreateCluster');
        return this;
    }
    /**
     * Runs and maintains a desired number of tasks from a specified task definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_CreateService.html
     */
    createService() {
        this.add('ecs:CreateService');
        return this;
    }
    /**
     * Creates a new Amazon ECS task set.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_CreateTaskSet.html
     */
    createTaskSet() {
        this.add('ecs:CreateTaskSet');
        return this;
    }
    /**
     * Modifies the ARN and resource ID format of a resource for a specified IAM user, IAM role, or the root user for an account. You can specify whether the new ARN and resource ID format are disabled for new resources that are created.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DeleteAccountSetting.html
     */
    deleteAccountSetting() {
        this.add('ecs:DeleteAccountSetting');
        return this;
    }
    /**
     * Deletes one or more custom attributes from an Amazon ECS resource.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DeleteAttributes.html
     */
    deleteAttributes() {
        this.add('ecs:DeleteAttributes');
        return this;
    }
    /**
     * Deletes the specified cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DeleteCluster.html
     */
    deleteCluster() {
        this.add('ecs:DeleteCluster');
        return this;
    }
    /**
     * Deletes a specified service within a cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DeleteService.html
     */
    deleteService() {
        this.add('ecs:DeleteService');
        return this;
    }
    /**
     * Deletes the specified task set.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DeleteTaskSet.html
     */
    deleteTaskSet() {
        this.add('ecs:DeleteTaskSet');
        return this;
    }
    /**
     * Deregisters an Amazon ECS container instance from the specified cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DeregisterContainerInstance.html
     */
    deregisterContainerInstance() {
        this.add('ecs:DeregisterContainerInstance');
        return this;
    }
    /**
     * Deregisters the specified task definition by family and revision.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DeregisterTaskDefinition.html
     */
    deregisterTaskDefinition() {
        this.add('ecs:DeregisterTaskDefinition');
        return this;
    }
    /**
     * Describes one or more of your clusters.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeClusters.html
     */
    describeClusters() {
        this.add('ecs:DescribeClusters');
        return this;
    }
    /**
     * Describes Amazon ECS container instances.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeContainerInstances.html
     */
    describeContainerInstances() {
        this.add('ecs:DescribeContainerInstances');
        return this;
    }
    /**
     * Describes the specified services running in your cluster.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeServices.html
     */
    describeServices() {
        this.add('ecs:DescribeServices');
        return this;
    }
    /**
     * Describes a task definition. You can specify a family and revision to find information about a specific task definition, or you can simply specify the family to find the latest ACTIVE revision in that family.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeTaskDefinition.html
     */
    describeTaskDefinition() {
        this.add('ecs:DescribeTaskDefinition');
        return this;
    }
    /**
     * Describes Amazon ECS task sets.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeTaskSets.html
     */
    describeTaskSets() {
        this.add('ecs:DescribeTaskSets');
        return this;
    }
    /**
     * Describes a specified task or tasks.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeTasks.html
     */
    describeTasks() {
        this.add('ecs:DescribeTasks');
        return this;
    }
    /**
     * Returns an endpoint for the Amazon ECS agent to poll for updates.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DiscoverPollEndpoint.html
     */
    discoverPollEndpoint() {
        this.add('ecs:DiscoverPollEndpoint');
        return this;
    }
    /**
     * Lists the account settings for an Amazon ECS resource for a specified principal.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListAccountSettings.html
     */
    listAccountSettings() {
        this.add('ecs:ListAccountSettings');
        return this;
    }
    /**
     * Lists the attributes for Amazon ECS resources within a specified target type and cluster.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListAttributes.html
     */
    listAttributes() {
        this.add('ecs:ListAttributes');
        return this;
    }
    /**
     * Returns a list of existing clusters.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListClusters.html
     */
    listClusters() {
        this.add('ecs:ListClusters');
        return this;
    }
    /**
     * Returns a list of container instances in a specified cluster.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListContainerInstances.html
     */
    listContainerInstances() {
        this.add('ecs:ListContainerInstances');
        return this;
    }
    /**
     * Lists the services that are running in a specified cluster.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListServices.html
     */
    listServices() {
        this.add('ecs:ListServices');
        return this;
    }
    /**
     * List tags for the specified resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('ecs:ListTagsForResource');
        return this;
    }
    /**
     * Returns a list of task definition families that are registered to your account (which may include task definition families that no longer have any ACTIVE task definitions).
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListServices.html
     */
    listTaskDefinitionFamilies() {
        this.add('ecs:ListTaskDefinitionFamilies');
        return this;
    }
    /**
     * Returns a list of task definitions that are registered to your account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListTaskDefinitions.html
     */
    listTaskDefinitions() {
        this.add('ecs:ListTaskDefinitions');
        return this;
    }
    /**
     * Returns a list of tasks for a specified cluster.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListTasks.html
     */
    listTasks() {
        this.add('ecs:ListTasks');
        return this;
    }
    /**
     * Grants permission to an agent to connect with the Amazon ECS service to report status and get commands.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/developerguide/instance_IAM_role.html
     */
    poll() {
        this.add('ecs:Poll');
        return this;
    }
    /**
     * Modifies the ARN and resource ID format of a resource for a specified IAM user, IAM role, or the root user for an account. You can specify whether the new ARN and resource ID format are enabled for new resources that are created. Enabling this setting is required to use new Amazon ECS features such as resource tagging.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_PutAccountSetting.html
     */
    putAccountSetting() {
        this.add('ecs:PutAccountSetting');
        return this;
    }
    /**
     * Modifies the ARN and resource ID format of a resource type for all IAM users on an account for which no individual account setting has been set. Enabling this setting is required to use new Amazon ECS features such as resource tagging.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_PutAccountSettingDefault.html
     */
    putAccountSettingDefault() {
        this.add('ecs:PutAccountSettingDefault');
        return this;
    }
    /**
     * Create or update an attribute on an Amazon ECS resource.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_PutAttributes.html
     */
    putAttributes() {
        this.add('ecs:PutAttributes');
        return this;
    }
    /**
     * Registers an EC2 instance into the specified cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_RegisterContainerInstance.html
     */
    registerContainerInstance() {
        this.add('ecs:RegisterContainerInstance');
        return this;
    }
    /**
     * Registers a new task definition from the supplied family and containerDefinitions.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_RegisterTaskDefinition.html
     */
    registerTaskDefinition() {
        this.add('ecs:RegisterTaskDefinition');
        return this;
    }
    /**
     * Start a task using random placement and the default Amazon ECS scheduler.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_RunTask.html
     */
    runTask() {
        this.add('ecs:RunTask');
        return this;
    }
    /**
     * Starts a new task from the specified task definition on the specified container instance or instances.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_StartTask.html
     */
    startTask() {
        this.add('ecs:StartTask');
        return this;
    }
    /**
     * Grants permission to start a telemetry session.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/developerguide/cloudwatch-metrics.html#enable_cloudwatch
     */
    startTelemetrySession() {
        this.add('ecs:StartTelemetrySession');
        return this;
    }
    /**
     * Stops a running task.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_StopTask.html
     */
    stopTask() {
        this.add('ecs:StopTask');
        return this;
    }
    /**
     * Sent to acknowledge that attachments changed states.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_SubmitAttachmentStateChanges.html
     */
    submitAttachmentStateChanges() {
        this.add('ecs:SubmitAttachmentStateChanges');
        return this;
    }
    /**
     * Sent to acknowledge that a container changed states.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_SubmitContainerStateChange.html
     */
    submitContainerStateChange() {
        this.add('ecs:SubmitContainerStateChange');
        return this;
    }
    /**
     * Sent to acknowledge that a task changed states.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_SubmitTaskStateChange.html
     */
    submitTaskStateChange() {
        this.add('ecs:SubmitTaskStateChange');
        return this;
    }
    /**
     * Tags the specified resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_TagResource.html
     */
    tagResource() {
        this.add('ecs:TagResource');
        return this;
    }
    /**
     * Untags the specified resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_UntagResource.html
     */
    untagResource() {
        this.add('ecs:UntagResource');
        return this;
    }
    /**
     * Updates the Amazon ECS container agent on a specified container instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_UpdateContainerAgent.html
     */
    updateContainerAgent() {
        this.add('ecs:UpdateContainerAgent');
        return this;
    }
    /**
     * Enables the user to modify the status of an Amazon ECS container instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_UpdateContainerInstancesState.html
     */
    updateContainerInstancesState() {
        this.add('ecs:UpdateContainerInstancesState');
        return this;
    }
    /**
     * Modifies the desired count, deployment configuration, or task definition used in a service.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_UpdateService.html
     */
    updateService() {
        this.add('ecs:UpdateService');
        return this;
    }
    /**
     * Modifies the primary task set used in a service.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_UpdateServicePrimaryTaskSet.html
     */
    updateServicePrimaryTaskSet() {
        this.add('ecs:UpdateServicePrimaryTaskSet');
        return this;
    }
    /**
     * Updates the specified task set.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_UpdateTaskSet.html
     */
    updateTaskSet() {
        this.add('ecs:UpdateTaskSet');
        return this;
    }
    /**
     * Adds a resource of type cluster to the statement
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ECS_clusters.html
     *
     * @param clusterName - Identifier for the clusterName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     *  - ecs:ResourceTag/${TagKey}
     */
    onCluster(clusterName, account, region, partition) {
        var arn = 'arn:${Partition}:ecs:${Region}:${Account}:cluster/${ClusterName}';
        arn = arn.replace('${ClusterName}', clusterName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type container-instance to the statement
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ECS_instances.html
     *
     * @param containerInstanceId - Identifier for the containerInstanceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     *  - ecs:ResourceTag/${TagKey}
     */
    onContainerInstance(containerInstanceId, account, region, partition) {
        var arn = 'arn:${Partition}:ecs:${Region}:${Account}:container-instance/${ContainerInstanceId}';
        arn = arn.replace('${ContainerInstanceId}', containerInstanceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type service to the statement
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs_services.html
     *
     * @param serviceName - Identifier for the serviceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     *  - ecs:ResourceTag/${TagKey}
     */
    onService(serviceName, account, region, partition) {
        var arn = 'arn:${Partition}:ecs:${Region}:${Account}:service/${ServiceName}';
        arn = arn.replace('${ServiceName}', serviceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type task to the statement
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/developerguide/scheduling_tasks.html
     *
     * @param taskId - Identifier for the taskId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     *  - ecs:ResourceTag/${TagKey}
     */
    onTask(taskId, account, region, partition) {
        var arn = 'arn:${Partition}:ecs:${Region}:${Account}:task/${TaskId}';
        arn = arn.replace('${TaskId}', taskId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type task-definition to the statement
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_definitions.html
     *
     * @param taskDefinitionFamilyName - Identifier for the taskDefinitionFamilyName.
     * @param taskDefinitionRevisionNumber - Identifier for the taskDefinitionRevisionNumber.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     *  - ecs:ResourceTag/${TagKey}
     */
    onTaskDefinition(taskDefinitionFamilyName, taskDefinitionRevisionNumber, account, region, partition) {
        var arn = 'arn:${Partition}:ecs:${Region}:${Account}:task-definition/${TaskDefinitionFamilyName}:${TaskDefinitionRevisionNumber}';
        arn = arn.replace('${TaskDefinitionFamilyName}', taskDefinitionFamilyName);
        arn = arn.replace('${TaskDefinitionRevisionNumber}', taskDefinitionRevisionNumber);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type task-set to the statement
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_sets.html
     *
     * @param clusterName - Identifier for the clusterName.
     * @param serviceName - Identifier for the serviceName.
     * @param taskSetId - Identifier for the taskSetId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     *  - ecs:ResourceTag/${TagKey}
     */
    onTaskSet(clusterName, serviceName, taskSetId, account, region, partition) {
        var arn = 'arn:${Partition}:ecs:${Region}:${Account}:task-set/${ClusterName}/${ServiceName}/${TaskSetId}';
        arn = arn.replace('${ClusterName}', clusterName);
        arn = arn.replace('${ServiceName}', serviceName);
        arn = arn.replace('${TaskSetId}', taskSetId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTag(tagKey, value, operator) {
        return this.if(`ecs:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * The ARN of an ECS cluster.
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/developerguide/iam-policy-structure.html#amazon-ecs-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifCluster(value, operator) {
        return this.if(`ecs:cluster`, value, operator || 'ArnEquals');
    }
    /**
     * The ARN of an ECS container instance.
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/developerguide/iam-policy-structure.html#amazon-ecs-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifContainerInstances(value, operator) {
        return this.if(`ecs:container-instances`, value, operator || 'ArnEquals');
    }
    /**
     * The ARN of an ECS service.
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/developerguide/iam-policy-structure.html#amazon-ecs-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifService(value, operator) {
        return this.if(`ecs:service`, value, operator || 'ArnEquals');
    }
    /**
     * The ARN of an ECS task definition.
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/developerguide/iam-policy-structure.html#amazon-ecs-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifTaskDefinition(value, operator) {
        return this.if(`ecs:task-definition`, value, operator || 'ArnEquals');
    }
}
exports.Ecs = Ecs;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYW1hem9uZWxhc3RpY2NvbnRhaW5lcnNlcnZpY2UuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJhbWF6b25lbGFzdGljY29udGFpbmVyc2VydmljZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUFBLHNDQUFvRTtBQUdwRTs7OztHQUlHO0FBQ0gsTUFBYSxHQUFJLFNBQVEsd0JBQWU7SUF1a0J0Qzs7OztPQUlHO0lBQ0gsWUFBYSxLQUE0QjtRQUN2QyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUM7UUE1a0JSLGtCQUFhLEdBQUcsS0FBSyxDQUFDO1FBQ3RCLFlBQU8sR0FBWTtZQUN4QixlQUFlLEVBQUU7Z0JBQ2YsS0FBSyxFQUFFLGtGQUFrRjtnQkFDekYsYUFBYSxFQUFFLG1DQUFtQztnQkFDbEQsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLFlBQVksRUFBRTtvQkFDWiwwQkFBMEI7b0JBQzFCLGFBQWE7aUJBQ2Q7YUFDRjtZQUNELGVBQWUsRUFBRTtnQkFDZixLQUFLLEVBQUUsa0ZBQWtGO2dCQUN6RixhQUFhLEVBQUUsZ0ZBQWdGO2dCQUMvRixhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLFNBQVMsRUFBRTt3QkFDVCxVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLGFBQWE7b0JBQ2IscUJBQXFCO29CQUNyQiwwQkFBMEI7b0JBQzFCLGFBQWE7aUJBQ2Q7YUFDRjtZQUNELGVBQWUsRUFBRTtnQkFDZixLQUFLLEVBQUUsa0ZBQWtGO2dCQUN6RixhQUFhLEVBQUUsb0NBQW9DO2dCQUNuRCxhQUFhLEVBQUUsT0FBTztnQkFDdEIsWUFBWSxFQUFFO29CQUNaLGFBQWE7b0JBQ2IsYUFBYTtvQkFDYixxQkFBcUI7aUJBQ3RCO2FBQ0Y7WUFDRCxzQkFBc0IsRUFBRTtnQkFDdEIsS0FBSyxFQUFFLHlGQUF5RjtnQkFDaEcsYUFBYSxFQUFFLHdPQUF3TztnQkFDdlAsYUFBYSxFQUFFLE9BQU87YUFDdkI7WUFDRCxrQkFBa0IsRUFBRTtnQkFDbEIsS0FBSyxFQUFFLHFGQUFxRjtnQkFDNUYsYUFBYSxFQUFFLG9FQUFvRTtnQkFDbkYsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixvQkFBb0IsRUFBRTt3QkFDcEIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixhQUFhO2lCQUNkO2FBQ0Y7WUFDRCxlQUFlLEVBQUU7Z0JBQ2YsS0FBSyxFQUFFLGtGQUFrRjtnQkFDekYsYUFBYSxFQUFFLGdDQUFnQztnQkFDL0MsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixTQUFTLEVBQUU7d0JBQ1QsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxlQUFlLEVBQUU7Z0JBQ2YsS0FBSyxFQUFFLGtGQUFrRjtnQkFDekYsYUFBYSxFQUFFLCtDQUErQztnQkFDOUQsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixTQUFTLEVBQUU7d0JBQ1QsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixhQUFhO2lCQUNkO2FBQ0Y7WUFDRCxlQUFlLEVBQUU7Z0JBQ2YsS0FBSyxFQUFFLGtGQUFrRjtnQkFDekYsYUFBYSxFQUFFLGlDQUFpQztnQkFDaEQsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixVQUFVLEVBQUU7d0JBQ1YsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixhQUFhO29CQUNiLGFBQWE7aUJBQ2Q7YUFDRjtZQUNELDZCQUE2QixFQUFFO2dCQUM3QixLQUFLLEVBQUUsZ0dBQWdHO2dCQUN2RyxhQUFhLEVBQUUsMEVBQTBFO2dCQUN6RixhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLFNBQVMsRUFBRTt3QkFDVCxVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELDBCQUEwQixFQUFFO2dCQUMxQixLQUFLLEVBQUUsNkZBQTZGO2dCQUNwRyxhQUFhLEVBQUUsbUVBQW1FO2dCQUNsRixhQUFhLEVBQUUsT0FBTzthQUN2QjtZQUNELGtCQUFrQixFQUFFO2dCQUNsQixLQUFLLEVBQUUscUZBQXFGO2dCQUM1RixhQUFhLEVBQUUseUNBQXlDO2dCQUN4RCxhQUFhLEVBQUUsTUFBTTtnQkFDckIsZUFBZSxFQUFFO29CQUNmLFNBQVMsRUFBRTt3QkFDVCxVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELDRCQUE0QixFQUFFO2dCQUM1QixLQUFLLEVBQUUsK0ZBQStGO2dCQUN0RyxhQUFhLEVBQUUsMkNBQTJDO2dCQUMxRCxhQUFhLEVBQUUsTUFBTTtnQkFDckIsZUFBZSxFQUFFO29CQUNmLG9CQUFvQixFQUFFO3dCQUNwQixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLGFBQWE7aUJBQ2Q7YUFDRjtZQUNELGtCQUFrQixFQUFFO2dCQUNsQixLQUFLLEVBQUUscUZBQXFGO2dCQUM1RixhQUFhLEVBQUUsMkRBQTJEO2dCQUMxRSxhQUFhLEVBQUUsTUFBTTtnQkFDckIsZUFBZSxFQUFFO29CQUNmLFNBQVMsRUFBRTt3QkFDVCxVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLGFBQWE7aUJBQ2Q7YUFDRjtZQUNELHdCQUF3QixFQUFFO2dCQUN4QixLQUFLLEVBQUUsMkZBQTJGO2dCQUNsRyxhQUFhLEVBQUUsa05BQWtOO2dCQUNqTyxhQUFhLEVBQUUsTUFBTTthQUN0QjtZQUNELGtCQUFrQixFQUFFO2dCQUNsQixLQUFLLEVBQUUscUZBQXFGO2dCQUM1RixhQUFhLEVBQUUsaUNBQWlDO2dCQUNoRCxhQUFhLEVBQUUsTUFBTTtnQkFDckIsZUFBZSxFQUFFO29CQUNmLFVBQVUsRUFBRTt3QkFDVixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLGFBQWE7b0JBQ2IsYUFBYTtpQkFDZDthQUNGO1lBQ0QsZUFBZSxFQUFFO2dCQUNmLEtBQUssRUFBRSxrRkFBa0Y7Z0JBQ3pGLGFBQWEsRUFBRSxzQ0FBc0M7Z0JBQ3JELGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsTUFBTSxFQUFFO3dCQUNOLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjtnQkFDRCxZQUFZLEVBQUU7b0JBQ1osYUFBYTtpQkFDZDthQUNGO1lBQ0Qsc0JBQXNCLEVBQUU7Z0JBQ3RCLEtBQUssRUFBRSx5RkFBeUY7Z0JBQ2hHLGFBQWEsRUFBRSxtRUFBbUU7Z0JBQ2xGLGFBQWEsRUFBRSxPQUFPO2FBQ3ZCO1lBQ0QscUJBQXFCLEVBQUU7Z0JBQ3JCLEtBQUssRUFBRSx3RkFBd0Y7Z0JBQy9GLGFBQWEsRUFBRSxrRkFBa0Y7Z0JBQ2pHLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0QsZ0JBQWdCLEVBQUU7Z0JBQ2hCLEtBQUssRUFBRSxtRkFBbUY7Z0JBQzFGLGFBQWEsRUFBRSwyRkFBMkY7Z0JBQzFHLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsU0FBUyxFQUFFO3dCQUNULFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsY0FBYyxFQUFFO2dCQUNkLEtBQUssRUFBRSxpRkFBaUY7Z0JBQ3hGLGFBQWEsRUFBRSxzQ0FBc0M7Z0JBQ3JELGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0Qsd0JBQXdCLEVBQUU7Z0JBQ3hCLEtBQUssRUFBRSwyRkFBMkY7Z0JBQ2xHLGFBQWEsRUFBRSwrREFBK0Q7Z0JBQzlFLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsU0FBUyxFQUFFO3dCQUNULFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsY0FBYyxFQUFFO2dCQUNkLEtBQUssRUFBRSxpRkFBaUY7Z0JBQ3hGLGFBQWEsRUFBRSw2REFBNkQ7Z0JBQzVFLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixZQUFZLEVBQUU7b0JBQ1osYUFBYTtpQkFDZDthQUNGO1lBQ0QscUJBQXFCLEVBQUU7Z0JBQ3JCLEtBQUssRUFBRSx3RkFBd0Y7Z0JBQy9GLGFBQWEsRUFBRSx1Q0FBdUM7Z0JBQ3RELGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsU0FBUyxFQUFFO3dCQUNULFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtvQkFDRCxvQkFBb0IsRUFBRTt3QkFDcEIsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO29CQUNELE1BQU0sRUFBRTt3QkFDTixVQUFVLEVBQUUsS0FBSztxQkFDbEI7b0JBQ0QsaUJBQWlCLEVBQUU7d0JBQ2pCLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtpQkFDRjthQUNGO1lBQ0QsNEJBQTRCLEVBQUU7Z0JBQzVCLEtBQUssRUFBRSxpRkFBaUY7Z0JBQ3hGLGFBQWEsRUFBRSw4S0FBOEs7Z0JBQzdMLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0QscUJBQXFCLEVBQUU7Z0JBQ3JCLEtBQUssRUFBRSx3RkFBd0Y7Z0JBQy9GLGFBQWEsRUFBRSx5RUFBeUU7Z0JBQ3hGLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0QsV0FBVyxFQUFFO2dCQUNYLEtBQUssRUFBRSw4RUFBOEU7Z0JBQ3JGLGFBQWEsRUFBRSxrREFBa0Q7Z0JBQ2pFLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2Ysb0JBQW9CLEVBQUU7d0JBQ3BCLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjtnQkFDRCxZQUFZLEVBQUU7b0JBQ1osYUFBYTtpQkFDZDthQUNGO1lBQ0QsTUFBTSxFQUFFO2dCQUNOLEtBQUssRUFBRSxvRkFBb0Y7Z0JBQzNGLGFBQWEsRUFBRSx5R0FBeUc7Z0JBQ3hILGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2Ysb0JBQW9CLEVBQUU7d0JBQ3BCLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjtnQkFDRCxZQUFZLEVBQUU7b0JBQ1osYUFBYTtpQkFDZDthQUNGO1lBQ0QsbUJBQW1CLEVBQUU7Z0JBQ25CLEtBQUssRUFBRSxzRkFBc0Y7Z0JBQzdGLGFBQWEsRUFBRSxrVUFBa1U7Z0JBQ2pWLGFBQWEsRUFBRSxPQUFPO2FBQ3ZCO1lBQ0QsMEJBQTBCLEVBQUU7Z0JBQzFCLEtBQUssRUFBRSw2RkFBNkY7Z0JBQ3BHLGFBQWEsRUFBRSw2T0FBNk87Z0JBQzVQLGFBQWEsRUFBRSxPQUFPO2FBQ3ZCO1lBQ0QsZUFBZSxFQUFFO2dCQUNmLEtBQUssRUFBRSxrRkFBa0Y7Z0JBQ3pGLGFBQWEsRUFBRSwwREFBMEQ7Z0JBQ3pFLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2Ysb0JBQW9CLEVBQUU7d0JBQ3BCLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjtnQkFDRCxZQUFZLEVBQUU7b0JBQ1osYUFBYTtpQkFDZDthQUNGO1lBQ0QsMkJBQTJCLEVBQUU7Z0JBQzNCLEtBQUssRUFBRSw4RkFBOEY7Z0JBQ3JHLGFBQWEsRUFBRSx1REFBdUQ7Z0JBQ3RFLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsU0FBUyxFQUFFO3dCQUNULFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjtnQkFDRCxZQUFZLEVBQUU7b0JBQ1osMEJBQTBCO29CQUMxQixhQUFhO2lCQUNkO2FBQ0Y7WUFDRCx3QkFBd0IsRUFBRTtnQkFDeEIsS0FBSyxFQUFFLDJGQUEyRjtnQkFDbEcsYUFBYSxFQUFFLG9GQUFvRjtnQkFDbkcsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLFlBQVksRUFBRTtvQkFDWiwwQkFBMEI7b0JBQzFCLGFBQWE7aUJBQ2Q7YUFDRjtZQUNELFNBQVMsRUFBRTtnQkFDVCxLQUFLLEVBQUUsNEVBQTRFO2dCQUNuRixhQUFhLEVBQUUsMkVBQTJFO2dCQUMxRixhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLGlCQUFpQixFQUFFO3dCQUNqQixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLGFBQWE7b0JBQ2IsMEJBQTBCO29CQUMxQixhQUFhO2lCQUNkO2FBQ0Y7WUFDRCxXQUFXLEVBQUU7Z0JBQ1gsS0FBSyxFQUFFLDhFQUE4RTtnQkFDckYsYUFBYSxFQUFFLHdHQUF3RztnQkFDdkgsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixpQkFBaUIsRUFBRTt3QkFDakIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixhQUFhO29CQUNiLHlCQUF5QjtvQkFDekIsMEJBQTBCO29CQUMxQixhQUFhO2lCQUNkO2FBQ0Y7WUFDRCx1QkFBdUIsRUFBRTtnQkFDdkIsS0FBSyxFQUFFLHVHQUF1RztnQkFDOUcsYUFBYSxFQUFFLGlEQUFpRDtnQkFDaEUsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixvQkFBb0IsRUFBRTt3QkFDcEIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixhQUFhO2lCQUNkO2FBQ0Y7WUFDRCxVQUFVLEVBQUU7Z0JBQ1YsS0FBSyxFQUFFLDZFQUE2RTtnQkFDcEYsYUFBYSxFQUFFLHVCQUF1QjtnQkFDdEMsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixNQUFNLEVBQUU7d0JBQ04sVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixhQUFhO2lCQUNkO2FBQ0Y7WUFDRCw4QkFBOEIsRUFBRTtnQkFDOUIsS0FBSyxFQUFFLGlHQUFpRztnQkFDeEcsYUFBYSxFQUFFLHNEQUFzRDtnQkFDckUsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixTQUFTLEVBQUU7d0JBQ1QsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCw0QkFBNEIsRUFBRTtnQkFDNUIsS0FBSyxFQUFFLCtGQUErRjtnQkFDdEcsYUFBYSxFQUFFLHNEQUFzRDtnQkFDckUsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixTQUFTLEVBQUU7d0JBQ1QsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCx1QkFBdUIsRUFBRTtnQkFDdkIsS0FBSyxFQUFFLDBGQUEwRjtnQkFDakcsYUFBYSxFQUFFLGlEQUFpRDtnQkFDaEUsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixTQUFTLEVBQUU7d0JBQ1QsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxhQUFhLEVBQUU7Z0JBQ2IsS0FBSyxFQUFFLGdGQUFnRjtnQkFDdkYsYUFBYSxFQUFFLDhCQUE4QjtnQkFDN0MsYUFBYSxFQUFFLFNBQVM7Z0JBQ3hCLGVBQWUsRUFBRTtvQkFDZixTQUFTLEVBQUU7d0JBQ1QsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO29CQUNELG9CQUFvQixFQUFFO3dCQUNwQixVQUFVLEVBQUUsS0FBSztxQkFDbEI7b0JBQ0QsU0FBUyxFQUFFO3dCQUNULFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtvQkFDRCxNQUFNLEVBQUU7d0JBQ04sVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO29CQUNELGlCQUFpQixFQUFFO3dCQUNqQixVQUFVLEVBQUUsS0FBSztxQkFDbEI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLGFBQWE7b0JBQ2IsMEJBQTBCO2lCQUMzQjthQUNGO1lBQ0QsZUFBZSxFQUFFO2dCQUNmLEtBQUssRUFBRSxrRkFBa0Y7Z0JBQ3pGLGFBQWEsRUFBRSxnQ0FBZ0M7Z0JBQy9DLGFBQWEsRUFBRSxTQUFTO2dCQUN4QixlQUFlLEVBQUU7b0JBQ2YsU0FBUyxFQUFFO3dCQUNULFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtvQkFDRCxvQkFBb0IsRUFBRTt3QkFDcEIsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO29CQUNELFNBQVMsRUFBRTt3QkFDVCxVQUFVLEVBQUUsS0FBSztxQkFDbEI7b0JBQ0QsTUFBTSxFQUFFO3dCQUNOLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtvQkFDRCxpQkFBaUIsRUFBRTt3QkFDakIsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixhQUFhO2lCQUNkO2FBQ0Y7WUFDRCxzQkFBc0IsRUFBRTtnQkFDdEIsS0FBSyxFQUFFLHlGQUF5RjtnQkFDaEcsYUFBYSxFQUFFLDJFQUEyRTtnQkFDMUYsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixvQkFBb0IsRUFBRTt3QkFDcEIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixhQUFhO2lCQUNkO2FBQ0Y7WUFDRCwrQkFBK0IsRUFBRTtnQkFDL0IsS0FBSyxFQUFFLGtHQUFrRztnQkFDekcsYUFBYSxFQUFFLDRFQUE0RTtnQkFDM0YsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixvQkFBb0IsRUFBRTt3QkFDcEIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixhQUFhO2lCQUNkO2FBQ0Y7WUFDRCxlQUFlLEVBQUU7Z0JBQ2YsS0FBSyxFQUFFLGtGQUFrRjtnQkFDekYsYUFBYSxFQUFFLDZGQUE2RjtnQkFDNUcsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixTQUFTLEVBQUU7d0JBQ1QsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixhQUFhO29CQUNiLHFCQUFxQjtpQkFDdEI7YUFDRjtZQUNELDZCQUE2QixFQUFFO2dCQUM3QixLQUFLLEVBQUUsZ0dBQWdHO2dCQUN2RyxhQUFhLEVBQUUsa0RBQWtEO2dCQUNqRSxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLFNBQVMsRUFBRTt3QkFDVCxVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLGFBQWE7aUJBQ2Q7YUFDRjtZQUNELGVBQWUsRUFBRTtnQkFDZixLQUFLLEVBQUUsa0ZBQWtGO2dCQUN6RixhQUFhLEVBQUUsaUNBQWlDO2dCQUNoRCxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLFVBQVUsRUFBRTt3QkFDVixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLGFBQWE7b0JBQ2IsYUFBYTtpQkFDZDthQUNGO1NBQ0YsQ0FBQztRQUNLLGtCQUFhLEdBQWtCO1lBQ3BDLFNBQVMsRUFBRTtnQkFDVCxNQUFNLEVBQUUsU0FBUztnQkFDakIsS0FBSyxFQUFFLCtFQUErRTtnQkFDdEYsS0FBSyxFQUFFLGtFQUFrRTtnQkFDekUsZUFBZSxFQUFFO29CQUNmLDJCQUEyQjtvQkFDM0IsMkJBQTJCO2lCQUM1QjthQUNGO1lBQ0Qsb0JBQW9CLEVBQUU7Z0JBQ3BCLE1BQU0sRUFBRSxvQkFBb0I7Z0JBQzVCLEtBQUssRUFBRSxnRkFBZ0Y7Z0JBQ3ZGLEtBQUssRUFBRSxxRkFBcUY7Z0JBQzVGLGVBQWUsRUFBRTtvQkFDZiwyQkFBMkI7b0JBQzNCLDJCQUEyQjtpQkFDNUI7YUFDRjtZQUNELFNBQVMsRUFBRTtnQkFDVCxNQUFNLEVBQUUsU0FBUztnQkFDakIsS0FBSyxFQUFFLCtFQUErRTtnQkFDdEYsS0FBSyxFQUFFLGtFQUFrRTtnQkFDekUsZUFBZSxFQUFFO29CQUNmLDJCQUEyQjtvQkFDM0IsMkJBQTJCO2lCQUM1QjthQUNGO1lBQ0QsTUFBTSxFQUFFO2dCQUNOLE1BQU0sRUFBRSxNQUFNO2dCQUNkLEtBQUssRUFBRSxtRkFBbUY7Z0JBQzFGLEtBQUssRUFBRSwwREFBMEQ7Z0JBQ2pFLGVBQWUsRUFBRTtvQkFDZiwyQkFBMkI7b0JBQzNCLDJCQUEyQjtpQkFDNUI7YUFDRjtZQUNELGlCQUFpQixFQUFFO2dCQUNqQixNQUFNLEVBQUUsaUJBQWlCO2dCQUN6QixLQUFLLEVBQUUsbUZBQW1GO2dCQUMxRixLQUFLLEVBQUUsdUhBQXVIO2dCQUM5SCxlQUFlLEVBQUU7b0JBQ2YsMkJBQTJCO29CQUMzQiwyQkFBMkI7aUJBQzVCO2FBQ0Y7WUFDRCxVQUFVLEVBQUU7Z0JBQ1YsTUFBTSxFQUFFLFVBQVU7Z0JBQ2xCLEtBQUssRUFBRSw0RUFBNEU7Z0JBQ25GLEtBQUssRUFBRSwrRkFBK0Y7Z0JBQ3RHLGVBQWUsRUFBRTtvQkFDZiwyQkFBMkI7b0JBQzNCLDJCQUEyQjtpQkFDNUI7YUFDRjtTQUNGLENBQUM7SUFTRixDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsR0FBRyxDQUFDLG1CQUFtQixDQUFDLENBQUM7UUFDOUIsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsR0FBRyxDQUFDLG1CQUFtQixDQUFDLENBQUM7UUFDOUIsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsR0FBRyxDQUFDLG1CQUFtQixDQUFDLENBQUM7UUFDOUIsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0JBQW9CO1FBQ3pCLElBQUksQ0FBQyxHQUFHLENBQUMsMEJBQTBCLENBQUMsQ0FBQztRQUNyQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO1FBQ2pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGFBQWE7UUFDbEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO1FBQzlCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGFBQWE7UUFDbEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO1FBQzlCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGFBQWE7UUFDbEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO1FBQzlCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDJCQUEyQjtRQUNoQyxJQUFJLENBQUMsR0FBRyxDQUFDLGlDQUFpQyxDQUFDLENBQUM7UUFDNUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksd0JBQXdCO1FBQzdCLElBQUksQ0FBQyxHQUFHLENBQUMsOEJBQThCLENBQUMsQ0FBQztRQUN6QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO1FBQ2pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDBCQUEwQjtRQUMvQixJQUFJLENBQUMsR0FBRyxDQUFDLGdDQUFnQyxDQUFDLENBQUM7UUFDM0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0JBQWdCO1FBQ3JCLElBQUksQ0FBQyxHQUFHLENBQUMsc0JBQXNCLENBQUMsQ0FBQztRQUNqQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxzQkFBc0I7UUFDM0IsSUFBSSxDQUFDLEdBQUcsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixJQUFJLENBQUMsR0FBRyxDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFDakMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsR0FBRyxDQUFDLG1CQUFtQixDQUFDLENBQUM7UUFDOUIsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0JBQW9CO1FBQ3pCLElBQUksQ0FBQyxHQUFHLENBQUMsMEJBQTBCLENBQUMsQ0FBQztRQUNyQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO1FBQ3BDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGNBQWM7UUFDbkIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO1FBQy9CLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFlBQVk7UUFDakIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO1FBQzdCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHNCQUFzQjtRQUMzQixJQUFJLENBQUMsR0FBRyxDQUFDLDRCQUE0QixDQUFDLENBQUM7UUFDdkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksWUFBWTtRQUNqQixJQUFJLENBQUMsR0FBRyxDQUFDLGtCQUFrQixDQUFDLENBQUM7UUFDN0IsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLElBQUksQ0FBQyxHQUFHLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwwQkFBMEI7UUFDL0IsSUFBSSxDQUFDLEdBQUcsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO1FBQzNDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQjtRQUN4QixJQUFJLENBQUMsR0FBRyxDQUFDLHlCQUF5QixDQUFDLENBQUM7UUFDcEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksU0FBUztRQUNkLElBQUksQ0FBQyxHQUFHLENBQUMsZUFBZSxDQUFDLENBQUM7UUFDMUIsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksSUFBSTtRQUNULElBQUksQ0FBQyxHQUFHLENBQUMsVUFBVSxDQUFDLENBQUM7UUFDckIsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksaUJBQWlCO1FBQ3RCLElBQUksQ0FBQyxHQUFHLENBQUMsdUJBQXVCLENBQUMsQ0FBQztRQUNsQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx3QkFBd0I7UUFDN0IsSUFBSSxDQUFDLEdBQUcsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO1FBQ3pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGFBQWE7UUFDbEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO1FBQzlCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHlCQUF5QjtRQUM5QixJQUFJLENBQUMsR0FBRyxDQUFDLCtCQUErQixDQUFDLENBQUM7UUFDMUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksc0JBQXNCO1FBQzNCLElBQUksQ0FBQyxHQUFHLENBQUMsNEJBQTRCLENBQUMsQ0FBQztRQUN2QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxPQUFPO1FBQ1osSUFBSSxDQUFDLEdBQUcsQ0FBQyxhQUFhLENBQUMsQ0FBQztRQUN4QixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxTQUFTO1FBQ2QsSUFBSSxDQUFDLEdBQUcsQ0FBQyxlQUFlLENBQUMsQ0FBQztRQUMxQixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsSUFBSSxDQUFDLEdBQUcsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO1FBQ3RDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFFBQVE7UUFDYixJQUFJLENBQUMsR0FBRyxDQUFDLGNBQWMsQ0FBQyxDQUFDO1FBQ3pCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDRCQUE0QjtRQUNqQyxJQUFJLENBQUMsR0FBRyxDQUFDLGtDQUFrQyxDQUFDLENBQUM7UUFDN0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMEJBQTBCO1FBQy9CLElBQUksQ0FBQyxHQUFHLENBQUMsZ0NBQWdDLENBQUMsQ0FBQztRQUMzQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsSUFBSSxDQUFDLEdBQUcsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO1FBQ3RDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFdBQVc7UUFDaEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO1FBQzVCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGFBQWE7UUFDbEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO1FBQzlCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG9CQUFvQjtRQUN6QixJQUFJLENBQUMsR0FBRyxDQUFDLDBCQUEwQixDQUFDLENBQUM7UUFDckMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksNkJBQTZCO1FBQ2xDLElBQUksQ0FBQyxHQUFHLENBQUMsbUNBQW1DLENBQUMsQ0FBQztRQUM5QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxhQUFhO1FBQ2xCLElBQUksQ0FBQyxHQUFHLENBQUMsbUJBQW1CLENBQUMsQ0FBQztRQUM5QixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwyQkFBMkI7UUFDaEMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxpQ0FBaUMsQ0FBQyxDQUFDO1FBQzVDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGFBQWE7UUFDbEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO1FBQzlCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSSxTQUFTLENBQUMsV0FBbUIsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUN6RixJQUFJLEdBQUcsR0FBRyxrRUFBa0UsQ0FBQztRQUM3RSxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxnQkFBZ0IsRUFBRSxXQUFXLENBQUMsQ0FBQztRQUNqRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsT0FBTyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQ2hELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxNQUFNLElBQUksR0FBRyxDQUFDLENBQUM7UUFDOUMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSSxtQkFBbUIsQ0FBQyxtQkFBMkIsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUMzRyxJQUFJLEdBQUcsR0FBRyxxRkFBcUYsQ0FBQztRQUNoRyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyx3QkFBd0IsRUFBRSxtQkFBbUIsQ0FBQyxDQUFDO1FBQ2pFLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7T0FhRztJQUNJLFNBQVMsQ0FBQyxXQUFtQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3pGLElBQUksR0FBRyxHQUFHLGtFQUFrRSxDQUFDO1FBQzdFLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGdCQUFnQixFQUFFLFdBQVcsQ0FBQyxDQUFDO1FBQ2pELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7T0FhRztJQUNJLE1BQU0sQ0FBQyxNQUFjLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDakYsSUFBSSxHQUFHLEdBQUcsMERBQTBELENBQUM7UUFDckUsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBQ3ZDLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7O09BY0c7SUFDSSxnQkFBZ0IsQ0FBQyx3QkFBZ0MsRUFBRSw0QkFBb0MsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNuSixJQUFJLEdBQUcsR0FBRyx1SEFBdUgsQ0FBQztRQUNsSSxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyw2QkFBNkIsRUFBRSx3QkFBd0IsQ0FBQyxDQUFDO1FBQzNFLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGlDQUFpQyxFQUFFLDRCQUE0QixDQUFDLENBQUM7UUFDbkYsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsWUFBWSxFQUFFLE9BQU8sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUNoRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxXQUFXLEVBQUUsTUFBTSxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQzlDLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGNBQWMsRUFBRSxTQUFTLElBQUksS0FBSyxDQUFDLENBQUM7UUFDdEQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ3RCLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7O09BZUc7SUFDSSxTQUFTLENBQUMsV0FBbUIsRUFBRSxXQUFtQixFQUFFLFNBQWlCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDakksSUFBSSxHQUFHLEdBQUcsK0ZBQStGLENBQUM7UUFDMUcsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsZ0JBQWdCLEVBQUUsV0FBVyxDQUFDLENBQUM7UUFDakQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsZ0JBQWdCLEVBQUUsV0FBVyxDQUFDLENBQUM7UUFDakQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQzdDLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLGFBQWEsQ0FBQyxNQUFjLEVBQUUsS0FBd0IsRUFBRSxRQUFpQjtRQUM5RSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW9CLE1BQU8sRUFBRSxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDakYsQ0FBQztJQUVEOzs7Ozs7O09BT0c7SUFDSSxTQUFTLENBQUMsS0FBd0IsRUFBRSxRQUFpQjtRQUMxRCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsYUFBYSxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksV0FBVyxDQUFDLENBQUM7SUFDaEUsQ0FBQztJQUVEOzs7Ozs7O09BT0c7SUFDSSxvQkFBb0IsQ0FBQyxLQUF3QixFQUFFLFFBQWlCO1FBQ3JFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx5QkFBeUIsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFdBQVcsQ0FBQyxDQUFDO0lBQzVFLENBQUM7SUFFRDs7Ozs7OztPQU9HO0lBQ0ksU0FBUyxDQUFDLEtBQXdCLEVBQUUsUUFBaUI7UUFDMUQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGFBQWEsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFdBQVcsQ0FBQyxDQUFDO0lBQ2hFLENBQUM7SUFFRDs7Ozs7OztPQU9HO0lBQ0ksZ0JBQWdCLENBQUMsS0FBd0IsRUFBRSxRQUFpQjtRQUNqRSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxXQUFXLENBQUMsQ0FBQztJQUN4RSxDQUFDO0NBQ0Y7QUFoMENELGtCQWcwQ0MiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBBY3Rpb25zLCBQb2xpY3lTdGF0ZW1lbnQsIFJlc291cmNlVHlwZXMgfSBmcm9tIFwiLi4vc2hhcmVkXCI7XG5pbXBvcnQgeyBQb2xpY3lTdGF0ZW1lbnRQcm9wcyB9IGZyb20gXCJAYXdzLWNkay9hd3MtaWFtXCI7XG5cbi8qKlxuICogQWN0aW9uIHByb3ZpZGVyIGZvciBzZXJ2aWNlIGVjc1xuICpcbiAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9saXN0X2FtYXpvbmVsYXN0aWNjb250YWluZXJzZXJ2aWNlLmh0bWxcbiAqL1xuZXhwb3J0IGNsYXNzIEVjcyBleHRlbmRzIFBvbGljeVN0YXRlbWVudCB7XG4gIHB1YmxpYyBzZXJ2aWNlUHJlZml4ID0gJ2Vjcyc7XG4gIHB1YmxpYyBhY3Rpb25zOiBBY3Rpb25zID0ge1xuICAgIFwiQ3JlYXRlQ2x1c3RlclwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25FQ1MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlQ2x1c3Rlci5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQ3JlYXRlcyBhIG5ldyBBbWF6b24gRUNTIGNsdXN0ZXIuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiYXdzOlJlcXVlc3RUYWcvJHtUYWdLZXl9XCIsXG4gICAgICAgIFwiYXdzOlRhZ0tleXNcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJDcmVhdGVTZXJ2aWNlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkVDUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVTZXJ2aWNlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSdW5zIGFuZCBtYWludGFpbnMgYSBkZXNpcmVkIG51bWJlciBvZiB0YXNrcyBmcm9tIGEgc3BlY2lmaWVkIHRhc2sgZGVmaW5pdGlvbi5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzZXJ2aWNlXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiZWNzOmNsdXN0ZXJcIixcbiAgICAgICAgXCJlY3M6dGFzay1kZWZpbml0aW9uXCIsXG4gICAgICAgIFwiYXdzOlJlcXVlc3RUYWcvJHtUYWdLZXl9XCIsXG4gICAgICAgIFwiYXdzOlRhZ0tleXNcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJDcmVhdGVUYXNrU2V0XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkVDUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVUYXNrU2V0Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJDcmVhdGVzIGEgbmV3IEFtYXpvbiBFQ1MgdGFzayBzZXQuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiZWNzOmNsdXN0ZXJcIixcbiAgICAgICAgXCJlY3M6c2VydmljZVwiLFxuICAgICAgICBcImVjczp0YXNrLWRlZmluaXRpb25cIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJEZWxldGVBY2NvdW50U2V0dGluZ1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25FQ1MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlQWNjb3VudFNldHRpbmcuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIk1vZGlmaWVzIHRoZSBBUk4gYW5kIHJlc291cmNlIElEIGZvcm1hdCBvZiBhIHJlc291cmNlIGZvciBhIHNwZWNpZmllZCBJQU0gdXNlciwgSUFNIHJvbGUsIG9yIHRoZSByb290IHVzZXIgZm9yIGFuIGFjY291bnQuIFlvdSBjYW4gc3BlY2lmeSB3aGV0aGVyIHRoZSBuZXcgQVJOIGFuZCByZXNvdXJjZSBJRCBmb3JtYXQgYXJlIGRpc2FibGVkIGZvciBuZXcgcmVzb3VyY2VzIHRoYXQgYXJlIGNyZWF0ZWQuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIlxuICAgIH0sXG4gICAgXCJEZWxldGVBdHRyaWJ1dGVzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkVDUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVBdHRyaWJ1dGVzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJEZWxldGVzIG9uZSBvciBtb3JlIGN1c3RvbSBhdHRyaWJ1dGVzIGZyb20gYW4gQW1hem9uIEVDUyByZXNvdXJjZS5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJjb250YWluZXItaW5zdGFuY2VcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJlY3M6Y2x1c3RlclwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIkRlbGV0ZUNsdXN0ZXJcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZUNsdXN0ZXIuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkRlbGV0ZXMgdGhlIHNwZWNpZmllZCBjbHVzdGVyLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImNsdXN0ZXJcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlbGV0ZVNlcnZpY2VcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZVNlcnZpY2UuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkRlbGV0ZXMgYSBzcGVjaWZpZWQgc2VydmljZSB3aXRoaW4gYSBjbHVzdGVyLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInNlcnZpY2VcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJlY3M6Y2x1c3RlclwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIkRlbGV0ZVRhc2tTZXRcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZVRhc2tTZXQuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkRlbGV0ZXMgdGhlIHNwZWNpZmllZCB0YXNrIHNldC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJ0YXNrLXNldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcImVjczpjbHVzdGVyXCIsXG4gICAgICAgIFwiZWNzOnNlcnZpY2VcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJEZXJlZ2lzdGVyQ29udGFpbmVySW5zdGFuY2VcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlcmVnaXN0ZXJDb250YWluZXJJbnN0YW5jZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVyZWdpc3RlcnMgYW4gQW1hem9uIEVDUyBjb250YWluZXIgaW5zdGFuY2UgZnJvbSB0aGUgc3BlY2lmaWVkIGNsdXN0ZXIuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiY2x1c3RlclwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVyZWdpc3RlclRhc2tEZWZpbml0aW9uXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkVDUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXJlZ2lzdGVyVGFza0RlZmluaXRpb24uaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkRlcmVnaXN0ZXJzIHRoZSBzcGVjaWZpZWQgdGFzayBkZWZpbml0aW9uIGJ5IGZhbWlseSBhbmQgcmV2aXNpb24uXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIlxuICAgIH0sXG4gICAgXCJEZXNjcmliZUNsdXN0ZXJzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkVDUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZUNsdXN0ZXJzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJEZXNjcmliZXMgb25lIG9yIG1vcmUgb2YgeW91ciBjbHVzdGVycy5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImNsdXN0ZXJcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlc2NyaWJlQ29udGFpbmVySW5zdGFuY2VzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkVDUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZUNvbnRhaW5lckluc3RhbmNlcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVzY3JpYmVzIEFtYXpvbiBFQ1MgY29udGFpbmVyIGluc3RhbmNlcy5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImNvbnRhaW5lci1pbnN0YW5jZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcImVjczpjbHVzdGVyXCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiRGVzY3JpYmVTZXJ2aWNlc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25FQ1MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVTZXJ2aWNlcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVzY3JpYmVzIHRoZSBzcGVjaWZpZWQgc2VydmljZXMgcnVubmluZyBpbiB5b3VyIGNsdXN0ZXIuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzZXJ2aWNlXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiZWNzOmNsdXN0ZXJcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJEZXNjcmliZVRhc2tEZWZpbml0aW9uXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkVDUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZVRhc2tEZWZpbml0aW9uLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJEZXNjcmliZXMgYSB0YXNrIGRlZmluaXRpb24uIFlvdSBjYW4gc3BlY2lmeSBhIGZhbWlseSBhbmQgcmV2aXNpb24gdG8gZmluZCBpbmZvcm1hdGlvbiBhYm91dCBhIHNwZWNpZmljIHRhc2sgZGVmaW5pdGlvbiwgb3IgeW91IGNhbiBzaW1wbHkgc3BlY2lmeSB0aGUgZmFtaWx5IHRvIGZpbmQgdGhlIGxhdGVzdCBBQ1RJVkUgcmV2aXNpb24gaW4gdGhhdCBmYW1pbHkuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiXG4gICAgfSxcbiAgICBcIkRlc2NyaWJlVGFza1NldHNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlVGFza1NldHMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkRlc2NyaWJlcyBBbWF6b24gRUNTIHRhc2sgc2V0cy5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInRhc2stc2V0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiZWNzOmNsdXN0ZXJcIixcbiAgICAgICAgXCJlY3M6c2VydmljZVwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIkRlc2NyaWJlVGFza3NcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlVGFza3MuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkRlc2NyaWJlcyBhIHNwZWNpZmllZCB0YXNrIG9yIHRhc2tzLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwidGFza1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcImVjczpjbHVzdGVyXCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiRGlzY292ZXJQb2xsRW5kcG9pbnRcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rpc2NvdmVyUG9sbEVuZHBvaW50Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZXR1cm5zIGFuIGVuZHBvaW50IGZvciB0aGUgQW1hem9uIEVDUyBhZ2VudCB0byBwb2xsIGZvciB1cGRhdGVzLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCJcbiAgICB9LFxuICAgIFwiTGlzdEFjY291bnRTZXR0aW5nc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25FQ1MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdEFjY291bnRTZXR0aW5ncy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiTGlzdHMgdGhlIGFjY291bnQgc2V0dGluZ3MgZm9yIGFuIEFtYXpvbiBFQ1MgcmVzb3VyY2UgZm9yIGEgc3BlY2lmaWVkIHByaW5jaXBhbC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCJcbiAgICB9LFxuICAgIFwiTGlzdEF0dHJpYnV0ZXNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RBdHRyaWJ1dGVzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJMaXN0cyB0aGUgYXR0cmlidXRlcyBmb3IgQW1hem9uIEVDUyByZXNvdXJjZXMgd2l0aGluIGEgc3BlY2lmaWVkIHRhcmdldCB0eXBlIGFuZCBjbHVzdGVyLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiY2x1c3RlclwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiTGlzdENsdXN0ZXJzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkVDUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0Q2x1c3RlcnMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHVybnMgYSBsaXN0IG9mIGV4aXN0aW5nIGNsdXN0ZXJzLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIlxuICAgIH0sXG4gICAgXCJMaXN0Q29udGFpbmVySW5zdGFuY2VzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkVDUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0Q29udGFpbmVySW5zdGFuY2VzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZXR1cm5zIGEgbGlzdCBvZiBjb250YWluZXIgaW5zdGFuY2VzIGluIGEgc3BlY2lmaWVkIGNsdXN0ZXIuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJjbHVzdGVyXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJMaXN0U2VydmljZXNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RTZXJ2aWNlcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiTGlzdHMgdGhlIHNlcnZpY2VzIHRoYXQgYXJlIHJ1bm5pbmcgaW4gYSBzcGVjaWZpZWQgY2x1c3Rlci5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCIsXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcImVjczpjbHVzdGVyXCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiTGlzdFRhZ3NGb3JSZXNvdXJjZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25FQ1MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdFRhZ3NGb3JSZXNvdXJjZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiTGlzdCB0YWdzIGZvciB0aGUgc3BlY2lmaWVkIHJlc291cmNlLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiY2x1c3RlclwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9LFxuICAgICAgICBcImNvbnRhaW5lci1pbnN0YW5jZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9LFxuICAgICAgICBcInRhc2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAgXCJ0YXNrLWRlZmluaXRpb25cIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJMaXN0VGFza0RlZmluaXRpb25GYW1pbGllc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25FQ1MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdFNlcnZpY2VzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZXR1cm5zIGEgbGlzdCBvZiB0YXNrIGRlZmluaXRpb24gZmFtaWxpZXMgdGhhdCBhcmUgcmVnaXN0ZXJlZCB0byB5b3VyIGFjY291bnQgKHdoaWNoIG1heSBpbmNsdWRlIHRhc2sgZGVmaW5pdGlvbiBmYW1pbGllcyB0aGF0IG5vIGxvbmdlciBoYXZlIGFueSBBQ1RJVkUgdGFzayBkZWZpbml0aW9ucykuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiXG4gICAgfSxcbiAgICBcIkxpc3RUYXNrRGVmaW5pdGlvbnNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RUYXNrRGVmaW5pdGlvbnMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHVybnMgYSBsaXN0IG9mIHRhc2sgZGVmaW5pdGlvbnMgdGhhdCBhcmUgcmVnaXN0ZXJlZCB0byB5b3VyIGFjY291bnQuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiXG4gICAgfSxcbiAgICBcIkxpc3RUYXNrc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25FQ1MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdFRhc2tzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZXR1cm5zIGEgbGlzdCBvZiB0YXNrcyBmb3IgYSBzcGVjaWZpZWQgY2x1c3Rlci5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImNvbnRhaW5lci1pbnN0YW5jZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcImVjczpjbHVzdGVyXCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiUG9sbFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25FQ1MvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL2luc3RhbmNlX0lBTV9yb2xlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBhbiBhZ2VudCB0byBjb25uZWN0IHdpdGggdGhlIEFtYXpvbiBFQ1Mgc2VydmljZSB0byByZXBvcnQgc3RhdHVzIGFuZCBnZXQgY29tbWFuZHMuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiY29udGFpbmVyLWluc3RhbmNlXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiZWNzOmNsdXN0ZXJcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJQdXRBY2NvdW50U2V0dGluZ1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25FQ1MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUHV0QWNjb3VudFNldHRpbmcuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIk1vZGlmaWVzIHRoZSBBUk4gYW5kIHJlc291cmNlIElEIGZvcm1hdCBvZiBhIHJlc291cmNlIGZvciBhIHNwZWNpZmllZCBJQU0gdXNlciwgSUFNIHJvbGUsIG9yIHRoZSByb290IHVzZXIgZm9yIGFuIGFjY291bnQuIFlvdSBjYW4gc3BlY2lmeSB3aGV0aGVyIHRoZSBuZXcgQVJOIGFuZCByZXNvdXJjZSBJRCBmb3JtYXQgYXJlIGVuYWJsZWQgZm9yIG5ldyByZXNvdXJjZXMgdGhhdCBhcmUgY3JlYXRlZC4gRW5hYmxpbmcgdGhpcyBzZXR0aW5nIGlzIHJlcXVpcmVkIHRvIHVzZSBuZXcgQW1hem9uIEVDUyBmZWF0dXJlcyBzdWNoIGFzIHJlc291cmNlIHRhZ2dpbmcuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIlxuICAgIH0sXG4gICAgXCJQdXRBY2NvdW50U2V0dGluZ0RlZmF1bHRcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1B1dEFjY291bnRTZXR0aW5nRGVmYXVsdC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiTW9kaWZpZXMgdGhlIEFSTiBhbmQgcmVzb3VyY2UgSUQgZm9ybWF0IG9mIGEgcmVzb3VyY2UgdHlwZSBmb3IgYWxsIElBTSB1c2VycyBvbiBhbiBhY2NvdW50IGZvciB3aGljaCBubyBpbmRpdmlkdWFsIGFjY291bnQgc2V0dGluZyBoYXMgYmVlbiBzZXQuIEVuYWJsaW5nIHRoaXMgc2V0dGluZyBpcyByZXF1aXJlZCB0byB1c2UgbmV3IEFtYXpvbiBFQ1MgZmVhdHVyZXMgc3VjaCBhcyByZXNvdXJjZSB0YWdnaW5nLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCJcbiAgICB9LFxuICAgIFwiUHV0QXR0cmlidXRlc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25FQ1MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUHV0QXR0cmlidXRlcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQ3JlYXRlIG9yIHVwZGF0ZSBhbiBhdHRyaWJ1dGUgb24gYW4gQW1hem9uIEVDUyByZXNvdXJjZS5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJjb250YWluZXItaW5zdGFuY2VcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJlY3M6Y2x1c3RlclwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIlJlZ2lzdGVyQ29udGFpbmVySW5zdGFuY2VcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1JlZ2lzdGVyQ29udGFpbmVySW5zdGFuY2UuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJlZ2lzdGVycyBhbiBFQzIgaW5zdGFuY2UgaW50byB0aGUgc3BlY2lmaWVkIGNsdXN0ZXIuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiY2x1c3RlclwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcImF3czpSZXF1ZXN0VGFnLyR7VGFnS2V5fVwiLFxuICAgICAgICBcImF3czpUYWdLZXlzXCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiUmVnaXN0ZXJUYXNrRGVmaW5pdGlvblwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25FQ1MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUmVnaXN0ZXJUYXNrRGVmaW5pdGlvbi5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmVnaXN0ZXJzIGEgbmV3IHRhc2sgZGVmaW5pdGlvbiBmcm9tIHRoZSBzdXBwbGllZCBmYW1pbHkgYW5kIGNvbnRhaW5lckRlZmluaXRpb25zLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcImF3czpSZXF1ZXN0VGFnLyR7VGFnS2V5fVwiLFxuICAgICAgICBcImF3czpUYWdLZXlzXCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiUnVuVGFza1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25FQ1MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUnVuVGFzay5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiU3RhcnQgYSB0YXNrIHVzaW5nIHJhbmRvbSBwbGFjZW1lbnQgYW5kIHRoZSBkZWZhdWx0IEFtYXpvbiBFQ1Mgc2NoZWR1bGVyLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInRhc2stZGVmaW5pdGlvblwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcImVjczpjbHVzdGVyXCIsXG4gICAgICAgIFwiYXdzOlJlcXVlc3RUYWcvJHtUYWdLZXl9XCIsXG4gICAgICAgIFwiYXdzOlRhZ0tleXNcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJTdGFydFRhc2tcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1N0YXJ0VGFzay5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiU3RhcnRzIGEgbmV3IHRhc2sgZnJvbSB0aGUgc3BlY2lmaWVkIHRhc2sgZGVmaW5pdGlvbiBvbiB0aGUgc3BlY2lmaWVkIGNvbnRhaW5lciBpbnN0YW5jZSBvciBpbnN0YW5jZXMuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwidGFzay1kZWZpbml0aW9uXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiZWNzOmNsdXN0ZXJcIixcbiAgICAgICAgXCJlY3M6Y29udGFpbmVyLWluc3RhbmNlc1wiLFxuICAgICAgICBcImF3czpSZXF1ZXN0VGFnLyR7VGFnS2V5fVwiLFxuICAgICAgICBcImF3czpUYWdLZXlzXCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiU3RhcnRUZWxlbWV0cnlTZXNzaW9uXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkVDUy9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvY2xvdWR3YXRjaC1tZXRyaWNzLmh0bWwjZW5hYmxlX2Nsb3Vkd2F0Y2hcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBzdGFydCBhIHRlbGVtZXRyeSBzZXNzaW9uLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImNvbnRhaW5lci1pbnN0YW5jZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcImVjczpjbHVzdGVyXCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiU3RvcFRhc2tcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1N0b3BUYXNrLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJTdG9wcyBhIHJ1bm5pbmcgdGFzay5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJ0YXNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiZWNzOmNsdXN0ZXJcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJTdWJtaXRBdHRhY2htZW50U3RhdGVDaGFuZ2VzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkVDUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9TdWJtaXRBdHRhY2htZW50U3RhdGVDaGFuZ2VzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJTZW50IHRvIGFja25vd2xlZGdlIHRoYXQgYXR0YWNobWVudHMgY2hhbmdlZCBzdGF0ZXMuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiY2x1c3RlclwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiU3VibWl0Q29udGFpbmVyU3RhdGVDaGFuZ2VcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1N1Ym1pdENvbnRhaW5lclN0YXRlQ2hhbmdlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJTZW50IHRvIGFja25vd2xlZGdlIHRoYXQgYSBjb250YWluZXIgY2hhbmdlZCBzdGF0ZXMuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiY2x1c3RlclwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiU3VibWl0VGFza1N0YXRlQ2hhbmdlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkVDUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9TdWJtaXRUYXNrU3RhdGVDaGFuZ2UuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlNlbnQgdG8gYWNrbm93bGVkZ2UgdGhhdCBhIHRhc2sgY2hhbmdlZCBzdGF0ZXMuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiY2x1c3RlclwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiVGFnUmVzb3VyY2VcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1RhZ1Jlc291cmNlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJUYWdzIHRoZSBzcGVjaWZpZWQgcmVzb3VyY2UuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiVGFnZ2luZ1wiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJjbHVzdGVyXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH0sXG4gICAgICAgIFwiY29udGFpbmVyLWluc3RhbmNlXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH0sXG4gICAgICAgIFwic2VydmljZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9LFxuICAgICAgICBcInRhc2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAgXCJ0YXNrLWRlZmluaXRpb25cIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiYXdzOlRhZ0tleXNcIixcbiAgICAgICAgXCJhd3M6UmVxdWVzdFRhZy8ke1RhZ0tleX1cIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJVbnRhZ1Jlc291cmNlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkVDUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VbnRhZ1Jlc291cmNlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJVbnRhZ3MgdGhlIHNwZWNpZmllZCByZXNvdXJjZS5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJUYWdnaW5nXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImNsdXN0ZXJcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAgXCJjb250YWluZXItaW5zdGFuY2VcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAgXCJzZXJ2aWNlXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH0sXG4gICAgICAgIFwidGFza1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9LFxuICAgICAgICBcInRhc2stZGVmaW5pdGlvblwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJhd3M6VGFnS2V5c1wiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIlVwZGF0ZUNvbnRhaW5lckFnZW50XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkVDUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVDb250YWluZXJBZ2VudC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiVXBkYXRlcyB0aGUgQW1hem9uIEVDUyBjb250YWluZXIgYWdlbnQgb24gYSBzcGVjaWZpZWQgY29udGFpbmVyIGluc3RhbmNlLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImNvbnRhaW5lci1pbnN0YW5jZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcImVjczpjbHVzdGVyXCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiVXBkYXRlQ29udGFpbmVySW5zdGFuY2VzU3RhdGVcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZUNvbnRhaW5lckluc3RhbmNlc1N0YXRlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJFbmFibGVzIHRoZSB1c2VyIHRvIG1vZGlmeSB0aGUgc3RhdHVzIG9mIGFuIEFtYXpvbiBFQ1MgY29udGFpbmVyIGluc3RhbmNlLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImNvbnRhaW5lci1pbnN0YW5jZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcImVjczpjbHVzdGVyXCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiVXBkYXRlU2VydmljZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25FQ1MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlU2VydmljZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiTW9kaWZpZXMgdGhlIGRlc2lyZWQgY291bnQsIGRlcGxveW1lbnQgY29uZmlndXJhdGlvbiwgb3IgdGFzayBkZWZpbml0aW9uIHVzZWQgaW4gYSBzZXJ2aWNlLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInNlcnZpY2VcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJlY3M6Y2x1c3RlclwiLFxuICAgICAgICBcImVjczp0YXNrLWRlZmluaXRpb25cIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJVcGRhdGVTZXJ2aWNlUHJpbWFyeVRhc2tTZXRcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZVNlcnZpY2VQcmltYXJ5VGFza1NldC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiTW9kaWZpZXMgdGhlIHByaW1hcnkgdGFzayBzZXQgdXNlZCBpbiBhIHNlcnZpY2UuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic2VydmljZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcImVjczpjbHVzdGVyXCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiVXBkYXRlVGFza1NldFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25FQ1MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlVGFza1NldC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiVXBkYXRlcyB0aGUgc3BlY2lmaWVkIHRhc2sgc2V0LlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInRhc2stc2V0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiZWNzOmNsdXN0ZXJcIixcbiAgICAgICAgXCJlY3M6c2VydmljZVwiXG4gICAgICBdXG4gICAgfVxuICB9O1xuICBwdWJsaWMgcmVzb3VyY2VUeXBlczogUmVzb3VyY2VUeXBlcyA9IHtcbiAgICBcImNsdXN0ZXJcIjoge1xuICAgICAgXCJuYW1lXCI6IFwiY2x1c3RlclwiLFxuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9FQ1NfY2x1c3RlcnMuaHRtbFwiLFxuICAgICAgXCJhcm5cIjogXCJhcm46JHtQYXJ0aXRpb259OmVjczoke1JlZ2lvbn06JHtBY2NvdW50fTpjbHVzdGVyLyR7Q2x1c3Rlck5hbWV9XCIsXG4gICAgICBcImNvbmRpdGlvbktleXNcIjogW1xuICAgICAgICBcImF3czpSZXNvdXJjZVRhZy8ke1RhZ0tleX1cIixcbiAgICAgICAgXCJlY3M6UmVzb3VyY2VUYWcvJHtUYWdLZXl9XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiY29udGFpbmVyLWluc3RhbmNlXCI6IHtcbiAgICAgIFwibmFtZVwiOiBcImNvbnRhaW5lci1pbnN0YW5jZVwiLFxuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9FQ1NfaW5zdGFuY2VzLmh0bWxcIixcbiAgICAgIFwiYXJuXCI6IFwiYXJuOiR7UGFydGl0aW9ufTplY3M6JHtSZWdpb259OiR7QWNjb3VudH06Y29udGFpbmVyLWluc3RhbmNlLyR7Q29udGFpbmVySW5zdGFuY2VJZH1cIixcbiAgICAgIFwiY29uZGl0aW9uS2V5c1wiOiBbXG4gICAgICAgIFwiYXdzOlJlc291cmNlVGFnLyR7VGFnS2V5fVwiLFxuICAgICAgICBcImVjczpSZXNvdXJjZVRhZy8ke1RhZ0tleX1cIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJzZXJ2aWNlXCI6IHtcbiAgICAgIFwibmFtZVwiOiBcInNlcnZpY2VcIixcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkVDUy9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvZWNzX3NlcnZpY2VzLmh0bWxcIixcbiAgICAgIFwiYXJuXCI6IFwiYXJuOiR7UGFydGl0aW9ufTplY3M6JHtSZWdpb259OiR7QWNjb3VudH06c2VydmljZS8ke1NlcnZpY2VOYW1lfVwiLFxuICAgICAgXCJjb25kaXRpb25LZXlzXCI6IFtcbiAgICAgICAgXCJhd3M6UmVzb3VyY2VUYWcvJHtUYWdLZXl9XCIsXG4gICAgICAgIFwiZWNzOlJlc291cmNlVGFnLyR7VGFnS2V5fVwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcInRhc2tcIjoge1xuICAgICAgXCJuYW1lXCI6IFwidGFza1wiLFxuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9zY2hlZHVsaW5nX3Rhc2tzLmh0bWxcIixcbiAgICAgIFwiYXJuXCI6IFwiYXJuOiR7UGFydGl0aW9ufTplY3M6JHtSZWdpb259OiR7QWNjb3VudH06dGFzay8ke1Rhc2tJZH1cIixcbiAgICAgIFwiY29uZGl0aW9uS2V5c1wiOiBbXG4gICAgICAgIFwiYXdzOlJlc291cmNlVGFnLyR7VGFnS2V5fVwiLFxuICAgICAgICBcImVjczpSZXNvdXJjZVRhZy8ke1RhZ0tleX1cIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJ0YXNrLWRlZmluaXRpb25cIjoge1xuICAgICAgXCJuYW1lXCI6IFwidGFzay1kZWZpbml0aW9uXCIsXG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25FQ1MvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3Rhc2tfZGVmaW5pdGlvbnMuaHRtbFwiLFxuICAgICAgXCJhcm5cIjogXCJhcm46JHtQYXJ0aXRpb259OmVjczoke1JlZ2lvbn06JHtBY2NvdW50fTp0YXNrLWRlZmluaXRpb24vJHtUYXNrRGVmaW5pdGlvbkZhbWlseU5hbWV9OiR7VGFza0RlZmluaXRpb25SZXZpc2lvbk51bWJlcn1cIixcbiAgICAgIFwiY29uZGl0aW9uS2V5c1wiOiBbXG4gICAgICAgIFwiYXdzOlJlc291cmNlVGFnLyR7VGFnS2V5fVwiLFxuICAgICAgICBcImVjczpSZXNvdXJjZVRhZy8ke1RhZ0tleX1cIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJ0YXNrLXNldFwiOiB7XG4gICAgICBcIm5hbWVcIjogXCJ0YXNrLXNldFwiLFxuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9kZXZlbG9wZXJndWlkZS90YXNrX3NldHMuaHRtbFwiLFxuICAgICAgXCJhcm5cIjogXCJhcm46JHtQYXJ0aXRpb259OmVjczoke1JlZ2lvbn06JHtBY2NvdW50fTp0YXNrLXNldC8ke0NsdXN0ZXJOYW1lfS8ke1NlcnZpY2VOYW1lfS8ke1Rhc2tTZXRJZH1cIixcbiAgICAgIFwiY29uZGl0aW9uS2V5c1wiOiBbXG4gICAgICAgIFwiYXdzOlJlc291cmNlVGFnLyR7VGFnS2V5fVwiLFxuICAgICAgICBcImVjczpSZXNvdXJjZVRhZy8ke1RhZ0tleX1cIlxuICAgICAgXVxuICAgIH1cbiAgfTtcblxuICAvKipcbiAgICogQWN0aW9uIHByb3ZpZGVyIGZvciBzZXJ2aWNlIGVjc1xuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvbGlzdF9hbWF6b25lbGFzdGljY29udGFpbmVyc2VydmljZS5odG1sXG4gICAqL1xuICBjb25zdHJ1Y3RvciAocHJvcHM/OiBQb2xpY3lTdGF0ZW1lbnRQcm9wcykge1xuICAgIHN1cGVyKHByb3BzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIGEgbmV3IEFtYXpvbiBFQ1MgY2x1c3Rlci5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZUNsdXN0ZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIGNyZWF0ZUNsdXN0ZXIoKSB7XG4gICAgdGhpcy5hZGQoJ2VjczpDcmVhdGVDbHVzdGVyJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUnVucyBhbmQgbWFpbnRhaW5zIGEgZGVzaXJlZCBudW1iZXIgb2YgdGFza3MgZnJvbSBhIHNwZWNpZmllZCB0YXNrIGRlZmluaXRpb24uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkVDUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVTZXJ2aWNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyBjcmVhdGVTZXJ2aWNlKCkge1xuICAgIHRoaXMuYWRkKCdlY3M6Q3JlYXRlU2VydmljZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgYSBuZXcgQW1hem9uIEVDUyB0YXNrIHNldC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZVRhc2tTZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIGNyZWF0ZVRhc2tTZXQoKSB7XG4gICAgdGhpcy5hZGQoJ2VjczpDcmVhdGVUYXNrU2V0Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogTW9kaWZpZXMgdGhlIEFSTiBhbmQgcmVzb3VyY2UgSUQgZm9ybWF0IG9mIGEgcmVzb3VyY2UgZm9yIGEgc3BlY2lmaWVkIElBTSB1c2VyLCBJQU0gcm9sZSwgb3IgdGhlIHJvb3QgdXNlciBmb3IgYW4gYWNjb3VudC4gWW91IGNhbiBzcGVjaWZ5IHdoZXRoZXIgdGhlIG5ldyBBUk4gYW5kIHJlc291cmNlIElEIGZvcm1hdCBhcmUgZGlzYWJsZWQgZm9yIG5ldyByZXNvdXJjZXMgdGhhdCBhcmUgY3JlYXRlZC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZUFjY291bnRTZXR0aW5nLmh0bWxcbiAgICovXG4gIHB1YmxpYyBkZWxldGVBY2NvdW50U2V0dGluZygpIHtcbiAgICB0aGlzLmFkZCgnZWNzOkRlbGV0ZUFjY291bnRTZXR0aW5nJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVsZXRlcyBvbmUgb3IgbW9yZSBjdXN0b20gYXR0cmlidXRlcyBmcm9tIGFuIEFtYXpvbiBFQ1MgcmVzb3VyY2UuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkVDUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVBdHRyaWJ1dGVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyBkZWxldGVBdHRyaWJ1dGVzKCkge1xuICAgIHRoaXMuYWRkKCdlY3M6RGVsZXRlQXR0cmlidXRlcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlbGV0ZXMgdGhlIHNwZWNpZmllZCBjbHVzdGVyLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25FQ1MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlQ2x1c3Rlci5odG1sXG4gICAqL1xuICBwdWJsaWMgZGVsZXRlQ2x1c3RlcigpIHtcbiAgICB0aGlzLmFkZCgnZWNzOkRlbGV0ZUNsdXN0ZXInKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWxldGVzIGEgc3BlY2lmaWVkIHNlcnZpY2Ugd2l0aGluIGEgY2x1c3Rlci5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZVNlcnZpY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIGRlbGV0ZVNlcnZpY2UoKSB7XG4gICAgdGhpcy5hZGQoJ2VjczpEZWxldGVTZXJ2aWNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVsZXRlcyB0aGUgc3BlY2lmaWVkIHRhc2sgc2V0LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25FQ1MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlVGFza1NldC5odG1sXG4gICAqL1xuICBwdWJsaWMgZGVsZXRlVGFza1NldCgpIHtcbiAgICB0aGlzLmFkZCgnZWNzOkRlbGV0ZVRhc2tTZXQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZXJlZ2lzdGVycyBhbiBBbWF6b24gRUNTIGNvbnRhaW5lciBpbnN0YW5jZSBmcm9tIHRoZSBzcGVjaWZpZWQgY2x1c3Rlci5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlcmVnaXN0ZXJDb250YWluZXJJbnN0YW5jZS5odG1sXG4gICAqL1xuICBwdWJsaWMgZGVyZWdpc3RlckNvbnRhaW5lckluc3RhbmNlKCkge1xuICAgIHRoaXMuYWRkKCdlY3M6RGVyZWdpc3RlckNvbnRhaW5lckluc3RhbmNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVyZWdpc3RlcnMgdGhlIHNwZWNpZmllZCB0YXNrIGRlZmluaXRpb24gYnkgZmFtaWx5IGFuZCByZXZpc2lvbi5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlcmVnaXN0ZXJUYXNrRGVmaW5pdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgZGVyZWdpc3RlclRhc2tEZWZpbml0aW9uKCkge1xuICAgIHRoaXMuYWRkKCdlY3M6RGVyZWdpc3RlclRhc2tEZWZpbml0aW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVzY3JpYmVzIG9uZSBvciBtb3JlIG9mIHlvdXIgY2x1c3RlcnMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlQ2x1c3RlcnMuaHRtbFxuICAgKi9cbiAgcHVibGljIGRlc2NyaWJlQ2x1c3RlcnMoKSB7XG4gICAgdGhpcy5hZGQoJ2VjczpEZXNjcmliZUNsdXN0ZXJzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVzY3JpYmVzIEFtYXpvbiBFQ1MgY29udGFpbmVyIGluc3RhbmNlcy5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25FQ1MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVDb250YWluZXJJbnN0YW5jZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIGRlc2NyaWJlQ29udGFpbmVySW5zdGFuY2VzKCkge1xuICAgIHRoaXMuYWRkKCdlY3M6RGVzY3JpYmVDb250YWluZXJJbnN0YW5jZXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZXNjcmliZXMgdGhlIHNwZWNpZmllZCBzZXJ2aWNlcyBydW5uaW5nIGluIHlvdXIgY2x1c3Rlci5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25FQ1MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVTZXJ2aWNlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgZGVzY3JpYmVTZXJ2aWNlcygpIHtcbiAgICB0aGlzLmFkZCgnZWNzOkRlc2NyaWJlU2VydmljZXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZXNjcmliZXMgYSB0YXNrIGRlZmluaXRpb24uIFlvdSBjYW4gc3BlY2lmeSBhIGZhbWlseSBhbmQgcmV2aXNpb24gdG8gZmluZCBpbmZvcm1hdGlvbiBhYm91dCBhIHNwZWNpZmljIHRhc2sgZGVmaW5pdGlvbiwgb3IgeW91IGNhbiBzaW1wbHkgc3BlY2lmeSB0aGUgZmFtaWx5IHRvIGZpbmQgdGhlIGxhdGVzdCBBQ1RJVkUgcmV2aXNpb24gaW4gdGhhdCBmYW1pbHkuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlVGFza0RlZmluaXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIGRlc2NyaWJlVGFza0RlZmluaXRpb24oKSB7XG4gICAgdGhpcy5hZGQoJ2VjczpEZXNjcmliZVRhc2tEZWZpbml0aW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVzY3JpYmVzIEFtYXpvbiBFQ1MgdGFzayBzZXRzLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkVDUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZVRhc2tTZXRzLmh0bWxcbiAgICovXG4gIHB1YmxpYyBkZXNjcmliZVRhc2tTZXRzKCkge1xuICAgIHRoaXMuYWRkKCdlY3M6RGVzY3JpYmVUYXNrU2V0cycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlc2NyaWJlcyBhIHNwZWNpZmllZCB0YXNrIG9yIHRhc2tzLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkVDUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZVRhc2tzLmh0bWxcbiAgICovXG4gIHB1YmxpYyBkZXNjcmliZVRhc2tzKCkge1xuICAgIHRoaXMuYWRkKCdlY3M6RGVzY3JpYmVUYXNrcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgYW4gZW5kcG9pbnQgZm9yIHRoZSBBbWF6b24gRUNTIGFnZW50IHRvIHBvbGwgZm9yIHVwZGF0ZXMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkVDUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EaXNjb3ZlclBvbGxFbmRwb2ludC5odG1sXG4gICAqL1xuICBwdWJsaWMgZGlzY292ZXJQb2xsRW5kcG9pbnQoKSB7XG4gICAgdGhpcy5hZGQoJ2VjczpEaXNjb3ZlclBvbGxFbmRwb2ludCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIExpc3RzIHRoZSBhY2NvdW50IHNldHRpbmdzIGZvciBhbiBBbWF6b24gRUNTIHJlc291cmNlIGZvciBhIHNwZWNpZmllZCBwcmluY2lwYWwuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RBY2NvdW50U2V0dGluZ3MuaHRtbFxuICAgKi9cbiAgcHVibGljIGxpc3RBY2NvdW50U2V0dGluZ3MoKSB7XG4gICAgdGhpcy5hZGQoJ2VjczpMaXN0QWNjb3VudFNldHRpbmdzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogTGlzdHMgdGhlIGF0dHJpYnV0ZXMgZm9yIEFtYXpvbiBFQ1MgcmVzb3VyY2VzIHdpdGhpbiBhIHNwZWNpZmllZCB0YXJnZXQgdHlwZSBhbmQgY2x1c3Rlci5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25FQ1MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdEF0dHJpYnV0ZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIGxpc3RBdHRyaWJ1dGVzKCkge1xuICAgIHRoaXMuYWRkKCdlY3M6TGlzdEF0dHJpYnV0ZXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGEgbGlzdCBvZiBleGlzdGluZyBjbHVzdGVycy5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25FQ1MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdENsdXN0ZXJzLmh0bWxcbiAgICovXG4gIHB1YmxpYyBsaXN0Q2x1c3RlcnMoKSB7XG4gICAgdGhpcy5hZGQoJ2VjczpMaXN0Q2x1c3RlcnMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGEgbGlzdCBvZiBjb250YWluZXIgaW5zdGFuY2VzIGluIGEgc3BlY2lmaWVkIGNsdXN0ZXIuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RDb250YWluZXJJbnN0YW5jZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIGxpc3RDb250YWluZXJJbnN0YW5jZXMoKSB7XG4gICAgdGhpcy5hZGQoJ2VjczpMaXN0Q29udGFpbmVySW5zdGFuY2VzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogTGlzdHMgdGhlIHNlcnZpY2VzIHRoYXQgYXJlIHJ1bm5pbmcgaW4gYSBzcGVjaWZpZWQgY2x1c3Rlci5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25FQ1MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdFNlcnZpY2VzLmh0bWxcbiAgICovXG4gIHB1YmxpYyBsaXN0U2VydmljZXMoKSB7XG4gICAgdGhpcy5hZGQoJ2VjczpMaXN0U2VydmljZXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBMaXN0IHRhZ3MgZm9yIHRoZSBzcGVjaWZpZWQgcmVzb3VyY2UuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RUYWdzRm9yUmVzb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIGxpc3RUYWdzRm9yUmVzb3VyY2UoKSB7XG4gICAgdGhpcy5hZGQoJ2VjczpMaXN0VGFnc0ZvclJlc291cmNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBhIGxpc3Qgb2YgdGFzayBkZWZpbml0aW9uIGZhbWlsaWVzIHRoYXQgYXJlIHJlZ2lzdGVyZWQgdG8geW91ciBhY2NvdW50ICh3aGljaCBtYXkgaW5jbHVkZSB0YXNrIGRlZmluaXRpb24gZmFtaWxpZXMgdGhhdCBubyBsb25nZXIgaGF2ZSBhbnkgQUNUSVZFIHRhc2sgZGVmaW5pdGlvbnMpLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkVDUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0U2VydmljZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIGxpc3RUYXNrRGVmaW5pdGlvbkZhbWlsaWVzKCkge1xuICAgIHRoaXMuYWRkKCdlY3M6TGlzdFRhc2tEZWZpbml0aW9uRmFtaWxpZXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGEgbGlzdCBvZiB0YXNrIGRlZmluaXRpb25zIHRoYXQgYXJlIHJlZ2lzdGVyZWQgdG8geW91ciBhY2NvdW50LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkVDUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0VGFza0RlZmluaXRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyBsaXN0VGFza0RlZmluaXRpb25zKCkge1xuICAgIHRoaXMuYWRkKCdlY3M6TGlzdFRhc2tEZWZpbml0aW9ucycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgYSBsaXN0IG9mIHRhc2tzIGZvciBhIHNwZWNpZmllZCBjbHVzdGVyLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkVDUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0VGFza3MuaHRtbFxuICAgKi9cbiAgcHVibGljIGxpc3RUYXNrcygpIHtcbiAgICB0aGlzLmFkZCgnZWNzOkxpc3RUYXNrcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGFuIGFnZW50IHRvIGNvbm5lY3Qgd2l0aCB0aGUgQW1hem9uIEVDUyBzZXJ2aWNlIHRvIHJlcG9ydCBzdGF0dXMgYW5kIGdldCBjb21tYW5kcy5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9pbnN0YW5jZV9JQU1fcm9sZS5odG1sXG4gICAqL1xuICBwdWJsaWMgcG9sbCgpIHtcbiAgICB0aGlzLmFkZCgnZWNzOlBvbGwnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBNb2RpZmllcyB0aGUgQVJOIGFuZCByZXNvdXJjZSBJRCBmb3JtYXQgb2YgYSByZXNvdXJjZSBmb3IgYSBzcGVjaWZpZWQgSUFNIHVzZXIsIElBTSByb2xlLCBvciB0aGUgcm9vdCB1c2VyIGZvciBhbiBhY2NvdW50LiBZb3UgY2FuIHNwZWNpZnkgd2hldGhlciB0aGUgbmV3IEFSTiBhbmQgcmVzb3VyY2UgSUQgZm9ybWF0IGFyZSBlbmFibGVkIGZvciBuZXcgcmVzb3VyY2VzIHRoYXQgYXJlIGNyZWF0ZWQuIEVuYWJsaW5nIHRoaXMgc2V0dGluZyBpcyByZXF1aXJlZCB0byB1c2UgbmV3IEFtYXpvbiBFQ1MgZmVhdHVyZXMgc3VjaCBhcyByZXNvdXJjZSB0YWdnaW5nLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25FQ1MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUHV0QWNjb3VudFNldHRpbmcuaHRtbFxuICAgKi9cbiAgcHVibGljIHB1dEFjY291bnRTZXR0aW5nKCkge1xuICAgIHRoaXMuYWRkKCdlY3M6UHV0QWNjb3VudFNldHRpbmcnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBNb2RpZmllcyB0aGUgQVJOIGFuZCByZXNvdXJjZSBJRCBmb3JtYXQgb2YgYSByZXNvdXJjZSB0eXBlIGZvciBhbGwgSUFNIHVzZXJzIG9uIGFuIGFjY291bnQgZm9yIHdoaWNoIG5vIGluZGl2aWR1YWwgYWNjb3VudCBzZXR0aW5nIGhhcyBiZWVuIHNldC4gRW5hYmxpbmcgdGhpcyBzZXR0aW5nIGlzIHJlcXVpcmVkIHRvIHVzZSBuZXcgQW1hem9uIEVDUyBmZWF0dXJlcyBzdWNoIGFzIHJlc291cmNlIHRhZ2dpbmcuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkVDUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9QdXRBY2NvdW50U2V0dGluZ0RlZmF1bHQuaHRtbFxuICAgKi9cbiAgcHVibGljIHB1dEFjY291bnRTZXR0aW5nRGVmYXVsdCgpIHtcbiAgICB0aGlzLmFkZCgnZWNzOlB1dEFjY291bnRTZXR0aW5nRGVmYXVsdCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZSBvciB1cGRhdGUgYW4gYXR0cmlidXRlIG9uIGFuIEFtYXpvbiBFQ1MgcmVzb3VyY2UuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkVDUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9QdXRBdHRyaWJ1dGVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyBwdXRBdHRyaWJ1dGVzKCkge1xuICAgIHRoaXMuYWRkKCdlY3M6UHV0QXR0cmlidXRlcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJlZ2lzdGVycyBhbiBFQzIgaW5zdGFuY2UgaW50byB0aGUgc3BlY2lmaWVkIGNsdXN0ZXIuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkVDUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9SZWdpc3RlckNvbnRhaW5lckluc3RhbmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyByZWdpc3RlckNvbnRhaW5lckluc3RhbmNlKCkge1xuICAgIHRoaXMuYWRkKCdlY3M6UmVnaXN0ZXJDb250YWluZXJJbnN0YW5jZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJlZ2lzdGVycyBhIG5ldyB0YXNrIGRlZmluaXRpb24gZnJvbSB0aGUgc3VwcGxpZWQgZmFtaWx5IGFuZCBjb250YWluZXJEZWZpbml0aW9ucy5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1JlZ2lzdGVyVGFza0RlZmluaXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHJlZ2lzdGVyVGFza0RlZmluaXRpb24oKSB7XG4gICAgdGhpcy5hZGQoJ2VjczpSZWdpc3RlclRhc2tEZWZpbml0aW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogU3RhcnQgYSB0YXNrIHVzaW5nIHJhbmRvbSBwbGFjZW1lbnQgYW5kIHRoZSBkZWZhdWx0IEFtYXpvbiBFQ1Mgc2NoZWR1bGVyLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25FQ1MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUnVuVGFzay5odG1sXG4gICAqL1xuICBwdWJsaWMgcnVuVGFzaygpIHtcbiAgICB0aGlzLmFkZCgnZWNzOlJ1blRhc2snKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBTdGFydHMgYSBuZXcgdGFzayBmcm9tIHRoZSBzcGVjaWZpZWQgdGFzayBkZWZpbml0aW9uIG9uIHRoZSBzcGVjaWZpZWQgY29udGFpbmVyIGluc3RhbmNlIG9yIGluc3RhbmNlcy5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1N0YXJ0VGFzay5odG1sXG4gICAqL1xuICBwdWJsaWMgc3RhcnRUYXNrKCkge1xuICAgIHRoaXMuYWRkKCdlY3M6U3RhcnRUYXNrJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc3RhcnQgYSB0ZWxlbWV0cnkgc2Vzc2lvbi5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9jbG91ZHdhdGNoLW1ldHJpY3MuaHRtbCNlbmFibGVfY2xvdWR3YXRjaFxuICAgKi9cbiAgcHVibGljIHN0YXJ0VGVsZW1ldHJ5U2Vzc2lvbigpIHtcbiAgICB0aGlzLmFkZCgnZWNzOlN0YXJ0VGVsZW1ldHJ5U2Vzc2lvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFN0b3BzIGEgcnVubmluZyB0YXNrLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25FQ1MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU3RvcFRhc2suaHRtbFxuICAgKi9cbiAgcHVibGljIHN0b3BUYXNrKCkge1xuICAgIHRoaXMuYWRkKCdlY3M6U3RvcFRhc2snKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBTZW50IHRvIGFja25vd2xlZGdlIHRoYXQgYXR0YWNobWVudHMgY2hhbmdlZCBzdGF0ZXMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkVDUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9TdWJtaXRBdHRhY2htZW50U3RhdGVDaGFuZ2VzLmh0bWxcbiAgICovXG4gIHB1YmxpYyBzdWJtaXRBdHRhY2htZW50U3RhdGVDaGFuZ2VzKCkge1xuICAgIHRoaXMuYWRkKCdlY3M6U3VibWl0QXR0YWNobWVudFN0YXRlQ2hhbmdlcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFNlbnQgdG8gYWNrbm93bGVkZ2UgdGhhdCBhIGNvbnRhaW5lciBjaGFuZ2VkIHN0YXRlcy5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1N1Ym1pdENvbnRhaW5lclN0YXRlQ2hhbmdlLmh0bWxcbiAgICovXG4gIHB1YmxpYyBzdWJtaXRDb250YWluZXJTdGF0ZUNoYW5nZSgpIHtcbiAgICB0aGlzLmFkZCgnZWNzOlN1Ym1pdENvbnRhaW5lclN0YXRlQ2hhbmdlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogU2VudCB0byBhY2tub3dsZWRnZSB0aGF0IGEgdGFzayBjaGFuZ2VkIHN0YXRlcy5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1N1Ym1pdFRhc2tTdGF0ZUNoYW5nZS5odG1sXG4gICAqL1xuICBwdWJsaWMgc3VibWl0VGFza1N0YXRlQ2hhbmdlKCkge1xuICAgIHRoaXMuYWRkKCdlY3M6U3VibWl0VGFza1N0YXRlQ2hhbmdlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVGFncyB0aGUgc3BlY2lmaWVkIHJlc291cmNlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFRhZ2dpbmdcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkVDUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9UYWdSZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdGFnUmVzb3VyY2UoKSB7XG4gICAgdGhpcy5hZGQoJ2VjczpUYWdSZXNvdXJjZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFVudGFncyB0aGUgc3BlY2lmaWVkIHJlc291cmNlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFRhZ2dpbmdcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkVDUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VbnRhZ1Jlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB1bnRhZ1Jlc291cmNlKCkge1xuICAgIHRoaXMuYWRkKCdlY3M6VW50YWdSZXNvdXJjZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFVwZGF0ZXMgdGhlIEFtYXpvbiBFQ1MgY29udGFpbmVyIGFnZW50IG9uIGEgc3BlY2lmaWVkIGNvbnRhaW5lciBpbnN0YW5jZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZUNvbnRhaW5lckFnZW50Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB1cGRhdGVDb250YWluZXJBZ2VudCgpIHtcbiAgICB0aGlzLmFkZCgnZWNzOlVwZGF0ZUNvbnRhaW5lckFnZW50Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRW5hYmxlcyB0aGUgdXNlciB0byBtb2RpZnkgdGhlIHN0YXR1cyBvZiBhbiBBbWF6b24gRUNTIGNvbnRhaW5lciBpbnN0YW5jZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZUNvbnRhaW5lckluc3RhbmNlc1N0YXRlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB1cGRhdGVDb250YWluZXJJbnN0YW5jZXNTdGF0ZSgpIHtcbiAgICB0aGlzLmFkZCgnZWNzOlVwZGF0ZUNvbnRhaW5lckluc3RhbmNlc1N0YXRlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogTW9kaWZpZXMgdGhlIGRlc2lyZWQgY291bnQsIGRlcGxveW1lbnQgY29uZmlndXJhdGlvbiwgb3IgdGFzayBkZWZpbml0aW9uIHVzZWQgaW4gYSBzZXJ2aWNlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25FQ1MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlU2VydmljZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdXBkYXRlU2VydmljZSgpIHtcbiAgICB0aGlzLmFkZCgnZWNzOlVwZGF0ZVNlcnZpY2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBNb2RpZmllcyB0aGUgcHJpbWFyeSB0YXNrIHNldCB1c2VkIGluIGEgc2VydmljZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZVNlcnZpY2VQcmltYXJ5VGFza1NldC5odG1sXG4gICAqL1xuICBwdWJsaWMgdXBkYXRlU2VydmljZVByaW1hcnlUYXNrU2V0KCkge1xuICAgIHRoaXMuYWRkKCdlY3M6VXBkYXRlU2VydmljZVByaW1hcnlUYXNrU2V0Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVXBkYXRlcyB0aGUgc3BlY2lmaWVkIHRhc2sgc2V0LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25FQ1MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlVGFza1NldC5odG1sXG4gICAqL1xuICBwdWJsaWMgdXBkYXRlVGFza1NldCgpIHtcbiAgICB0aGlzLmFkZCgnZWNzOlVwZGF0ZVRhc2tTZXQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBjbHVzdGVyIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkVDUy9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvRUNTX2NsdXN0ZXJzLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGNsdXN0ZXJOYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIGNsdXN0ZXJOYW1lLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIHJlZ2lvbnMuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb24ga2V5czpcbiAgICogIC0gYXdzOlJlc291cmNlVGFnLyR7VGFnS2V5fVxuICAgKiAgLSBlY3M6UmVzb3VyY2VUYWcvJHtUYWdLZXl9XG4gICAqL1xuICBwdWJsaWMgb25DbHVzdGVyKGNsdXN0ZXJOYW1lOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OmVjczoke1JlZ2lvbn06JHtBY2NvdW50fTpjbHVzdGVyLyR7Q2x1c3Rlck5hbWV9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtDbHVzdGVyTmFtZX0nLCBjbHVzdGVyTmFtZSk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QWNjb3VudH0nLCBhY2NvdW50IHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVnaW9ufScsIHJlZ2lvbiB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgY29udGFpbmVyLWluc3RhbmNlIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkVDUy9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvRUNTX2luc3RhbmNlcy5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBjb250YWluZXJJbnN0YW5jZUlkIC0gSWRlbnRpZmllciBmb3IgdGhlIGNvbnRhaW5lckluc3RhbmNlSWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbiBrZXlzOlxuICAgKiAgLSBhd3M6UmVzb3VyY2VUYWcvJHtUYWdLZXl9XG4gICAqICAtIGVjczpSZXNvdXJjZVRhZy8ke1RhZ0tleX1cbiAgICovXG4gIHB1YmxpYyBvbkNvbnRhaW5lckluc3RhbmNlKGNvbnRhaW5lckluc3RhbmNlSWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06ZWNzOiR7UmVnaW9ufToke0FjY291bnR9OmNvbnRhaW5lci1pbnN0YW5jZS8ke0NvbnRhaW5lckluc3RhbmNlSWR9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtDb250YWluZXJJbnN0YW5jZUlkfScsIGNvbnRhaW5lckluc3RhbmNlSWQpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1JlZ2lvbn0nLCByZWdpb24gfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIHNlcnZpY2UgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9lY3Nfc2VydmljZXMuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gc2VydmljZU5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgc2VydmljZU5hbWUuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbiBrZXlzOlxuICAgKiAgLSBhd3M6UmVzb3VyY2VUYWcvJHtUYWdLZXl9XG4gICAqICAtIGVjczpSZXNvdXJjZVRhZy8ke1RhZ0tleX1cbiAgICovXG4gIHB1YmxpYyBvblNlcnZpY2Uoc2VydmljZU5hbWU6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06ZWNzOiR7UmVnaW9ufToke0FjY291bnR9OnNlcnZpY2UvJHtTZXJ2aWNlTmFtZX0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1NlcnZpY2VOYW1lfScsIHNlcnZpY2VOYW1lKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZWdpb259JywgcmVnaW9uIHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGFydGl0aW9ufScsIHBhcnRpdGlvbiB8fCAnYXdzJyk7XG4gICAgcmV0dXJuIHRoaXMub24oYXJuKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSB0YXNrIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkVDUy9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvc2NoZWR1bGluZ190YXNrcy5odG1sXG4gICAqXG4gICAqIEBwYXJhbSB0YXNrSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgdGFza0lkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIHJlZ2lvbnMuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb24ga2V5czpcbiAgICogIC0gYXdzOlJlc291cmNlVGFnLyR7VGFnS2V5fVxuICAgKiAgLSBlY3M6UmVzb3VyY2VUYWcvJHtUYWdLZXl9XG4gICAqL1xuICBwdWJsaWMgb25UYXNrKHRhc2tJZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHZhciBhcm4gPSAnYXJuOiR7UGFydGl0aW9ufTplY3M6JHtSZWdpb259OiR7QWNjb3VudH06dGFzay8ke1Rhc2tJZH0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1Rhc2tJZH0nLCB0YXNrSWQpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1JlZ2lvbn0nLCByZWdpb24gfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIHRhc2stZGVmaW5pdGlvbiB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25FQ1MvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3Rhc2tfZGVmaW5pdGlvbnMuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gdGFza0RlZmluaXRpb25GYW1pbHlOYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIHRhc2tEZWZpbml0aW9uRmFtaWx5TmFtZS5cbiAgICogQHBhcmFtIHRhc2tEZWZpbml0aW9uUmV2aXNpb25OdW1iZXIgLSBJZGVudGlmaWVyIGZvciB0aGUgdGFza0RlZmluaXRpb25SZXZpc2lvbk51bWJlci5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgYWNjb3VudHMuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCByZWdpb25zLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uIGtleXM6XG4gICAqICAtIGF3czpSZXNvdXJjZVRhZy8ke1RhZ0tleX1cbiAgICogIC0gZWNzOlJlc291cmNlVGFnLyR7VGFnS2V5fVxuICAgKi9cbiAgcHVibGljIG9uVGFza0RlZmluaXRpb24odGFza0RlZmluaXRpb25GYW1pbHlOYW1lOiBzdHJpbmcsIHRhc2tEZWZpbml0aW9uUmV2aXNpb25OdW1iZXI6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06ZWNzOiR7UmVnaW9ufToke0FjY291bnR9OnRhc2stZGVmaW5pdGlvbi8ke1Rhc2tEZWZpbml0aW9uRmFtaWx5TmFtZX06JHtUYXNrRGVmaW5pdGlvblJldmlzaW9uTnVtYmVyfSc7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7VGFza0RlZmluaXRpb25GYW1pbHlOYW1lfScsIHRhc2tEZWZpbml0aW9uRmFtaWx5TmFtZSk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7VGFza0RlZmluaXRpb25SZXZpc2lvbk51bWJlcn0nLCB0YXNrRGVmaW5pdGlvblJldmlzaW9uTnVtYmVyKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZWdpb259JywgcmVnaW9uIHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGFydGl0aW9ufScsIHBhcnRpdGlvbiB8fCAnYXdzJyk7XG4gICAgcmV0dXJuIHRoaXMub24oYXJuKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSB0YXNrLXNldCB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25FQ1MvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3Rhc2tfc2V0cy5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBjbHVzdGVyTmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBjbHVzdGVyTmFtZS5cbiAgICogQHBhcmFtIHNlcnZpY2VOYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIHNlcnZpY2VOYW1lLlxuICAgKiBAcGFyYW0gdGFza1NldElkIC0gSWRlbnRpZmllciBmb3IgdGhlIHRhc2tTZXRJZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgYWNjb3VudHMuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCByZWdpb25zLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uIGtleXM6XG4gICAqICAtIGF3czpSZXNvdXJjZVRhZy8ke1RhZ0tleX1cbiAgICogIC0gZWNzOlJlc291cmNlVGFnLyR7VGFnS2V5fVxuICAgKi9cbiAgcHVibGljIG9uVGFza1NldChjbHVzdGVyTmFtZTogc3RyaW5nLCBzZXJ2aWNlTmFtZTogc3RyaW5nLCB0YXNrU2V0SWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06ZWNzOiR7UmVnaW9ufToke0FjY291bnR9OnRhc2stc2V0LyR7Q2x1c3Rlck5hbWV9LyR7U2VydmljZU5hbWV9LyR7VGFza1NldElkfSc7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7Q2x1c3Rlck5hbWV9JywgY2x1c3Rlck5hbWUpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1NlcnZpY2VOYW1lfScsIHNlcnZpY2VOYW1lKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtUYXNrU2V0SWR9JywgdGFza1NldElkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZWdpb259JywgcmVnaW9uIHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGFydGl0aW9ufScsIHBhcnRpdGlvbiB8fCAnYXdzJyk7XG4gICAgcmV0dXJuIHRoaXMub24oYXJuKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBAcGFyYW0gdGFnS2V5IFRoZSB0YWcga2V5IHRvIGNoZWNrXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmUmVzb3VyY2VUYWcodGFnS2V5OiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgZWNzOlJlc291cmNlVGFnLyR7IHRhZ0tleSB9YCwgdmFsdWUsIG9wZXJhdG9yIHx8ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogVGhlIEFSTiBvZiBhbiBFQ1MgY2x1c3Rlci5cbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkVDUy9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvaWFtLXBvbGljeS1zdHJ1Y3R1cmUuaHRtbCNhbWF6b24tZWNzLWtleXNcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbYXJuIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19BUk4pLiAqKkRlZmF1bHQ6KiogYEFybkVxdWFsc2BcbiAgICovXG4gIHB1YmxpYyBpZkNsdXN0ZXIodmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBlY3M6Y2x1c3RlcmAsIHZhbHVlLCBvcGVyYXRvciB8fCAnQXJuRXF1YWxzJyk7XG4gIH1cblxuICAvKipcbiAgICogVGhlIEFSTiBvZiBhbiBFQ1MgY29udGFpbmVyIGluc3RhbmNlLlxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9pYW0tcG9saWN5LXN0cnVjdHVyZS5odG1sI2FtYXpvbi1lY3Mta2V5c1xuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFthcm4gb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX0FSTikuICoqRGVmYXVsdDoqKiBgQXJuRXF1YWxzYFxuICAgKi9cbiAgcHVibGljIGlmQ29udGFpbmVySW5zdGFuY2VzKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgZWNzOmNvbnRhaW5lci1pbnN0YW5jZXNgLCB2YWx1ZSwgb3BlcmF0b3IgfHwgJ0FybkVxdWFscycpO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBBUk4gb2YgYW4gRUNTIHNlcnZpY2UuXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25FQ1MvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL2lhbS1wb2xpY3ktc3RydWN0dXJlLmh0bWwjYW1hem9uLWVjcy1rZXlzXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW2FybiBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfQVJOKS4gKipEZWZhdWx0OioqIGBBcm5FcXVhbHNgXG4gICAqL1xuICBwdWJsaWMgaWZTZXJ2aWNlKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgZWNzOnNlcnZpY2VgLCB2YWx1ZSwgb3BlcmF0b3IgfHwgJ0FybkVxdWFscycpO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBBUk4gb2YgYW4gRUNTIHRhc2sgZGVmaW5pdGlvbi5cbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkVDUy9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvaWFtLXBvbGljeS1zdHJ1Y3R1cmUuaHRtbCNhbWF6b24tZWNzLWtleXNcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbYXJuIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19BUk4pLiAqKkRlZmF1bHQ6KiogYEFybkVxdWFsc2BcbiAgICovXG4gIHB1YmxpYyBpZlRhc2tEZWZpbml0aW9uKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgZWNzOnRhc2stZGVmaW5pdGlvbmAsIHZhbHVlLCBvcGVyYXRvciB8fCAnQXJuRXF1YWxzJyk7XG4gIH1cbn1cbiJdfQ==