"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service elasticfilesystem
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelasticfilesystem.html
 */
class Elasticfilesystem extends shared_1.PolicyStatement {
    /**
     * Action provider for service elasticfilesystem
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelasticfilesystem.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'elasticfilesystem';
        this.actions = {
            "Backup": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/efs-backup-solutions.html",
                "description": "Starts a backup job for an existing file system.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "ClientMount": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/efs-client-authorization.html",
                "description": "Permission for allowing read-access to a file system.",
                "accessLevel": "Read",
                "conditions": [
                    "elasticfilesystem:AccessPointArn"
                ]
            },
            "ClientRootAccess": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/efs-client-authorization.html",
                "description": "Permission for allowing root-access to a file system.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                },
                "conditions": [
                    "elasticfilesystem:AccessPointArn"
                ]
            },
            "ClientWrite": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/efs-client-authorization.html",
                "description": "Permission for allowing write-access to a file system.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                },
                "conditions": [
                    "elasticfilesystem:AccessPointArn"
                ]
            },
            "CreateAccessPoint": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_CreateAccessPoint.html",
                "description": "Creates an access point for the specified file system.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "CreateFileSystem": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_CreateFileSystem.html",
                "description": "Creates a new, empty file system.",
                "accessLevel": "Tagging",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateMountTarget": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_CreateMountTarget.html",
                "description": "Creates a mount target for a file system.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "CreateTags": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_CreateTags.html",
                "description": "Creates or overwrites tags associated with a file system.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteAccessPoint": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_DeleteAccessPoint.html",
                "description": "Deletes the specified access point.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "access-point": {
                        "required": true
                    }
                }
            },
            "DeleteFileSystem": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_DeleteFileSystem.html",
                "description": "Deletes a file system, permanently severing access to its contents.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "DeleteFileSystemPolicy": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_DeleteFileSystemPolicy.html",
                "description": "Clears the resource-level policy for a given file system.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "DeleteMountTarget": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_DeleteMountTarget.html",
                "description": "Deletes the specified mount target.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "DeleteTags": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_DeleteTags.html",
                "description": "Deletes the specified tags from a file system.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "DescribeAccessPoints": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_DescribeAccessPoints.html",
                "description": "Returns the descriptions of Amazon EFS access points.",
                "accessLevel": "List",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "DescribeFileSystemPolicy": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_DescribeFileSystemPolicy.html",
                "description": "Returns the current resource-level policy for a given file system.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "file-system": {
                        "required": false
                    }
                }
            },
            "DescribeFileSystems": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_DescribeFileSystems.html",
                "description": "Returns the description of a specific Amazon EFS file system if either the file system CreationToken or the FileSystemId is provided; otherwise, returns descriptions of all file systems owned by the caller's AWS account in the AWS region of the endpoint that you're calling.",
                "accessLevel": "List",
                "resourceTypes": {
                    "file-system": {
                        "required": false
                    }
                }
            },
            "DescribeLifecycleConfiguration": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_DescribeLifecycleConfiguration.html",
                "description": "Returns the current LifecycleConfiguration object for the specified Amazon EFS file system.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "DescribeMountTargetSecurityGroups": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_DescribeMountTargetSecurityGroups.html",
                "description": "Returns the security groups currently in effect for a mount target.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "DescribeMountTargets": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_DescribeMountTargets.html",
                "description": "Returns the descriptions of all the current mount targets, or a specific mount target, for a file system.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "DescribeTags": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_DescribeTags.html",
                "description": "Returns the tags associated with a file system.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_ListTagsForResource.html",
                "description": "Returns the tags associated with the specified Amazon EFS resource.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "ModifyMountTargetSecurityGroups": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_ModifyMountTargetSecurityGroups.html",
                "description": "Modifies the set of security groups in effect for a mount target.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "PutFileSystemPolicy": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_PutFileSystemPolicy.html",
                "description": "Apply a resource-level policy granting and/or restricting actions from given actors for the specified file system.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "PutLifecycleConfiguration": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_PutLifecycleConfiguration.html",
                "description": "Enables lifecycle management by creating a new LifecycleConfiguration object.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "Restore": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/efs-backup-solutions.html",
                "description": "Starts a restore job for an existing file system.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_TagResource.html",
                "description": "Creates or overwrites tags associated with the specified Amazon EFS resource.",
                "accessLevel": "Tagging"
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_UntagResource.html",
                "description": "Deletes the specified tags from a specified Amazon EFS resource.",
                "accessLevel": "Tagging"
            },
            "UpdateFileSystem": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_UpdateFileSystem.html",
                "description": "Updates the throughput mode or the amount of provisioned throughput of an existing file system.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "file-system": {
                "name": "file-system",
                "url": "https://docs.aws.amazon.com/efs/latest/ug/access-control-overview.html#access-control-resources",
                "arn": "arn:${Partition}:elasticfilesystem:${Region}:${Account}:file-system/${FileSystemId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "access-point": {
                "name": "access-point",
                "url": "https://docs.aws.amazon.com/efs/latest/ug/access-control-overview.html#access-control-resources",
                "arn": "arn:${Partition}:elasticfilesystem:${Region}:${Account}:access-point/${AccessPointId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Starts a backup job for an existing file system.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/efs-backup-solutions.html
     */
    backup() {
        this.add('elasticfilesystem:Backup');
        return this;
    }
    /**
     * Permission for allowing read-access to a file system.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/efs/latest/ug/efs-client-authorization.html
     */
    clientMount() {
        this.add('elasticfilesystem:ClientMount');
        return this;
    }
    /**
     * Permission for allowing root-access to a file system.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/efs-client-authorization.html
     */
    clientRootAccess() {
        this.add('elasticfilesystem:ClientRootAccess');
        return this;
    }
    /**
     * Permission for allowing write-access to a file system.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/efs-client-authorization.html
     */
    clientWrite() {
        this.add('elasticfilesystem:ClientWrite');
        return this;
    }
    /**
     * Creates an access point for the specified file system.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_CreateAccessPoint.html
     */
    createAccessPoint() {
        this.add('elasticfilesystem:CreateAccessPoint');
        return this;
    }
    /**
     * Creates a new, empty file system.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_CreateFileSystem.html
     */
    createFileSystem() {
        this.add('elasticfilesystem:CreateFileSystem');
        return this;
    }
    /**
     * Creates a mount target for a file system.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_CreateMountTarget.html
     */
    createMountTarget() {
        this.add('elasticfilesystem:CreateMountTarget');
        return this;
    }
    /**
     * Creates or overwrites tags associated with a file system.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_CreateTags.html
     */
    createTags() {
        this.add('elasticfilesystem:CreateTags');
        return this;
    }
    /**
     * Deletes the specified access point.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DeleteAccessPoint.html
     */
    deleteAccessPoint() {
        this.add('elasticfilesystem:DeleteAccessPoint');
        return this;
    }
    /**
     * Deletes a file system, permanently severing access to its contents.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DeleteFileSystem.html
     */
    deleteFileSystem() {
        this.add('elasticfilesystem:DeleteFileSystem');
        return this;
    }
    /**
     * Clears the resource-level policy for a given file system.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DeleteFileSystemPolicy.html
     */
    deleteFileSystemPolicy() {
        this.add('elasticfilesystem:DeleteFileSystemPolicy');
        return this;
    }
    /**
     * Deletes the specified mount target.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DeleteMountTarget.html
     */
    deleteMountTarget() {
        this.add('elasticfilesystem:DeleteMountTarget');
        return this;
    }
    /**
     * Deletes the specified tags from a file system.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DeleteTags.html
     */
    deleteTags() {
        this.add('elasticfilesystem:DeleteTags');
        return this;
    }
    /**
     * Returns the descriptions of Amazon EFS access points.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DescribeAccessPoints.html
     */
    describeAccessPoints() {
        this.add('elasticfilesystem:DescribeAccessPoints');
        return this;
    }
    /**
     * Returns the current resource-level policy for a given file system.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DescribeFileSystemPolicy.html
     */
    describeFileSystemPolicy() {
        this.add('elasticfilesystem:DescribeFileSystemPolicy');
        return this;
    }
    /**
     * Returns the description of a specific Amazon EFS file system if either the file system CreationToken or the FileSystemId is provided; otherwise, returns descriptions of all file systems owned by the caller's AWS account in the AWS region of the endpoint that you're calling.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DescribeFileSystems.html
     */
    describeFileSystems() {
        this.add('elasticfilesystem:DescribeFileSystems');
        return this;
    }
    /**
     * Returns the current LifecycleConfiguration object for the specified Amazon EFS file system.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DescribeLifecycleConfiguration.html
     */
    describeLifecycleConfiguration() {
        this.add('elasticfilesystem:DescribeLifecycleConfiguration');
        return this;
    }
    /**
     * Returns the security groups currently in effect for a mount target.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DescribeMountTargetSecurityGroups.html
     */
    describeMountTargetSecurityGroups() {
        this.add('elasticfilesystem:DescribeMountTargetSecurityGroups');
        return this;
    }
    /**
     * Returns the descriptions of all the current mount targets, or a specific mount target, for a file system.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DescribeMountTargets.html
     */
    describeMountTargets() {
        this.add('elasticfilesystem:DescribeMountTargets');
        return this;
    }
    /**
     * Returns the tags associated with a file system.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DescribeTags.html
     */
    describeTags() {
        this.add('elasticfilesystem:DescribeTags');
        return this;
    }
    /**
     * Returns the tags associated with the specified Amazon EFS resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('elasticfilesystem:ListTagsForResource');
        return this;
    }
    /**
     * Modifies the set of security groups in effect for a mount target.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_ModifyMountTargetSecurityGroups.html
     */
    modifyMountTargetSecurityGroups() {
        this.add('elasticfilesystem:ModifyMountTargetSecurityGroups');
        return this;
    }
    /**
     * Apply a resource-level policy granting and/or restricting actions from given actors for the specified file system.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_PutFileSystemPolicy.html
     */
    putFileSystemPolicy() {
        this.add('elasticfilesystem:PutFileSystemPolicy');
        return this;
    }
    /**
     * Enables lifecycle management by creating a new LifecycleConfiguration object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_PutLifecycleConfiguration.html
     */
    putLifecycleConfiguration() {
        this.add('elasticfilesystem:PutLifecycleConfiguration');
        return this;
    }
    /**
     * Starts a restore job for an existing file system.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/efs-backup-solutions.html
     */
    restore() {
        this.add('elasticfilesystem:Restore');
        return this;
    }
    /**
     * Creates or overwrites tags associated with the specified Amazon EFS resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_TagResource.html
     */
    tagResource() {
        this.add('elasticfilesystem:TagResource');
        return this;
    }
    /**
     * Deletes the specified tags from a specified Amazon EFS resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_UntagResource.html
     */
    untagResource() {
        this.add('elasticfilesystem:UntagResource');
        return this;
    }
    /**
     * Updates the throughput mode or the amount of provisioned throughput of an existing file system.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_UpdateFileSystem.html
     */
    updateFileSystem() {
        this.add('elasticfilesystem:UpdateFileSystem');
        return this;
    }
    /**
     * Adds a resource of type file-system to the statement
     *
     * https://docs.aws.amazon.com/efs/latest/ug/access-control-overview.html#access-control-resources
     *
     * @param fileSystemId - Identifier for the fileSystemId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onFileSystem(fileSystemId, account, region, partition) {
        var arn = 'arn:${Partition}:elasticfilesystem:${Region}:${Account}:file-system/${FileSystemId}';
        arn = arn.replace('${FileSystemId}', fileSystemId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type access-point to the statement
     *
     * https://docs.aws.amazon.com/efs/latest/ug/access-control-overview.html#access-control-resources
     *
     * @param accessPointId - Identifier for the accessPointId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onAccessPoint(accessPointId, account, region, partition) {
        var arn = 'arn:${Partition}:elasticfilesystem:${Region}:${Account}:access-point/${AccessPointId}';
        arn = arn.replace('${AccessPointId}', accessPointId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * The ARN of the access point used to mount the file system
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAccessPointArn(value, operator) {
        return this.if(`elasticfilesystem:AccessPointArn`, value, operator || 'StringLike');
    }
}
exports.Elasticfilesystem = Elasticfilesystem;
//# sourceMappingURL=data:application/json;base64,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