"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service es
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelasticsearchservice.html
 */
class Es extends shared_1.PolicyStatement {
    /**
     * Action provider for service es
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelasticsearchservice.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'es';
        this.actions = {
            "AcceptInboundCrossClusterSearchConnection": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-acceptinboundcrossclustersearchconnection",
                "description": "Allows the destination domain owner to accept an inbound cross-cluster search connection request",
                "accessLevel": "Write"
            },
            "AddTags": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-addtags",
                "description": "Grants permission to attach resource tags to an Amazon ES domain.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "CreateElasticsearchDomain": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-createelasticsearchdomain",
                "description": "Grants permission to create an Amazon ES domain.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": false
                    }
                }
            },
            "CreateElasticsearchServiceRole": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-createelasticsearchservicerole",
                "description": "Grants permission to create the service-linked role required for Amazon ES domains that use VPC access.",
                "accessLevel": "Write"
            },
            "CreateOutboundCrossClusterSearchConnection": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-createoutboundcrossclustersearchconnection",
                "description": "Creates a new cross-cluster search connection from a source domain to a destination domain",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DeleteElasticsearchDomain": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-deleteelasticsearchdomain",
                "description": "Grants permission to delete an Amazon ES domain and all of its data.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DeleteElasticsearchServiceRole": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-deleteelasticsearchservicerole",
                "description": "Grants permission to delete the service-linked role required for Amazon ES domains that use VPC access.",
                "accessLevel": "Write"
            },
            "DeleteInboundCrossClusterSearchConnection": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-deleteinboundcrossclustersearchconnection",
                "description": "Allows the destination domain owner to delete an existing inbound cross-cluster search connection",
                "accessLevel": "Write"
            },
            "DeleteOutboundCrossClusterSearchConnection": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-deleteoutboundcrossclustersearchconnection",
                "description": "Allows the source domain owner to delete an existing outbound cross-cluster search connection",
                "accessLevel": "Write"
            },
            "DescribeElasticsearchDomain": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describeelasticsearchdomain",
                "description": "Grants permission to view a description of the domain configuration for the specified Amazon ES domain, including the domain ID, domain service endpoint, and domain ARN.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DescribeElasticsearchDomainConfig": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describeelasticsearchdomainconfig",
                "description": "Grants permission to view a description of the configuration options and status of an Amazon ES domain.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DescribeElasticsearchDomains": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describeelasticsearchdomain",
                "description": "Grants permission to view a description of the domain configuration for up to five specified Amazon ES domains.",
                "accessLevel": "List",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DescribeElasticsearchInstanceTypeLimits": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describeinstancetypelimits",
                "description": "Grants permission to view the instance count, storage, and master node limits for a given Elasticsearch version and instance type.",
                "accessLevel": "List"
            },
            "DescribeInboundCrossClusterSearchConnections": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describeinboundcrossclustersearchconnections",
                "description": "Lists all the inbound cross-cluster search connections for a destination domain",
                "accessLevel": "List"
            },
            "DescribeOutboundCrossClusterSearchConnections": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describeoutboundcrossclustersearchconnections",
                "description": "Lists all the outbound cross-cluster search connections for a source domain",
                "accessLevel": "List"
            },
            "DescribeReservedElasticsearchInstanceOfferings": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describereservedelasticsearchinstanceofferings",
                "description": "Grants permission to fetch reserved instance offerings for ES",
                "accessLevel": "List"
            },
            "DescribeReservedElasticsearchInstances": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describereservedelasticsearchinstances",
                "description": "Grants permission to fetch ES reserved instances already purchased by customer",
                "accessLevel": "List"
            },
            "ESCrossClusterGet": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference",
                "description": "Grants permission to send cross-cluster requests to a destination domain.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": false
                    }
                }
            },
            "ESHttpDelete": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference",
                "description": "Grants permission to send HTTP DELETE requests to the Elasticsearch APIs.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": false
                    }
                }
            },
            "ESHttpGet": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference",
                "description": "Grants permission to send HTTP GET requests to the Elasticsearch APIs.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": false
                    }
                }
            },
            "ESHttpHead": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference",
                "description": "Grants permission to send HTTP HEAD requests to the Elasticsearch APIs.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": false
                    }
                }
            },
            "ESHttpPatch": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference",
                "description": "Grants permission to send HTTP PATCH requests to the Elasticsearch APIs.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": false
                    }
                }
            },
            "ESHttpPost": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference",
                "description": "Grants permission to send HTTP POST requests to the Elasticsearch APIs.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": false
                    }
                }
            },
            "ESHttpPut": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference",
                "description": "Grants permission to send HTTP PUT requests to the Elasticsearch APIs.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": false
                    }
                }
            },
            "GetCompatibleElasticsearchVersions": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-getcompatibleelasticsearchversions",
                "description": "Grants permission to fetch list of compatible elastic search versions to which Amazon ES domain can be upgraded",
                "accessLevel": "List",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "GetUpgradeHistory": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-getupgradehistory",
                "description": "Grants permission to fetch upgrade history for given ES domain",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "GetUpgradeStatus": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-getupgradestatus",
                "description": "Grants permission to fetch upgrade status for given ES domain",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "ListDomainNames": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-listdomainnames",
                "description": "Grants permission to display the names of all Amazon ES domains that the current user owns.",
                "accessLevel": "List"
            },
            "ListElasticsearchInstanceTypeDetails": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-listelasticsearchinstancetypedetails",
                "description": "Grants permission to list all instance types and available features for a given Elasticsearch version.",
                "accessLevel": "List"
            },
            "ListElasticsearchInstanceTypes": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-listelasticsearchinstancetypes",
                "description": "Grants permission to list all Elasticsearch instance types that are supported for a given Elasticsearch version.",
                "accessLevel": "List"
            },
            "ListElasticsearchVersions": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-listelasticsearchversions",
                "description": "Grants permission to list all supported Elasticsearch versions on Amazon ES.",
                "accessLevel": "List"
            },
            "ListTags": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-listtags",
                "description": "Grants permission to display all of the tags for an Amazon ES domain.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "PurchaseReservedElasticsearchInstanceOffering": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-purchasereservedelasticsearchinstance",
                "description": "Grants permission to purchase ES reserved instances",
                "accessLevel": "Write"
            },
            "RejectInboundCrossClusterSearchConnection": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-rejectinboundcrossclustersearchconnection",
                "description": "Allows the destination domain owner to reject an inbound cross-cluster search connection request",
                "accessLevel": "Write"
            },
            "RemoveTags": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-listtags",
                "description": "Grants permission to remove tags from Amazon ES domains.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "UpdateElasticsearchDomainConfig": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-updateelasticsearchdomainconfig",
                "description": "Grants permission to modify the configuration of an Amazon ES domain, such as the instance type or number of instances.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "UpgradeElasticsearchDomain": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-upgradeelasticsearchdomain",
                "description": "Grants permission to initiate upgrade of elastic search domain to given version",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "domain": {
                "name": "domain",
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html",
                "arn": "arn:${Partition}:es:${Region}:${Account}:domain/${DomainName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Allows the destination domain owner to accept an inbound cross-cluster search connection request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-acceptinboundcrossclustersearchconnection
     */
    acceptInboundCrossClusterSearchConnection() {
        this.add('es:AcceptInboundCrossClusterSearchConnection');
        return this;
    }
    /**
     * Grants permission to attach resource tags to an Amazon ES domain.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-addtags
     */
    addTags() {
        this.add('es:AddTags');
        return this;
    }
    /**
     * Grants permission to create an Amazon ES domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-createelasticsearchdomain
     */
    createElasticsearchDomain() {
        this.add('es:CreateElasticsearchDomain');
        return this;
    }
    /**
     * Grants permission to create the service-linked role required for Amazon ES domains that use VPC access.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-createelasticsearchservicerole
     */
    createElasticsearchServiceRole() {
        this.add('es:CreateElasticsearchServiceRole');
        return this;
    }
    /**
     * Creates a new cross-cluster search connection from a source domain to a destination domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-createoutboundcrossclustersearchconnection
     */
    createOutboundCrossClusterSearchConnection() {
        this.add('es:CreateOutboundCrossClusterSearchConnection');
        return this;
    }
    /**
     * Grants permission to delete an Amazon ES domain and all of its data.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-deleteelasticsearchdomain
     */
    deleteElasticsearchDomain() {
        this.add('es:DeleteElasticsearchDomain');
        return this;
    }
    /**
     * Grants permission to delete the service-linked role required for Amazon ES domains that use VPC access.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-deleteelasticsearchservicerole
     */
    deleteElasticsearchServiceRole() {
        this.add('es:DeleteElasticsearchServiceRole');
        return this;
    }
    /**
     * Allows the destination domain owner to delete an existing inbound cross-cluster search connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-deleteinboundcrossclustersearchconnection
     */
    deleteInboundCrossClusterSearchConnection() {
        this.add('es:DeleteInboundCrossClusterSearchConnection');
        return this;
    }
    /**
     * Allows the source domain owner to delete an existing outbound cross-cluster search connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-deleteoutboundcrossclustersearchconnection
     */
    deleteOutboundCrossClusterSearchConnection() {
        this.add('es:DeleteOutboundCrossClusterSearchConnection');
        return this;
    }
    /**
     * Grants permission to view a description of the domain configuration for the specified Amazon ES domain, including the domain ID, domain service endpoint, and domain ARN.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describeelasticsearchdomain
     */
    describeElasticsearchDomain() {
        this.add('es:DescribeElasticsearchDomain');
        return this;
    }
    /**
     * Grants permission to view a description of the configuration options and status of an Amazon ES domain.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describeelasticsearchdomainconfig
     */
    describeElasticsearchDomainConfig() {
        this.add('es:DescribeElasticsearchDomainConfig');
        return this;
    }
    /**
     * Grants permission to view a description of the domain configuration for up to five specified Amazon ES domains.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describeelasticsearchdomain
     */
    describeElasticsearchDomains() {
        this.add('es:DescribeElasticsearchDomains');
        return this;
    }
    /**
     * Grants permission to view the instance count, storage, and master node limits for a given Elasticsearch version and instance type.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describeinstancetypelimits
     */
    describeElasticsearchInstanceTypeLimits() {
        this.add('es:DescribeElasticsearchInstanceTypeLimits');
        return this;
    }
    /**
     * Lists all the inbound cross-cluster search connections for a destination domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describeinboundcrossclustersearchconnections
     */
    describeInboundCrossClusterSearchConnections() {
        this.add('es:DescribeInboundCrossClusterSearchConnections');
        return this;
    }
    /**
     * Lists all the outbound cross-cluster search connections for a source domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describeoutboundcrossclustersearchconnections
     */
    describeOutboundCrossClusterSearchConnections() {
        this.add('es:DescribeOutboundCrossClusterSearchConnections');
        return this;
    }
    /**
     * Grants permission to fetch reserved instance offerings for ES
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describereservedelasticsearchinstanceofferings
     */
    describeReservedElasticsearchInstanceOfferings() {
        this.add('es:DescribeReservedElasticsearchInstanceOfferings');
        return this;
    }
    /**
     * Grants permission to fetch ES reserved instances already purchased by customer
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describereservedelasticsearchinstances
     */
    describeReservedElasticsearchInstances() {
        this.add('es:DescribeReservedElasticsearchInstances');
        return this;
    }
    /**
     * Grants permission to send cross-cluster requests to a destination domain.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference
     */
    eSCrossClusterGet() {
        this.add('es:ESCrossClusterGet');
        return this;
    }
    /**
     * Grants permission to send HTTP DELETE requests to the Elasticsearch APIs.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference
     */
    eSHttpDelete() {
        this.add('es:ESHttpDelete');
        return this;
    }
    /**
     * Grants permission to send HTTP GET requests to the Elasticsearch APIs.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference
     */
    eSHttpGet() {
        this.add('es:ESHttpGet');
        return this;
    }
    /**
     * Grants permission to send HTTP HEAD requests to the Elasticsearch APIs.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference
     */
    eSHttpHead() {
        this.add('es:ESHttpHead');
        return this;
    }
    /**
     * Grants permission to send HTTP PATCH requests to the Elasticsearch APIs.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference
     */
    eSHttpPatch() {
        this.add('es:ESHttpPatch');
        return this;
    }
    /**
     * Grants permission to send HTTP POST requests to the Elasticsearch APIs.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference
     */
    eSHttpPost() {
        this.add('es:ESHttpPost');
        return this;
    }
    /**
     * Grants permission to send HTTP PUT requests to the Elasticsearch APIs.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference
     */
    eSHttpPut() {
        this.add('es:ESHttpPut');
        return this;
    }
    /**
     * Grants permission to fetch list of compatible elastic search versions to which Amazon ES domain can be upgraded
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-getcompatibleelasticsearchversions
     */
    getCompatibleElasticsearchVersions() {
        this.add('es:GetCompatibleElasticsearchVersions');
        return this;
    }
    /**
     * Grants permission to fetch upgrade history for given ES domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-getupgradehistory
     */
    getUpgradeHistory() {
        this.add('es:GetUpgradeHistory');
        return this;
    }
    /**
     * Grants permission to fetch upgrade status for given ES domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-getupgradestatus
     */
    getUpgradeStatus() {
        this.add('es:GetUpgradeStatus');
        return this;
    }
    /**
     * Grants permission to display the names of all Amazon ES domains that the current user owns.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-listdomainnames
     */
    listDomainNames() {
        this.add('es:ListDomainNames');
        return this;
    }
    /**
     * Grants permission to list all instance types and available features for a given Elasticsearch version.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-listelasticsearchinstancetypedetails
     */
    listElasticsearchInstanceTypeDetails() {
        this.add('es:ListElasticsearchInstanceTypeDetails');
        return this;
    }
    /**
     * Grants permission to list all Elasticsearch instance types that are supported for a given Elasticsearch version.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-listelasticsearchinstancetypes
     */
    listElasticsearchInstanceTypes() {
        this.add('es:ListElasticsearchInstanceTypes');
        return this;
    }
    /**
     * Grants permission to list all supported Elasticsearch versions on Amazon ES.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-listelasticsearchversions
     */
    listElasticsearchVersions() {
        this.add('es:ListElasticsearchVersions');
        return this;
    }
    /**
     * Grants permission to display all of the tags for an Amazon ES domain.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-listtags
     */
    listTags() {
        this.add('es:ListTags');
        return this;
    }
    /**
     * Grants permission to purchase ES reserved instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-purchasereservedelasticsearchinstance
     */
    purchaseReservedElasticsearchInstanceOffering() {
        this.add('es:PurchaseReservedElasticsearchInstanceOffering');
        return this;
    }
    /**
     * Allows the destination domain owner to reject an inbound cross-cluster search connection request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-rejectinboundcrossclustersearchconnection
     */
    rejectInboundCrossClusterSearchConnection() {
        this.add('es:RejectInboundCrossClusterSearchConnection');
        return this;
    }
    /**
     * Grants permission to remove tags from Amazon ES domains.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-listtags
     */
    removeTags() {
        this.add('es:RemoveTags');
        return this;
    }
    /**
     * Grants permission to modify the configuration of an Amazon ES domain, such as the instance type or number of instances.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-updateelasticsearchdomainconfig
     */
    updateElasticsearchDomainConfig() {
        this.add('es:UpdateElasticsearchDomainConfig');
        return this;
    }
    /**
     * Grants permission to initiate upgrade of elastic search domain to given version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-upgradeelasticsearchdomain
     */
    upgradeElasticsearchDomain() {
        this.add('es:UpgradeElasticsearchDomain');
        return this;
    }
    /**
     * Adds a resource of type domain to the statement
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html
     *
     * @param domainName - Identifier for the domainName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDomain(domainName, account, region, partition) {
        var arn = 'arn:${Partition}:es:${Region}:${Account}:domain/${DomainName}';
        arn = arn.replace('${DomainName}', domainName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Es = Es;
//# sourceMappingURL=data:application/json;base64,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