"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service glacier
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonglacier.html
 */
class Glacier extends shared_1.PolicyStatement {
    /**
     * Action provider for service glacier
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonglacier.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'glacier';
        this.actions = {
            "AbortMultipartUpload": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-multipart-abort-upload.html",
                "description": "Aborts a multipart upload identified by the upload ID",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "AbortVaultLock": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-AbortVaultLock.html",
                "description": "Aborts the vault locking process if the vault lock is not in the Locked state",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "AddTagsToVault": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-AddTagsToVault.html",
                "description": "Adds the specified tags to a vault",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "CompleteMultipartUpload": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-multipart-complete-upload.html",
                "description": "Completes a multipart upload process",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "CompleteVaultLock": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-CompleteVaultLock.html",
                "description": "Completes the vault locking process",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "CreateVault": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-put.html",
                "description": "Creates a new vault with the specified name",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "DeleteArchive": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-archive-delete.html",
                "description": "Deletes an archive from a vault",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                },
                "conditions": [
                    "glacier:ArchiveAgeInDays"
                ]
            },
            "DeleteVault": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-delete.html",
                "description": "Deletes a vault",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "DeleteVaultAccessPolicy": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-DeleteVaultAccessPolicy.html",
                "description": "Deletes the access policy associated with the specified vault",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "DeleteVaultNotifications": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-notifications-delete.html",
                "description": "Deletes the notification configuration set for a vault",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "DescribeJob": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-describe-job-get.html",
                "description": "Returns information about a job you previously initiated",
                "accessLevel": "Read",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "DescribeVault": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-get..html",
                "description": "Returns information about a vault",
                "accessLevel": "Read",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "GetDataRetrievalPolicy": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-GetDataRetrievalPolicy.html",
                "description": "Returns the current data retrieval policy for the account and region specified in the GET request",
                "accessLevel": "Read"
            },
            "GetJobOutput": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-job-output-get.html",
                "description": "Downloads the output of the job you initiated",
                "accessLevel": "Read",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "GetVaultAccessPolicy": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-GetVaultAccessPolicy.html",
                "description": "Retrieves the access-policy subresource set on the vault",
                "accessLevel": "Read",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "GetVaultLock": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-GetVaultLock.html",
                "description": "Retrieves attributes from the lock-policy subresource set on the specified vault",
                "accessLevel": "Read",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "GetVaultNotifications": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-notifications-get.html",
                "description": "Retrieves the notification-configuration subresource set on the vault",
                "accessLevel": "Read",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "InitiateJob": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-initiate-job-post.html",
                "description": "Initiates a job of the specified type",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                },
                "conditions": [
                    "glacier:ArchiveAgeInDays"
                ]
            },
            "InitiateMultipartUpload": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-multipart-initiate-upload.html",
                "description": "Initiates a multipart upload",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "InitiateVaultLock": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-InitiateVaultLock.html",
                "description": "Initiates the vault locking process",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "ListJobs": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-jobs-get.html",
                "description": "Lists jobs for a vault that are in-progress and jobs that have recently finished",
                "accessLevel": "List",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "ListMultipartUploads": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-multipart-list-uploads.html",
                "description": "Lists in-progress multipart uploads for the specified vault",
                "accessLevel": "List",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "ListParts": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-multipart-list-parts.html",
                "description": "Lists the parts of an archive that have been uploaded in a specific multipart upload",
                "accessLevel": "List",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "ListProvisionedCapacity": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-ListProvisionedCapacity.html",
                "description": "This operation lists the provisioned capacity for the specified AWS account.",
                "accessLevel": "List"
            },
            "ListTagsForVault": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-ListTagsForVault.html",
                "description": "Lists all the tags attached to a vault",
                "accessLevel": "List",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "ListVaults": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vaults-get.html",
                "description": "Lists all vaults",
                "accessLevel": "List"
            },
            "PurchaseProvisionedCapacity": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-PurchaseProvisionedCapacity.html",
                "description": "This operation purchases a provisioned capacity unit for an AWS account.",
                "accessLevel": "Write"
            },
            "RemoveTagsFromVault": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-RemoveTagsFromVault.html",
                "description": "Removes one or more tags from the set of tags attached to a vault",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "SetDataRetrievalPolicy": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-SetDataRetrievalPolicy.html",
                "description": "Sets and then enacts a data retrieval policy in the region specified in the PUT request",
                "accessLevel": "Permissions management"
            },
            "SetVaultAccessPolicy": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-SetVaultAccessPolicy.html",
                "description": "Configures an access policy for a vault and will overwrite an existing policy",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "SetVaultNotifications": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-notifications-put.html",
                "description": "Configures vault notifications",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "UploadArchive": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-archive-post.html",
                "description": "Adds an archive to a vault",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "UploadMultipartPart": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-upload-part.html",
                "description": "Uploads a part of an archive",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "vault": {
                "name": "vault",
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/working-with-vaults.html",
                "arn": "arn:${Partition}:glacier:${Region}:${Account}:vaults/${VaultName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Aborts a multipart upload identified by the upload ID
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-multipart-abort-upload.html
     */
    abortMultipartUpload() {
        this.add('glacier:AbortMultipartUpload');
        return this;
    }
    /**
     * Aborts the vault locking process if the vault lock is not in the Locked state
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-AbortVaultLock.html
     */
    abortVaultLock() {
        this.add('glacier:AbortVaultLock');
        return this;
    }
    /**
     * Adds the specified tags to a vault
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-AddTagsToVault.html
     */
    addTagsToVault() {
        this.add('glacier:AddTagsToVault');
        return this;
    }
    /**
     * Completes a multipart upload process
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-multipart-complete-upload.html
     */
    completeMultipartUpload() {
        this.add('glacier:CompleteMultipartUpload');
        return this;
    }
    /**
     * Completes the vault locking process
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-CompleteVaultLock.html
     */
    completeVaultLock() {
        this.add('glacier:CompleteVaultLock');
        return this;
    }
    /**
     * Creates a new vault with the specified name
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-put.html
     */
    createVault() {
        this.add('glacier:CreateVault');
        return this;
    }
    /**
     * Deletes an archive from a vault
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-archive-delete.html
     */
    deleteArchive() {
        this.add('glacier:DeleteArchive');
        return this;
    }
    /**
     * Deletes a vault
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-delete.html
     */
    deleteVault() {
        this.add('glacier:DeleteVault');
        return this;
    }
    /**
     * Deletes the access policy associated with the specified vault
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-DeleteVaultAccessPolicy.html
     */
    deleteVaultAccessPolicy() {
        this.add('glacier:DeleteVaultAccessPolicy');
        return this;
    }
    /**
     * Deletes the notification configuration set for a vault
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-notifications-delete.html
     */
    deleteVaultNotifications() {
        this.add('glacier:DeleteVaultNotifications');
        return this;
    }
    /**
     * Returns information about a job you previously initiated
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-describe-job-get.html
     */
    describeJob() {
        this.add('glacier:DescribeJob');
        return this;
    }
    /**
     * Returns information about a vault
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-get..html
     */
    describeVault() {
        this.add('glacier:DescribeVault');
        return this;
    }
    /**
     * Returns the current data retrieval policy for the account and region specified in the GET request
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-GetDataRetrievalPolicy.html
     */
    getDataRetrievalPolicy() {
        this.add('glacier:GetDataRetrievalPolicy');
        return this;
    }
    /**
     * Downloads the output of the job you initiated
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-job-output-get.html
     */
    getJobOutput() {
        this.add('glacier:GetJobOutput');
        return this;
    }
    /**
     * Retrieves the access-policy subresource set on the vault
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-GetVaultAccessPolicy.html
     */
    getVaultAccessPolicy() {
        this.add('glacier:GetVaultAccessPolicy');
        return this;
    }
    /**
     * Retrieves attributes from the lock-policy subresource set on the specified vault
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-GetVaultLock.html
     */
    getVaultLock() {
        this.add('glacier:GetVaultLock');
        return this;
    }
    /**
     * Retrieves the notification-configuration subresource set on the vault
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-notifications-get.html
     */
    getVaultNotifications() {
        this.add('glacier:GetVaultNotifications');
        return this;
    }
    /**
     * Initiates a job of the specified type
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-initiate-job-post.html
     */
    initiateJob() {
        this.add('glacier:InitiateJob');
        return this;
    }
    /**
     * Initiates a multipart upload
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-multipart-initiate-upload.html
     */
    initiateMultipartUpload() {
        this.add('glacier:InitiateMultipartUpload');
        return this;
    }
    /**
     * Initiates the vault locking process
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-InitiateVaultLock.html
     */
    initiateVaultLock() {
        this.add('glacier:InitiateVaultLock');
        return this;
    }
    /**
     * Lists jobs for a vault that are in-progress and jobs that have recently finished
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-jobs-get.html
     */
    listJobs() {
        this.add('glacier:ListJobs');
        return this;
    }
    /**
     * Lists in-progress multipart uploads for the specified vault
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-multipart-list-uploads.html
     */
    listMultipartUploads() {
        this.add('glacier:ListMultipartUploads');
        return this;
    }
    /**
     * Lists the parts of an archive that have been uploaded in a specific multipart upload
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-multipart-list-parts.html
     */
    listParts() {
        this.add('glacier:ListParts');
        return this;
    }
    /**
     * This operation lists the provisioned capacity for the specified AWS account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-ListProvisionedCapacity.html
     */
    listProvisionedCapacity() {
        this.add('glacier:ListProvisionedCapacity');
        return this;
    }
    /**
     * Lists all the tags attached to a vault
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-ListTagsForVault.html
     */
    listTagsForVault() {
        this.add('glacier:ListTagsForVault');
        return this;
    }
    /**
     * Lists all vaults
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vaults-get.html
     */
    listVaults() {
        this.add('glacier:ListVaults');
        return this;
    }
    /**
     * This operation purchases a provisioned capacity unit for an AWS account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-PurchaseProvisionedCapacity.html
     */
    purchaseProvisionedCapacity() {
        this.add('glacier:PurchaseProvisionedCapacity');
        return this;
    }
    /**
     * Removes one or more tags from the set of tags attached to a vault
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-RemoveTagsFromVault.html
     */
    removeTagsFromVault() {
        this.add('glacier:RemoveTagsFromVault');
        return this;
    }
    /**
     * Sets and then enacts a data retrieval policy in the region specified in the PUT request
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-SetDataRetrievalPolicy.html
     */
    setDataRetrievalPolicy() {
        this.add('glacier:SetDataRetrievalPolicy');
        return this;
    }
    /**
     * Configures an access policy for a vault and will overwrite an existing policy
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-SetVaultAccessPolicy.html
     */
    setVaultAccessPolicy() {
        this.add('glacier:SetVaultAccessPolicy');
        return this;
    }
    /**
     * Configures vault notifications
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-notifications-put.html
     */
    setVaultNotifications() {
        this.add('glacier:SetVaultNotifications');
        return this;
    }
    /**
     * Adds an archive to a vault
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-archive-post.html
     */
    uploadArchive() {
        this.add('glacier:UploadArchive');
        return this;
    }
    /**
     * Uploads a part of an archive
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-upload-part.html
     */
    uploadMultipartPart() {
        this.add('glacier:UploadMultipartPart');
        return this;
    }
    /**
     * Adds a resource of type vault to the statement
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/working-with-vaults.html
     *
     * @param vaultName - Identifier for the vaultName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onVault(vaultName, account, region, partition) {
        var arn = 'arn:${Partition}:glacier:${Region}:${Account}:vaults/${VaultName}';
        arn = arn.replace('${VaultName}', vaultName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * How long an archive has been stored in the vault, in days.
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/access-control-overview.html#specifying-conditions
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifArchiveAgeInDays(value, operator) {
        return this.if(`glacier:ArchiveAgeInDays`, value, operator || 'StringLike');
    }
    /**
     * A customer-defined tag.
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/access-control-overview.html#specifying-conditions
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTagExists(value, operator) {
        return this.if(`glacier:ResourceTag/`, value, operator || 'StringLike');
    }
}
exports.Glacier = Glacier;
//# sourceMappingURL=data:application/json;base64,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