"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service guardduty
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonguardduty.html
 */
class Guardduty extends shared_1.PolicyStatement {
    /**
     * Action provider for service guardduty
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonguardduty.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'guardduty';
        this.actions = {
            "AcceptInvitation": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_AcceptInvitation.html",
                "description": "Grants permission to accept invitations to become a GuardDuty member account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "ArchiveFindings": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ArchiveFindings.html",
                "description": "Grants permission to archive GuardDuty findings.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "CreateDetector": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateDetector.html",
                "description": "Grants permission to create a detector.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateFilter": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateFilter.html",
                "description": "Grants permission to create GuardDuty filters. A filters defines finding attributes and conditions used to filter findings.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateIPSet": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateIPSet.html",
                "description": "Grants permission to create an IPSet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateMembers": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateMembers.html",
                "description": "Grants permission to create GuardDuty member accounts. The account used to create a member becomes the GuardDuty master account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "CreatePublishingDestination": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreatePublishingDestination.html",
                "description": "Grants permission to create a publishing destination.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "CreateSampleFindings": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateSampleFindings.html",
                "description": "Grants permission to create sample findings.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "CreateThreatIntelSet": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateThreatIntelSet.html",
                "description": "Grants permission to create GuardDuty ThreatIntelSets. A ThreatIntelSet consists of known malicious IP addresses used by GuardDuty to generate findings.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeclineInvitations": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeclineInvitations.html",
                "description": "Grants permission to decline invitations to become a GuardDuty member account.",
                "accessLevel": "Write"
            },
            "DeleteDetector": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteDetector.html",
                "description": "Grants permission to delete GuardDuty detectors.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "DeleteFilter": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteFilter.html",
                "description": "Grants permission to delete GuardDuty filters.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    },
                    "filter": {
                        "required": true
                    }
                }
            },
            "DeleteIPSet": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteIPSet.html",
                "description": "Grants permission to delete GuardDuty IPSets.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    },
                    "ipset": {
                        "required": true
                    }
                }
            },
            "DeleteInvitations": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteInvitations.html",
                "description": "Grants permission to delete invitations to become a GuardDuty member account.",
                "accessLevel": "Write"
            },
            "DeleteMembers": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteMembers.html",
                "description": "Grants permission to delete GuardDuty member accounts.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "DeletePublishingDestination": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeletePublishingDestination.html",
                "description": "Grants permission to delete a publishing destination.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    },
                    "publishingdestination": {
                        "required": true
                    }
                }
            },
            "DeleteThreatIntelSet": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteThreatIntelSet.html",
                "description": "Grants permission to delete GuardDuty ThreatIntelSets.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    },
                    "threatintelset": {
                        "required": true
                    }
                }
            },
            "DescribeOrganizationConfiguration": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DescribeOrganizationConfiguration.html",
                "description": "Grants permission to retrieve details about the delegated administrator associated with a GuardDuty detector.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "DescribePublishingDestination": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DescribePublishingDestination.html",
                "description": "Grants permission to retrieve details about a publishing destination.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    },
                    "publishingdestination": {
                        "required": true
                    }
                }
            },
            "DisableOrganizationAdminAccount": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DisableOrganizationAdminAccount.html",
                "description": "Grants permission to disable the organization delegated administrator for GuardDuty.",
                "accessLevel": "Write"
            },
            "DisassociateFromMasterAccount": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DisassociateFromMasterAccount.html",
                "description": "Grants permission to disassociate a GuardDuty member account from its GuardDuty master account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "DisassociateMembers": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DisassociateMembers.html",
                "description": "Grants permission to disassociate GuardDuty member accounts from their master GuardDuty account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "EnableOrganizationAdminAccount": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_EnableOrganizationAdminAccount.html",
                "description": "Grants permission to enable an organization delegated administrator for GuardDuty.",
                "accessLevel": "Write"
            },
            "GetDetector": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetDetector.html",
                "description": "Grants permission to retrieve GuardDuty detectors.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "GetFilter": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetFilter.html",
                "description": "Grants permission to retrieve GuardDuty filters.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    },
                    "filter": {
                        "required": true
                    }
                }
            },
            "GetFindings": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetFindings.html",
                "description": "Grants permission to retrieve GuardDuty findings.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "GetFindingsStatistics": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetFindingsStatistics.html",
                "description": "Grants permission to retrieve a list of GuardDuty finding statistics.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "GetIPSet": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetIPSet.html",
                "description": "Grants permsission to retrieve GuardDuty IPSets.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    },
                    "ipset": {
                        "required": true
                    }
                }
            },
            "GetInvitationsCount": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetInvitationsCount.html",
                "description": "Grants permission to retrieve the count of all GuardDuty invitations sent to a specified account. The count does not include an accepted invitation.",
                "accessLevel": "Read"
            },
            "GetMasterAccount": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetMasterAccount.html",
                "description": "Grants permission to retrieve details of the GuardDuty master account associated with a member account.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "GetMembers": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetMembers.html",
                "description": "Grants permission to retrieve the member accounts associated with a master account.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "GetThreatIntelSet": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetThreatIntelSet.html",
                "description": "Grants permission to retrieve GuardDuty ThreatIntelSets.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    },
                    "threatintelset": {
                        "required": true
                    }
                }
            },
            "InviteMembers": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_InviteMembers.html",
                "description": "Grants permission to invite other AWS accounts to enable GuardDuty and become GuardDuty member accounts.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "ListDetectors": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListDetectors.html",
                "description": "Grants permission to retrieve a list of GuardDuty detectors.",
                "accessLevel": "List"
            },
            "ListFilters": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListFilters.html",
                "description": "Grants permission to retrieve a list of GuardDuty filters.",
                "accessLevel": "List",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "ListFindings": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListFindings.html",
                "description": "Grants permission to retrieve a list of GuardDuty findings.",
                "accessLevel": "List",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "ListInvitations": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListInvitations.html",
                "description": "Grants permission to retrieve a lists of all of the GuardDuty membership invitations that were sent to an AWS account.",
                "accessLevel": "List"
            },
            "ListMembers": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListMembers.html",
                "description": "Grants permission to retrierve a lsit of GuardDuty member accounts associated with a master account.",
                "accessLevel": "List",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "ListOrganizationAdminAccounts": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListOrganizationAdminAccounts.html",
                "description": "Grants permission to list details about the organization delegated administrator for GuardDuty.",
                "accessLevel": "List"
            },
            "ListPublishingDestinations": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListPublishingDestinations.html",
                "description": "Grants permission to retrieve a list of publishing destinations.",
                "accessLevel": "List",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListTagsForResource.html",
                "description": "Grants permission to retrieve a list of tags associated with a GuardDuty resource.",
                "accessLevel": "List",
                "resourceTypes": {
                    "detector": {
                        "required": false
                    },
                    "filter": {
                        "required": false
                    },
                    "ipset": {
                        "required": false
                    },
                    "threatintelset": {
                        "required": false
                    }
                }
            },
            "ListThreatIntelSets": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListThreatIntelSets.html",
                "description": "Grants permission to retrieve a list of GuardDuty ThreatIntelSets.",
                "accessLevel": "List",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "StartMonitoringMembers": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_StartMonitoringMembers.html",
                "description": "Grants permission to a master account to monitor findings from GuardDuty member accounts. Use this after disabling monitoring of member accounts using the StopMonitoringMembers operation.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "StopMonitoringMembers": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_StopMonitoringMembers.html",
                "description": "Grants permission to disable monitoring findings from member accounts.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_TagResource.html",
                "description": "Grants permission to add tags to a GuardDuty resource. There is a limit of 50 tags per resource.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": false
                    },
                    "filter": {
                        "required": false
                    },
                    "ipset": {
                        "required": false
                    },
                    "threatintelset": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UnarchiveFindings": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UnarchiveFindings.html",
                "description": "Grants permission to unarchive GuardDuty findings.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UntagResource.html",
                "description": "Grants permission to remove tags from a GuardDuty resource.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": false
                    },
                    "filter": {
                        "required": false
                    },
                    "ipset": {
                        "required": false
                    },
                    "threatintelset": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateDetector": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateDetector.html",
                "description": "Grants permission to update GuardDuty detectors.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "UpdateFilter": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateFilter.html",
                "description": "Grants permission to updates GuardDuty filters.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    },
                    "filter": {
                        "required": true
                    }
                }
            },
            "UpdateFindingsFeedback": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateFindingsFeedback.html",
                "description": "Grants permission to update findings feedback to mark GuardDuty findings as useful or not useful.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "UpdateIPSet": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateIPSet.html",
                "description": "Grants permission to update GuardDuty IPSets.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    },
                    "ipset": {
                        "required": true
                    }
                }
            },
            "UpdateOrganizationConfiguration": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateOrganizationConfiguration.html",
                "description": "Grants permission to update the delegated administrator configuration associated with a GuardDuty detector.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "UpdatePublishingDestination": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdatePublishingDestination.html",
                "description": "Grants permission to update a publishing destination.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    },
                    "publishingdestination": {
                        "required": true
                    }
                }
            },
            "UpdateThreatIntelSet": {
                "url": "https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateThreatIntelSet.html",
                "description": "Grants permission to updates the GuardDuty ThreatIntelSets.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    },
                    "threatintelset": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "detector": {
                "name": "detector",
                "url": "https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_managing_access.html#guardduty-resources",
                "arn": "arn:${Partition}:guardduty:${Region}:${Account}:detector/${DetectorId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "filter": {
                "name": "filter",
                "url": "https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_managing_access.html#guardduty-resources",
                "arn": "arn:${Partition}:guardduty:${Region}:${Account}:detector/${DetectorId}/filter/${FilterName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "ipset": {
                "name": "ipset",
                "url": "https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_managing_access.html#guardduty-resources",
                "arn": "arn:${Partition}:guardduty:${Region}:${Account}:detector/${DetectorId}/ipset/${IPSetId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "threatintelset": {
                "name": "threatintelset",
                "url": "https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_managing_access.html#guardduty-resources",
                "arn": "arn:${Partition}:guardduty:${Region}:${Account}:detector/${DetectorId}/threatintelset/${ThreatIntelSetId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "publishingdestination": {
                "name": "publishingdestination",
                "url": "https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_managing_access.html#guardduty-resources",
                "arn": "arn:${Partition}:guardduty:${Region}:${Account}:detector/${DetectorId}/threatintelset/${PublishingDestinationId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to accept invitations to become a GuardDuty member account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_AcceptInvitation.html
     */
    acceptInvitation() {
        this.add('guardduty:AcceptInvitation');
        return this;
    }
    /**
     * Grants permission to archive GuardDuty findings.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ArchiveFindings.html
     */
    archiveFindings() {
        this.add('guardduty:ArchiveFindings');
        return this;
    }
    /**
     * Grants permission to create a detector.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateDetector.html
     */
    createDetector() {
        this.add('guardduty:CreateDetector');
        return this;
    }
    /**
     * Grants permission to create GuardDuty filters. A filters defines finding attributes and conditions used to filter findings.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateFilter.html
     */
    createFilter() {
        this.add('guardduty:CreateFilter');
        return this;
    }
    /**
     * Grants permission to create an IPSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateIPSet.html
     */
    createIPSet() {
        this.add('guardduty:CreateIPSet');
        return this;
    }
    /**
     * Grants permission to create GuardDuty member accounts. The account used to create a member becomes the GuardDuty master account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateMembers.html
     */
    createMembers() {
        this.add('guardduty:CreateMembers');
        return this;
    }
    /**
     * Grants permission to create a publishing destination.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreatePublishingDestination.html
     */
    createPublishingDestination() {
        this.add('guardduty:CreatePublishingDestination');
        return this;
    }
    /**
     * Grants permission to create sample findings.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateSampleFindings.html
     */
    createSampleFindings() {
        this.add('guardduty:CreateSampleFindings');
        return this;
    }
    /**
     * Grants permission to create GuardDuty ThreatIntelSets. A ThreatIntelSet consists of known malicious IP addresses used by GuardDuty to generate findings.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateThreatIntelSet.html
     */
    createThreatIntelSet() {
        this.add('guardduty:CreateThreatIntelSet');
        return this;
    }
    /**
     * Grants permission to decline invitations to become a GuardDuty member account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeclineInvitations.html
     */
    declineInvitations() {
        this.add('guardduty:DeclineInvitations');
        return this;
    }
    /**
     * Grants permission to delete GuardDuty detectors.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteDetector.html
     */
    deleteDetector() {
        this.add('guardduty:DeleteDetector');
        return this;
    }
    /**
     * Grants permission to delete GuardDuty filters.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteFilter.html
     */
    deleteFilter() {
        this.add('guardduty:DeleteFilter');
        return this;
    }
    /**
     * Grants permission to delete GuardDuty IPSets.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteIPSet.html
     */
    deleteIPSet() {
        this.add('guardduty:DeleteIPSet');
        return this;
    }
    /**
     * Grants permission to delete invitations to become a GuardDuty member account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteInvitations.html
     */
    deleteInvitations() {
        this.add('guardduty:DeleteInvitations');
        return this;
    }
    /**
     * Grants permission to delete GuardDuty member accounts.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteMembers.html
     */
    deleteMembers() {
        this.add('guardduty:DeleteMembers');
        return this;
    }
    /**
     * Grants permission to delete a publishing destination.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeletePublishingDestination.html
     */
    deletePublishingDestination() {
        this.add('guardduty:DeletePublishingDestination');
        return this;
    }
    /**
     * Grants permission to delete GuardDuty ThreatIntelSets.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteThreatIntelSet.html
     */
    deleteThreatIntelSet() {
        this.add('guardduty:DeleteThreatIntelSet');
        return this;
    }
    /**
     * Grants permission to retrieve details about the delegated administrator associated with a GuardDuty detector.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DescribeOrganizationConfiguration.html
     */
    describeOrganizationConfiguration() {
        this.add('guardduty:DescribeOrganizationConfiguration');
        return this;
    }
    /**
     * Grants permission to retrieve details about a publishing destination.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DescribePublishingDestination.html
     */
    describePublishingDestination() {
        this.add('guardduty:DescribePublishingDestination');
        return this;
    }
    /**
     * Grants permission to disable the organization delegated administrator for GuardDuty.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DisableOrganizationAdminAccount.html
     */
    disableOrganizationAdminAccount() {
        this.add('guardduty:DisableOrganizationAdminAccount');
        return this;
    }
    /**
     * Grants permission to disassociate a GuardDuty member account from its GuardDuty master account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DisassociateFromMasterAccount.html
     */
    disassociateFromMasterAccount() {
        this.add('guardduty:DisassociateFromMasterAccount');
        return this;
    }
    /**
     * Grants permission to disassociate GuardDuty member accounts from their master GuardDuty account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DisassociateMembers.html
     */
    disassociateMembers() {
        this.add('guardduty:DisassociateMembers');
        return this;
    }
    /**
     * Grants permission to enable an organization delegated administrator for GuardDuty.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_EnableOrganizationAdminAccount.html
     */
    enableOrganizationAdminAccount() {
        this.add('guardduty:EnableOrganizationAdminAccount');
        return this;
    }
    /**
     * Grants permission to retrieve GuardDuty detectors.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetDetector.html
     */
    getDetector() {
        this.add('guardduty:GetDetector');
        return this;
    }
    /**
     * Grants permission to retrieve GuardDuty filters.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetFilter.html
     */
    getFilter() {
        this.add('guardduty:GetFilter');
        return this;
    }
    /**
     * Grants permission to retrieve GuardDuty findings.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetFindings.html
     */
    getFindings() {
        this.add('guardduty:GetFindings');
        return this;
    }
    /**
     * Grants permission to retrieve a list of GuardDuty finding statistics.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetFindingsStatistics.html
     */
    getFindingsStatistics() {
        this.add('guardduty:GetFindingsStatistics');
        return this;
    }
    /**
     * Grants permsission to retrieve GuardDuty IPSets.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetIPSet.html
     */
    getIPSet() {
        this.add('guardduty:GetIPSet');
        return this;
    }
    /**
     * Grants permission to retrieve the count of all GuardDuty invitations sent to a specified account. The count does not include an accepted invitation.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetInvitationsCount.html
     */
    getInvitationsCount() {
        this.add('guardduty:GetInvitationsCount');
        return this;
    }
    /**
     * Grants permission to retrieve details of the GuardDuty master account associated with a member account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetMasterAccount.html
     */
    getMasterAccount() {
        this.add('guardduty:GetMasterAccount');
        return this;
    }
    /**
     * Grants permission to retrieve the member accounts associated with a master account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetMembers.html
     */
    getMembers() {
        this.add('guardduty:GetMembers');
        return this;
    }
    /**
     * Grants permission to retrieve GuardDuty ThreatIntelSets.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetThreatIntelSet.html
     */
    getThreatIntelSet() {
        this.add('guardduty:GetThreatIntelSet');
        return this;
    }
    /**
     * Grants permission to invite other AWS accounts to enable GuardDuty and become GuardDuty member accounts.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_InviteMembers.html
     */
    inviteMembers() {
        this.add('guardduty:InviteMembers');
        return this;
    }
    /**
     * Grants permission to retrieve a list of GuardDuty detectors.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListDetectors.html
     */
    listDetectors() {
        this.add('guardduty:ListDetectors');
        return this;
    }
    /**
     * Grants permission to retrieve a list of GuardDuty filters.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListFilters.html
     */
    listFilters() {
        this.add('guardduty:ListFilters');
        return this;
    }
    /**
     * Grants permission to retrieve a list of GuardDuty findings.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListFindings.html
     */
    listFindings() {
        this.add('guardduty:ListFindings');
        return this;
    }
    /**
     * Grants permission to retrieve a lists of all of the GuardDuty membership invitations that were sent to an AWS account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListInvitations.html
     */
    listInvitations() {
        this.add('guardduty:ListInvitations');
        return this;
    }
    /**
     * Grants permission to retrierve a lsit of GuardDuty member accounts associated with a master account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListMembers.html
     */
    listMembers() {
        this.add('guardduty:ListMembers');
        return this;
    }
    /**
     * Grants permission to list details about the organization delegated administrator for GuardDuty.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListOrganizationAdminAccounts.html
     */
    listOrganizationAdminAccounts() {
        this.add('guardduty:ListOrganizationAdminAccounts');
        return this;
    }
    /**
     * Grants permission to retrieve a list of publishing destinations.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListPublishingDestinations.html
     */
    listPublishingDestinations() {
        this.add('guardduty:ListPublishingDestinations');
        return this;
    }
    /**
     * Grants permission to retrieve a list of tags associated with a GuardDuty resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('guardduty:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to retrieve a list of GuardDuty ThreatIntelSets.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListThreatIntelSets.html
     */
    listThreatIntelSets() {
        this.add('guardduty:ListThreatIntelSets');
        return this;
    }
    /**
     * Grants permission to a master account to monitor findings from GuardDuty member accounts. Use this after disabling monitoring of member accounts using the StopMonitoringMembers operation.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_StartMonitoringMembers.html
     */
    startMonitoringMembers() {
        this.add('guardduty:StartMonitoringMembers');
        return this;
    }
    /**
     * Grants permission to disable monitoring findings from member accounts.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_StopMonitoringMembers.html
     */
    stopMonitoringMembers() {
        this.add('guardduty:StopMonitoringMembers');
        return this;
    }
    /**
     * Grants permission to add tags to a GuardDuty resource. There is a limit of 50 tags per resource.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_TagResource.html
     */
    tagResource() {
        this.add('guardduty:TagResource');
        return this;
    }
    /**
     * Grants permission to unarchive GuardDuty findings.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UnarchiveFindings.html
     */
    unarchiveFindings() {
        this.add('guardduty:UnarchiveFindings');
        return this;
    }
    /**
     * Grants permission to remove tags from a GuardDuty resource.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UntagResource.html
     */
    untagResource() {
        this.add('guardduty:UntagResource');
        return this;
    }
    /**
     * Grants permission to update GuardDuty detectors.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateDetector.html
     */
    updateDetector() {
        this.add('guardduty:UpdateDetector');
        return this;
    }
    /**
     * Grants permission to updates GuardDuty filters.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateFilter.html
     */
    updateFilter() {
        this.add('guardduty:UpdateFilter');
        return this;
    }
    /**
     * Grants permission to update findings feedback to mark GuardDuty findings as useful or not useful.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateFindingsFeedback.html
     */
    updateFindingsFeedback() {
        this.add('guardduty:UpdateFindingsFeedback');
        return this;
    }
    /**
     * Grants permission to update GuardDuty IPSets.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateIPSet.html
     */
    updateIPSet() {
        this.add('guardduty:UpdateIPSet');
        return this;
    }
    /**
     * Grants permission to update the delegated administrator configuration associated with a GuardDuty detector.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateOrganizationConfiguration.html
     */
    updateOrganizationConfiguration() {
        this.add('guardduty:UpdateOrganizationConfiguration');
        return this;
    }
    /**
     * Grants permission to update a publishing destination.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdatePublishingDestination.html
     */
    updatePublishingDestination() {
        this.add('guardduty:UpdatePublishingDestination');
        return this;
    }
    /**
     * Grants permission to updates the GuardDuty ThreatIntelSets.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateThreatIntelSet.html
     */
    updateThreatIntelSet() {
        this.add('guardduty:UpdateThreatIntelSet');
        return this;
    }
    /**
     * Adds a resource of type detector to the statement
     *
     * https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_managing_access.html#guardduty-resources
     *
     * @param detectorId - Identifier for the detectorId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onDetector(detectorId, account, region, partition) {
        var arn = 'arn:${Partition}:guardduty:${Region}:${Account}:detector/${DetectorId}';
        arn = arn.replace('${DetectorId}', detectorId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type filter to the statement
     *
     * https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_managing_access.html#guardduty-resources
     *
     * @param detectorId - Identifier for the detectorId.
     * @param filterName - Identifier for the filterName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onFilter(detectorId, filterName, account, region, partition) {
        var arn = 'arn:${Partition}:guardduty:${Region}:${Account}:detector/${DetectorId}/filter/${FilterName}';
        arn = arn.replace('${DetectorId}', detectorId);
        arn = arn.replace('${FilterName}', filterName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type ipset to the statement
     *
     * https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_managing_access.html#guardduty-resources
     *
     * @param detectorId - Identifier for the detectorId.
     * @param iPSetId - Identifier for the iPSetId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onIpset(detectorId, iPSetId, account, region, partition) {
        var arn = 'arn:${Partition}:guardduty:${Region}:${Account}:detector/${DetectorId}/ipset/${IPSetId}';
        arn = arn.replace('${DetectorId}', detectorId);
        arn = arn.replace('${IPSetId}', iPSetId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type threatintelset to the statement
     *
     * https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_managing_access.html#guardduty-resources
     *
     * @param detectorId - Identifier for the detectorId.
     * @param threatIntelSetId - Identifier for the threatIntelSetId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onThreatintelset(detectorId, threatIntelSetId, account, region, partition) {
        var arn = 'arn:${Partition}:guardduty:${Region}:${Account}:detector/${DetectorId}/threatintelset/${ThreatIntelSetId}';
        arn = arn.replace('${DetectorId}', detectorId);
        arn = arn.replace('${ThreatIntelSetId}', threatIntelSetId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type publishingdestination to the statement
     *
     * https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_managing_access.html#guardduty-resources
     *
     * @param detectorId - Identifier for the detectorId.
     * @param publishingDestinationId - Identifier for the publishingDestinationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onPublishingdestination(detectorId, publishingDestinationId, account, region, partition) {
        var arn = 'arn:${Partition}:guardduty:${Region}:${Account}:detector/${DetectorId}/threatintelset/${PublishingDestinationId}';
        arn = arn.replace('${DetectorId}', detectorId);
        arn = arn.replace('${PublishingDestinationId}', publishingDestinationId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Guardduty = Guardduty;
//# sourceMappingURL=data:application/json;base64,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