"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service kinesisanalytics-v2
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonkinesisanalyticsv2.html
 */
class KinesisanalyticsV2 extends shared_1.PolicyStatement {
    /**
     * Action provider for service kinesisanalytics-v2
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonkinesisanalyticsv2.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'kinesisanalytics-v2';
        this.actions = {
            "AddApplicationCloudWatchLoggingOption": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_AddApplicationCloudWatchLoggingOption.html",
                "description": "Adds cloudwatch logging option to the application.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "AddApplicationInput": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_AddApplicationInput.html",
                "description": "Adds input to the application.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "AddApplicationInputProcessingConfiguration": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_AddApplicationInputProcessingConfiguration.html",
                "description": "Adds input processing configuration to the application.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "AddApplicationOutput": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_AddApplicationOutput.html",
                "description": "Adds output to the application.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "AddApplicationReferenceDataSource": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_AddApplicationReferenceDataSource.html",
                "description": "Adds reference data source to the application.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "AddApplicationVpcConfiguration": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_AddApplicationVpcConfiguration.html",
                "description": "Adds VPC configuration to the application.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "CreateApplication": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_CreateApplication.html",
                "description": "Creates an application.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateApplicationSnapshot": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_CreateApplicationSnapshot.html",
                "description": "Creates a snapshot for an application.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "DeleteApplication": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_DeleteApplication.html",
                "description": "Deletes the application.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "DeleteApplicationCloudWatchLoggingOption": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_DeleteApplicationCloudWatchLoggingOption.html",
                "description": "Deletes the specified cloudwatch logging option of the application.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "DeleteApplicationInputProcessingConfiguration": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_DeleteApplicationInputProcessingConfiguration.html",
                "description": "Deletes the specified input processing configuration of the application.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "DeleteApplicationOutput": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_DeleteApplicationOutput.html",
                "description": "Deletes the specified output of the application.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "DeleteApplicationReferenceDataSource": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_DeleteApplicationReferenceDataSource.html",
                "description": "Deletes the specified reference data source of the application.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "DeleteApplicationSnapshot": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_DeleteApplicationSnapshot.html",
                "description": "Deletes a snapshot for an application.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "DeleteApplicationVpcConfiguration": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_DeleteApplicationVpcConfiguration.html",
                "description": "Deletes the specified VPC configuration of the application.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "DescribeApplication": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_DescribeApplication.html",
                "description": "Describes the specified application.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "DescribeApplicationSnapshot": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_DescribeApplicationSnapshot.html",
                "description": "Describes an application snapshot.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "DiscoverInputSchema": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_DiscoverInputSchema.html",
                "description": "Discovers the input schema for the application.",
                "accessLevel": "Read"
            },
            "ListApplicationSnapshots": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_ListApplicationSnapshots.html",
                "description": "Lists the snapshots for an application.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "ListApplications": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_ListApplications.html",
                "description": "List applications for the account",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_ListTagsForResource.html",
                "description": "Fetch the tags associated with the application.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "StartApplication": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_StartsApplication.html",
                "description": "Starts the application.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "StopApplication": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_StopApplication.html",
                "description": "Stops the application.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_TagResource.html",
                "description": "Add tags to the application.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_UntagResource.html",
                "description": "Remove the specified tags from the application.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateApplication": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_UpdateApplication.html",
                "description": "Updates the application.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "application": {
                "name": "application",
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/how-it-works.html",
                "arn": "arn:${Partition}:kinesisanalytics:${Region}:${Account}:application/${ApplicationName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Adds cloudwatch logging option to the application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_AddApplicationCloudWatchLoggingOption.html
     */
    addApplicationCloudWatchLoggingOption() {
        this.add('kinesisanalytics-v2:AddApplicationCloudWatchLoggingOption');
        return this;
    }
    /**
     * Adds input to the application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_AddApplicationInput.html
     */
    addApplicationInput() {
        this.add('kinesisanalytics-v2:AddApplicationInput');
        return this;
    }
    /**
     * Adds input processing configuration to the application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_AddApplicationInputProcessingConfiguration.html
     */
    addApplicationInputProcessingConfiguration() {
        this.add('kinesisanalytics-v2:AddApplicationInputProcessingConfiguration');
        return this;
    }
    /**
     * Adds output to the application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_AddApplicationOutput.html
     */
    addApplicationOutput() {
        this.add('kinesisanalytics-v2:AddApplicationOutput');
        return this;
    }
    /**
     * Adds reference data source to the application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_AddApplicationReferenceDataSource.html
     */
    addApplicationReferenceDataSource() {
        this.add('kinesisanalytics-v2:AddApplicationReferenceDataSource');
        return this;
    }
    /**
     * Adds VPC configuration to the application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_AddApplicationVpcConfiguration.html
     */
    addApplicationVpcConfiguration() {
        this.add('kinesisanalytics-v2:AddApplicationVpcConfiguration');
        return this;
    }
    /**
     * Creates an application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_CreateApplication.html
     */
    createApplication() {
        this.add('kinesisanalytics-v2:CreateApplication');
        return this;
    }
    /**
     * Creates a snapshot for an application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_CreateApplicationSnapshot.html
     */
    createApplicationSnapshot() {
        this.add('kinesisanalytics-v2:CreateApplicationSnapshot');
        return this;
    }
    /**
     * Deletes the application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_DeleteApplication.html
     */
    deleteApplication() {
        this.add('kinesisanalytics-v2:DeleteApplication');
        return this;
    }
    /**
     * Deletes the specified cloudwatch logging option of the application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_DeleteApplicationCloudWatchLoggingOption.html
     */
    deleteApplicationCloudWatchLoggingOption() {
        this.add('kinesisanalytics-v2:DeleteApplicationCloudWatchLoggingOption');
        return this;
    }
    /**
     * Deletes the specified input processing configuration of the application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_DeleteApplicationInputProcessingConfiguration.html
     */
    deleteApplicationInputProcessingConfiguration() {
        this.add('kinesisanalytics-v2:DeleteApplicationInputProcessingConfiguration');
        return this;
    }
    /**
     * Deletes the specified output of the application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_DeleteApplicationOutput.html
     */
    deleteApplicationOutput() {
        this.add('kinesisanalytics-v2:DeleteApplicationOutput');
        return this;
    }
    /**
     * Deletes the specified reference data source of the application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_DeleteApplicationReferenceDataSource.html
     */
    deleteApplicationReferenceDataSource() {
        this.add('kinesisanalytics-v2:DeleteApplicationReferenceDataSource');
        return this;
    }
    /**
     * Deletes a snapshot for an application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_DeleteApplicationSnapshot.html
     */
    deleteApplicationSnapshot() {
        this.add('kinesisanalytics-v2:DeleteApplicationSnapshot');
        return this;
    }
    /**
     * Deletes the specified VPC configuration of the application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_DeleteApplicationVpcConfiguration.html
     */
    deleteApplicationVpcConfiguration() {
        this.add('kinesisanalytics-v2:DeleteApplicationVpcConfiguration');
        return this;
    }
    /**
     * Describes the specified application.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_DescribeApplication.html
     */
    describeApplication() {
        this.add('kinesisanalytics-v2:DescribeApplication');
        return this;
    }
    /**
     * Describes an application snapshot.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_DescribeApplicationSnapshot.html
     */
    describeApplicationSnapshot() {
        this.add('kinesisanalytics-v2:DescribeApplicationSnapshot');
        return this;
    }
    /**
     * Discovers the input schema for the application.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_DiscoverInputSchema.html
     */
    discoverInputSchema() {
        this.add('kinesisanalytics-v2:DiscoverInputSchema');
        return this;
    }
    /**
     * Lists the snapshots for an application.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_ListApplicationSnapshots.html
     */
    listApplicationSnapshots() {
        this.add('kinesisanalytics-v2:ListApplicationSnapshots');
        return this;
    }
    /**
     * List applications for the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_ListApplications.html
     */
    listApplications() {
        this.add('kinesisanalytics-v2:ListApplications');
        return this;
    }
    /**
     * Fetch the tags associated with the application.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('kinesisanalytics-v2:ListTagsForResource');
        return this;
    }
    /**
     * Starts the application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_StartsApplication.html
     */
    startApplication() {
        this.add('kinesisanalytics-v2:StartApplication');
        return this;
    }
    /**
     * Stops the application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_StopApplication.html
     */
    stopApplication() {
        this.add('kinesisanalytics-v2:StopApplication');
        return this;
    }
    /**
     * Add tags to the application.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_TagResource.html
     */
    tagResource() {
        this.add('kinesisanalytics-v2:TagResource');
        return this;
    }
    /**
     * Remove the specified tags from the application.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_UntagResource.html
     */
    untagResource() {
        this.add('kinesisanalytics-v2:UntagResource');
        return this;
    }
    /**
     * Updates the application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_UpdateApplication.html
     */
    updateApplication() {
        this.add('kinesisanalytics-v2:UpdateApplication');
        return this;
    }
    /**
     * Adds a resource of type application to the statement
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/how-it-works.html
     *
     * @param applicationName - Identifier for the applicationName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onApplication(applicationName, account, region, partition) {
        var arn = 'arn:${Partition}:kinesisanalytics:${Region}:${Account}:application/${ApplicationName}';
        arn = arn.replace('${ApplicationName}', applicationName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.KinesisanalyticsV2 = KinesisanalyticsV2;
//# sourceMappingURL=data:application/json;base64,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