"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service lightsail
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonlightsail.html
 */
class Lightsail extends shared_1.PolicyStatement {
    /**
     * Action provider for service lightsail
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonlightsail.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'lightsail';
        this.actions = {
            "AllocateStaticIp": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AllocateStaticIp.html",
                "description": "Creates a static IP address that can be attached to an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "StaticIp": {
                        "required": true
                    }
                }
            },
            "AttachDisk": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AttachDisk.html",
                "description": "Attaches a disk to an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Disk": {
                        "required": true
                    },
                    "Instance": {
                        "required": true
                    }
                }
            },
            "AttachInstancesToLoadBalancer": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AttachInstancesToLoadBalancer.html",
                "description": "Attaches one or more instances to a load balancer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    },
                    "LoadBalancer": {
                        "required": true
                    }
                }
            },
            "AttachLoadBalancerTlsCertificate": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AttachLoadBalancerTlsCertificate.html",
                "description": "Attaches a TLS certificate to a load balancer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "LoadBalancer": {
                        "required": true
                    }
                }
            },
            "AttachStaticIp": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AttachStaticIp.html",
                "description": "Attaches a static IP address to an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    },
                    "StaticIp": {
                        "required": true
                    }
                }
            },
            "CloseInstancePublicPorts": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CloseInstancePublicPorts.html",
                "description": "Closes a public port of an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "CopySnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CopySnapshot.html",
                "description": "Copies a snapshot from one AWS Region to another in Amazon Lightsail.",
                "accessLevel": "Write"
            },
            "CreateCloudFormationStack": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateCloudFormationStack.html",
                "description": "Creates a new Amazon EC2 instance from an exported Amazon Lightsail snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ExportSnapshotRecord": {
                        "required": true
                    }
                }
            },
            "CreateDisk": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDisk.html",
                "description": "Creates a disk.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Disk": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateDiskFromSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDiskFromSnapshot.html",
                "description": "Creates a disk from snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Disk": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateDiskSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDiskSnapshot.html",
                "description": "Creates a disk snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Disk": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateDomain": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDomain.html",
                "description": "Creates a domain resource for the specified domain name.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateDomainEntry": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDomainEntry.html",
                "description": "Creates one or more DNS record entries for a domain resource: Address (A), canonical name (CNAME), mail exchanger (MX), name server (NS), start of authority (SOA), service locator (SRV), or text (TXT).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Domain": {
                        "required": true
                    }
                }
            },
            "CreateInstanceSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateInstanceSnapshot.html",
                "description": "Creates an instance snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    },
                    "InstanceSnapshot": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateInstances": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateInstances.html",
                "description": "Creates one or more instances.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "KeyPair": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateInstancesFromSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateInstancesFromSnapshot.html",
                "description": "Creates one or more instances based on an instance snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    },
                    "InstanceSnapshot": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateKeyPair": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateKeyPair.html",
                "description": "Creates a key pair used to authenticate and connect to an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "KeyPair": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateLoadBalancer": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateLoadBalancer.html",
                "description": "Creates a load balancer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "LoadBalancer": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateLoadBalancerTlsCertificate": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateLoadBalancerTlsCertificate.html",
                "description": "Creates a load balancer TLS certificate.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "LoadBalancer": {
                        "required": true
                    }
                }
            },
            "CreateRelationalDatabase": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateRelationalDatabase.html",
                "description": "Creates a new relational database.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "RelationalDatabase": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateRelationalDatabaseFromSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateRelationalDatabaseFromSnapshot.html",
                "description": "Creates a new relational database from a snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "RelationalDatabase": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateRelationalDatabaseSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateRelationalDatabaseSnapshot.html",
                "description": "Creates a relational database snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "RelationalDatabaseSnapshot": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteDisk": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteDisk.html",
                "description": "Deletes a disk.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Disk": {
                        "required": true
                    }
                }
            },
            "DeleteDiskSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteDiskSnapshot.html",
                "description": "Deletes a disk snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Disk": {
                        "required": true
                    }
                }
            },
            "DeleteDomain": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteDomain.html",
                "description": "Deletes a domain resource and all of its DNS records.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Domain": {
                        "required": true
                    }
                }
            },
            "DeleteDomainEntry": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteDomainEntry.html",
                "description": "Deletes a DNS record entry for a domain resource.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Domain": {
                        "required": true
                    }
                }
            },
            "DeleteInstance": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteInstance.html",
                "description": "Deletes an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "DeleteInstanceSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteInstanceSnapshot.html",
                "description": "Deletes an instance snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "InstanceSnapshot": {
                        "required": true
                    }
                }
            },
            "DeleteKeyPair": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteKeyPair.html",
                "description": "Deletes a key pair used to authenticate and connect to an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "KeyPair": {
                        "required": true
                    }
                }
            },
            "DeleteKnownHostKeys": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteKnownHostKeys.html",
                "description": "Deletes the known host key or certificate used by the Amazon Lightsail browser-based SSH or RDP clients to authenticate an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "DeleteLoadBalancer": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteLoadBalancer.html",
                "description": "Deletes a load balancer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "LoadBalancer": {
                        "required": true
                    }
                }
            },
            "DeleteLoadBalancerTlsCertificate": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteLoadBalancerTlsCertificate.html",
                "description": "Deletes a load balancer TLS certificate.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "LoadBalancer": {
                        "required": true
                    }
                }
            },
            "DeleteRelationalDatabase": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteRelationalDatabase.html",
                "description": "Deletes a relational database.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "RelationalDatabase": {
                        "required": true
                    }
                }
            },
            "DeleteRelationalDatabaseSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteRelationalDatabaseSnapshot.html",
                "description": "Deletes relational database snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "RelationalDatabaseSnapshot": {
                        "required": true
                    }
                }
            },
            "DetachDisk": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DetachDisk.html",
                "description": "Detaches a disk from an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Disk": {
                        "required": true
                    }
                }
            },
            "DetachInstancesFromLoadBalancer": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DetachInstancesFromLoadBalancer.html",
                "description": "Detaches one or more instances from a load balancer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    },
                    "LoadBalancer": {
                        "required": true
                    }
                }
            },
            "DetachStaticIp": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DetachStaticIp.html",
                "description": "Detaches a static IP from an instance to which it is attached.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    },
                    "StaticIp": {
                        "required": true
                    }
                }
            },
            "DownloadDefaultKeyPair": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DownloadDefaultKeyPair.html",
                "description": "Downloads the default key pair used to authenticate and connect to instances in a specific AWS Region.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "KeyPair": {
                        "required": true
                    }
                }
            },
            "ExportSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_ExportSnapshot.html",
                "description": "Exports an Amazon Lightsail snapshot to Amazon EC2.",
                "accessLevel": "Write"
            },
            "GetActiveNames": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetActiveNames.html",
                "description": "Returns the names of all active (not deleted) resources.",
                "accessLevel": "Read"
            },
            "GetBlueprints": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetBlueprints.html",
                "description": "Returns a list of instance images, or blueprints. You can use a blueprint to create a new instance already running a specific operating system, as well as a pre-installed application or development stack. The software that runs on your instance depends on the blueprint you define when creating the instance.",
                "accessLevel": "List"
            },
            "GetBundles": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetBundles.html",
                "description": "Returns a list of instance bundles. You can use a bundle to create a new instance with a set of performance specifications, such as CPU count, disk size, RAM size, and network transfer allowance. The cost of your instance depends on the bundle you define when creating the instance.",
                "accessLevel": "List"
            },
            "GetCloudFormationStackRecords": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetCloudFormationStackRecords.html",
                "description": "Returns information about all CloudFormation stacks used to create Amazon EC2 resources from exported Amazon Lightsail snapshots.",
                "accessLevel": "List",
                "resourceTypes": {
                    "CloudFormationStackRecord": {
                        "required": true
                    }
                }
            },
            "GetDisk": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDisk.html",
                "description": "Returns information about a disk.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Disk": {
                        "required": true
                    }
                }
            },
            "GetDiskSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDiskSnapshot.html",
                "description": "Returns information about a disk snapshot.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Disk": {
                        "required": true
                    }
                }
            },
            "GetDiskSnapshots": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDiskSnapshots.html",
                "description": "Returns information about all disk snapshots.",
                "accessLevel": "List",
                "resourceTypes": {
                    "Disk": {
                        "required": true
                    }
                }
            },
            "GetDisks": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDisks.html",
                "description": "Returns information about all disks.",
                "accessLevel": "List"
            },
            "GetDomain": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDomain.html",
                "description": "Returns DNS records for a domain resource.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Domain": {
                        "required": true
                    }
                }
            },
            "GetDomains": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDomains.html",
                "description": "Returns DNS records for all domain resources.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Domain": {
                        "required": true
                    }
                }
            },
            "GetExportSnapshotRecords": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetExportSnapshotRecords.html",
                "description": "Returns information about all records to export Amazon Lightsail snapshots to Amazon EC2.",
                "accessLevel": "List",
                "resourceTypes": {
                    "ExportSnapshotRecord": {
                        "required": true
                    }
                }
            },
            "GetInstance": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstance.html",
                "description": "Returns information about an instance.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "GetInstanceAccessDetails": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceAccessDetails.html",
                "description": "Returns temporary keys you can use to authenticate and connect to an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "GetInstanceMetricData": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceMetricData.html",
                "description": "Returns the data points for the specified metric of an instance.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "GetInstancePortStates": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstancePortStates.html",
                "description": "Returns the port states of an instance.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "GetInstanceSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceSnapshot.html",
                "description": "Returns information about an instance snapshot.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "InstanceSnapshot": {
                        "required": true
                    }
                }
            },
            "GetInstanceSnapshots": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceSnapshots.html",
                "description": "Returns information about all instance snapshots.",
                "accessLevel": "List",
                "resourceTypes": {
                    "InstanceSnapshot": {
                        "required": true
                    }
                }
            },
            "GetInstanceState": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceState.html",
                "description": "Returns the state of an instance.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "GetInstances": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstances.html",
                "description": "Returns information about all instances.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "GetKeyPair": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetKeyPair.html",
                "description": "Returns information about a key pair.",
                "accessLevel": "List",
                "resourceTypes": {
                    "KeyPair": {
                        "required": true
                    }
                }
            },
            "GetKeyPairs": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetKeyPairs.html",
                "description": "Returns information about all key pairs.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "KeyPair": {
                        "required": true
                    }
                }
            },
            "GetLoadBalancer": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetLoadBalancer.html",
                "description": "Returns information about a load balancer.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "LoadBalancer": {
                        "required": true
                    }
                }
            },
            "GetLoadBalancerMetricData": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetLoadBalancerMetricData.html",
                "description": "Returns the data points for the specified metric of a load balancer.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "LoadBalancer": {
                        "required": true
                    }
                }
            },
            "GetLoadBalancerTlsCertificates": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteLoadBalancerTlsCertificate.html",
                "description": "Returns information about a load balancer TLS certificate.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "LoadBalancer": {
                        "required": true
                    }
                }
            },
            "GetLoadBalancers": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetLoadBalancers.html",
                "description": "Returns information about load balancers.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "LoadBalancer": {
                        "required": true
                    }
                }
            },
            "GetOperation": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetOperation.html",
                "description": "Returns information about an operation. Operations include events such as when you create an instance, allocate a static IP, attach a static IP, and so on.",
                "accessLevel": "Read"
            },
            "GetOperations": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetOperations.html",
                "description": "Returns information about all operations. Operations include events such as when you create an instance, allocate a static IP, attach a static IP, and so on.",
                "accessLevel": "Read"
            },
            "GetOperationsForResource": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetOperationsForResource.html",
                "description": "Returns operations for a resource.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Domain": {
                        "required": false
                    },
                    "Instance": {
                        "required": false
                    },
                    "InstanceSnapshot": {
                        "required": false
                    },
                    "KeyPair": {
                        "required": false
                    },
                    "StaticIp": {
                        "required": false
                    }
                }
            },
            "GetRegions": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRegions.html",
                "description": "Returns a list of all valid AWS Regions for Amazon Lightsail.",
                "accessLevel": "List"
            },
            "GetRelationalDatabase": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabase.html",
                "description": "Returns information about a relational database.",
                "accessLevel": "List",
                "resourceTypes": {
                    "RelationalDatabase": {
                        "required": true
                    }
                }
            },
            "GetRelationalDatabaseBlueprints": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseBlueprints.html",
                "description": "Returns a list of relational database images, or blueprints. You can use a blueprint to create a new database running a specific database engine. The database engine that runs on your database depends on the blueprint you define when creating the relational database.",
                "accessLevel": "List"
            },
            "GetRelationalDatabaseBundles": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseBundles.html",
                "description": "Returns a list of relational database bundles. You can use a bundle to create a new database with a set of performance specifications, such as CPU count, disk size, RAM size, network transfer allowance, and standard of high availability. The cost of your database depends on the bundle you define when creating the relational database.",
                "accessLevel": "List"
            },
            "GetRelationalDatabaseEvents": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseEvents.html",
                "description": "Returns events for a relational database.",
                "accessLevel": "Read"
            },
            "GetRelationalDatabaseLogEvents": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseLogEvents.html",
                "description": "Returns events for the specified log stream of a relational database.",
                "accessLevel": "Read"
            },
            "GetRelationalDatabaseLogStreams": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseLogStreams.html",
                "description": "Returns the log streams available for a relational database.",
                "accessLevel": "Read"
            },
            "GetRelationalDatabaseMasterUserPassword": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseMasterUserPassword.html",
                "description": "Returns the master user password of a relational database.",
                "accessLevel": "Write"
            },
            "GetRelationalDatabaseMetricData": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseMetricData.html",
                "description": "Returns the data points for the specified metric of a relational database.",
                "accessLevel": "Read"
            },
            "GetRelationalDatabaseParameters": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseParameters.html",
                "description": "Returns the parameters of a relational database.",
                "accessLevel": "List"
            },
            "GetRelationalDatabaseSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseSnapshot.html",
                "description": "Returns information about a relational database snapshot.",
                "accessLevel": "List",
                "resourceTypes": {
                    "RelationalDatabase": {
                        "required": true
                    }
                }
            },
            "GetRelationalDatabaseSnapshots": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseSnapshots.html",
                "description": "Returns information about all relational database snapshots.",
                "accessLevel": "List",
                "resourceTypes": {
                    "RelationalDatabase": {
                        "required": true
                    }
                }
            },
            "GetRelationalDatabases": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabases.html",
                "description": "Return information about all relational databases.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "RelationalDatabase": {
                        "required": true
                    }
                }
            },
            "GetStaticIp": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetStaticIp.html",
                "description": "Returns information about a static IP.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "StaticIp": {
                        "required": true
                    }
                }
            },
            "GetStaticIps": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetStaticIps.html",
                "description": "Returns information about all static IPs.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "StaticIp": {
                        "required": true
                    }
                }
            },
            "ImportKeyPair": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_ImportKeyPair.html",
                "description": "Imports a public key from a key pair.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "KeyPair": {
                        "required": true
                    }
                }
            },
            "IsVpcPeered": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_IsVpcPeered.html",
                "description": "Returns a boolean value indicating whether the Amazon Lightsail virtual private cloud (VPC) is peered.",
                "accessLevel": "Read"
            },
            "OpenInstancePublicPorts": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_OpenInstancePublicPorts.html",
                "description": "Adds, or opens a public port of an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "PeerVpc": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_PeerVpc.html",
                "description": "Tries to peer the Amazon Lightsail virtual private cloud (VPC) with the default VPC.",
                "accessLevel": "Write"
            },
            "PutInstancePublicPorts": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_PutInstancePublicPorts.html",
                "description": "Sets the specified open ports for an instance, and closes all ports for every protocol not included in the request.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "RebootInstance": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_RebootInstance.html",
                "description": "Reboots an instance that is in a running state.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "RebootRelationalDatabase": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_RebootRelationalDatabase.html",
                "description": "Reboots a relational database that is in a running state.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "RelationalDatabase": {
                        "required": true
                    }
                }
            },
            "ReleaseStaticIp": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_ReleaseStaticIp.html",
                "description": "Deletes a static IP.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "StaticIp": {
                        "required": true
                    }
                }
            },
            "StartInstance": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StartInstance.html",
                "description": "Starts an instance that is in a stopped state.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "StartRelationalDatabase": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StartRelationalDatabase.html",
                "description": "Starts a relational database that is in a stopped state.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "RelationalDatabase": {
                        "required": true
                    }
                }
            },
            "StopInstance": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StopInstance.html",
                "description": "Stops an instance that is in a running state.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "StopRelationalDatabase": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StopRelationalDatabase.html",
                "description": "Stops a relational database that is in a running state.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "RelationalDatabase": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_TagResource.html",
                "description": "Tags a resource.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Disk": {
                        "required": false
                    },
                    "DiskSnapshot": {
                        "required": false
                    },
                    "Domain": {
                        "required": false
                    },
                    "Instance": {
                        "required": false
                    },
                    "InstanceSnapshot": {
                        "required": false
                    },
                    "KeyPair": {
                        "required": false
                    },
                    "LoadBalancer": {
                        "required": false
                    },
                    "RelationalDatabase": {
                        "required": false
                    },
                    "RelationalDatabaseSnapshot": {
                        "required": false
                    },
                    "StaticIp": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UnpeerVpc": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UnpeerVpc.html",
                "description": "Attempts to unpeer the Amazon Lightsail virtual private cloud (VPC) from the default VPC.",
                "accessLevel": "Write"
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UntagResource.html",
                "description": "Untags a resource.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Disk": {
                        "required": false
                    },
                    "DiskSnapshot": {
                        "required": false
                    },
                    "Domain": {
                        "required": false
                    },
                    "Instance": {
                        "required": false
                    },
                    "InstanceSnapshot": {
                        "required": false
                    },
                    "KeyPair": {
                        "required": false
                    },
                    "LoadBalancer": {
                        "required": false
                    },
                    "RelationalDatabase": {
                        "required": false
                    },
                    "RelationalDatabaseSnapshot": {
                        "required": false
                    },
                    "StaticIp": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UpdateDomainEntry": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UpdateDomainEntry.html",
                "description": "Updates a domain recordset after it is created.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Domain": {
                        "required": true
                    }
                }
            },
            "UpdateLoadBalancerAttribute": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DetachInstancesFromLoadBalancer.html",
                "description": "Updates a load balancer attribute, such as the health check path and session stickiness.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "LoadBalancer": {
                        "required": true
                    }
                }
            },
            "UpdateRelationalDatabase": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UpdateRelationalDatabase.html",
                "description": "Updates a relational database.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "RelationalDatabase": {
                        "required": true
                    }
                }
            },
            "UpdateRelationalDatabaseParameters": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UpdateRelationalDatabaseParameters.html",
                "description": "Updates the parameters of a relational database.",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "Domain": {
                "name": "Domain",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_Domain.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:Domain/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "Instance": {
                "name": "Instance",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_Instance.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:Instance/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "InstanceSnapshot": {
                "name": "InstanceSnapshot",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_InstanceSnapshot.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:InstanceSnapshot/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "KeyPair": {
                "name": "KeyPair",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_KeyPair.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:KeyPair/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "StaticIp": {
                "name": "StaticIp",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StaticIp.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:StaticIp/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "Disk": {
                "name": "Disk",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_Disk.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:Disk/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "DiskSnapshot": {
                "name": "DiskSnapshot",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DiskSnapshot.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:DiskSnapshot/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "LoadBalancer": {
                "name": "LoadBalancer",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_LoadBalancer.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:LoadBalancer/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "PeeredVpc": {
                "name": "PeeredVpc",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/PeeredVpc.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:PeeredVpc/${Id}",
                "conditionKeys": []
            },
            "LoadBalancerTlsCertificate": {
                "name": "LoadBalancerTlsCertificate",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/LoadBalancerTlsCertificate.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:LoadBalancerTlsCertificate/${Id}",
                "conditionKeys": []
            },
            "ExportSnapshotRecord": {
                "name": "ExportSnapshotRecord",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_ExportSnapshotRecord.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:ExportSnapshotRecord/${Id}",
                "conditionKeys": []
            },
            "CloudFormationStackRecord": {
                "name": "CloudFormationStackRecord",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CloudFormationStackRecord.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:CloudFormationStackRecord/${Id}",
                "conditionKeys": []
            },
            "RelationalDatabase": {
                "name": "RelationalDatabase",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_RelationalDatabase.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:RelationalDatabase/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "RelationalDatabaseSnapshot": {
                "name": "RelationalDatabaseSnapshot",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_RelationalDatabaseSnapshot.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:RelationalDatabaseSnapshot/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Creates a static IP address that can be attached to an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AllocateStaticIp.html
     */
    allocateStaticIp() {
        this.add('lightsail:AllocateStaticIp');
        return this;
    }
    /**
     * Attaches a disk to an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AttachDisk.html
     */
    attachDisk() {
        this.add('lightsail:AttachDisk');
        return this;
    }
    /**
     * Attaches one or more instances to a load balancer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AttachInstancesToLoadBalancer.html
     */
    attachInstancesToLoadBalancer() {
        this.add('lightsail:AttachInstancesToLoadBalancer');
        return this;
    }
    /**
     * Attaches a TLS certificate to a load balancer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AttachLoadBalancerTlsCertificate.html
     */
    attachLoadBalancerTlsCertificate() {
        this.add('lightsail:AttachLoadBalancerTlsCertificate');
        return this;
    }
    /**
     * Attaches a static IP address to an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AttachStaticIp.html
     */
    attachStaticIp() {
        this.add('lightsail:AttachStaticIp');
        return this;
    }
    /**
     * Closes a public port of an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CloseInstancePublicPorts.html
     */
    closeInstancePublicPorts() {
        this.add('lightsail:CloseInstancePublicPorts');
        return this;
    }
    /**
     * Copies a snapshot from one AWS Region to another in Amazon Lightsail.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CopySnapshot.html
     */
    copySnapshot() {
        this.add('lightsail:CopySnapshot');
        return this;
    }
    /**
     * Creates a new Amazon EC2 instance from an exported Amazon Lightsail snapshot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateCloudFormationStack.html
     */
    createCloudFormationStack() {
        this.add('lightsail:CreateCloudFormationStack');
        return this;
    }
    /**
     * Creates a disk.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDisk.html
     */
    createDisk() {
        this.add('lightsail:CreateDisk');
        return this;
    }
    /**
     * Creates a disk from snapshot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDiskFromSnapshot.html
     */
    createDiskFromSnapshot() {
        this.add('lightsail:CreateDiskFromSnapshot');
        return this;
    }
    /**
     * Creates a disk snapshot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDiskSnapshot.html
     */
    createDiskSnapshot() {
        this.add('lightsail:CreateDiskSnapshot');
        return this;
    }
    /**
     * Creates a domain resource for the specified domain name.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDomain.html
     */
    createDomain() {
        this.add('lightsail:CreateDomain');
        return this;
    }
    /**
     * Creates one or more DNS record entries for a domain resource: Address (A), canonical name (CNAME), mail exchanger (MX), name server (NS), start of authority (SOA), service locator (SRV), or text (TXT).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDomainEntry.html
     */
    createDomainEntry() {
        this.add('lightsail:CreateDomainEntry');
        return this;
    }
    /**
     * Creates an instance snapshot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateInstanceSnapshot.html
     */
    createInstanceSnapshot() {
        this.add('lightsail:CreateInstanceSnapshot');
        return this;
    }
    /**
     * Creates one or more instances.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateInstances.html
     */
    createInstances() {
        this.add('lightsail:CreateInstances');
        return this;
    }
    /**
     * Creates one or more instances based on an instance snapshot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateInstancesFromSnapshot.html
     */
    createInstancesFromSnapshot() {
        this.add('lightsail:CreateInstancesFromSnapshot');
        return this;
    }
    /**
     * Creates a key pair used to authenticate and connect to an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateKeyPair.html
     */
    createKeyPair() {
        this.add('lightsail:CreateKeyPair');
        return this;
    }
    /**
     * Creates a load balancer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateLoadBalancer.html
     */
    createLoadBalancer() {
        this.add('lightsail:CreateLoadBalancer');
        return this;
    }
    /**
     * Creates a load balancer TLS certificate.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateLoadBalancerTlsCertificate.html
     */
    createLoadBalancerTlsCertificate() {
        this.add('lightsail:CreateLoadBalancerTlsCertificate');
        return this;
    }
    /**
     * Creates a new relational database.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateRelationalDatabase.html
     */
    createRelationalDatabase() {
        this.add('lightsail:CreateRelationalDatabase');
        return this;
    }
    /**
     * Creates a new relational database from a snapshot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateRelationalDatabaseFromSnapshot.html
     */
    createRelationalDatabaseFromSnapshot() {
        this.add('lightsail:CreateRelationalDatabaseFromSnapshot');
        return this;
    }
    /**
     * Creates a relational database snapshot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateRelationalDatabaseSnapshot.html
     */
    createRelationalDatabaseSnapshot() {
        this.add('lightsail:CreateRelationalDatabaseSnapshot');
        return this;
    }
    /**
     * Deletes a disk.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteDisk.html
     */
    deleteDisk() {
        this.add('lightsail:DeleteDisk');
        return this;
    }
    /**
     * Deletes a disk snapshot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteDiskSnapshot.html
     */
    deleteDiskSnapshot() {
        this.add('lightsail:DeleteDiskSnapshot');
        return this;
    }
    /**
     * Deletes a domain resource and all of its DNS records.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteDomain.html
     */
    deleteDomain() {
        this.add('lightsail:DeleteDomain');
        return this;
    }
    /**
     * Deletes a DNS record entry for a domain resource.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteDomainEntry.html
     */
    deleteDomainEntry() {
        this.add('lightsail:DeleteDomainEntry');
        return this;
    }
    /**
     * Deletes an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteInstance.html
     */
    deleteInstance() {
        this.add('lightsail:DeleteInstance');
        return this;
    }
    /**
     * Deletes an instance snapshot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteInstanceSnapshot.html
     */
    deleteInstanceSnapshot() {
        this.add('lightsail:DeleteInstanceSnapshot');
        return this;
    }
    /**
     * Deletes a key pair used to authenticate and connect to an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteKeyPair.html
     */
    deleteKeyPair() {
        this.add('lightsail:DeleteKeyPair');
        return this;
    }
    /**
     * Deletes the known host key or certificate used by the Amazon Lightsail browser-based SSH or RDP clients to authenticate an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteKnownHostKeys.html
     */
    deleteKnownHostKeys() {
        this.add('lightsail:DeleteKnownHostKeys');
        return this;
    }
    /**
     * Deletes a load balancer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteLoadBalancer.html
     */
    deleteLoadBalancer() {
        this.add('lightsail:DeleteLoadBalancer');
        return this;
    }
    /**
     * Deletes a load balancer TLS certificate.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteLoadBalancerTlsCertificate.html
     */
    deleteLoadBalancerTlsCertificate() {
        this.add('lightsail:DeleteLoadBalancerTlsCertificate');
        return this;
    }
    /**
     * Deletes a relational database.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteRelationalDatabase.html
     */
    deleteRelationalDatabase() {
        this.add('lightsail:DeleteRelationalDatabase');
        return this;
    }
    /**
     * Deletes relational database snapshot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteRelationalDatabaseSnapshot.html
     */
    deleteRelationalDatabaseSnapshot() {
        this.add('lightsail:DeleteRelationalDatabaseSnapshot');
        return this;
    }
    /**
     * Detaches a disk from an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DetachDisk.html
     */
    detachDisk() {
        this.add('lightsail:DetachDisk');
        return this;
    }
    /**
     * Detaches one or more instances from a load balancer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DetachInstancesFromLoadBalancer.html
     */
    detachInstancesFromLoadBalancer() {
        this.add('lightsail:DetachInstancesFromLoadBalancer');
        return this;
    }
    /**
     * Detaches a static IP from an instance to which it is attached.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DetachStaticIp.html
     */
    detachStaticIp() {
        this.add('lightsail:DetachStaticIp');
        return this;
    }
    /**
     * Downloads the default key pair used to authenticate and connect to instances in a specific AWS Region.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DownloadDefaultKeyPair.html
     */
    downloadDefaultKeyPair() {
        this.add('lightsail:DownloadDefaultKeyPair');
        return this;
    }
    /**
     * Exports an Amazon Lightsail snapshot to Amazon EC2.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_ExportSnapshot.html
     */
    exportSnapshot() {
        this.add('lightsail:ExportSnapshot');
        return this;
    }
    /**
     * Returns the names of all active (not deleted) resources.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetActiveNames.html
     */
    getActiveNames() {
        this.add('lightsail:GetActiveNames');
        return this;
    }
    /**
     * Returns a list of instance images, or blueprints. You can use a blueprint to create a new instance already running a specific operating system, as well as a pre-installed application or development stack. The software that runs on your instance depends on the blueprint you define when creating the instance.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetBlueprints.html
     */
    getBlueprints() {
        this.add('lightsail:GetBlueprints');
        return this;
    }
    /**
     * Returns a list of instance bundles. You can use a bundle to create a new instance with a set of performance specifications, such as CPU count, disk size, RAM size, and network transfer allowance. The cost of your instance depends on the bundle you define when creating the instance.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetBundles.html
     */
    getBundles() {
        this.add('lightsail:GetBundles');
        return this;
    }
    /**
     * Returns information about all CloudFormation stacks used to create Amazon EC2 resources from exported Amazon Lightsail snapshots.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetCloudFormationStackRecords.html
     */
    getCloudFormationStackRecords() {
        this.add('lightsail:GetCloudFormationStackRecords');
        return this;
    }
    /**
     * Returns information about a disk.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDisk.html
     */
    getDisk() {
        this.add('lightsail:GetDisk');
        return this;
    }
    /**
     * Returns information about a disk snapshot.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDiskSnapshot.html
     */
    getDiskSnapshot() {
        this.add('lightsail:GetDiskSnapshot');
        return this;
    }
    /**
     * Returns information about all disk snapshots.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDiskSnapshots.html
     */
    getDiskSnapshots() {
        this.add('lightsail:GetDiskSnapshots');
        return this;
    }
    /**
     * Returns information about all disks.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDisks.html
     */
    getDisks() {
        this.add('lightsail:GetDisks');
        return this;
    }
    /**
     * Returns DNS records for a domain resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDomain.html
     */
    getDomain() {
        this.add('lightsail:GetDomain');
        return this;
    }
    /**
     * Returns DNS records for all domain resources.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDomains.html
     */
    getDomains() {
        this.add('lightsail:GetDomains');
        return this;
    }
    /**
     * Returns information about all records to export Amazon Lightsail snapshots to Amazon EC2.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetExportSnapshotRecords.html
     */
    getExportSnapshotRecords() {
        this.add('lightsail:GetExportSnapshotRecords');
        return this;
    }
    /**
     * Returns information about an instance.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstance.html
     */
    getInstance() {
        this.add('lightsail:GetInstance');
        return this;
    }
    /**
     * Returns temporary keys you can use to authenticate and connect to an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceAccessDetails.html
     */
    getInstanceAccessDetails() {
        this.add('lightsail:GetInstanceAccessDetails');
        return this;
    }
    /**
     * Returns the data points for the specified metric of an instance.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceMetricData.html
     */
    getInstanceMetricData() {
        this.add('lightsail:GetInstanceMetricData');
        return this;
    }
    /**
     * Returns the port states of an instance.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstancePortStates.html
     */
    getInstancePortStates() {
        this.add('lightsail:GetInstancePortStates');
        return this;
    }
    /**
     * Returns information about an instance snapshot.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceSnapshot.html
     */
    getInstanceSnapshot() {
        this.add('lightsail:GetInstanceSnapshot');
        return this;
    }
    /**
     * Returns information about all instance snapshots.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceSnapshots.html
     */
    getInstanceSnapshots() {
        this.add('lightsail:GetInstanceSnapshots');
        return this;
    }
    /**
     * Returns the state of an instance.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceState.html
     */
    getInstanceState() {
        this.add('lightsail:GetInstanceState');
        return this;
    }
    /**
     * Returns information about all instances.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstances.html
     */
    getInstances() {
        this.add('lightsail:GetInstances');
        return this;
    }
    /**
     * Returns information about a key pair.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetKeyPair.html
     */
    getKeyPair() {
        this.add('lightsail:GetKeyPair');
        return this;
    }
    /**
     * Returns information about all key pairs.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetKeyPairs.html
     */
    getKeyPairs() {
        this.add('lightsail:GetKeyPairs');
        return this;
    }
    /**
     * Returns information about a load balancer.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetLoadBalancer.html
     */
    getLoadBalancer() {
        this.add('lightsail:GetLoadBalancer');
        return this;
    }
    /**
     * Returns the data points for the specified metric of a load balancer.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetLoadBalancerMetricData.html
     */
    getLoadBalancerMetricData() {
        this.add('lightsail:GetLoadBalancerMetricData');
        return this;
    }
    /**
     * Returns information about a load balancer TLS certificate.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteLoadBalancerTlsCertificate.html
     */
    getLoadBalancerTlsCertificates() {
        this.add('lightsail:GetLoadBalancerTlsCertificates');
        return this;
    }
    /**
     * Returns information about load balancers.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetLoadBalancers.html
     */
    getLoadBalancers() {
        this.add('lightsail:GetLoadBalancers');
        return this;
    }
    /**
     * Returns information about an operation. Operations include events such as when you create an instance, allocate a static IP, attach a static IP, and so on.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetOperation.html
     */
    getOperation() {
        this.add('lightsail:GetOperation');
        return this;
    }
    /**
     * Returns information about all operations. Operations include events such as when you create an instance, allocate a static IP, attach a static IP, and so on.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetOperations.html
     */
    getOperations() {
        this.add('lightsail:GetOperations');
        return this;
    }
    /**
     * Returns operations for a resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetOperationsForResource.html
     */
    getOperationsForResource() {
        this.add('lightsail:GetOperationsForResource');
        return this;
    }
    /**
     * Returns a list of all valid AWS Regions for Amazon Lightsail.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRegions.html
     */
    getRegions() {
        this.add('lightsail:GetRegions');
        return this;
    }
    /**
     * Returns information about a relational database.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabase.html
     */
    getRelationalDatabase() {
        this.add('lightsail:GetRelationalDatabase');
        return this;
    }
    /**
     * Returns a list of relational database images, or blueprints. You can use a blueprint to create a new database running a specific database engine. The database engine that runs on your database depends on the blueprint you define when creating the relational database.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseBlueprints.html
     */
    getRelationalDatabaseBlueprints() {
        this.add('lightsail:GetRelationalDatabaseBlueprints');
        return this;
    }
    /**
     * Returns a list of relational database bundles. You can use a bundle to create a new database with a set of performance specifications, such as CPU count, disk size, RAM size, network transfer allowance, and standard of high availability. The cost of your database depends on the bundle you define when creating the relational database.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseBundles.html
     */
    getRelationalDatabaseBundles() {
        this.add('lightsail:GetRelationalDatabaseBundles');
        return this;
    }
    /**
     * Returns events for a relational database.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseEvents.html
     */
    getRelationalDatabaseEvents() {
        this.add('lightsail:GetRelationalDatabaseEvents');
        return this;
    }
    /**
     * Returns events for the specified log stream of a relational database.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseLogEvents.html
     */
    getRelationalDatabaseLogEvents() {
        this.add('lightsail:GetRelationalDatabaseLogEvents');
        return this;
    }
    /**
     * Returns the log streams available for a relational database.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseLogStreams.html
     */
    getRelationalDatabaseLogStreams() {
        this.add('lightsail:GetRelationalDatabaseLogStreams');
        return this;
    }
    /**
     * Returns the master user password of a relational database.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseMasterUserPassword.html
     */
    getRelationalDatabaseMasterUserPassword() {
        this.add('lightsail:GetRelationalDatabaseMasterUserPassword');
        return this;
    }
    /**
     * Returns the data points for the specified metric of a relational database.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseMetricData.html
     */
    getRelationalDatabaseMetricData() {
        this.add('lightsail:GetRelationalDatabaseMetricData');
        return this;
    }
    /**
     * Returns the parameters of a relational database.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseParameters.html
     */
    getRelationalDatabaseParameters() {
        this.add('lightsail:GetRelationalDatabaseParameters');
        return this;
    }
    /**
     * Returns information about a relational database snapshot.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseSnapshot.html
     */
    getRelationalDatabaseSnapshot() {
        this.add('lightsail:GetRelationalDatabaseSnapshot');
        return this;
    }
    /**
     * Returns information about all relational database snapshots.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseSnapshots.html
     */
    getRelationalDatabaseSnapshots() {
        this.add('lightsail:GetRelationalDatabaseSnapshots');
        return this;
    }
    /**
     * Return information about all relational databases.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabases.html
     */
    getRelationalDatabases() {
        this.add('lightsail:GetRelationalDatabases');
        return this;
    }
    /**
     * Returns information about a static IP.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetStaticIp.html
     */
    getStaticIp() {
        this.add('lightsail:GetStaticIp');
        return this;
    }
    /**
     * Returns information about all static IPs.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetStaticIps.html
     */
    getStaticIps() {
        this.add('lightsail:GetStaticIps');
        return this;
    }
    /**
     * Imports a public key from a key pair.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_ImportKeyPair.html
     */
    importKeyPair() {
        this.add('lightsail:ImportKeyPair');
        return this;
    }
    /**
     * Returns a boolean value indicating whether the Amazon Lightsail virtual private cloud (VPC) is peered.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_IsVpcPeered.html
     */
    isVpcPeered() {
        this.add('lightsail:IsVpcPeered');
        return this;
    }
    /**
     * Adds, or opens a public port of an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_OpenInstancePublicPorts.html
     */
    openInstancePublicPorts() {
        this.add('lightsail:OpenInstancePublicPorts');
        return this;
    }
    /**
     * Tries to peer the Amazon Lightsail virtual private cloud (VPC) with the default VPC.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_PeerVpc.html
     */
    peerVpc() {
        this.add('lightsail:PeerVpc');
        return this;
    }
    /**
     * Sets the specified open ports for an instance, and closes all ports for every protocol not included in the request.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_PutInstancePublicPorts.html
     */
    putInstancePublicPorts() {
        this.add('lightsail:PutInstancePublicPorts');
        return this;
    }
    /**
     * Reboots an instance that is in a running state.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_RebootInstance.html
     */
    rebootInstance() {
        this.add('lightsail:RebootInstance');
        return this;
    }
    /**
     * Reboots a relational database that is in a running state.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_RebootRelationalDatabase.html
     */
    rebootRelationalDatabase() {
        this.add('lightsail:RebootRelationalDatabase');
        return this;
    }
    /**
     * Deletes a static IP.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_ReleaseStaticIp.html
     */
    releaseStaticIp() {
        this.add('lightsail:ReleaseStaticIp');
        return this;
    }
    /**
     * Starts an instance that is in a stopped state.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StartInstance.html
     */
    startInstance() {
        this.add('lightsail:StartInstance');
        return this;
    }
    /**
     * Starts a relational database that is in a stopped state.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StartRelationalDatabase.html
     */
    startRelationalDatabase() {
        this.add('lightsail:StartRelationalDatabase');
        return this;
    }
    /**
     * Stops an instance that is in a running state.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StopInstance.html
     */
    stopInstance() {
        this.add('lightsail:StopInstance');
        return this;
    }
    /**
     * Stops a relational database that is in a running state.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StopRelationalDatabase.html
     */
    stopRelationalDatabase() {
        this.add('lightsail:StopRelationalDatabase');
        return this;
    }
    /**
     * Tags a resource.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_TagResource.html
     */
    tagResource() {
        this.add('lightsail:TagResource');
        return this;
    }
    /**
     * Attempts to unpeer the Amazon Lightsail virtual private cloud (VPC) from the default VPC.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UnpeerVpc.html
     */
    unpeerVpc() {
        this.add('lightsail:UnpeerVpc');
        return this;
    }
    /**
     * Untags a resource.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UntagResource.html
     */
    untagResource() {
        this.add('lightsail:UntagResource');
        return this;
    }
    /**
     * Updates a domain recordset after it is created.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UpdateDomainEntry.html
     */
    updateDomainEntry() {
        this.add('lightsail:UpdateDomainEntry');
        return this;
    }
    /**
     * Updates a load balancer attribute, such as the health check path and session stickiness.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DetachInstancesFromLoadBalancer.html
     */
    updateLoadBalancerAttribute() {
        this.add('lightsail:UpdateLoadBalancerAttribute');
        return this;
    }
    /**
     * Updates a relational database.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UpdateRelationalDatabase.html
     */
    updateRelationalDatabase() {
        this.add('lightsail:UpdateRelationalDatabase');
        return this;
    }
    /**
     * Updates the parameters of a relational database.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UpdateRelationalDatabaseParameters.html
     */
    updateRelationalDatabaseParameters() {
        this.add('lightsail:UpdateRelationalDatabaseParameters');
        return this;
    }
    /**
     * Adds a resource of type Domain to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_Domain.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onDomain(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:Domain/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Instance to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_Instance.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onInstance(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:Instance/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type InstanceSnapshot to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_InstanceSnapshot.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onInstanceSnapshot(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:InstanceSnapshot/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type KeyPair to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_KeyPair.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onKeyPair(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:KeyPair/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type StaticIp to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StaticIp.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onStaticIp(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:StaticIp/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Disk to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_Disk.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onDisk(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:Disk/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type DiskSnapshot to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DiskSnapshot.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onDiskSnapshot(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:DiskSnapshot/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type LoadBalancer to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_LoadBalancer.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onLoadBalancer(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:LoadBalancer/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type PeeredVpc to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/PeeredVpc.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onPeeredVpc(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:PeeredVpc/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type LoadBalancerTlsCertificate to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/LoadBalancerTlsCertificate.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onLoadBalancerTlsCertificate(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:LoadBalancerTlsCertificate/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type ExportSnapshotRecord to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_ExportSnapshotRecord.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onExportSnapshotRecord(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:ExportSnapshotRecord/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type CloudFormationStackRecord to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CloudFormationStackRecord.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCloudFormationStackRecord(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:CloudFormationStackRecord/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type RelationalDatabase to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_RelationalDatabase.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onRelationalDatabase(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:RelationalDatabase/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type RelationalDatabaseSnapshot to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_RelationalDatabaseSnapshot.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onRelationalDatabaseSnapshot(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:RelationalDatabaseSnapshot/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Lightsail = Lightsail;
//# sourceMappingURL=data:application/json;base64,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