"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service managedblockchain
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonmanagedblockchain.html
 */
class Managedblockchain extends shared_1.PolicyStatement {
    /**
     * Action provider for service managedblockchain
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonmanagedblockchain.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'managedblockchain';
        this.actions = {
            "CreateMember": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_CreateMember.html",
                "description": "Grants permission to create a member of an Amazon Managed Blockchain network.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "network": {
                        "required": true
                    }
                }
            },
            "CreateNetwork": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_CreateNetwork.html",
                "description": "Grants permission to create an Amazon Managed Blockchain network.",
                "accessLevel": "Write"
            },
            "CreateNode": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_CreateNode.html",
                "description": "Grants permission to create a node within a member of an Amazon Managed Blockchain network.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "member": {
                        "required": true
                    }
                }
            },
            "CreateProposal": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_CreateProposal.html",
                "description": "Grants permission to create a proposal that other blockchain network members can vote on to add or remove a member in an Amazon Managed Blockchain network.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "network": {
                        "required": true
                    }
                }
            },
            "DeleteMember": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_DeleteMember.html",
                "description": "Grants permission to delete a member and all associated resources from an Amazon Managed Blockchain network.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "member": {
                        "required": true
                    }
                }
            },
            "DeleteNode": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_DeleteNode.html",
                "description": "Grants permission to delete a node from a member of an Amazon Managed Blockchain network.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "node": {
                        "required": true
                    }
                }
            },
            "GetMember": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_GetMember.html",
                "description": "Grants permission to return detailed information about a member of an Amazon Managed Blockchain network.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "member": {
                        "required": true
                    }
                }
            },
            "GetNetwork": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_GetNetwork.html",
                "description": "Grants permission to return detailed information about an Amazon Managed Blockchain network.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "network": {
                        "required": true
                    }
                }
            },
            "GetNode": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_GetNode.html",
                "description": "Grants permission to return detailed information about a node within a member of an Amazon Managed Blockchain network.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "node": {
                        "required": true
                    }
                }
            },
            "GetProposal": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_GetProposal.html",
                "description": "Grants permission to return detailed information about a proposal of an Amazon Managed Blockchain network.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "proposal": {
                        "required": true
                    }
                }
            },
            "ListInvitations": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_ListInvitations.html",
                "description": "Grants permission to list the invitations extended to the active AWS account from any Managed Blockchain network.",
                "accessLevel": "List"
            },
            "ListMembers": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_ListMembers.html",
                "description": "Grants permission to list the members of an Amazon Managed Blockchain network and the properties of their memberships.",
                "accessLevel": "List",
                "resourceTypes": {
                    "network": {
                        "required": true
                    }
                }
            },
            "ListNetworks": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_ListNetworks.html",
                "description": "Grants permission to return information about the Amazon Managed Blockchain networks in which the current AWS account has members.",
                "accessLevel": "List"
            },
            "ListNodes": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_ListNodes.html",
                "description": "Grants permission to list the nodes within a member of an Amazon Managed Blockchain network.",
                "accessLevel": "List",
                "resourceTypes": {
                    "member": {
                        "required": true
                    }
                }
            },
            "ListProposalVotes": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_ListProposalVotes.html",
                "description": "Grants permission to list all votes for a proposal, including the value of the vote and the unique identifier of the member that cast the vote for the given Amazon Managed Blockchain network.",
                "accessLevel": "List",
                "resourceTypes": {
                    "proposal": {
                        "required": true
                    }
                }
            },
            "ListProposals": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_ListProposals.html",
                "description": "Grants permission to list proposals for the given Amazon Managed Blockchain network.",
                "accessLevel": "List",
                "resourceTypes": {
                    "network": {
                        "required": true
                    }
                }
            },
            "RejectInvitation": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_RejectInvitation.html",
                "description": "Grants permission to reject the invitation to join the blockchain network.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "invitation": {
                        "required": true
                    }
                }
            },
            "UpdateMember": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_UpdateMember.html",
                "description": "Grants permission to update a member of an Amazon Managed Blockchain network.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "member": {
                        "required": true
                    }
                }
            },
            "UpdateNode": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_UpdateNode.html",
                "description": "Grants permission to update a node from a member of an Amazon Managed Blockchain network.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "node": {
                        "required": true
                    }
                }
            },
            "VoteOnProposal": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_VoteOnProposal.html",
                "description": "Grants permission to cast a vote for a proposal on behalf of the blockchain network member specified.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "proposal": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "network": {
                "name": "network",
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_Network.html",
                "arn": "arn:${Partition}:managedblockchain:${Region}::networks/${NetworkId}",
                "conditionKeys": []
            },
            "member": {
                "name": "member",
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_Member.html",
                "arn": "arn:${Partition}:managedblockchain:${Region}:${Account}:members/${MemberId}",
                "conditionKeys": []
            },
            "node": {
                "name": "node",
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_Node.html",
                "arn": "arn:${Partition}:managedblockchain:${Region}:${Account}:nodes/${NodeId}",
                "conditionKeys": []
            },
            "proposal": {
                "name": "proposal",
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_Proposal.html",
                "arn": "arn:${Partition}:managedblockchain:${Region}::proposals/${ProposalId}",
                "conditionKeys": []
            },
            "invitation": {
                "name": "invitation",
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_Invitation.html",
                "arn": "arn:${Partition}:managedblockchain:${Region}:${Account}:invitations/${InvitationId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to create a member of an Amazon Managed Blockchain network.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_CreateMember.html
     */
    createMember() {
        this.add('managedblockchain:CreateMember');
        return this;
    }
    /**
     * Grants permission to create an Amazon Managed Blockchain network.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_CreateNetwork.html
     */
    createNetwork() {
        this.add('managedblockchain:CreateNetwork');
        return this;
    }
    /**
     * Grants permission to create a node within a member of an Amazon Managed Blockchain network.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_CreateNode.html
     */
    createNode() {
        this.add('managedblockchain:CreateNode');
        return this;
    }
    /**
     * Grants permission to create a proposal that other blockchain network members can vote on to add or remove a member in an Amazon Managed Blockchain network.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_CreateProposal.html
     */
    createProposal() {
        this.add('managedblockchain:CreateProposal');
        return this;
    }
    /**
     * Grants permission to delete a member and all associated resources from an Amazon Managed Blockchain network.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_DeleteMember.html
     */
    deleteMember() {
        this.add('managedblockchain:DeleteMember');
        return this;
    }
    /**
     * Grants permission to delete a node from a member of an Amazon Managed Blockchain network.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_DeleteNode.html
     */
    deleteNode() {
        this.add('managedblockchain:DeleteNode');
        return this;
    }
    /**
     * Grants permission to return detailed information about a member of an Amazon Managed Blockchain network.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_GetMember.html
     */
    getMember() {
        this.add('managedblockchain:GetMember');
        return this;
    }
    /**
     * Grants permission to return detailed information about an Amazon Managed Blockchain network.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_GetNetwork.html
     */
    getNetwork() {
        this.add('managedblockchain:GetNetwork');
        return this;
    }
    /**
     * Grants permission to return detailed information about a node within a member of an Amazon Managed Blockchain network.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_GetNode.html
     */
    getNode() {
        this.add('managedblockchain:GetNode');
        return this;
    }
    /**
     * Grants permission to return detailed information about a proposal of an Amazon Managed Blockchain network.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_GetProposal.html
     */
    getProposal() {
        this.add('managedblockchain:GetProposal');
        return this;
    }
    /**
     * Grants permission to list the invitations extended to the active AWS account from any Managed Blockchain network.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_ListInvitations.html
     */
    listInvitations() {
        this.add('managedblockchain:ListInvitations');
        return this;
    }
    /**
     * Grants permission to list the members of an Amazon Managed Blockchain network and the properties of their memberships.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_ListMembers.html
     */
    listMembers() {
        this.add('managedblockchain:ListMembers');
        return this;
    }
    /**
     * Grants permission to return information about the Amazon Managed Blockchain networks in which the current AWS account has members.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_ListNetworks.html
     */
    listNetworks() {
        this.add('managedblockchain:ListNetworks');
        return this;
    }
    /**
     * Grants permission to list the nodes within a member of an Amazon Managed Blockchain network.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_ListNodes.html
     */
    listNodes() {
        this.add('managedblockchain:ListNodes');
        return this;
    }
    /**
     * Grants permission to list all votes for a proposal, including the value of the vote and the unique identifier of the member that cast the vote for the given Amazon Managed Blockchain network.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_ListProposalVotes.html
     */
    listProposalVotes() {
        this.add('managedblockchain:ListProposalVotes');
        return this;
    }
    /**
     * Grants permission to list proposals for the given Amazon Managed Blockchain network.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_ListProposals.html
     */
    listProposals() {
        this.add('managedblockchain:ListProposals');
        return this;
    }
    /**
     * Grants permission to reject the invitation to join the blockchain network.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_RejectInvitation.html
     */
    rejectInvitation() {
        this.add('managedblockchain:RejectInvitation');
        return this;
    }
    /**
     * Grants permission to update a member of an Amazon Managed Blockchain network.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_UpdateMember.html
     */
    updateMember() {
        this.add('managedblockchain:UpdateMember');
        return this;
    }
    /**
     * Grants permission to update a node from a member of an Amazon Managed Blockchain network.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_UpdateNode.html
     */
    updateNode() {
        this.add('managedblockchain:UpdateNode');
        return this;
    }
    /**
     * Grants permission to cast a vote for a proposal on behalf of the blockchain network member specified.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_VoteOnProposal.html
     */
    voteOnProposal() {
        this.add('managedblockchain:VoteOnProposal');
        return this;
    }
    /**
     * Adds a resource of type network to the statement
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_Network.html
     *
     * @param networkId - Identifier for the networkId.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onNetwork(networkId, region, partition) {
        var arn = 'arn:${Partition}:managedblockchain:${Region}::networks/${NetworkId}';
        arn = arn.replace('${NetworkId}', networkId);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type member to the statement
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_Member.html
     *
     * @param memberId - Identifier for the memberId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onMember(memberId, account, region, partition) {
        var arn = 'arn:${Partition}:managedblockchain:${Region}:${Account}:members/${MemberId}';
        arn = arn.replace('${MemberId}', memberId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type node to the statement
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_Node.html
     *
     * @param nodeId - Identifier for the nodeId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onNode(nodeId, account, region, partition) {
        var arn = 'arn:${Partition}:managedblockchain:${Region}:${Account}:nodes/${NodeId}';
        arn = arn.replace('${NodeId}', nodeId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type proposal to the statement
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_Proposal.html
     *
     * @param proposalId - Identifier for the proposalId.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onProposal(proposalId, region, partition) {
        var arn = 'arn:${Partition}:managedblockchain:${Region}::proposals/${ProposalId}';
        arn = arn.replace('${ProposalId}', proposalId);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type invitation to the statement
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_Invitation.html
     *
     * @param invitationId - Identifier for the invitationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onInvitation(invitationId, account, region, partition) {
        var arn = 'arn:${Partition}:managedblockchain:${Region}:${Account}:invitations/${InvitationId}';
        arn = arn.replace('${InvitationId}', invitationId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Managedblockchain = Managedblockchain;
//# sourceMappingURL=data:application/json;base64,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