"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service quicksight
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonquicksight.html
 */
class Quicksight extends shared_1.PolicyStatement {
    /**
     * Action provider for service quicksight
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonquicksight.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'quicksight';
        this.actions = {
            "CreateAdmin": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html",
                "description": "CreateAdmin enables the user to provision Amazon QuickSight administrators, authors, and readers.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "CreateDashboard": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateDashboard.html",
                "description": "Creates a dashboard from a template",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dashboard": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateGroup": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateGroup.html",
                "description": "Create a QuickSight group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "CreateGroupMembership": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateGroupMembership.html",
                "description": "Add a QuickSight user to a QuickSight group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true,
                        "conditions": [
                            "quicksight:UserName"
                        ]
                    }
                }
            },
            "CreateIAMPolicyAssignment": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateIAMPolicyAssignment.html",
                "description": "Creates an assignment with one specified IAM Policy ARN that will be assigned to specified groups or users of QuickSight.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "assignment": {
                        "required": true
                    }
                }
            },
            "CreateReader": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html",
                "description": "CreateReader enables the user to provision Amazon QuickSight readers.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "CreateTemplate": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateTemplate.html",
                "description": "Creates a template from an existing QuickSight analysis or template",
                "accessLevel": "Write",
                "resourceTypes": {
                    "template": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateTemplateAlias": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateTemplateAlias.html",
                "description": "Creates a template alias for a template",
                "accessLevel": "Write",
                "resourceTypes": {
                    "template": {
                        "required": true
                    }
                }
            },
            "CreateTheme": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateTheme.html",
                "description": "Creates a QuickSight theme",
                "accessLevel": "Write",
                "resourceTypes": {
                    "theme": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateThemeAlias": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateThemeAlias.html",
                "description": "Creates a theme alias for a theme",
                "accessLevel": "Write",
                "resourceTypes": {
                    "theme": {
                        "required": true
                    }
                }
            },
            "CreateUser": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html",
                "description": "CreateUser enables the user to provision Amazon QuickSight authors and readers.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "DeleteDashboard": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteDashboard.html",
                "description": "Deletes a dashboard",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dashboard": {
                        "required": true
                    }
                }
            },
            "DeleteGroup": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteGroup.html",
                "description": "Remove a user group from QuickSight.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "DeleteGroupMembership": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteGroupMembership.html",
                "description": "Remove a user from a group so that he/she is no longer a member of the group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true,
                        "conditions": [
                            "quicksight:UserName"
                        ]
                    }
                }
            },
            "DeleteIAMPolicyAssignment": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteIAMPolicyAssignment.html",
                "description": "Update an existing assignment.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "assignment": {
                        "required": true
                    }
                }
            },
            "DeleteTemplate": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteTemplate.html",
                "description": "Deletes a template",
                "accessLevel": "Write",
                "resourceTypes": {
                    "template": {
                        "required": true
                    }
                }
            },
            "DeleteTemplateAlias": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteTemplateAlias.html",
                "description": "Deletes the item that the specified template alias points to",
                "accessLevel": "Write",
                "resourceTypes": {
                    "template": {
                        "required": true
                    }
                }
            },
            "DeleteTheme": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteTheme.html",
                "description": "Deletes a theme",
                "accessLevel": "Write",
                "resourceTypes": {
                    "theme": {
                        "required": true
                    }
                }
            },
            "DeleteThemeAlias": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteThemeAlias.html",
                "description": "Deletes the item that the specified theme alias points to",
                "accessLevel": "Write",
                "resourceTypes": {
                    "theme": {
                        "required": true
                    }
                }
            },
            "DeleteUser": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteUser.html",
                "description": "Delete the QuickSight user that is associated with the identity of the IAM user/role making the call. The IAM user is not deleted as a result of this call.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "DeleteUserByPrincipalId": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteUserByPrincipalId.html",
                "description": "Deletes a user identified by its principal ID.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "DescribeDashboard": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeDashboard.html",
                "description": "Provides a summary for a dashboard",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dashboard": {
                        "required": true
                    }
                }
            },
            "DescribeDashboardPermissions": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeDashboardPermissions.html",
                "description": "Describes read and write permissions for a dashboard",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dashboard": {
                        "required": true
                    }
                }
            },
            "DescribeGroup": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeGroup.html",
                "description": "Return a QuickSight group’s description and ARN.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "DescribeIAMPolicyAssignment": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeIAMPolicyAssignment.html",
                "description": "Describe an existing assignment.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "assignment": {
                        "required": true
                    }
                }
            },
            "DescribeTemplate": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeTemplate.html",
                "description": "Describes a template's metadata",
                "accessLevel": "Read",
                "resourceTypes": {
                    "template": {
                        "required": true
                    }
                }
            },
            "DescribeTemplateAlias": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeTemplateAlias.html",
                "description": "Describes the template alias for a template",
                "accessLevel": "Read",
                "resourceTypes": {
                    "template": {
                        "required": true
                    }
                }
            },
            "DescribeTemplatePermissions": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeTemplatePermissions.html",
                "description": "Describes read and write permissions on a template",
                "accessLevel": "Read",
                "resourceTypes": {
                    "template": {
                        "required": true
                    }
                }
            },
            "DescribeTheme": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeTheme.html",
                "description": "Describes a theme's metadata",
                "accessLevel": "Read",
                "resourceTypes": {
                    "theme": {
                        "required": true
                    }
                }
            },
            "DescribeThemeAlias": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeThemeAlias.html",
                "description": "Describes the theme alias for a theme",
                "accessLevel": "Read",
                "resourceTypes": {
                    "theme": {
                        "required": true
                    }
                }
            },
            "DescribeThemePermissions": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeThemePermissions.html",
                "description": "Describes read and write permissions on a theme",
                "accessLevel": "Read",
                "resourceTypes": {
                    "theme": {
                        "required": true
                    }
                }
            },
            "DescribeUser": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeUser.html",
                "description": "Return information about a user, given the user name.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "GetAuthCode": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html",
                "description": "Return an auth code representing a QuickSight user.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "GetDashboardEmbedUrl": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_GetDashboardEmbedUrl.html",
                "description": "Return a QuickSight dashboard embedding URL.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dashboard": {
                        "required": true
                    }
                }
            },
            "GetGroupMapping": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html",
                "description": "GetGroupMapping is used only in Amazon QuickSight Enterprise edition accounts. It enables the user to use Amazon QuickSight to identify and display the Microsoft Active Directory (Microsoft Active Directory) directory groups that are mapped to roles in Amazon QuickSight.",
                "accessLevel": "Read"
            },
            "ListDashboardVersions": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListDashboardVersions.html",
                "description": "Lists all the versions of the dashboards in the QuickSight subscription",
                "accessLevel": "List",
                "resourceTypes": {
                    "dashboard": {
                        "required": true
                    }
                }
            },
            "ListDashboards": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListDashboards.html",
                "description": "Lists dashboards in an AWS account",
                "accessLevel": "List",
                "resourceTypes": {
                    "dashboard": {
                        "required": true
                    }
                }
            },
            "ListGroupMemberships": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListGroupMemberships.html",
                "description": "Return a list of member users in a group.",
                "accessLevel": "List",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "ListGroups": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListGroups.html",
                "description": "Get a list of all user groups in QuickSight.",
                "accessLevel": "List",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "ListIAMPolicyAssignments": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListIAMPolicyAssignments.html",
                "description": "List all assignments in the current Amazon QuickSight account.",
                "accessLevel": "List",
                "resourceTypes": {
                    "assignment": {
                        "required": true
                    }
                }
            },
            "ListIAMPolicyAssignmentsForUser": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListIAMPolicyAssignmentsForUser.html",
                "description": "List all assignments assigned to a user and the groups it belongs",
                "accessLevel": "List",
                "resourceTypes": {
                    "assignment": {
                        "required": true
                    }
                }
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListTagsForResource.html",
                "description": "List tags of a QuickSight resource.",
                "accessLevel": "List",
                "resourceTypes": {
                    "dashboard": {
                        "required": false
                    },
                    "template": {
                        "required": false
                    },
                    "theme": {
                        "required": false
                    }
                }
            },
            "ListTemplateAliases": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListTemplateAliases.html",
                "description": "Lists all the aliases of a template",
                "accessLevel": "List",
                "resourceTypes": {
                    "template": {
                        "required": true
                    }
                }
            },
            "ListTemplateVersions": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListTemplateVersions.html",
                "description": "Lists all the versions of the templates in the current Amazon QuickSight account",
                "accessLevel": "List",
                "resourceTypes": {
                    "template": {
                        "required": true
                    }
                }
            },
            "ListTemplates": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListTemplates.html",
                "description": "Lists all the templates in the current Amazon QuickSight account",
                "accessLevel": "List",
                "resourceTypes": {
                    "template": {
                        "required": true
                    }
                }
            },
            "ListThemeAliases": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListThemeAliases.html",
                "description": "Lists all the aliases of a theme",
                "accessLevel": "List",
                "resourceTypes": {
                    "theme": {
                        "required": true
                    }
                }
            },
            "ListThemeVersions": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListThemeVersions.html",
                "description": "Lists all the versions of a theme",
                "accessLevel": "List",
                "resourceTypes": {
                    "theme": {
                        "required": true
                    }
                }
            },
            "ListThemes": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListThemes.html",
                "description": "Lists all the themes in the current Amazon QuickSight account",
                "accessLevel": "List",
                "resourceTypes": {
                    "theme": {
                        "required": true
                    }
                }
            },
            "ListUserGroups": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListUserGroups.html",
                "description": "Return a list of groups that a given user is a member of.",
                "accessLevel": "List",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "ListUsers": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListUsers.html",
                "description": "Return a list of all of the QuickSight users belonging to this account.",
                "accessLevel": "List",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "RegisterUser": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_RegisterUser.html",
                "description": "Create a QuickSight user, whose identity is associated with the IAM identity/role specified in the request.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true,
                        "conditions": [
                            "quicksight:IamArn",
                            "quicksight:SessionName"
                        ]
                    }
                }
            },
            "SearchDirectoryGroups": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html",
                "description": "SearchDirectoryGroups is used only in Amazon QuickSight Enterprise edition accounts. It enables the user to use Amazon QuickSight to display your Microsoft Active Directory directory groups so that you can choose which ones to map to roles in Amazon QuickSight.",
                "accessLevel": "Write"
            },
            "SetGroupMapping": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html",
                "description": "SearchDirectoryGroups is used only in Amazon QuickSight Enterprise edition accounts. It enables the user to use Amazon QuickSight to display your Microsoft Active Directory directory groups so that you can choose which ones to map to roles in Amazon QuickSight.",
                "accessLevel": "Write"
            },
            "Subscribe": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html",
                "description": "Subscribe enables the user to subscribe to Amazon QuickSight. Enabling this action also allows the user to upgrade the subscription to Enterprise edition.",
                "accessLevel": "Write"
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_TagResource.html",
                "description": "Add tags to a QuickSight resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "dashboard": {
                        "required": false
                    },
                    "template": {
                        "required": false
                    },
                    "theme": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "Unsubscribe": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html",
                "description": "Unsubscribe enables the user to unsubscribe from Amazon QuickSight, which permanently deletes all users and their resources from Amazon QuickSight.",
                "accessLevel": "Write"
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UntagResource.html",
                "description": "Remove tags from a QuickSight resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "dashboard": {
                        "required": false
                    },
                    "template": {
                        "required": false
                    },
                    "theme": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateDashboard": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateDashboard.html",
                "description": "Updates a dashboard in an AWS account",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dashboard": {
                        "required": true
                    }
                }
            },
            "UpdateDashboardPermissions": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateDashboardPermissions.html",
                "description": "Updates read and write permissions on a dashboard",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dashboard": {
                        "required": true
                    }
                }
            },
            "UpdateDashboardPublishedVersion": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateDashboardPublishedVersion.html",
                "description": "Updates the published version of a dashboard",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dashboard": {
                        "required": true
                    }
                }
            },
            "UpdateGroup": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateGroup.html",
                "description": "Change group description.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "UpdateIAMPolicyAssignment": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateIAMPolicyAssignment.html",
                "description": "Update an existing assignment.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "assignment": {
                        "required": true
                    }
                }
            },
            "UpdateTemplate": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateTemplate.html",
                "description": "Updates a template from an existing Amazon QuickSight analysis or another template",
                "accessLevel": "Write",
                "resourceTypes": {
                    "template": {
                        "required": true
                    }
                }
            },
            "UpdateTemplateAlias": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateTemplateAlias.html",
                "description": "Updates the template alias of a template",
                "accessLevel": "Write",
                "resourceTypes": {
                    "template": {
                        "required": true
                    }
                }
            },
            "UpdateTemplatePermissions": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateTemplatePermissions.html",
                "description": "Updates the resource permissions for a template",
                "accessLevel": "Write",
                "resourceTypes": {
                    "template": {
                        "required": true
                    }
                }
            },
            "UpdateTheme": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateTheme.html",
                "description": "Updates a theme",
                "accessLevel": "Write",
                "resourceTypes": {
                    "theme": {
                        "required": true
                    }
                }
            },
            "UpdateThemeAlias": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateThemeAlias.html",
                "description": "Updates the theme alias of a theme",
                "accessLevel": "Write",
                "resourceTypes": {
                    "theme": {
                        "required": true
                    }
                }
            },
            "UpdateThemePermissions": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateThemePermissions.html",
                "description": "Updates the resource permissions for a theme",
                "accessLevel": "Write",
                "resourceTypes": {
                    "theme": {
                        "required": true
                    }
                }
            },
            "UpdateUser": {
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateUser.html",
                "description": "Updates an Amazon QuickSight user.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "user": {
                "name": "user",
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_User.html",
                "arn": "arn:${Partition}:quicksight:${Region}:${Account}:user/${ResourceId}",
                "conditionKeys": []
            },
            "group": {
                "name": "group",
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_Group.html",
                "arn": "arn:${Partition}:quicksight:${Region}:${Account}:group/${ResourceId}",
                "conditionKeys": []
            },
            "dashboard": {
                "name": "dashboard",
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_Dashboard.html",
                "arn": "arn:${Partition}:quicksight:${Region}:${Account}:dashboard/${ResourceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "template": {
                "name": "template",
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_Template.html",
                "arn": "arn:${Partition}:quicksight:${Region}:${Account}:template/${ResourceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "theme": {
                "name": "theme",
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_Theme.html",
                "arn": "arn:${Partition}:quicksight:${Region}:${Account}:theme/${ResourceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "assignment": {
                "name": "assignment",
                "url": "https://docs.aws.amazon.com/quicksight/latest/APIReference/API_IAMPolicyAssignment.html",
                "arn": "arn:${Partition}:quicksight::${Account}:assignment/${ResourceId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * CreateAdmin enables the user to provision Amazon QuickSight administrators, authors, and readers.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    createAdmin() {
        this.add('quicksight:CreateAdmin');
        return this;
    }
    /**
     * Creates a dashboard from a template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateDashboard.html
     */
    createDashboard() {
        this.add('quicksight:CreateDashboard');
        return this;
    }
    /**
     * Create a QuickSight group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateGroup.html
     */
    createGroup() {
        this.add('quicksight:CreateGroup');
        return this;
    }
    /**
     * Add a QuickSight user to a QuickSight group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateGroupMembership.html
     */
    createGroupMembership() {
        this.add('quicksight:CreateGroupMembership');
        return this;
    }
    /**
     * Creates an assignment with one specified IAM Policy ARN that will be assigned to specified groups or users of QuickSight.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateIAMPolicyAssignment.html
     */
    createIAMPolicyAssignment() {
        this.add('quicksight:CreateIAMPolicyAssignment');
        return this;
    }
    /**
     * CreateReader enables the user to provision Amazon QuickSight readers.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    createReader() {
        this.add('quicksight:CreateReader');
        return this;
    }
    /**
     * Creates a template from an existing QuickSight analysis or template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateTemplate.html
     */
    createTemplate() {
        this.add('quicksight:CreateTemplate');
        return this;
    }
    /**
     * Creates a template alias for a template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateTemplateAlias.html
     */
    createTemplateAlias() {
        this.add('quicksight:CreateTemplateAlias');
        return this;
    }
    /**
     * Creates a QuickSight theme
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateTheme.html
     */
    createTheme() {
        this.add('quicksight:CreateTheme');
        return this;
    }
    /**
     * Creates a theme alias for a theme
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateThemeAlias.html
     */
    createThemeAlias() {
        this.add('quicksight:CreateThemeAlias');
        return this;
    }
    /**
     * CreateUser enables the user to provision Amazon QuickSight authors and readers.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    createUser() {
        this.add('quicksight:CreateUser');
        return this;
    }
    /**
     * Deletes a dashboard
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteDashboard.html
     */
    deleteDashboard() {
        this.add('quicksight:DeleteDashboard');
        return this;
    }
    /**
     * Remove a user group from QuickSight.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteGroup.html
     */
    deleteGroup() {
        this.add('quicksight:DeleteGroup');
        return this;
    }
    /**
     * Remove a user from a group so that he/she is no longer a member of the group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteGroupMembership.html
     */
    deleteGroupMembership() {
        this.add('quicksight:DeleteGroupMembership');
        return this;
    }
    /**
     * Update an existing assignment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteIAMPolicyAssignment.html
     */
    deleteIAMPolicyAssignment() {
        this.add('quicksight:DeleteIAMPolicyAssignment');
        return this;
    }
    /**
     * Deletes a template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteTemplate.html
     */
    deleteTemplate() {
        this.add('quicksight:DeleteTemplate');
        return this;
    }
    /**
     * Deletes the item that the specified template alias points to
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteTemplateAlias.html
     */
    deleteTemplateAlias() {
        this.add('quicksight:DeleteTemplateAlias');
        return this;
    }
    /**
     * Deletes a theme
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteTheme.html
     */
    deleteTheme() {
        this.add('quicksight:DeleteTheme');
        return this;
    }
    /**
     * Deletes the item that the specified theme alias points to
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteThemeAlias.html
     */
    deleteThemeAlias() {
        this.add('quicksight:DeleteThemeAlias');
        return this;
    }
    /**
     * Delete the QuickSight user that is associated with the identity of the IAM user/role making the call. The IAM user is not deleted as a result of this call.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteUser.html
     */
    deleteUser() {
        this.add('quicksight:DeleteUser');
        return this;
    }
    /**
     * Deletes a user identified by its principal ID.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteUserByPrincipalId.html
     */
    deleteUserByPrincipalId() {
        this.add('quicksight:DeleteUserByPrincipalId');
        return this;
    }
    /**
     * Provides a summary for a dashboard
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeDashboard.html
     */
    describeDashboard() {
        this.add('quicksight:DescribeDashboard');
        return this;
    }
    /**
     * Describes read and write permissions for a dashboard
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeDashboardPermissions.html
     */
    describeDashboardPermissions() {
        this.add('quicksight:DescribeDashboardPermissions');
        return this;
    }
    /**
     * Return a QuickSight group’s description and ARN.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeGroup.html
     */
    describeGroup() {
        this.add('quicksight:DescribeGroup');
        return this;
    }
    /**
     * Describe an existing assignment.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeIAMPolicyAssignment.html
     */
    describeIAMPolicyAssignment() {
        this.add('quicksight:DescribeIAMPolicyAssignment');
        return this;
    }
    /**
     * Describes a template's metadata
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeTemplate.html
     */
    describeTemplate() {
        this.add('quicksight:DescribeTemplate');
        return this;
    }
    /**
     * Describes the template alias for a template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeTemplateAlias.html
     */
    describeTemplateAlias() {
        this.add('quicksight:DescribeTemplateAlias');
        return this;
    }
    /**
     * Describes read and write permissions on a template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeTemplatePermissions.html
     */
    describeTemplatePermissions() {
        this.add('quicksight:DescribeTemplatePermissions');
        return this;
    }
    /**
     * Describes a theme's metadata
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeTheme.html
     */
    describeTheme() {
        this.add('quicksight:DescribeTheme');
        return this;
    }
    /**
     * Describes the theme alias for a theme
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeThemeAlias.html
     */
    describeThemeAlias() {
        this.add('quicksight:DescribeThemeAlias');
        return this;
    }
    /**
     * Describes read and write permissions on a theme
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeThemePermissions.html
     */
    describeThemePermissions() {
        this.add('quicksight:DescribeThemePermissions');
        return this;
    }
    /**
     * Return information about a user, given the user name.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeUser.html
     */
    describeUser() {
        this.add('quicksight:DescribeUser');
        return this;
    }
    /**
     * Return an auth code representing a QuickSight user.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    getAuthCode() {
        this.add('quicksight:GetAuthCode');
        return this;
    }
    /**
     * Return a QuickSight dashboard embedding URL.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_GetDashboardEmbedUrl.html
     */
    getDashboardEmbedUrl() {
        this.add('quicksight:GetDashboardEmbedUrl');
        return this;
    }
    /**
     * GetGroupMapping is used only in Amazon QuickSight Enterprise edition accounts. It enables the user to use Amazon QuickSight to identify and display the Microsoft Active Directory (Microsoft Active Directory) directory groups that are mapped to roles in Amazon QuickSight.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    getGroupMapping() {
        this.add('quicksight:GetGroupMapping');
        return this;
    }
    /**
     * Lists all the versions of the dashboards in the QuickSight subscription
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListDashboardVersions.html
     */
    listDashboardVersions() {
        this.add('quicksight:ListDashboardVersions');
        return this;
    }
    /**
     * Lists dashboards in an AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListDashboards.html
     */
    listDashboards() {
        this.add('quicksight:ListDashboards');
        return this;
    }
    /**
     * Return a list of member users in a group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListGroupMemberships.html
     */
    listGroupMemberships() {
        this.add('quicksight:ListGroupMemberships');
        return this;
    }
    /**
     * Get a list of all user groups in QuickSight.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListGroups.html
     */
    listGroups() {
        this.add('quicksight:ListGroups');
        return this;
    }
    /**
     * List all assignments in the current Amazon QuickSight account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListIAMPolicyAssignments.html
     */
    listIAMPolicyAssignments() {
        this.add('quicksight:ListIAMPolicyAssignments');
        return this;
    }
    /**
     * List all assignments assigned to a user and the groups it belongs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListIAMPolicyAssignmentsForUser.html
     */
    listIAMPolicyAssignmentsForUser() {
        this.add('quicksight:ListIAMPolicyAssignmentsForUser');
        return this;
    }
    /**
     * List tags of a QuickSight resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('quicksight:ListTagsForResource');
        return this;
    }
    /**
     * Lists all the aliases of a template
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListTemplateAliases.html
     */
    listTemplateAliases() {
        this.add('quicksight:ListTemplateAliases');
        return this;
    }
    /**
     * Lists all the versions of the templates in the current Amazon QuickSight account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListTemplateVersions.html
     */
    listTemplateVersions() {
        this.add('quicksight:ListTemplateVersions');
        return this;
    }
    /**
     * Lists all the templates in the current Amazon QuickSight account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListTemplates.html
     */
    listTemplates() {
        this.add('quicksight:ListTemplates');
        return this;
    }
    /**
     * Lists all the aliases of a theme
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListThemeAliases.html
     */
    listThemeAliases() {
        this.add('quicksight:ListThemeAliases');
        return this;
    }
    /**
     * Lists all the versions of a theme
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListThemeVersions.html
     */
    listThemeVersions() {
        this.add('quicksight:ListThemeVersions');
        return this;
    }
    /**
     * Lists all the themes in the current Amazon QuickSight account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListThemes.html
     */
    listThemes() {
        this.add('quicksight:ListThemes');
        return this;
    }
    /**
     * Return a list of groups that a given user is a member of.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListUserGroups.html
     */
    listUserGroups() {
        this.add('quicksight:ListUserGroups');
        return this;
    }
    /**
     * Return a list of all of the QuickSight users belonging to this account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListUsers.html
     */
    listUsers() {
        this.add('quicksight:ListUsers');
        return this;
    }
    /**
     * Create a QuickSight user, whose identity is associated with the IAM identity/role specified in the request.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_RegisterUser.html
     */
    registerUser() {
        this.add('quicksight:RegisterUser');
        return this;
    }
    /**
     * SearchDirectoryGroups is used only in Amazon QuickSight Enterprise edition accounts. It enables the user to use Amazon QuickSight to display your Microsoft Active Directory directory groups so that you can choose which ones to map to roles in Amazon QuickSight.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    searchDirectoryGroups() {
        this.add('quicksight:SearchDirectoryGroups');
        return this;
    }
    /**
     * SearchDirectoryGroups is used only in Amazon QuickSight Enterprise edition accounts. It enables the user to use Amazon QuickSight to display your Microsoft Active Directory directory groups so that you can choose which ones to map to roles in Amazon QuickSight.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    setGroupMapping() {
        this.add('quicksight:SetGroupMapping');
        return this;
    }
    /**
     * Subscribe enables the user to subscribe to Amazon QuickSight. Enabling this action also allows the user to upgrade the subscription to Enterprise edition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    subscribe() {
        this.add('quicksight:Subscribe');
        return this;
    }
    /**
     * Add tags to a QuickSight resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_TagResource.html
     */
    tagResource() {
        this.add('quicksight:TagResource');
        return this;
    }
    /**
     * Unsubscribe enables the user to unsubscribe from Amazon QuickSight, which permanently deletes all users and their resources from Amazon QuickSight.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    unsubscribe() {
        this.add('quicksight:Unsubscribe');
        return this;
    }
    /**
     * Remove tags from a QuickSight resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UntagResource.html
     */
    untagResource() {
        this.add('quicksight:UntagResource');
        return this;
    }
    /**
     * Updates a dashboard in an AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateDashboard.html
     */
    updateDashboard() {
        this.add('quicksight:UpdateDashboard');
        return this;
    }
    /**
     * Updates read and write permissions on a dashboard
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateDashboardPermissions.html
     */
    updateDashboardPermissions() {
        this.add('quicksight:UpdateDashboardPermissions');
        return this;
    }
    /**
     * Updates the published version of a dashboard
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateDashboardPublishedVersion.html
     */
    updateDashboardPublishedVersion() {
        this.add('quicksight:UpdateDashboardPublishedVersion');
        return this;
    }
    /**
     * Change group description.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateGroup.html
     */
    updateGroup() {
        this.add('quicksight:UpdateGroup');
        return this;
    }
    /**
     * Update an existing assignment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateIAMPolicyAssignment.html
     */
    updateIAMPolicyAssignment() {
        this.add('quicksight:UpdateIAMPolicyAssignment');
        return this;
    }
    /**
     * Updates a template from an existing Amazon QuickSight analysis or another template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateTemplate.html
     */
    updateTemplate() {
        this.add('quicksight:UpdateTemplate');
        return this;
    }
    /**
     * Updates the template alias of a template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateTemplateAlias.html
     */
    updateTemplateAlias() {
        this.add('quicksight:UpdateTemplateAlias');
        return this;
    }
    /**
     * Updates the resource permissions for a template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateTemplatePermissions.html
     */
    updateTemplatePermissions() {
        this.add('quicksight:UpdateTemplatePermissions');
        return this;
    }
    /**
     * Updates a theme
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateTheme.html
     */
    updateTheme() {
        this.add('quicksight:UpdateTheme');
        return this;
    }
    /**
     * Updates the theme alias of a theme
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateThemeAlias.html
     */
    updateThemeAlias() {
        this.add('quicksight:UpdateThemeAlias');
        return this;
    }
    /**
     * Updates the resource permissions for a theme
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateThemePermissions.html
     */
    updateThemePermissions() {
        this.add('quicksight:UpdateThemePermissions');
        return this;
    }
    /**
     * Updates an Amazon QuickSight user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateUser.html
     */
    updateUser() {
        this.add('quicksight:UpdateUser');
        return this;
    }
    /**
     * Adds a resource of type user to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_User.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onUser(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:quicksight:${Region}:${Account}:user/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type group to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_Group.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onGroup(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:quicksight:${Region}:${Account}:group/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type dashboard to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_Dashboard.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onDashboard(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:quicksight:${Region}:${Account}:dashboard/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type template to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_Template.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onTemplate(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:quicksight:${Region}:${Account}:template/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type theme to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_Theme.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onTheme(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:quicksight:${Region}:${Account}:theme/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type assignment to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_IAMPolicyAssignment.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAssignment(resourceId, account, partition) {
        var arn = 'arn:${Partition}:quicksight::${Account}:assignment/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * IAM user ARN or role ARN.
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifIamArn(value, operator) {
        return this.if(`quicksight:IamArn`, value, operator || 'StringLike');
    }
    /**
     * The session name.
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSessionName(value, operator) {
        return this.if(`quicksight:SessionName`, value, operator || 'StringLike');
    }
    /**
     * The user name.
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifUserName(value, operator) {
        return this.if(`quicksight:UserName`, value, operator || 'StringLike');
    }
}
exports.Quicksight = Quicksight;
//# sourceMappingURL=data:application/json;base64,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