"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service route53
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonroute53.html
 */
class Route53 extends shared_1.PolicyStatement {
    /**
     * Action provider for service route53
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonroute53.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'route53';
        this.actions = {
            "AssociateVPCWithHostedZone": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_AssociateVPCWithHostedZone.html",
                "description": "Grants permission to associate an additional Amazon VPC with a private hosted zone",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hostedzone": {
                        "required": true
                    }
                }
            },
            "ChangeResourceRecordSets": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_ChangeResourceRecordSets.html",
                "description": "Grants permission to create, update, or delete a record, which contains authoritative DNS information for a specified domain or subdomain name",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hostedzone": {
                        "required": true
                    }
                }
            },
            "ChangeTagsForResource": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_ChangeTagsForResource.html",
                "description": "Grants permission to add, edit, or delete tags for a health check or a hosted zone",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "healthcheck": {
                        "required": true
                    },
                    "hostedzone": {
                        "required": true
                    }
                }
            },
            "CreateHealthCheck": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateHealthCheck.html",
                "description": "Grants permission to create a new health check, which monitors the health and performance of your web applications, web servers, and other resources",
                "accessLevel": "Write"
            },
            "CreateHostedZone": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateHostedZone.html",
                "description": "Grants permission to create a public hosted zone, which you use to specify how the Domain Name System (DNS) routes traffic on the Internet for a domain, such as example.com, and its subdomains",
                "accessLevel": "Write"
            },
            "CreateQueryLoggingConfig": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateQueryLoggingConfig.html",
                "description": "Grants permission to create a configuration for DNS query logging",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hostedzone": {
                        "required": true
                    }
                }
            },
            "CreateReusableDelegationSet": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateReusableDelegationSet.html",
                "description": "Grants permission to create a delegation set (a group of four name servers) that can be reused by multiple hosted zones",
                "accessLevel": "Write"
            },
            "CreateTrafficPolicy": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateTrafficPolicy.html",
                "description": "Grants permission to create a traffic policy, which you use to create multiple DNS records for one domain name (such as example.com) or one subdomain name (such as www.example.com)",
                "accessLevel": "Write"
            },
            "CreateTrafficPolicyInstance": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateTrafficPolicyInstance.html",
                "description": "Grants permission to create records in a specified hosted zone based on the settings in a specified traffic policy version",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hostedzone": {
                        "required": true
                    },
                    "trafficpolicy": {
                        "required": true
                    }
                }
            },
            "CreateTrafficPolicyVersion": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateTrafficPolicyVersion.html",
                "description": "Grants permission to create a new version of an existing traffic policy",
                "accessLevel": "Write",
                "resourceTypes": {
                    "trafficpolicy": {
                        "required": true
                    }
                }
            },
            "CreateVPCAssociationAuthorization": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateVPCAssociationAuthorization.html",
                "description": "Grants permission to authorize the AWS account that created a specified VPC to submit an AssociateVPCWithHostedZone request, which associates the VPC with a specified hosted zone that was created by a different account",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hostedzone": {
                        "required": true
                    }
                }
            },
            "DeleteHealthCheck": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeleteHealthCheck.html",
                "description": "Grants permission to delete a health check",
                "accessLevel": "Write",
                "resourceTypes": {
                    "healthcheck": {
                        "required": true
                    }
                }
            },
            "DeleteHostedZone": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeleteHostedZone.html",
                "description": "Grants permission to delete a hosted zone",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hostedzone": {
                        "required": true
                    }
                }
            },
            "DeleteQueryLoggingConfig": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeleteQueryLoggingConfig.html",
                "description": "Grants permission to delete a configuration for DNS query logging",
                "accessLevel": "Write",
                "resourceTypes": {
                    "queryloggingconfig": {
                        "required": true
                    }
                }
            },
            "DeleteReusableDelegationSet": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeleteReusableDelegationSet.html",
                "description": "Grants permission to delete a reusable delegation set",
                "accessLevel": "Write",
                "resourceTypes": {
                    "delegationset": {
                        "required": true
                    }
                }
            },
            "DeleteTrafficPolicy": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeleteTrafficPolicy.html",
                "description": "Grants permission to delete a traffic policy",
                "accessLevel": "Write",
                "resourceTypes": {
                    "trafficpolicy": {
                        "required": true
                    }
                }
            },
            "DeleteTrafficPolicyInstance": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeleteTrafficPolicyInstance.html",
                "description": "Grants permission to delete a traffic policy instance and all the records that Route 53 created when you created the instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "trafficpolicyinstance": {
                        "required": true
                    }
                }
            },
            "DeleteVPCAssociationAuthorization": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeleteVPCAssociationAuthorization.html",
                "description": "Grants permission to remove authorization for associating an Amazon Virtual Private Cloud with a Route 53 private hosted zone",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hostedzone": {
                        "required": true
                    }
                }
            },
            "DisassociateVPCFromHostedZone": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_DisassociateVPCFromHostedZone.html",
                "description": "Grants permission to disassociate an Amazon Virtual Private Cloud from a Route 53 private hosted zone",
                "accessLevel": "Write"
            },
            "GetAccountLimit": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetAccountLimit.html",
                "description": "Grants permission to get the specified limit for the current account, for example, the maximum number of health checks that you can create using the account",
                "accessLevel": "Read"
            },
            "GetChange": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetChange.html",
                "description": "Grants permission to get the current status of a request to create, update, or delete one or more records",
                "accessLevel": "List",
                "resourceTypes": {
                    "change": {
                        "required": true
                    }
                }
            },
            "GetCheckerIpRanges": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetCheckerIpRanges.html",
                "description": "Grants permission to get a list of the IP ranges that are used by Route 53 health checkers to check the health of your resources",
                "accessLevel": "List"
            },
            "GetGeoLocation": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetGeoLocation.html",
                "description": "Grants permission to get information about whether a specified geographic location is supported for Route 53 geolocation records",
                "accessLevel": "List"
            },
            "GetHealthCheck": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetHealthCheck.html",
                "description": "Grants permission to get information about a specified health check",
                "accessLevel": "Read",
                "resourceTypes": {
                    "healthcheck": {
                        "required": true
                    }
                }
            },
            "GetHealthCheckCount": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetHealthCheckCount.html",
                "description": "Grants permission to get the number of health checks that are associated with the current AWS account",
                "accessLevel": "List"
            },
            "GetHealthCheckLastFailureReason": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetHealthCheckLastFailureReason.html",
                "description": "Grants permission to get the reason that a specified health check failed most recently",
                "accessLevel": "List",
                "resourceTypes": {
                    "healthcheck": {
                        "required": true
                    }
                }
            },
            "GetHealthCheckStatus": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetHealthCheckStatus.html",
                "description": "Grants permission to get the status of a specified health check",
                "accessLevel": "List",
                "resourceTypes": {
                    "healthcheck": {
                        "required": true
                    }
                }
            },
            "GetHostedZone": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetHostedZone.html",
                "description": "Grants permission to get information about a specified hosted zone including the four name servers that Route 53 assigned to the hosted zone",
                "accessLevel": "List",
                "resourceTypes": {
                    "hostedzone": {
                        "required": true
                    }
                }
            },
            "GetHostedZoneCount": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetHostedZoneCount.html",
                "description": "Grants permission to get the number of hosted zones that are associated with the current AWS account",
                "accessLevel": "List"
            },
            "GetHostedZoneLimit": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetHostedZoneLimit.html",
                "description": "Grants permission to get the specified limit for a specified hosted zone",
                "accessLevel": "Read",
                "resourceTypes": {
                    "hostedzone": {
                        "required": true
                    }
                }
            },
            "GetQueryLoggingConfig": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetQueryLoggingConfig.html",
                "description": "Grants permission to get information about a specified configuration for DNS query logging",
                "accessLevel": "Read",
                "resourceTypes": {
                    "queryloggingconfig": {
                        "required": true
                    }
                }
            },
            "GetReusableDelegationSet": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetReusableDelegationSet.html",
                "description": "Grants permission to get information about a specified reusable delegation set, including the four name servers that are assigned to the delegation set",
                "accessLevel": "List",
                "resourceTypes": {
                    "delegationset": {
                        "required": true
                    }
                }
            },
            "GetReusableDelegationSetLimit": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetReusableDelegationSetLimit.html",
                "description": "Grants permission to get the maximum number of hosted zones that you can associate with the specified reusable delegation set",
                "accessLevel": "Read",
                "resourceTypes": {
                    "delegationset": {
                        "required": true
                    }
                }
            },
            "GetTrafficPolicy": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetTrafficPolicy.html",
                "description": "Grants permission to get information about a specified traffic policy version",
                "accessLevel": "Read",
                "resourceTypes": {
                    "trafficpolicy": {
                        "required": true
                    }
                }
            },
            "GetTrafficPolicyInstance": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetTrafficPolicyInstance.html",
                "description": "Grants permission to get information about a specified traffic policy instance",
                "accessLevel": "Read",
                "resourceTypes": {
                    "trafficpolicyinstance": {
                        "required": true
                    }
                }
            },
            "GetTrafficPolicyInstanceCount": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetTrafficPolicyInstanceCount.html",
                "description": "Grants permission to get the number of traffic policy instances that are associated with the current AWS account",
                "accessLevel": "Read"
            },
            "ListGeoLocations": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListGeoLocations.html",
                "description": "Grants permission to get a list of geographic locations that Route 53 supports for geolocation",
                "accessLevel": "List"
            },
            "ListHealthChecks": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListHealthChecks.html",
                "description": "Grants permission to get a list of the health checks that are associated with the current AWS account",
                "accessLevel": "List"
            },
            "ListHostedZones": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListHostedZones.html",
                "description": "Grants permission to get a list of the public and private hosted zones that are associated with the current AWS account",
                "accessLevel": "List"
            },
            "ListHostedZonesByName": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListHostedZonesByName.html",
                "description": "Grants permission to get a list of your hosted zones in lexicographic order. Hosted zones are sorted by name with the labels reversed, for example, com.example.www.",
                "accessLevel": "List"
            },
            "ListQueryLoggingConfigs": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListQueryLoggingConfigs.html",
                "description": "Grants permission to list the configurations for DNS query logging that are associated with the current AWS account or the configuration that is associated with a specified hosted zone.",
                "accessLevel": "List",
                "resourceTypes": {
                    "queryloggingconfig": {
                        "required": true
                    }
                }
            },
            "ListResourceRecordSets": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListResourceRecordSets.html",
                "description": "Grants permission to list the records in a specified hosted zone",
                "accessLevel": "List",
                "resourceTypes": {
                    "hostedzone": {
                        "required": true
                    }
                }
            },
            "ListReusableDelegationSets": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListReusableDelegationSets.html",
                "description": "Grants permission to list the reusable delegation sets that are associated with the current AWS account.",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListTagsForResource.html",
                "description": "Grants permission to list tags for one health check or hosted zone",
                "accessLevel": "Read",
                "resourceTypes": {
                    "healthcheck": {
                        "required": false
                    },
                    "hostedzone": {
                        "required": false
                    }
                }
            },
            "ListTagsForResources": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListTagsForResources.html",
                "description": "Grants permission to list tags for up to 10 health checks or hosted zones",
                "accessLevel": "Read",
                "resourceTypes": {
                    "healthcheck": {
                        "required": false
                    },
                    "hostedzone": {
                        "required": false
                    }
                }
            },
            "ListTrafficPolicies": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListTrafficPolicies.html",
                "description": "Grants permission to get information about the latest version for every traffic policy that is associated with the current AWS account. Policies are listed in the order in which they were created.",
                "accessLevel": "Read"
            },
            "ListTrafficPolicyInstances": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListTrafficPolicyInstances.html",
                "description": "Grants permission to get information about the traffic policy instances that you created by using the current AWS account",
                "accessLevel": "Read"
            },
            "ListTrafficPolicyInstancesByHostedZone": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListTrafficPolicyInstancesByHostedZone.html",
                "description": "Grants permission to get information about the traffic policy instances that you created in a specified hosted zone",
                "accessLevel": "Read",
                "resourceTypes": {
                    "hostedzone": {
                        "required": true
                    }
                }
            },
            "ListTrafficPolicyInstancesByPolicy": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListTrafficPolicyInstancesByPolicy.html",
                "description": "Grants permission to get information about the traffic policy instances that you created using a specified traffic policy version",
                "accessLevel": "Read",
                "resourceTypes": {
                    "trafficpolicy": {
                        "required": true
                    }
                }
            },
            "ListTrafficPolicyVersions": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListTrafficPolicyVersions.html",
                "description": "Grants permission to get information about all the versions for a specified traffic policy",
                "accessLevel": "Read",
                "resourceTypes": {
                    "trafficpolicy": {
                        "required": true
                    }
                }
            },
            "ListVPCAssociationAuthorizations": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListVPCAssociationAuthorizations.html",
                "description": "Grants permission to get a list of the VPCs that were created by other accounts and that can be associated with a specified hosted zone",
                "accessLevel": "Read",
                "resourceTypes": {
                    "hostedzone": {
                        "required": true
                    }
                }
            },
            "TestDNSAnswer": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_TestDNSAnswer.html",
                "description": "Grants permission to get the value that Route 53 returns in response to a DNS query for a specified record name and type",
                "accessLevel": "Read"
            },
            "UpdateHealthCheck": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_UpdateHealthCheck.html",
                "description": "Grants permission to update an existing health check",
                "accessLevel": "Write",
                "resourceTypes": {
                    "healthcheck": {
                        "required": true
                    }
                }
            },
            "UpdateHostedZoneComment": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_UpdateHostedZoneComment.html",
                "description": "Grants permission to update the comment for a specified hosted zone",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hostedzone": {
                        "required": true
                    }
                }
            },
            "UpdateTrafficPolicyComment": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_UpdateTrafficPolicyComment.html",
                "description": "Grants permission to update the comment for a specified traffic policy version",
                "accessLevel": "Write",
                "resourceTypes": {
                    "trafficpolicy": {
                        "required": true
                    }
                }
            },
            "UpdateTrafficPolicyInstance": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_UpdateTrafficPolicyInstance.html",
                "description": "Grants permission to update the records in a specified hosted zone that were created based on the settings in a specified traffic policy version",
                "accessLevel": "Write",
                "resourceTypes": {
                    "trafficpolicyinstance": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "change": {
                "name": "change",
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_Change.html",
                "arn": "arn:${Partition}:route53:::change/${Id}",
                "conditionKeys": []
            },
            "delegationset": {
                "name": "delegationset",
                "url": "https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/route-53-concepts.html#route-53-concepts-reusable-delegation-set",
                "arn": "arn:${Partition}:route53:::delegationset/${Id}",
                "conditionKeys": []
            },
            "healthcheck": {
                "name": "healthcheck",
                "url": "https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/route-53-concepts.html#route-53-concepts-health-check",
                "arn": "arn:${Partition}:route53:::healthcheck/${Id}",
                "conditionKeys": []
            },
            "hostedzone": {
                "name": "hostedzone",
                "url": "https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/route-53-concepts.html#route-53-concepts-hosted-zone",
                "arn": "arn:${Partition}:route53:::hostedzone/${Id}",
                "conditionKeys": []
            },
            "trafficpolicy": {
                "name": "trafficpolicy",
                "url": "https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/traffic-policies.html",
                "arn": "arn:${Partition}:route53:::trafficpolicy/${Id}",
                "conditionKeys": []
            },
            "trafficpolicyinstance": {
                "name": "trafficpolicyinstance",
                "url": "https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/traffic-policy-records.html",
                "arn": "arn:${Partition}:route53:::trafficpolicyinstance/${Id}",
                "conditionKeys": []
            },
            "queryloggingconfig": {
                "name": "queryloggingconfig",
                "url": "https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/query-logs.html",
                "arn": "arn:${Partition}:route53:::queryloggingconfig/${Id}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to associate an additional Amazon VPC with a private hosted zone
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_AssociateVPCWithHostedZone.html
     */
    associateVPCWithHostedZone() {
        this.add('route53:AssociateVPCWithHostedZone');
        return this;
    }
    /**
     * Grants permission to create, update, or delete a record, which contains authoritative DNS information for a specified domain or subdomain name
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ChangeResourceRecordSets.html
     */
    changeResourceRecordSets() {
        this.add('route53:ChangeResourceRecordSets');
        return this;
    }
    /**
     * Grants permission to add, edit, or delete tags for a health check or a hosted zone
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ChangeTagsForResource.html
     */
    changeTagsForResource() {
        this.add('route53:ChangeTagsForResource');
        return this;
    }
    /**
     * Grants permission to create a new health check, which monitors the health and performance of your web applications, web servers, and other resources
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateHealthCheck.html
     */
    createHealthCheck() {
        this.add('route53:CreateHealthCheck');
        return this;
    }
    /**
     * Grants permission to create a public hosted zone, which you use to specify how the Domain Name System (DNS) routes traffic on the Internet for a domain, such as example.com, and its subdomains
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateHostedZone.html
     */
    createHostedZone() {
        this.add('route53:CreateHostedZone');
        return this;
    }
    /**
     * Grants permission to create a configuration for DNS query logging
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateQueryLoggingConfig.html
     */
    createQueryLoggingConfig() {
        this.add('route53:CreateQueryLoggingConfig');
        return this;
    }
    /**
     * Grants permission to create a delegation set (a group of four name servers) that can be reused by multiple hosted zones
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateReusableDelegationSet.html
     */
    createReusableDelegationSet() {
        this.add('route53:CreateReusableDelegationSet');
        return this;
    }
    /**
     * Grants permission to create a traffic policy, which you use to create multiple DNS records for one domain name (such as example.com) or one subdomain name (such as www.example.com)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateTrafficPolicy.html
     */
    createTrafficPolicy() {
        this.add('route53:CreateTrafficPolicy');
        return this;
    }
    /**
     * Grants permission to create records in a specified hosted zone based on the settings in a specified traffic policy version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateTrafficPolicyInstance.html
     */
    createTrafficPolicyInstance() {
        this.add('route53:CreateTrafficPolicyInstance');
        return this;
    }
    /**
     * Grants permission to create a new version of an existing traffic policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateTrafficPolicyVersion.html
     */
    createTrafficPolicyVersion() {
        this.add('route53:CreateTrafficPolicyVersion');
        return this;
    }
    /**
     * Grants permission to authorize the AWS account that created a specified VPC to submit an AssociateVPCWithHostedZone request, which associates the VPC with a specified hosted zone that was created by a different account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateVPCAssociationAuthorization.html
     */
    createVPCAssociationAuthorization() {
        this.add('route53:CreateVPCAssociationAuthorization');
        return this;
    }
    /**
     * Grants permission to delete a health check
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeleteHealthCheck.html
     */
    deleteHealthCheck() {
        this.add('route53:DeleteHealthCheck');
        return this;
    }
    /**
     * Grants permission to delete a hosted zone
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeleteHostedZone.html
     */
    deleteHostedZone() {
        this.add('route53:DeleteHostedZone');
        return this;
    }
    /**
     * Grants permission to delete a configuration for DNS query logging
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeleteQueryLoggingConfig.html
     */
    deleteQueryLoggingConfig() {
        this.add('route53:DeleteQueryLoggingConfig');
        return this;
    }
    /**
     * Grants permission to delete a reusable delegation set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeleteReusableDelegationSet.html
     */
    deleteReusableDelegationSet() {
        this.add('route53:DeleteReusableDelegationSet');
        return this;
    }
    /**
     * Grants permission to delete a traffic policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeleteTrafficPolicy.html
     */
    deleteTrafficPolicy() {
        this.add('route53:DeleteTrafficPolicy');
        return this;
    }
    /**
     * Grants permission to delete a traffic policy instance and all the records that Route 53 created when you created the instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeleteTrafficPolicyInstance.html
     */
    deleteTrafficPolicyInstance() {
        this.add('route53:DeleteTrafficPolicyInstance');
        return this;
    }
    /**
     * Grants permission to remove authorization for associating an Amazon Virtual Private Cloud with a Route 53 private hosted zone
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeleteVPCAssociationAuthorization.html
     */
    deleteVPCAssociationAuthorization() {
        this.add('route53:DeleteVPCAssociationAuthorization');
        return this;
    }
    /**
     * Grants permission to disassociate an Amazon Virtual Private Cloud from a Route 53 private hosted zone
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_DisassociateVPCFromHostedZone.html
     */
    disassociateVPCFromHostedZone() {
        this.add('route53:DisassociateVPCFromHostedZone');
        return this;
    }
    /**
     * Grants permission to get the specified limit for the current account, for example, the maximum number of health checks that you can create using the account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetAccountLimit.html
     */
    getAccountLimit() {
        this.add('route53:GetAccountLimit');
        return this;
    }
    /**
     * Grants permission to get the current status of a request to create, update, or delete one or more records
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetChange.html
     */
    getChange() {
        this.add('route53:GetChange');
        return this;
    }
    /**
     * Grants permission to get a list of the IP ranges that are used by Route 53 health checkers to check the health of your resources
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetCheckerIpRanges.html
     */
    getCheckerIpRanges() {
        this.add('route53:GetCheckerIpRanges');
        return this;
    }
    /**
     * Grants permission to get information about whether a specified geographic location is supported for Route 53 geolocation records
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetGeoLocation.html
     */
    getGeoLocation() {
        this.add('route53:GetGeoLocation');
        return this;
    }
    /**
     * Grants permission to get information about a specified health check
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetHealthCheck.html
     */
    getHealthCheck() {
        this.add('route53:GetHealthCheck');
        return this;
    }
    /**
     * Grants permission to get the number of health checks that are associated with the current AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetHealthCheckCount.html
     */
    getHealthCheckCount() {
        this.add('route53:GetHealthCheckCount');
        return this;
    }
    /**
     * Grants permission to get the reason that a specified health check failed most recently
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetHealthCheckLastFailureReason.html
     */
    getHealthCheckLastFailureReason() {
        this.add('route53:GetHealthCheckLastFailureReason');
        return this;
    }
    /**
     * Grants permission to get the status of a specified health check
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetHealthCheckStatus.html
     */
    getHealthCheckStatus() {
        this.add('route53:GetHealthCheckStatus');
        return this;
    }
    /**
     * Grants permission to get information about a specified hosted zone including the four name servers that Route 53 assigned to the hosted zone
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetHostedZone.html
     */
    getHostedZone() {
        this.add('route53:GetHostedZone');
        return this;
    }
    /**
     * Grants permission to get the number of hosted zones that are associated with the current AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetHostedZoneCount.html
     */
    getHostedZoneCount() {
        this.add('route53:GetHostedZoneCount');
        return this;
    }
    /**
     * Grants permission to get the specified limit for a specified hosted zone
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetHostedZoneLimit.html
     */
    getHostedZoneLimit() {
        this.add('route53:GetHostedZoneLimit');
        return this;
    }
    /**
     * Grants permission to get information about a specified configuration for DNS query logging
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetQueryLoggingConfig.html
     */
    getQueryLoggingConfig() {
        this.add('route53:GetQueryLoggingConfig');
        return this;
    }
    /**
     * Grants permission to get information about a specified reusable delegation set, including the four name servers that are assigned to the delegation set
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetReusableDelegationSet.html
     */
    getReusableDelegationSet() {
        this.add('route53:GetReusableDelegationSet');
        return this;
    }
    /**
     * Grants permission to get the maximum number of hosted zones that you can associate with the specified reusable delegation set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetReusableDelegationSetLimit.html
     */
    getReusableDelegationSetLimit() {
        this.add('route53:GetReusableDelegationSetLimit');
        return this;
    }
    /**
     * Grants permission to get information about a specified traffic policy version
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetTrafficPolicy.html
     */
    getTrafficPolicy() {
        this.add('route53:GetTrafficPolicy');
        return this;
    }
    /**
     * Grants permission to get information about a specified traffic policy instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetTrafficPolicyInstance.html
     */
    getTrafficPolicyInstance() {
        this.add('route53:GetTrafficPolicyInstance');
        return this;
    }
    /**
     * Grants permission to get the number of traffic policy instances that are associated with the current AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetTrafficPolicyInstanceCount.html
     */
    getTrafficPolicyInstanceCount() {
        this.add('route53:GetTrafficPolicyInstanceCount');
        return this;
    }
    /**
     * Grants permission to get a list of geographic locations that Route 53 supports for geolocation
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListGeoLocations.html
     */
    listGeoLocations() {
        this.add('route53:ListGeoLocations');
        return this;
    }
    /**
     * Grants permission to get a list of the health checks that are associated with the current AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListHealthChecks.html
     */
    listHealthChecks() {
        this.add('route53:ListHealthChecks');
        return this;
    }
    /**
     * Grants permission to get a list of the public and private hosted zones that are associated with the current AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListHostedZones.html
     */
    listHostedZones() {
        this.add('route53:ListHostedZones');
        return this;
    }
    /**
     * Grants permission to get a list of your hosted zones in lexicographic order. Hosted zones are sorted by name with the labels reversed, for example, com.example.www.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListHostedZonesByName.html
     */
    listHostedZonesByName() {
        this.add('route53:ListHostedZonesByName');
        return this;
    }
    /**
     * Grants permission to list the configurations for DNS query logging that are associated with the current AWS account or the configuration that is associated with a specified hosted zone.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListQueryLoggingConfigs.html
     */
    listQueryLoggingConfigs() {
        this.add('route53:ListQueryLoggingConfigs');
        return this;
    }
    /**
     * Grants permission to list the records in a specified hosted zone
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListResourceRecordSets.html
     */
    listResourceRecordSets() {
        this.add('route53:ListResourceRecordSets');
        return this;
    }
    /**
     * Grants permission to list the reusable delegation sets that are associated with the current AWS account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListReusableDelegationSets.html
     */
    listReusableDelegationSets() {
        this.add('route53:ListReusableDelegationSets');
        return this;
    }
    /**
     * Grants permission to list tags for one health check or hosted zone
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('route53:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to list tags for up to 10 health checks or hosted zones
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListTagsForResources.html
     */
    listTagsForResources() {
        this.add('route53:ListTagsForResources');
        return this;
    }
    /**
     * Grants permission to get information about the latest version for every traffic policy that is associated with the current AWS account. Policies are listed in the order in which they were created.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListTrafficPolicies.html
     */
    listTrafficPolicies() {
        this.add('route53:ListTrafficPolicies');
        return this;
    }
    /**
     * Grants permission to get information about the traffic policy instances that you created by using the current AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListTrafficPolicyInstances.html
     */
    listTrafficPolicyInstances() {
        this.add('route53:ListTrafficPolicyInstances');
        return this;
    }
    /**
     * Grants permission to get information about the traffic policy instances that you created in a specified hosted zone
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListTrafficPolicyInstancesByHostedZone.html
     */
    listTrafficPolicyInstancesByHostedZone() {
        this.add('route53:ListTrafficPolicyInstancesByHostedZone');
        return this;
    }
    /**
     * Grants permission to get information about the traffic policy instances that you created using a specified traffic policy version
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListTrafficPolicyInstancesByPolicy.html
     */
    listTrafficPolicyInstancesByPolicy() {
        this.add('route53:ListTrafficPolicyInstancesByPolicy');
        return this;
    }
    /**
     * Grants permission to get information about all the versions for a specified traffic policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListTrafficPolicyVersions.html
     */
    listTrafficPolicyVersions() {
        this.add('route53:ListTrafficPolicyVersions');
        return this;
    }
    /**
     * Grants permission to get a list of the VPCs that were created by other accounts and that can be associated with a specified hosted zone
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListVPCAssociationAuthorizations.html
     */
    listVPCAssociationAuthorizations() {
        this.add('route53:ListVPCAssociationAuthorizations');
        return this;
    }
    /**
     * Grants permission to get the value that Route 53 returns in response to a DNS query for a specified record name and type
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_TestDNSAnswer.html
     */
    testDNSAnswer() {
        this.add('route53:TestDNSAnswer');
        return this;
    }
    /**
     * Grants permission to update an existing health check
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_UpdateHealthCheck.html
     */
    updateHealthCheck() {
        this.add('route53:UpdateHealthCheck');
        return this;
    }
    /**
     * Grants permission to update the comment for a specified hosted zone
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_UpdateHostedZoneComment.html
     */
    updateHostedZoneComment() {
        this.add('route53:UpdateHostedZoneComment');
        return this;
    }
    /**
     * Grants permission to update the comment for a specified traffic policy version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_UpdateTrafficPolicyComment.html
     */
    updateTrafficPolicyComment() {
        this.add('route53:UpdateTrafficPolicyComment');
        return this;
    }
    /**
     * Grants permission to update the records in a specified hosted zone that were created based on the settings in a specified traffic policy version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_UpdateTrafficPolicyInstance.html
     */
    updateTrafficPolicyInstance() {
        this.add('route53:UpdateTrafficPolicyInstance');
        return this;
    }
    /**
     * Adds a resource of type change to the statement
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_Change.html
     *
     * @param id - Identifier for the id.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onChange(id, partition) {
        var arn = 'arn:${Partition}:route53:::change/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type delegationset to the statement
     *
     * https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/route-53-concepts.html#route-53-concepts-reusable-delegation-set
     *
     * @param id - Identifier for the id.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDelegationset(id, partition) {
        var arn = 'arn:${Partition}:route53:::delegationset/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type healthcheck to the statement
     *
     * https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/route-53-concepts.html#route-53-concepts-health-check
     *
     * @param id - Identifier for the id.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onHealthcheck(id, partition) {
        var arn = 'arn:${Partition}:route53:::healthcheck/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type hostedzone to the statement
     *
     * https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/route-53-concepts.html#route-53-concepts-hosted-zone
     *
     * @param id - Identifier for the id.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onHostedzone(id, partition) {
        var arn = 'arn:${Partition}:route53:::hostedzone/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type trafficpolicy to the statement
     *
     * https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/traffic-policies.html
     *
     * @param id - Identifier for the id.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTrafficpolicy(id, partition) {
        var arn = 'arn:${Partition}:route53:::trafficpolicy/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type trafficpolicyinstance to the statement
     *
     * https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/traffic-policy-records.html
     *
     * @param id - Identifier for the id.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTrafficpolicyinstance(id, partition) {
        var arn = 'arn:${Partition}:route53:::trafficpolicyinstance/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type queryloggingconfig to the statement
     *
     * https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/query-logs.html
     *
     * @param id - Identifier for the id.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onQueryloggingconfig(id, partition) {
        var arn = 'arn:${Partition}:route53:::queryloggingconfig/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Route53 = Route53;
//# sourceMappingURL=data:application/json;base64,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