"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service s3
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazons3.html
 */
class S3 extends shared_1.PolicyStatement {
    /**
     * Action provider for service s3
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazons3.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 's3';
        this.actions = {
            "AbortMultipartUpload": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_AbortMultipartUpload.html",
                "description": "Grants permission to abort a multipart upload",
                "accessLevel": "Write",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointArn",
                    "s3:DataAccessPointAccount",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "BypassGovernanceRetention": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/dev/object-lock-managing.html#object-lock-managing-bypass",
                "description": "Grants permission to allow circumvention of governance-mode object retention settings",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:RequestObjectTag/<key>",
                    "s3:RequestObjectTagKeys",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-acl",
                    "s3:x-amz-content-sha256",
                    "s3:x-amz-copy-source",
                    "s3:x-amz-grant-full-control",
                    "s3:x-amz-grant-read",
                    "s3:x-amz-grant-read-acp",
                    "s3:x-amz-grant-write",
                    "s3:x-amz-grant-write-acp",
                    "s3:x-amz-metadata-directive",
                    "s3:x-amz-server-side-encryption",
                    "s3:x-amz-server-side-encryption-aws-kms-key-id",
                    "s3:x-amz-storage-class",
                    "s3:x-amz-website-redirect-location",
                    "s3:object-lock-mode",
                    "s3:object-lock-retain-until-date",
                    "s3:object-lock-remaining-retention-days",
                    "s3:object-lock-legal-hold"
                ]
            },
            "CreateAccessPoint": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateAccessPoint.html",
                "description": "Grants permission to create a new access point",
                "accessLevel": "Write",
                "resourceTypes": {
                    "accesspoint": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:locationconstraint",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-acl",
                    "s3:x-amz-content-sha256"
                ]
            },
            "CreateBucket": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_CreateBucket.html",
                "description": "Grants permission to create a new bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:locationconstraint",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-acl",
                    "s3:x-amz-content-sha256",
                    "s3:x-amz-grant-full-control",
                    "s3:x-amz-grant-read",
                    "s3:x-amz-grant-read-acp",
                    "s3:x-amz-grant-write",
                    "s3:x-amz-grant-write-acp"
                ]
            },
            "CreateJob": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateJob.html",
                "description": "Grants permission to create a new Amazon S3 Batch Operations job",
                "accessLevel": "Write",
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256",
                    "s3:RequestJobPriority",
                    "s3:RequestJobOperation",
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "DeleteAccessPoint": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPoint.html",
                "description": "Grants permission to delete the access point named in the URI",
                "accessLevel": "Write",
                "resourceTypes": {
                    "accesspoint": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointArn",
                    "s3:DataAccessPointAccount",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "DeleteAccessPointPolicy": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPointPolicy.html",
                "description": "Grants permission to delete the policy on a specified access point",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "accesspoint": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointArn",
                    "s3:DataAccessPointAccount",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "DeleteBucket": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteBucket.html",
                "description": "Grants permission to delete the bucket named in the URI",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "DeleteBucketPolicy": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteBucketPolicy.html",
                "description": "Grants permission to delete the policy on a specified bucket",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "DeleteBucketWebsite": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteBucketWebsite.html",
                "description": "Grants permission to remove the website configuration for a bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "DeleteJobTagging": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteJobTagging.html",
                "description": "Grants permission to remove tags from an existing Amazon S3 Batch Operations job",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "job": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256",
                    "s3:ExistingJobPriority",
                    "s3:ExistingJobOperation"
                ]
            },
            "DeleteObject": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObject.html",
                "description": "Grants permission to remove the null version of an object and insert a delete marker, which becomes the current version of the object",
                "accessLevel": "Write",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "DeleteObjectTagging": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObjectTagging.html",
                "description": "Grants permission to use the tagging subresource to remove the entire tag set from the specified object",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:ExistingObjectTag/<key>",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "DeleteObjectVersion": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObject.html",
                "description": "Grants permission to remove a specific version of an object",
                "accessLevel": "Write",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:versionid",
                    "s3:x-amz-content-sha256"
                ]
            },
            "DeleteObjectVersionTagging": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObjectTagging.html",
                "description": "Grants permission to remove the entire tag set for a specific version of the object",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:ExistingObjectTag/<key>",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:versionid",
                    "s3:x-amz-content-sha256"
                ]
            },
            "DescribeJob": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DescribeJob.html",
                "description": "Grants permission to retrieve the configuration parameters and status for a batch operations job.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "job": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetAccelerateConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketAccelerateConfiguration.html",
                "description": "Grants permission to uses the accelerate subresource to return the Transfer Acceleration state of a bucket, which is either Enabled or Suspended",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetAccessPoint": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPoint.html",
                "description": "Grants permission to return configuration information about the specified access point",
                "accessLevel": "Read",
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetAccessPointPolicy": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPointPolicy.html",
                "description": "Grants permission to returns the access point policy associated with the specified access point",
                "accessLevel": "Read",
                "resourceTypes": {
                    "accesspoint": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetAccessPointPolicyStatus": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPointPolicyStatus.html",
                "description": "Grants permission to return the policy status for a specific access point policy",
                "accessLevel": "Read",
                "resourceTypes": {
                    "accesspoint": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetAccountPublicAccessBlock": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetPublicAccessBlock.html",
                "description": "Grants permission to retrieve the PublicAccessBlock configuration for an AWS account",
                "accessLevel": "Read",
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetAnalyticsConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketAnalyticsConfiguration.html",
                "description": "Grants permission to get an analytics configuration from an Amazon S3 bucket, identified by the analytics configuration ID",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetBucketAcl": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketAcl.html",
                "description": "Grants permission to use the acl subresource to return the access control list (ACL) of an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetBucketCORS": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketCors.html",
                "description": "Grants permission to return the CORS configuration information set for an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetBucketLocation": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketLocation.html",
                "description": "Grants permission to return the Region that an Amazon S3 bucket resides in",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                }
            },
            "GetBucketLogging": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketLogging.html",
                "description": "Grants permission to return the logging status of an Amazon S3 bucket and the permissions users have to view or modify that status",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetBucketNotification": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketNotification.html",
                "description": "Grants permission to get the notification configuration of an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetBucketObjectLockConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectLockConfiguration.html",
                "description": "Grants permission to get the Object Lock configuration of an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion"
                ]
            },
            "GetBucketPolicy": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketPolicy.html",
                "description": "Grants permission to return the policy of the specified bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetBucketPolicyStatus": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketPolicyStatus.html",
                "description": "Grants permission to retrieve the policy status for a specific Amazon S3 bucket, which indicates whether the bucket is public",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetBucketPublicAccessBlock": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetPublicAccessBlock.html",
                "description": "Grants permission to retrieve the PublicAccessBlock configuration for an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetBucketRequestPayment": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketRequestPayment.html",
                "description": "Grants permission to return the request payment configuration for an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetBucketTagging": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketTagging.html",
                "description": "Grants permission to return the tag set associated with an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetBucketVersioning": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketVersioning.html",
                "description": "Grants permission to return the versioning state of an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetBucketWebsite": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketWebsite.html",
                "description": "Grants permission to return the website configuration for an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetEncryptionConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketEncryption.html",
                "description": "Grants permission to return the default encryption configuration an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetInventoryConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketInventoryConfiguration.html",
                "description": "Grants permission to return an inventory configuration from an Amazon S3 bucket, identified by the inventory configuration ID",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetJobTagging": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetJobTagging.html",
                "description": "Grants permission to return the tag set of an existing Amazon S3 Batch Operations job",
                "accessLevel": "Read",
                "resourceTypes": {
                    "job": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetLifecycleConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketLifecycleConfiguration.html",
                "description": "Grants permission to return the lifecycle configuration information set on an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetMetricsConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketMetricsConfiguration.html",
                "description": "Grants permission to get a metrics configuration from an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetObject": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObject.html",
                "description": "Grants permission to retrieve objects from Amazon S3",
                "accessLevel": "Read",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:ExistingObjectTag/<key>",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetObjectAcl": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectAcl.html",
                "description": "Grants permission to return the access control list (ACL) of an object",
                "accessLevel": "Read",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:ExistingObjectTag/<key>",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetObjectLegalHold": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectLegalHold.html",
                "description": "Grants permission to get an object's current Legal Hold status",
                "accessLevel": "Read",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetObjectRetention": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectRetention.html",
                "description": "Grants permission to retrieve the retention settings for an object",
                "accessLevel": "Read",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetObjectTagging": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectTagging.html",
                "description": "Grants permission to return the tag set of an object",
                "accessLevel": "Read",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:ExistingObjectTag/<key>",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetObjectTorrent": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectTorrent.html",
                "description": "Grants permission to return torrent files from an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetObjectVersion": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObject.html",
                "description": "Grants permission to retrieve a specific version of an object",
                "accessLevel": "Read",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:ExistingObjectTag/<key>",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:versionid",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetObjectVersionAcl": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectAcl.html",
                "description": "Grants permission to return the access control list (ACL) of a specific object version",
                "accessLevel": "Read",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:ExistingObjectTag/<key>",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:versionid",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetObjectVersionForReplication": {
                "url": "",
                "description": "Grants permission to replicate both unencrypted objects and objects encrypted with SSE-S3 or SSE-KMS",
                "accessLevel": "Read",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetObjectVersionTagging": {
                "url": "",
                "description": "Grants permission to return the tag set for a specific version of the object",
                "accessLevel": "Read",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:ExistingObjectTag/<key>",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:versionid",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetObjectVersionTorrent": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectTorrent.html",
                "description": "Grants permission to get Torrent files about a different version using the versionId subresource",
                "accessLevel": "Read",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:versionid",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetReplicationConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketReplication.html",
                "description": "Grants permission to get the replication configuration information set on an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "ListAccessPoints": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListAccessPoints.html",
                "description": "Grants permission to list access points",
                "accessLevel": "Read",
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "ListAllMyBuckets": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListBuckets.html",
                "description": "Grants permission to list all buckets owned by the authenticated sender of the request",
                "accessLevel": "List",
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "ListBucket": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListObjectsV2.html",
                "description": "Grants permission to list some or all of the objects in an Amazon S3 bucket (up to 1000)",
                "accessLevel": "List",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:delimiter",
                    "s3:max-keys",
                    "s3:prefix",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "ListBucketMultipartUploads": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListMultipartUploads.html",
                "description": "Grants permission to list in-progress multipart uploads",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "ListBucketVersions": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListObjectVersions.html",
                "description": "Grants permission to list metadata about all the versions of objects in an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:delimiter",
                    "s3:max-keys",
                    "s3:prefix",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "ListJobs": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListJobs.html",
                "description": "Grants permission to list current jobs and jobs that have ended recently",
                "accessLevel": "Read",
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "ListMultipartUploadParts": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListParts.html",
                "description": "Grants permission to list the parts that have been uploaded for a specific multipart upload",
                "accessLevel": "Read",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "ObjectOwnerOverrideToBucketOwner": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/dev/replication-change-owner.html#repl-ownership-add-role-permission",
                "description": "Grants permission to change replica ownership",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutAccelerateConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketAccelerateConfiguration.html",
                "description": "Grants permission to use the accelerate subresource to set the Transfer Acceleration state of an existing S3 bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutAccessPointPolicy": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutAccessPointPolicy.html",
                "description": "Grants permission to associate an access policy with a specified access point",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "accesspoint": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutAccountPublicAccessBlock": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutPublicAccessBlock.html",
                "description": "Grants permission to create or modify the PublicAccessBlock configuration for an AWS account",
                "accessLevel": "Permissions management",
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutAnalyticsConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketAnalyticsConfiguration.html",
                "description": "Grants permission to set an analytics configuration for the bucket, specified by the analytics configuration ID",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutBucketAcl": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketAcl.html",
                "description": "Grants permission to set the permissions on an existing bucket using access control lists (ACLs)",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-acl",
                    "s3:x-amz-content-sha256",
                    "s3:x-amz-grant-full-control",
                    "s3:x-amz-grant-read",
                    "s3:x-amz-grant-read-acp",
                    "s3:x-amz-grant-write",
                    "s3:x-amz-grant-write-acp"
                ]
            },
            "PutBucketCORS": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketCors.html",
                "description": "Grants permission to set the CORS configuration for an Amazon S3 bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutBucketLogging": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketLogging.html",
                "description": "Grants permission to set the logging parameters for an Amazon S3 bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutBucketNotification": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketNotification.html",
                "description": "Grants permission to receive notifications when certain events happen in an Amazon S3 bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutBucketObjectLockConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectLockConfiguration.html",
                "description": "Grants permission to put Object Lock configuration on a specific bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion"
                ]
            },
            "PutBucketPolicy": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketPolicy.html",
                "description": "Grants permission to add or replace a bucket policy on a bucket",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutBucketPublicAccessBlock": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutPublicAccessBlock.html",
                "description": "Grants permission to create or modify the PublicAccessBlock configuration for a specific Amazon S3 bucket",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutBucketRequestPayment": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketRequestPayment.html",
                "description": "Grants permission to set the request payment configuration of a bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutBucketTagging": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketTagging.html",
                "description": "Grants permission to add a set of tags to an existing Amazon S3 bucket",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutBucketVersioning": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketVersioning.html",
                "description": "Grants permission to set the versioning state of an existing Amazon S3 bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutBucketWebsite": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketWebsite.html",
                "description": "Grants permission to set the configuration of the website that is specified in the website subresource",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutEncryptionConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketEncryption.html",
                "description": "Grants permission to set the encryption configuration for an Amazon S3 bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutInventoryConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketInventoryConfiguration.html",
                "description": "Grants permission to add an inventory configuration to the bucket, identified by the inventory ID",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutJobTagging": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutJobTagging.html",
                "description": "Grants permission to replace tags on an existing Amazon S3 Batch Operations job",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "job": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256",
                    "s3:ExistingJobPriority",
                    "s3:ExistingJobOperation",
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "PutLifecycleConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketLifecycleConfiguration.html",
                "description": "Grants permission to create a new lifecycle configuration for the bucket or replace an existing lifecycle configuration",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutMetricsConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketMetricsConfiguration.html",
                "description": "Grants permission to set or update a metrics configuration for the CloudWatch request metrics from an Amazon S3 bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutObject": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObject.html",
                "description": "Grants permission to add an object to a bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:RequestObjectTag/<key>",
                    "s3:RequestObjectTagKeys",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-acl",
                    "s3:x-amz-content-sha256",
                    "s3:x-amz-copy-source",
                    "s3:x-amz-grant-full-control",
                    "s3:x-amz-grant-read",
                    "s3:x-amz-grant-read-acp",
                    "s3:x-amz-grant-write",
                    "s3:x-amz-grant-write-acp",
                    "s3:x-amz-metadata-directive",
                    "s3:x-amz-server-side-encryption",
                    "s3:x-amz-server-side-encryption-aws-kms-key-id",
                    "s3:x-amz-storage-class",
                    "s3:x-amz-website-redirect-location",
                    "s3:object-lock-mode",
                    "s3:object-lock-retain-until-date",
                    "s3:object-lock-remaining-retention-days",
                    "s3:object-lock-legal-hold"
                ]
            },
            "PutObjectAcl": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectAcl.html",
                "description": "Grants permission to set the access control list (ACL) permissions for an object that already exists in a bucket",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:ExistingObjectTag/<key>",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-acl",
                    "s3:x-amz-content-sha256",
                    "s3:x-amz-grant-full-control",
                    "s3:x-amz-grant-read",
                    "s3:x-amz-grant-read-acp",
                    "s3:x-amz-grant-write",
                    "s3:x-amz-grant-write-acp",
                    "s3:x-amz-storage-class"
                ]
            },
            "PutObjectLegalHold": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectLegalHold.html",
                "description": "Grants permission to apply a Legal Hold configuration to the specified object",
                "accessLevel": "Write",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256",
                    "s3:object-lock-legal-hold"
                ]
            },
            "PutObjectRetention": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectRetention.html",
                "description": "Grants permission to place an Object Retention configuration on an object",
                "accessLevel": "Write",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256",
                    "s3:object-lock-mode",
                    "s3:object-lock-retain-until-date",
                    "s3:object-lock-remaining-retention-days"
                ]
            },
            "PutObjectTagging": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectTagging.html",
                "description": "Grants permission to set the supplied tag-set to an object that already exists in a bucket",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:ExistingObjectTag/<key>",
                    "s3:RequestObjectTag/<key>",
                    "s3:RequestObjectTagKeys",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutObjectVersionAcl": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectAcl.html",
                "description": "Grants permission to use the acl subresource to set the access control list (ACL) permissions for an object that already exists in a bucket",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:ExistingObjectTag/<key>",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:versionid",
                    "s3:x-amz-acl",
                    "s3:x-amz-content-sha256",
                    "s3:x-amz-grant-full-control",
                    "s3:x-amz-grant-read",
                    "s3:x-amz-grant-read-acp",
                    "s3:x-amz-grant-write",
                    "s3:x-amz-grant-write-acp",
                    "s3:x-amz-storage-class"
                ]
            },
            "PutObjectVersionTagging": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectTagging.html",
                "description": "Grants permission to set the supplied tag-set for a specific version of an object",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:ExistingObjectTag/<key>",
                    "s3:RequestObjectTag/<key>",
                    "s3:RequestObjectTagKeys",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:versionid",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutReplicationConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketReplication.html",
                "description": "Grants permission to create a new replication configuration or replace an existing one",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "ReplicateDelete": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/dev/setting-repl-config-perm-overview.html",
                "description": "Grants permission to replicate delete markers to the destination bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "ReplicateObject": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/dev/setting-repl-config-perm-overview.html",
                "description": "Grants permission to replicate objects and object tags to the destination bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256",
                    "s3:x-amz-server-side-encryption",
                    "s3:x-amz-server-side-encryption-aws-kms-key-id"
                ]
            },
            "ReplicateTags": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/dev/setting-repl-config-perm-overview.html",
                "description": "Grants permission to replicate object tags to the destination bucket",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "RestoreObject": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_RestoreObject.html",
                "description": "Grants permission to restore an archived copy of an object back into Amazon S3",
                "accessLevel": "Write",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "UpdateJobPriority": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_UpdateJobPriority.html",
                "description": "Grants permission to update the priority of an existing job",
                "accessLevel": "Write",
                "resourceTypes": {
                    "job": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256",
                    "s3:RequestJobPriority",
                    "s3:ExistingJobPriority",
                    "s3:ExistingJobOperation"
                ]
            },
            "UpdateJobStatus": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_UpdateJobStatus.html",
                "description": "Grants permission to update the status for the specified job",
                "accessLevel": "Write",
                "resourceTypes": {
                    "job": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256",
                    "s3:ExistingJobPriority",
                    "s3:ExistingJobOperation",
                    "s3:JobSuspendedCause"
                ]
            }
        };
        this.resourceTypes = {
            "accesspoint": {
                "name": "accesspoint",
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/dev/access-points.html",
                "arn": "arn:${Partition}:s3:${Region}:${Account}:accesspoint/${AccessPointName}",
                "conditionKeys": []
            },
            "bucket": {
                "name": "bucket",
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingBucket.html",
                "arn": "arn:${Partition}:s3:::${BucketName}",
                "conditionKeys": []
            },
            "object": {
                "name": "object",
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingObjects.html",
                "arn": "arn:${Partition}:s3:::${BucketName}/${ObjectName}",
                "conditionKeys": []
            },
            "job": {
                "name": "job",
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-managing-jobs.html",
                "arn": "arn:${Partition}:s3:${Region}:${Account}:job/${JobId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to abort a multipart upload
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_AbortMultipartUpload.html
     */
    abortMultipartUpload() {
        this.add('s3:AbortMultipartUpload');
        return this;
    }
    /**
     * Grants permission to allow circumvention of governance-mode object retention settings
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/object-lock-managing.html#object-lock-managing-bypass
     */
    bypassGovernanceRetention() {
        this.add('s3:BypassGovernanceRetention');
        return this;
    }
    /**
     * Grants permission to create a new access point
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateAccessPoint.html
     */
    createAccessPoint() {
        this.add('s3:CreateAccessPoint');
        return this;
    }
    /**
     * Grants permission to create a new bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_CreateBucket.html
     */
    createBucket() {
        this.add('s3:CreateBucket');
        return this;
    }
    /**
     * Grants permission to create a new Amazon S3 Batch Operations job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateJob.html
     */
    createJob() {
        this.add('s3:CreateJob');
        return this;
    }
    /**
     * Grants permission to delete the access point named in the URI
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPoint.html
     */
    deleteAccessPoint() {
        this.add('s3:DeleteAccessPoint');
        return this;
    }
    /**
     * Grants permission to delete the policy on a specified access point
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPointPolicy.html
     */
    deleteAccessPointPolicy() {
        this.add('s3:DeleteAccessPointPolicy');
        return this;
    }
    /**
     * Grants permission to delete the bucket named in the URI
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteBucket.html
     */
    deleteBucket() {
        this.add('s3:DeleteBucket');
        return this;
    }
    /**
     * Grants permission to delete the policy on a specified bucket
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteBucketPolicy.html
     */
    deleteBucketPolicy() {
        this.add('s3:DeleteBucketPolicy');
        return this;
    }
    /**
     * Grants permission to remove the website configuration for a bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteBucketWebsite.html
     */
    deleteBucketWebsite() {
        this.add('s3:DeleteBucketWebsite');
        return this;
    }
    /**
     * Grants permission to remove tags from an existing Amazon S3 Batch Operations job
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteJobTagging.html
     */
    deleteJobTagging() {
        this.add('s3:DeleteJobTagging');
        return this;
    }
    /**
     * Grants permission to remove the null version of an object and insert a delete marker, which becomes the current version of the object
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObject.html
     */
    deleteObject() {
        this.add('s3:DeleteObject');
        return this;
    }
    /**
     * Grants permission to use the tagging subresource to remove the entire tag set from the specified object
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObjectTagging.html
     */
    deleteObjectTagging() {
        this.add('s3:DeleteObjectTagging');
        return this;
    }
    /**
     * Grants permission to remove a specific version of an object
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObject.html
     */
    deleteObjectVersion() {
        this.add('s3:DeleteObjectVersion');
        return this;
    }
    /**
     * Grants permission to remove the entire tag set for a specific version of the object
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObjectTagging.html
     */
    deleteObjectVersionTagging() {
        this.add('s3:DeleteObjectVersionTagging');
        return this;
    }
    /**
     * Grants permission to retrieve the configuration parameters and status for a batch operations job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DescribeJob.html
     */
    describeJob() {
        this.add('s3:DescribeJob');
        return this;
    }
    /**
     * Grants permission to uses the accelerate subresource to return the Transfer Acceleration state of a bucket, which is either Enabled or Suspended
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketAccelerateConfiguration.html
     */
    getAccelerateConfiguration() {
        this.add('s3:GetAccelerateConfiguration');
        return this;
    }
    /**
     * Grants permission to return configuration information about the specified access point
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPoint.html
     */
    getAccessPoint() {
        this.add('s3:GetAccessPoint');
        return this;
    }
    /**
     * Grants permission to returns the access point policy associated with the specified access point
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPointPolicy.html
     */
    getAccessPointPolicy() {
        this.add('s3:GetAccessPointPolicy');
        return this;
    }
    /**
     * Grants permission to return the policy status for a specific access point policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPointPolicyStatus.html
     */
    getAccessPointPolicyStatus() {
        this.add('s3:GetAccessPointPolicyStatus');
        return this;
    }
    /**
     * Grants permission to retrieve the PublicAccessBlock configuration for an AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetPublicAccessBlock.html
     */
    getAccountPublicAccessBlock() {
        this.add('s3:GetAccountPublicAccessBlock');
        return this;
    }
    /**
     * Grants permission to get an analytics configuration from an Amazon S3 bucket, identified by the analytics configuration ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketAnalyticsConfiguration.html
     */
    getAnalyticsConfiguration() {
        this.add('s3:GetAnalyticsConfiguration');
        return this;
    }
    /**
     * Grants permission to use the acl subresource to return the access control list (ACL) of an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketAcl.html
     */
    getBucketAcl() {
        this.add('s3:GetBucketAcl');
        return this;
    }
    /**
     * Grants permission to return the CORS configuration information set for an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketCors.html
     */
    getBucketCORS() {
        this.add('s3:GetBucketCORS');
        return this;
    }
    /**
     * Grants permission to return the Region that an Amazon S3 bucket resides in
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketLocation.html
     */
    getBucketLocation() {
        this.add('s3:GetBucketLocation');
        return this;
    }
    /**
     * Grants permission to return the logging status of an Amazon S3 bucket and the permissions users have to view or modify that status
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketLogging.html
     */
    getBucketLogging() {
        this.add('s3:GetBucketLogging');
        return this;
    }
    /**
     * Grants permission to get the notification configuration of an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketNotification.html
     */
    getBucketNotification() {
        this.add('s3:GetBucketNotification');
        return this;
    }
    /**
     * Grants permission to get the Object Lock configuration of an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectLockConfiguration.html
     */
    getBucketObjectLockConfiguration() {
        this.add('s3:GetBucketObjectLockConfiguration');
        return this;
    }
    /**
     * Grants permission to return the policy of the specified bucket
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketPolicy.html
     */
    getBucketPolicy() {
        this.add('s3:GetBucketPolicy');
        return this;
    }
    /**
     * Grants permission to retrieve the policy status for a specific Amazon S3 bucket, which indicates whether the bucket is public
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketPolicyStatus.html
     */
    getBucketPolicyStatus() {
        this.add('s3:GetBucketPolicyStatus');
        return this;
    }
    /**
     * Grants permission to retrieve the PublicAccessBlock configuration for an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetPublicAccessBlock.html
     */
    getBucketPublicAccessBlock() {
        this.add('s3:GetBucketPublicAccessBlock');
        return this;
    }
    /**
     * Grants permission to return the request payment configuration for an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketRequestPayment.html
     */
    getBucketRequestPayment() {
        this.add('s3:GetBucketRequestPayment');
        return this;
    }
    /**
     * Grants permission to return the tag set associated with an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketTagging.html
     */
    getBucketTagging() {
        this.add('s3:GetBucketTagging');
        return this;
    }
    /**
     * Grants permission to return the versioning state of an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketVersioning.html
     */
    getBucketVersioning() {
        this.add('s3:GetBucketVersioning');
        return this;
    }
    /**
     * Grants permission to return the website configuration for an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketWebsite.html
     */
    getBucketWebsite() {
        this.add('s3:GetBucketWebsite');
        return this;
    }
    /**
     * Grants permission to return the default encryption configuration an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketEncryption.html
     */
    getEncryptionConfiguration() {
        this.add('s3:GetEncryptionConfiguration');
        return this;
    }
    /**
     * Grants permission to return an inventory configuration from an Amazon S3 bucket, identified by the inventory configuration ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketInventoryConfiguration.html
     */
    getInventoryConfiguration() {
        this.add('s3:GetInventoryConfiguration');
        return this;
    }
    /**
     * Grants permission to return the tag set of an existing Amazon S3 Batch Operations job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetJobTagging.html
     */
    getJobTagging() {
        this.add('s3:GetJobTagging');
        return this;
    }
    /**
     * Grants permission to return the lifecycle configuration information set on an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketLifecycleConfiguration.html
     */
    getLifecycleConfiguration() {
        this.add('s3:GetLifecycleConfiguration');
        return this;
    }
    /**
     * Grants permission to get a metrics configuration from an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketMetricsConfiguration.html
     */
    getMetricsConfiguration() {
        this.add('s3:GetMetricsConfiguration');
        return this;
    }
    /**
     * Grants permission to retrieve objects from Amazon S3
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObject.html
     */
    getObject() {
        this.add('s3:GetObject');
        return this;
    }
    /**
     * Grants permission to return the access control list (ACL) of an object
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectAcl.html
     */
    getObjectAcl() {
        this.add('s3:GetObjectAcl');
        return this;
    }
    /**
     * Grants permission to get an object's current Legal Hold status
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectLegalHold.html
     */
    getObjectLegalHold() {
        this.add('s3:GetObjectLegalHold');
        return this;
    }
    /**
     * Grants permission to retrieve the retention settings for an object
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectRetention.html
     */
    getObjectRetention() {
        this.add('s3:GetObjectRetention');
        return this;
    }
    /**
     * Grants permission to return the tag set of an object
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectTagging.html
     */
    getObjectTagging() {
        this.add('s3:GetObjectTagging');
        return this;
    }
    /**
     * Grants permission to return torrent files from an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectTorrent.html
     */
    getObjectTorrent() {
        this.add('s3:GetObjectTorrent');
        return this;
    }
    /**
     * Grants permission to retrieve a specific version of an object
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObject.html
     */
    getObjectVersion() {
        this.add('s3:GetObjectVersion');
        return this;
    }
    /**
     * Grants permission to return the access control list (ACL) of a specific object version
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectAcl.html
     */
    getObjectVersionAcl() {
        this.add('s3:GetObjectVersionAcl');
        return this;
    }
    /**
     * Grants permission to replicate both unencrypted objects and objects encrypted with SSE-S3 or SSE-KMS
     *
     * Access Level: Read
     */
    getObjectVersionForReplication() {
        this.add('s3:GetObjectVersionForReplication');
        return this;
    }
    /**
     * Grants permission to return the tag set for a specific version of the object
     *
     * Access Level: Read
     */
    getObjectVersionTagging() {
        this.add('s3:GetObjectVersionTagging');
        return this;
    }
    /**
     * Grants permission to get Torrent files about a different version using the versionId subresource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectTorrent.html
     */
    getObjectVersionTorrent() {
        this.add('s3:GetObjectVersionTorrent');
        return this;
    }
    /**
     * Grants permission to get the replication configuration information set on an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketReplication.html
     */
    getReplicationConfiguration() {
        this.add('s3:GetReplicationConfiguration');
        return this;
    }
    /**
     * Grants permission to list access points
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListAccessPoints.html
     */
    listAccessPoints() {
        this.add('s3:ListAccessPoints');
        return this;
    }
    /**
     * Grants permission to list all buckets owned by the authenticated sender of the request
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListBuckets.html
     */
    listAllMyBuckets() {
        this.add('s3:ListAllMyBuckets');
        return this;
    }
    /**
     * Grants permission to list some or all of the objects in an Amazon S3 bucket (up to 1000)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListObjectsV2.html
     */
    listBucket() {
        this.add('s3:ListBucket');
        return this;
    }
    /**
     * Grants permission to list in-progress multipart uploads
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListMultipartUploads.html
     */
    listBucketMultipartUploads() {
        this.add('s3:ListBucketMultipartUploads');
        return this;
    }
    /**
     * Grants permission to list metadata about all the versions of objects in an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListObjectVersions.html
     */
    listBucketVersions() {
        this.add('s3:ListBucketVersions');
        return this;
    }
    /**
     * Grants permission to list current jobs and jobs that have ended recently
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListJobs.html
     */
    listJobs() {
        this.add('s3:ListJobs');
        return this;
    }
    /**
     * Grants permission to list the parts that have been uploaded for a specific multipart upload
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListParts.html
     */
    listMultipartUploadParts() {
        this.add('s3:ListMultipartUploadParts');
        return this;
    }
    /**
     * Grants permission to change replica ownership
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/replication-change-owner.html#repl-ownership-add-role-permission
     */
    objectOwnerOverrideToBucketOwner() {
        this.add('s3:ObjectOwnerOverrideToBucketOwner');
        return this;
    }
    /**
     * Grants permission to use the accelerate subresource to set the Transfer Acceleration state of an existing S3 bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketAccelerateConfiguration.html
     */
    putAccelerateConfiguration() {
        this.add('s3:PutAccelerateConfiguration');
        return this;
    }
    /**
     * Grants permission to associate an access policy with a specified access point
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutAccessPointPolicy.html
     */
    putAccessPointPolicy() {
        this.add('s3:PutAccessPointPolicy');
        return this;
    }
    /**
     * Grants permission to create or modify the PublicAccessBlock configuration for an AWS account
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutPublicAccessBlock.html
     */
    putAccountPublicAccessBlock() {
        this.add('s3:PutAccountPublicAccessBlock');
        return this;
    }
    /**
     * Grants permission to set an analytics configuration for the bucket, specified by the analytics configuration ID
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketAnalyticsConfiguration.html
     */
    putAnalyticsConfiguration() {
        this.add('s3:PutAnalyticsConfiguration');
        return this;
    }
    /**
     * Grants permission to set the permissions on an existing bucket using access control lists (ACLs)
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketAcl.html
     */
    putBucketAcl() {
        this.add('s3:PutBucketAcl');
        return this;
    }
    /**
     * Grants permission to set the CORS configuration for an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketCors.html
     */
    putBucketCORS() {
        this.add('s3:PutBucketCORS');
        return this;
    }
    /**
     * Grants permission to set the logging parameters for an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketLogging.html
     */
    putBucketLogging() {
        this.add('s3:PutBucketLogging');
        return this;
    }
    /**
     * Grants permission to receive notifications when certain events happen in an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketNotification.html
     */
    putBucketNotification() {
        this.add('s3:PutBucketNotification');
        return this;
    }
    /**
     * Grants permission to put Object Lock configuration on a specific bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectLockConfiguration.html
     */
    putBucketObjectLockConfiguration() {
        this.add('s3:PutBucketObjectLockConfiguration');
        return this;
    }
    /**
     * Grants permission to add or replace a bucket policy on a bucket
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketPolicy.html
     */
    putBucketPolicy() {
        this.add('s3:PutBucketPolicy');
        return this;
    }
    /**
     * Grants permission to create or modify the PublicAccessBlock configuration for a specific Amazon S3 bucket
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutPublicAccessBlock.html
     */
    putBucketPublicAccessBlock() {
        this.add('s3:PutBucketPublicAccessBlock');
        return this;
    }
    /**
     * Grants permission to set the request payment configuration of a bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketRequestPayment.html
     */
    putBucketRequestPayment() {
        this.add('s3:PutBucketRequestPayment');
        return this;
    }
    /**
     * Grants permission to add a set of tags to an existing Amazon S3 bucket
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketTagging.html
     */
    putBucketTagging() {
        this.add('s3:PutBucketTagging');
        return this;
    }
    /**
     * Grants permission to set the versioning state of an existing Amazon S3 bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketVersioning.html
     */
    putBucketVersioning() {
        this.add('s3:PutBucketVersioning');
        return this;
    }
    /**
     * Grants permission to set the configuration of the website that is specified in the website subresource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketWebsite.html
     */
    putBucketWebsite() {
        this.add('s3:PutBucketWebsite');
        return this;
    }
    /**
     * Grants permission to set the encryption configuration for an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketEncryption.html
     */
    putEncryptionConfiguration() {
        this.add('s3:PutEncryptionConfiguration');
        return this;
    }
    /**
     * Grants permission to add an inventory configuration to the bucket, identified by the inventory ID
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketInventoryConfiguration.html
     */
    putInventoryConfiguration() {
        this.add('s3:PutInventoryConfiguration');
        return this;
    }
    /**
     * Grants permission to replace tags on an existing Amazon S3 Batch Operations job
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutJobTagging.html
     */
    putJobTagging() {
        this.add('s3:PutJobTagging');
        return this;
    }
    /**
     * Grants permission to create a new lifecycle configuration for the bucket or replace an existing lifecycle configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketLifecycleConfiguration.html
     */
    putLifecycleConfiguration() {
        this.add('s3:PutLifecycleConfiguration');
        return this;
    }
    /**
     * Grants permission to set or update a metrics configuration for the CloudWatch request metrics from an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketMetricsConfiguration.html
     */
    putMetricsConfiguration() {
        this.add('s3:PutMetricsConfiguration');
        return this;
    }
    /**
     * Grants permission to add an object to a bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObject.html
     */
    putObject() {
        this.add('s3:PutObject');
        return this;
    }
    /**
     * Grants permission to set the access control list (ACL) permissions for an object that already exists in a bucket
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectAcl.html
     */
    putObjectAcl() {
        this.add('s3:PutObjectAcl');
        return this;
    }
    /**
     * Grants permission to apply a Legal Hold configuration to the specified object
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectLegalHold.html
     */
    putObjectLegalHold() {
        this.add('s3:PutObjectLegalHold');
        return this;
    }
    /**
     * Grants permission to place an Object Retention configuration on an object
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectRetention.html
     */
    putObjectRetention() {
        this.add('s3:PutObjectRetention');
        return this;
    }
    /**
     * Grants permission to set the supplied tag-set to an object that already exists in a bucket
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectTagging.html
     */
    putObjectTagging() {
        this.add('s3:PutObjectTagging');
        return this;
    }
    /**
     * Grants permission to use the acl subresource to set the access control list (ACL) permissions for an object that already exists in a bucket
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectAcl.html
     */
    putObjectVersionAcl() {
        this.add('s3:PutObjectVersionAcl');
        return this;
    }
    /**
     * Grants permission to set the supplied tag-set for a specific version of an object
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectTagging.html
     */
    putObjectVersionTagging() {
        this.add('s3:PutObjectVersionTagging');
        return this;
    }
    /**
     * Grants permission to create a new replication configuration or replace an existing one
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketReplication.html
     */
    putReplicationConfiguration() {
        this.add('s3:PutReplicationConfiguration');
        return this;
    }
    /**
     * Grants permission to replicate delete markers to the destination bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/setting-repl-config-perm-overview.html
     */
    replicateDelete() {
        this.add('s3:ReplicateDelete');
        return this;
    }
    /**
     * Grants permission to replicate objects and object tags to the destination bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/setting-repl-config-perm-overview.html
     */
    replicateObject() {
        this.add('s3:ReplicateObject');
        return this;
    }
    /**
     * Grants permission to replicate object tags to the destination bucket
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/setting-repl-config-perm-overview.html
     */
    replicateTags() {
        this.add('s3:ReplicateTags');
        return this;
    }
    /**
     * Grants permission to restore an archived copy of an object back into Amazon S3
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_RestoreObject.html
     */
    restoreObject() {
        this.add('s3:RestoreObject');
        return this;
    }
    /**
     * Grants permission to update the priority of an existing job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_UpdateJobPriority.html
     */
    updateJobPriority() {
        this.add('s3:UpdateJobPriority');
        return this;
    }
    /**
     * Grants permission to update the status for the specified job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_UpdateJobStatus.html
     */
    updateJobStatus() {
        this.add('s3:UpdateJobStatus');
        return this;
    }
    /**
     * Adds a resource of type accesspoint to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/access-points.html
     *
     * @param accessPointName - Identifier for the accessPointName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAccesspoint(accessPointName, account, region, partition) {
        var arn = 'arn:${Partition}:s3:${Region}:${Account}:accesspoint/${AccessPointName}';
        arn = arn.replace('${AccessPointName}', accessPointName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type bucket to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingBucket.html
     *
     * @param bucketName - Identifier for the bucketName.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onBucket(bucketName, partition) {
        var arn = 'arn:${Partition}:s3:::${BucketName}';
        arn = arn.replace('${BucketName}', bucketName);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type object to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingObjects.html
     *
     * @param bucketName - Identifier for the bucketName.
     * @param objectName - Identifier for the objectName.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onObject(bucketName, objectName, partition) {
        var arn = 'arn:${Partition}:s3:::${BucketName}/${ObjectName}';
        arn = arn.replace('${BucketName}', bucketName);
        arn = arn.replace('${ObjectName}', objectName);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type job to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-managing-jobs.html
     *
     * @param jobId - Identifier for the jobId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onJob(jobId, account, region, partition) {
        var arn = 'arn:${Partition}:s3:${Region}:${Account}:job/${JobId}';
        arn = arn.replace('${JobId}', jobId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters access by the network origin (Internet or VPC)
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/creating-access-points.html#access-points-policies
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAccessPointNetworkOrigin(value, operator) {
        return this.if(`s3:AccessPointNetworkOrigin`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the AWS Account ID that owns the access point
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/creating-access-points.html#access-points-policies
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDataAccessPointAccount(value, operator) {
        return this.if(`s3:DataAccessPointAccount`, value, operator || 'StringLike');
    }
    /**
     * Filters access by an access point Amazon Resource Name (ARN)
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDataAccessPointArn(value, operator) {
        return this.if(`s3:DataAccessPointArn`, value, operator || 'StringLike');
    }
    /**
     * Filters access to updating the job priority by operation
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-job-tags-examples.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifExistingJobOperation(value, operator) {
        return this.if(`s3:ExistingJobOperation`, value, operator || 'StringLike');
    }
    /**
     * Filters access to cancelling existing jobs by priority range
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-job-tags-examples.html
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifExistingJobPriority(value, operator) {
        return this.if(`s3:ExistingJobPriority`, value, operator || 'NumericEquals');
    }
    /**
     * Requires that an existing object tag has a specific tag key and value.
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/object-tagging.html#tagging-and-policies
     *
     * @param key The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifExistingObjectTag(key, value, operator) {
        return this.if(`s3:ExistingObjectTag/${key}`, value, operator || 'StringLike');
    }
    /**
     * Filters access to cancelling suspended jobs by a specific job suspended cause (for example, AWAITING_CONFIRMATION)
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-job-tags-examples.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifJobSuspendedCause(value, operator) {
        return this.if(`s3:JobSuspendedCause`, value, operator || 'StringLike');
    }
    /**
     * Filters access by a specific Region
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/amazon-s3-policy-keys.html#condition-key-bucket-ops-1
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifLocationConstraint(value, operator) {
        return this.if(`s3:LocationConstraint`, value, operator || 'StringLike');
    }
    /**
     * Filters access to creating jobs by operation
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-job-tags-examples.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestJobOperation(value, operator) {
        return this.if(`s3:RequestJobOperation`, value, operator || 'StringLike');
    }
    /**
     * Filters access to creating new jobs by priority range
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-job-tags-examples.html
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifRequestJobPriority(value, operator) {
        return this.if(`s3:RequestJobPriority`, value, operator || 'NumericEquals');
    }
    /**
     * Restricts the tag keys and values allowed on objects
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/object-tagging.html#tagging-and-policies
     *
     * @param key The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestObjectTag(key, value, operator) {
        return this.if(`s3:RequestObjectTag/${key}`, value, operator || 'StringLike');
    }
    /**
     * Restricts the tag keys allowed on objects
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/object-tagging.html#tagging-and-policies
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestObjectTagKeys(value, operator) {
        return this.if(`s3:RequestObjectTagKeys`, value, operator || 'StringLike');
    }
    /**
     * Filters access by a specific object version
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/amazon-s3-policy-keys.html#getobjectversion-limit-access-to-specific-version-3
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVersionId(value, operator) {
        return this.if(`s3:VersionId`, value, operator || 'StringLike');
    }
    /**
     * Restricts incoming requests to a specific authentication method
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAuthType(value, operator) {
        return this.if(`s3:authType`, value, operator || 'StringLike');
    }
    /**
     * Requires the delimiter parameter
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/walkthrough1.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDelimiter(value, operator) {
        return this.if(`s3:delimiter`, value, operator || 'StringLike');
    }
    /**
     * Filters access by a specific Region
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/amazon-s3-policy-keys.html#condition-key-bucket-ops-1
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifLocationconstraint(value, operator) {
        return this.if(`s3:locationconstraint`, value, operator || 'StringLike');
    }
    /**
     * Limits the maximum number of keys returned in a ListBucket request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/amazon-s3-policy-keys.html#example-numeric-condition-operators
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifMaxKeys(value, operator) {
        return this.if(`s3:max-keys`, value, operator || 'NumericEquals');
    }
    /**
     * Enables enforcement of the specified object legal hold status
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/object-lock-overview.html#object-lock-legal-holds
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifObjectLockLegalHold(value, operator) {
        return this.if(`s3:object-lock-legal-hold`, value, operator || 'StringLike');
    }
    /**
     * Enables enforcement of the specified object retention mode (COMPLIANCE or GOVERNANCE)
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifObjectLockMode(value, operator) {
        return this.if(`s3:object-lock-mode`, value, operator || 'StringLike');
    }
    /**
     * Enables enforcement of an object relative to the remaining retention days
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/object-lock-managing.html#object-lock-managing-retention-limits
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifObjectLockRemainingRetentionDays(value, operator) {
        return this.if(`s3:object-lock-remaining-retention-days`, value, operator || 'StringLike');
    }
    /**
     * Enables enforcement of a specific retain-until-date
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/object-lock-overview.html#object-lock-retention-periods
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifObjectLockRetainUntilDate(value, operator) {
        return this.if(`s3:object-lock-retain-until-date`, value, operator || 'StringLike');
    }
    /**
     * Filters access by key name prefix
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/amazon-s3-policy-keys.html#condition-key-bucket-ops-2
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPrefix(value, operator) {
        return this.if(`s3:prefix`, value, operator || 'StringLike');
    }
    /**
     * Identifies the length of time, in milliseconds, that a signature is valid in an authenticated request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifSignatureAge(value, operator) {
        return this.if(`s3:signatureAge`, value, operator || 'NumericEquals');
    }
    /**
     * Identifies the version of AWS Signature that is supported for authenticated requests
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSignatureversion(value, operator) {
        return this.if(`s3:signatureversion`, value, operator || 'StringLike');
    }
    /**
     * Filters access by a specific object version
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/amazon-s3-policy-keys.html#getobjectversion-limit-access-to-specific-version-3
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVersionid(value, operator) {
        return this.if(`s3:versionid`, value, operator || 'StringLike');
    }
    /**
     * Requires the x-amz-acl header with a specific canned ACL in a request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#permissions
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzAcl(value, operator) {
        return this.if(`s3:x-amz-acl`, value, operator || 'StringLike');
    }
    /**
     * Disallows unsigned content in your bucket
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzContentSha256(value, operator) {
        return this.if(`s3:x-amz-content-sha256`, value, operator || 'StringLike');
    }
    /**
     * Restricts the copy source to a specific bucket, prefix, or object
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/amazon-s3-policy-keys.html#putobject-limit-copy-source-3
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzCopySource(value, operator) {
        return this.if(`s3:x-amz-copy-source`, value, operator || 'StringLike');
    }
    /**
     * Requires the x-amz-grant-full-control (full control) header in a request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#permissions
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzGrantFullControl(value, operator) {
        return this.if(`s3:x-amz-grant-full-control`, value, operator || 'StringLike');
    }
    /**
     * Requires the x-amz-grant-read (read access) header in a request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#permissions
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzGrantRead(value, operator) {
        return this.if(`s3:x-amz-grant-read`, value, operator || 'StringLike');
    }
    /**
     * Requires the x-amz-grant-read-acp (read permissions for the ACL) header in a request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#permissions
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzGrantReadAcp(value, operator) {
        return this.if(`s3:x-amz-grant-read-acp`, value, operator || 'StringLike');
    }
    /**
     * Requires the x-amz-grant-write (write access) header in a request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#permissions
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzGrantWrite(value, operator) {
        return this.if(`s3:x-amz-grant-write`, value, operator || 'StringLike');
    }
    /**
     * Requires the x-amz-grant-write-acp (write permissions for the ACL) header in a request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#permissions
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzGrantWriteAcp(value, operator) {
        return this.if(`s3:x-amz-grant-write-acp`, value, operator || 'StringLike');
    }
    /**
     * Enables enforcement of object metadata behavior (COPY or REPLACE) when objects are copied
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_CopyObject.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzMetadataDirective(value, operator) {
        return this.if(`s3:x-amz-metadata-directive`, value, operator || 'StringLike');
    }
    /**
     * Requires server-side encryption
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingServerSideEncryption.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzServerSideEncryption(value, operator) {
        return this.if(`s3:x-amz-server-side-encryption`, value, operator || 'StringLike');
    }
    /**
     * Requires a specific AWS KMS customer managed CMK for server-side encryption
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingKMSEncryption.html#require-sse-kms
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzServerSideEncryptionAwsKmsKeyId(value, operator) {
        return this.if(`s3:x-amz-server-side-encryption-aws-kms-key-id`, value, operator || 'StringLike');
    }
    /**
     * Filters access by storage class
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/storage-class-intro.html#sc-howtoset
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzStorageClass(value, operator) {
        return this.if(`s3:x-amz-storage-class`, value, operator || 'StringLike');
    }
    /**
     * Filters access by a specific website redirect location for buckets that are configured as static websites
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/how-to-page-redirect.html#page-redirect-using-rest-api
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzWebsiteRedirectLocation(value, operator) {
        return this.if(`s3:x-amz-website-redirect-location`, value, operator || 'StringLike');
    }
}
exports.S3 = S3;
//# sourceMappingURL=data:application/json;base64,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