"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service sdb
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsimpledb.html
 */
class Sdb extends shared_1.PolicyStatement {
    /**
     * Action provider for service sdb
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsimpledb.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'sdb';
        this.actions = {
            "BatchDeleteAttributes": {
                "url": "https://docs.aws.amazon.com/AmazonSimpleDB/latest/DeveloperGuide/API_BatchDeleteAttributes.html",
                "description": "Performs multiple DeleteAttributes operations in a single call, which reduces round trips and latencies.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "BatchPutAttributes": {
                "url": "https://docs.aws.amazon.com/AmazonSimpleDB/latest/DeveloperGuide/API_BatchPutAttributes.html",
                "description": "With the BatchPutAttributes operation, you can perform multiple PutAttribute operations in a single call. With the BatchPutAttributes operation, you can perform multiple PutAttribute operations in a single call.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "CreateDomain": {
                "url": "https://docs.aws.amazon.com/AmazonSimpleDB/latest/DeveloperGuide/API_CreateDomain.html",
                "description": "The CreateDomain operation creates a new domain.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DeleteAttributes": {
                "url": "https://docs.aws.amazon.com/AmazonSimpleDB/latest/DeveloperGuide/API_DeleteAttributes.html",
                "description": "Deletes one or more attributes associated with the item.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DeleteDomain": {
                "url": "https://docs.aws.amazon.com/AmazonSimpleDB/latest/DeveloperGuide/API_DeleteDomain.html",
                "description": "The DeleteDomain operation deletes a domain.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DomainMetadata": {
                "url": "https://docs.aws.amazon.com/AmazonSimpleDB/latest/DeveloperGuide/API_DomainMetadata.html",
                "description": "Returns information about the domain, including when the domain was created, the number of items and attributes, and the size of attribute names and values.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "GetAttributes": {
                "url": "https://docs.aws.amazon.com/AmazonSimpleDB/latest/DeveloperGuide/API_GetAttributes.html",
                "description": "Returns all of the attributes associated with the item.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "ListDomains": {
                "url": "https://docs.aws.amazon.com/AmazonSimpleDB/latest/DeveloperGuide/API_ListDomains.html",
                "description": "Description for ListDomains",
                "accessLevel": "List"
            },
            "PutAttributes": {
                "url": "https://docs.aws.amazon.com/AmazonSimpleDB/latest/DeveloperGuide/API_PutAttributes.html",
                "description": "The PutAttributes operation creates or replaces attributes in an item.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "Select": {
                "url": "https://docs.aws.amazon.com/AmazonSimpleDB/latest/DeveloperGuide/API_Select.html",
                "description": "Description for Select",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "domain": {
                "name": "domain",
                "url": "https://docs.aws.amazon.com/sdb/latest/APIReference/DataModel.html",
                "arn": "arn:${Partition}:sdb:${Region}:${Account}:domain/${DomainName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Performs multiple DeleteAttributes operations in a single call, which reduces round trips and latencies.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonSimpleDB/latest/DeveloperGuide/API_BatchDeleteAttributes.html
     */
    batchDeleteAttributes() {
        this.add('sdb:BatchDeleteAttributes');
        return this;
    }
    /**
     * With the BatchPutAttributes operation, you can perform multiple PutAttribute operations in a single call. With the BatchPutAttributes operation, you can perform multiple PutAttribute operations in a single call.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonSimpleDB/latest/DeveloperGuide/API_BatchPutAttributes.html
     */
    batchPutAttributes() {
        this.add('sdb:BatchPutAttributes');
        return this;
    }
    /**
     * The CreateDomain operation creates a new domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonSimpleDB/latest/DeveloperGuide/API_CreateDomain.html
     */
    createDomain() {
        this.add('sdb:CreateDomain');
        return this;
    }
    /**
     * Deletes one or more attributes associated with the item.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonSimpleDB/latest/DeveloperGuide/API_DeleteAttributes.html
     */
    deleteAttributes() {
        this.add('sdb:DeleteAttributes');
        return this;
    }
    /**
     * The DeleteDomain operation deletes a domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonSimpleDB/latest/DeveloperGuide/API_DeleteDomain.html
     */
    deleteDomain() {
        this.add('sdb:DeleteDomain');
        return this;
    }
    /**
     * Returns information about the domain, including when the domain was created, the number of items and attributes, and the size of attribute names and values.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonSimpleDB/latest/DeveloperGuide/API_DomainMetadata.html
     */
    domainMetadata() {
        this.add('sdb:DomainMetadata');
        return this;
    }
    /**
     * Returns all of the attributes associated with the item.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonSimpleDB/latest/DeveloperGuide/API_GetAttributes.html
     */
    getAttributes() {
        this.add('sdb:GetAttributes');
        return this;
    }
    /**
     * Description for ListDomains
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonSimpleDB/latest/DeveloperGuide/API_ListDomains.html
     */
    listDomains() {
        this.add('sdb:ListDomains');
        return this;
    }
    /**
     * The PutAttributes operation creates or replaces attributes in an item.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonSimpleDB/latest/DeveloperGuide/API_PutAttributes.html
     */
    putAttributes() {
        this.add('sdb:PutAttributes');
        return this;
    }
    /**
     * Description for Select
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonSimpleDB/latest/DeveloperGuide/API_Select.html
     */
    select() {
        this.add('sdb:Select');
        return this;
    }
    /**
     * Adds a resource of type domain to the statement
     *
     * https://docs.aws.amazon.com/sdb/latest/APIReference/DataModel.html
     *
     * @param domainName - Identifier for the domainName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDomain(domainName, account, region, partition) {
        var arn = 'arn:${Partition}:sdb:${Region}:${Account}:domain/${DomainName}';
        arn = arn.replace('${DomainName}', domainName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Sdb = Sdb;
//# sourceMappingURL=data:application/json;base64,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