"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service sns
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsns.html
 */
class Sns extends shared_1.PolicyStatement {
    /**
     * Action provider for service sns
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsns.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'sns';
        this.actions = {
            "AddPermission": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_AddPermission.html",
                "description": "Adds a statement to a topic's access control policy, granting access for the specified AWS accounts to the specified actions.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "topic": {
                        "required": true
                    }
                }
            },
            "CheckIfPhoneNumberIsOptedOut": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_CheckIfPhoneNumberIsOptedOut.html",
                "description": "Accepts a phone number and indicates whether the phone holder has opted out of receiving SMS messages from your account.",
                "accessLevel": "Read"
            },
            "ConfirmSubscription": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_ConfirmSubscription.html",
                "description": "Verifies an endpoint owner's intent to receive messages by validating the token sent to the endpoint by an earlier Subscribe action.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "topic": {
                        "required": true
                    }
                }
            },
            "CreatePlatformApplication": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_CreatePlatformApplication.html",
                "description": "Creates a platform application object for one of the supported push notification services, such as APNS and GCM, to which devices and mobile apps may register.",
                "accessLevel": "Write"
            },
            "CreatePlatformEndpoint": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_CreatePlatformEndpoint.html",
                "description": "Creates an endpoint for a device and mobile app on one of the supported push notification services, such as GCM and APNS.",
                "accessLevel": "Write"
            },
            "CreateTopic": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_CreateTopic.html",
                "description": "Creates a topic to which notifications can be published.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "topic": {
                        "required": true
                    }
                }
            },
            "DeleteEndpoint": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_DeleteEndpoint.html",
                "description": "Deletes the endpoint for a device and mobile app from Amazon SNS.",
                "accessLevel": "Write"
            },
            "DeletePlatformApplication": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_DeletePlatformApplication.html",
                "description": "Deletes a platform application object for one of the supported push notification services, such as APNS and GCM.",
                "accessLevel": "Write"
            },
            "DeleteTopic": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_DeleteTopic.html",
                "description": "Deletes a topic and all its subscriptions.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "topic": {
                        "required": true
                    }
                }
            },
            "GetEndpointAttributes": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_GetEndpointAttributes.html",
                "description": "Retrieves the endpoint attributes for a device on one of the supported push notification services, such as GCM and APNS.",
                "accessLevel": "Read"
            },
            "GetPlatformApplicationAttributes": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_GetPlatformApplicationAttributes.html",
                "description": "Retrieves the attributes of the platform application object for the supported push notification services, such as APNS and GCM.",
                "accessLevel": "Read"
            },
            "GetSMSAttributes": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_GetSMSAttributes.html",
                "description": "Returns the settings for sending SMS messages from your account.",
                "accessLevel": "Read"
            },
            "GetSubscriptionAttributes": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_GetSubscriptionAttributes.html",
                "description": "Returns all of the properties of a subscription.",
                "accessLevel": "Read"
            },
            "GetTopicAttributes": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_GetTopicAttributes.html",
                "description": "Returns all of the properties of a topic. Topic properties returned might differ based on the authorization of the user.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "topic": {
                        "required": true
                    }
                }
            },
            "ListEndpointsByPlatformApplication": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_ListEndpointsByPlatformApplication.html",
                "description": "Lists the endpoints and endpoint attributes for devices in a supported push notification service, such as GCM and APNS.",
                "accessLevel": "List"
            },
            "ListPhoneNumbersOptedOut": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_ListPhoneNumbersOptedOut.html",
                "description": "Returns a list of phone numbers that are opted out, meaning you cannot send SMS messages to them.",
                "accessLevel": "Read"
            },
            "ListPlatformApplications": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_ListPlatformApplications.html",
                "description": "Lists the platform application objects for the supported push notification services, such as APNS and GCM.",
                "accessLevel": "List"
            },
            "ListSubscriptions": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_ListSubscriptions.html",
                "description": "Returns a list of the requester's subscriptions.",
                "accessLevel": "List"
            },
            "ListSubscriptionsByTopic": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_ListSubscriptionsByTopic.html",
                "description": "Returns a list of the subscriptions to a specific topic.",
                "accessLevel": "List",
                "resourceTypes": {
                    "topic": {
                        "required": true
                    }
                }
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_ListTagsForResource.html",
                "description": "List all tags added to the specified Amazon SNS topic.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "topic": {
                        "required": false
                    }
                }
            },
            "ListTopics": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_ListTopics.html",
                "description": "Returns a list of the requester's topics. Each call returns a limited list of topics, up to 100.",
                "accessLevel": "List"
            },
            "OptInPhoneNumber": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_OptInPhoneNumber.html",
                "description": "Opts in a phone number that is currently opted out, which enables you to resume sending SMS messages to the number.",
                "accessLevel": "Write"
            },
            "Publish": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_Publish.html",
                "description": "Sends a message to all of a topic's subscribed endpoints.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "topic": {
                        "required": true
                    }
                }
            },
            "RemovePermission": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_RemovePermission.html",
                "description": "Removes a statement from a topic's access control policy.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "topic": {
                        "required": true
                    }
                }
            },
            "SetEndpointAttributes": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_SetEndpointAttributes.html",
                "description": "Sets the attributes for an endpoint for a device on one of the supported push notification services, such as GCM and APNS.",
                "accessLevel": "Write"
            },
            "SetPlatformApplicationAttributes": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_SetPlatformApplicationAttributes.html",
                "description": "Sets the attributes of the platform application object for the supported push notification services, such as APNS and GCM.",
                "accessLevel": "Write"
            },
            "SetSMSAttributes": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_SetSMSAttributes.html",
                "description": "Set the default settings for sending SMS messages and receiving daily SMS usage reports.",
                "accessLevel": "Write"
            },
            "SetSubscriptionAttributes": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_SetSubscriptionAttributes.html",
                "description": "Allows a subscription owner to set an attribute of the topic to a new value.",
                "accessLevel": "Write"
            },
            "SetTopicAttributes": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_SetTopicAttributes.html",
                "description": "Allows a topic owner to set an attribute of the topic to a new value.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "topic": {
                        "required": true
                    }
                }
            },
            "Subscribe": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_Subscribe.html",
                "description": "Prepares to subscribe an endpoint by sending the endpoint a confirmation message.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "topic": {
                        "required": true
                    }
                },
                "conditions": [
                    "sns:Endpoint",
                    "sns:Protocol"
                ]
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_TagResource.html",
                "description": "Add tags to the specified Amazon SNS topic.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "topic": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "Unsubscribe": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_Unsubscribe.html",
                "description": "Deletes a subscription. If the subscription requires authentication for deletion, only the owner of the subscription or the topic's owner can unsubscribe, and an AWS signature is required.",
                "accessLevel": "Write"
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/sns/latest/api/API_UntagResource.html",
                "description": "Remove tags from the specified Amazon SNS topic.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "topic": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            }
        };
        this.resourceTypes = {
            "topic": {
                "name": "topic",
                "url": "https://docs.aws.amazon.com/sns/latest/dg/CreateTopic.html",
                "arn": "arn:${Partition}:sns:${Region}:${Account}:${TopicName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Adds a statement to a topic's access control policy, granting access for the specified AWS accounts to the specified actions.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_AddPermission.html
     */
    addPermission() {
        this.add('sns:AddPermission');
        return this;
    }
    /**
     * Accepts a phone number and indicates whether the phone holder has opted out of receiving SMS messages from your account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_CheckIfPhoneNumberIsOptedOut.html
     */
    checkIfPhoneNumberIsOptedOut() {
        this.add('sns:CheckIfPhoneNumberIsOptedOut');
        return this;
    }
    /**
     * Verifies an endpoint owner's intent to receive messages by validating the token sent to the endpoint by an earlier Subscribe action.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_ConfirmSubscription.html
     */
    confirmSubscription() {
        this.add('sns:ConfirmSubscription');
        return this;
    }
    /**
     * Creates a platform application object for one of the supported push notification services, such as APNS and GCM, to which devices and mobile apps may register.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_CreatePlatformApplication.html
     */
    createPlatformApplication() {
        this.add('sns:CreatePlatformApplication');
        return this;
    }
    /**
     * Creates an endpoint for a device and mobile app on one of the supported push notification services, such as GCM and APNS.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_CreatePlatformEndpoint.html
     */
    createPlatformEndpoint() {
        this.add('sns:CreatePlatformEndpoint');
        return this;
    }
    /**
     * Creates a topic to which notifications can be published.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_CreateTopic.html
     */
    createTopic() {
        this.add('sns:CreateTopic');
        return this;
    }
    /**
     * Deletes the endpoint for a device and mobile app from Amazon SNS.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_DeleteEndpoint.html
     */
    deleteEndpoint() {
        this.add('sns:DeleteEndpoint');
        return this;
    }
    /**
     * Deletes a platform application object for one of the supported push notification services, such as APNS and GCM.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_DeletePlatformApplication.html
     */
    deletePlatformApplication() {
        this.add('sns:DeletePlatformApplication');
        return this;
    }
    /**
     * Deletes a topic and all its subscriptions.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_DeleteTopic.html
     */
    deleteTopic() {
        this.add('sns:DeleteTopic');
        return this;
    }
    /**
     * Retrieves the endpoint attributes for a device on one of the supported push notification services, such as GCM and APNS.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_GetEndpointAttributes.html
     */
    getEndpointAttributes() {
        this.add('sns:GetEndpointAttributes');
        return this;
    }
    /**
     * Retrieves the attributes of the platform application object for the supported push notification services, such as APNS and GCM.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_GetPlatformApplicationAttributes.html
     */
    getPlatformApplicationAttributes() {
        this.add('sns:GetPlatformApplicationAttributes');
        return this;
    }
    /**
     * Returns the settings for sending SMS messages from your account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_GetSMSAttributes.html
     */
    getSMSAttributes() {
        this.add('sns:GetSMSAttributes');
        return this;
    }
    /**
     * Returns all of the properties of a subscription.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_GetSubscriptionAttributes.html
     */
    getSubscriptionAttributes() {
        this.add('sns:GetSubscriptionAttributes');
        return this;
    }
    /**
     * Returns all of the properties of a topic. Topic properties returned might differ based on the authorization of the user.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_GetTopicAttributes.html
     */
    getTopicAttributes() {
        this.add('sns:GetTopicAttributes');
        return this;
    }
    /**
     * Lists the endpoints and endpoint attributes for devices in a supported push notification service, such as GCM and APNS.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_ListEndpointsByPlatformApplication.html
     */
    listEndpointsByPlatformApplication() {
        this.add('sns:ListEndpointsByPlatformApplication');
        return this;
    }
    /**
     * Returns a list of phone numbers that are opted out, meaning you cannot send SMS messages to them.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_ListPhoneNumbersOptedOut.html
     */
    listPhoneNumbersOptedOut() {
        this.add('sns:ListPhoneNumbersOptedOut');
        return this;
    }
    /**
     * Lists the platform application objects for the supported push notification services, such as APNS and GCM.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_ListPlatformApplications.html
     */
    listPlatformApplications() {
        this.add('sns:ListPlatformApplications');
        return this;
    }
    /**
     * Returns a list of the requester's subscriptions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_ListSubscriptions.html
     */
    listSubscriptions() {
        this.add('sns:ListSubscriptions');
        return this;
    }
    /**
     * Returns a list of the subscriptions to a specific topic.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_ListSubscriptionsByTopic.html
     */
    listSubscriptionsByTopic() {
        this.add('sns:ListSubscriptionsByTopic');
        return this;
    }
    /**
     * List all tags added to the specified Amazon SNS topic.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('sns:ListTagsForResource');
        return this;
    }
    /**
     * Returns a list of the requester's topics. Each call returns a limited list of topics, up to 100.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_ListTopics.html
     */
    listTopics() {
        this.add('sns:ListTopics');
        return this;
    }
    /**
     * Opts in a phone number that is currently opted out, which enables you to resume sending SMS messages to the number.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_OptInPhoneNumber.html
     */
    optInPhoneNumber() {
        this.add('sns:OptInPhoneNumber');
        return this;
    }
    /**
     * Sends a message to all of a topic's subscribed endpoints.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_Publish.html
     */
    publish() {
        this.add('sns:Publish');
        return this;
    }
    /**
     * Removes a statement from a topic's access control policy.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_RemovePermission.html
     */
    removePermission() {
        this.add('sns:RemovePermission');
        return this;
    }
    /**
     * Sets the attributes for an endpoint for a device on one of the supported push notification services, such as GCM and APNS.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_SetEndpointAttributes.html
     */
    setEndpointAttributes() {
        this.add('sns:SetEndpointAttributes');
        return this;
    }
    /**
     * Sets the attributes of the platform application object for the supported push notification services, such as APNS and GCM.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_SetPlatformApplicationAttributes.html
     */
    setPlatformApplicationAttributes() {
        this.add('sns:SetPlatformApplicationAttributes');
        return this;
    }
    /**
     * Set the default settings for sending SMS messages and receiving daily SMS usage reports.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_SetSMSAttributes.html
     */
    setSMSAttributes() {
        this.add('sns:SetSMSAttributes');
        return this;
    }
    /**
     * Allows a subscription owner to set an attribute of the topic to a new value.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_SetSubscriptionAttributes.html
     */
    setSubscriptionAttributes() {
        this.add('sns:SetSubscriptionAttributes');
        return this;
    }
    /**
     * Allows a topic owner to set an attribute of the topic to a new value.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_SetTopicAttributes.html
     */
    setTopicAttributes() {
        this.add('sns:SetTopicAttributes');
        return this;
    }
    /**
     * Prepares to subscribe an endpoint by sending the endpoint a confirmation message.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_Subscribe.html
     */
    subscribe() {
        this.add('sns:Subscribe');
        return this;
    }
    /**
     * Add tags to the specified Amazon SNS topic.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_TagResource.html
     */
    tagResource() {
        this.add('sns:TagResource');
        return this;
    }
    /**
     * Deletes a subscription. If the subscription requires authentication for deletion, only the owner of the subscription or the topic's owner can unsubscribe, and an AWS signature is required.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_Unsubscribe.html
     */
    unsubscribe() {
        this.add('sns:Unsubscribe');
        return this;
    }
    /**
     * Remove tags from the specified Amazon SNS topic.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_UntagResource.html
     */
    untagResource() {
        this.add('sns:UntagResource');
        return this;
    }
    /**
     * Adds a resource of type topic to the statement
     *
     * https://docs.aws.amazon.com/sns/latest/dg/CreateTopic.html
     *
     * @param topicName - Identifier for the topicName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTopic(topicName, account, region, partition) {
        var arn = 'arn:${Partition}:sns:${Region}:${Account}:${TopicName}';
        arn = arn.replace('${TopicName}', topicName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * The URL, email address, or ARN from a Subscribe request or a previously confirmed subscription.
     *
     * https://docs.aws.amazon.com/sns/latest/dg/UsingIAMwithSNS.html#w2ab1c11c23c19
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEndpoint(value, operator) {
        return this.if(`sns:Endpoint`, value, operator || 'StringLike');
    }
    /**
     * The protocol value from a Subscribe request or a previously confirmed subscription.
     *
     * https://docs.aws.amazon.com/sns/latest/dg/UsingIAMwithSNS.html#w2ab1c11c23c19
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifProtocol(value, operator) {
        return this.if(`sns:Protocol`, value, operator || 'StringLike');
    }
}
exports.Sns = Sns;
//# sourceMappingURL=data:application/json;base64,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