"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service amplify
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsamplify.html
 */
class Amplify extends shared_1.PolicyStatement {
    /**
     * Action provider for service amplify
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsamplify.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'amplify';
        this.actions = {
            "CreateApp": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Creates a new Amplify App.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateBackendEnvironment": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Creates a new backend environment for an Amplify App.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "CreateBranch": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Creates a new Branch for an Amplify App.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateDeployment": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Create a deployment for manual deploy apps. (Apps are not connected to repository)",
                "accessLevel": "Write",
                "resourceTypes": {
                    "branches": {
                        "required": true
                    }
                }
            },
            "CreateDomainAssociation": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Create a new DomainAssociation on an App",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateWebHook": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Create a new webhook on an App.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "branches": {
                        "required": true
                    }
                }
            },
            "DeleteApp": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Delete an existing Amplify App by appId.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "DeleteBackendEnvironment": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Deletes a branch for an Amplify App.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "DeleteBranch": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Deletes a branch for an Amplify App.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "branches": {
                        "required": true
                    }
                }
            },
            "DeleteDomainAssociation": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Deletes a DomainAssociation.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domains": {
                        "required": true
                    }
                }
            },
            "DeleteJob": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Delete a job, for an Amplify branch, part of Amplify App.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "jobs": {
                        "required": true
                    }
                }
            },
            "DeleteWebHook": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Delete a webhook by id.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GenerateAccessLogs": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Generate website access logs for a specific time range via a pre-signed URL.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetApp": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Retrieves an existing Amplify App by appId.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetArtifactUrl": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Retrieves artifact info that corresponds to a artifactId.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetBackendEnvironment": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Retrieves a backend environment for an Amplify App.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetBranch": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Retrieves a branch for an Amplify App.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "branches": {
                        "required": true
                    }
                }
            },
            "GetDomainAssociation": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Retrieves domain info that corresponds to an appId and domainName.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domains": {
                        "required": true
                    }
                }
            },
            "GetJob": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Get a job for a branch, part of an Amplify App.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "jobs": {
                        "required": true
                    }
                }
            },
            "GetWebHook": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Retrieves webhook info that corresponds to a webhookId.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "ListApps": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Lists existing Amplify Apps.",
                "accessLevel": "List"
            },
            "ListArtifacts": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "List artifacts with an app, a branch, a job and an artifact type.",
                "accessLevel": "List",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "ListBackendEnvironments": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Lists backend environments for an Amplify App.",
                "accessLevel": "List",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "ListBranches": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Lists branches for an Amplify App.",
                "accessLevel": "List",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "ListDomainAssociations": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "List domains with an app",
                "accessLevel": "List",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "ListJobs": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "List Jobs for a branch, part of an Amplify App.",
                "accessLevel": "List",
                "resourceTypes": {
                    "branches": {
                        "required": true
                    }
                }
            },
            "ListWebHooks": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "List webhooks on an App.",
                "accessLevel": "List",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "StartDeployment": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Start a deployment for manual deploy apps. (Apps are not connected to repository)",
                "accessLevel": "Write",
                "resourceTypes": {
                    "branches": {
                        "required": true
                    }
                }
            },
            "StartJob": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Starts a new job for a branch, part of an Amplify App.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "jobs": {
                        "required": true
                    }
                }
            },
            "StopJob": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Stop a job that is in progress, for an Amplify branch, part of Amplify App.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "jobs": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "This action tags an AWS Amplify Console resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "apps": {
                        "required": false
                    },
                    "branches": {
                        "required": false
                    },
                    "jobs": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "This action removes a tag from an AWS Amplify Console resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "apps": {
                        "required": false
                    },
                    "branches": {
                        "required": false
                    },
                    "jobs": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateApp": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Updates an existing Amplify App.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "UpdateBranch": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Updates a branch for an Amplify App.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "branches": {
                        "required": true
                    }
                }
            },
            "UpdateDomainAssociation": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Update a DomainAssociation on an App.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domains": {
                        "required": true
                    }
                }
            },
            "UpdateWebHook": {
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "description": "Update a webhook.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "apps": {
                "name": "apps",
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "arn": "arn:${Partition}:amplify:${Region}:${Account}:apps/${AppId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "branches": {
                "name": "branches",
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "arn": "arn:${Partition}:amplify:${Region}:${Account}:apps/${AppId}/branches/${BranchName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "jobs": {
                "name": "jobs",
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "arn": "arn:${Partition}:amplify:${Region}:${Account}:apps/${AppId}/branches/${BranchName}/jobs/${JobId}",
                "conditionKeys": []
            },
            "domains": {
                "name": "domains",
                "url": "https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html",
                "arn": "arn:${Partition}:amplify:${Region}:${Account}:apps/${AppId}/domains/${DomainName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Creates a new Amplify App.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    createApp() {
        this.add('amplify:CreateApp');
        return this;
    }
    /**
     * Creates a new backend environment for an Amplify App.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    createBackendEnvironment() {
        this.add('amplify:CreateBackendEnvironment');
        return this;
    }
    /**
     * Creates a new Branch for an Amplify App.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    createBranch() {
        this.add('amplify:CreateBranch');
        return this;
    }
    /**
     * Create a deployment for manual deploy apps. (Apps are not connected to repository)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    createDeployment() {
        this.add('amplify:CreateDeployment');
        return this;
    }
    /**
     * Create a new DomainAssociation on an App
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    createDomainAssociation() {
        this.add('amplify:CreateDomainAssociation');
        return this;
    }
    /**
     * Create a new webhook on an App.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    createWebHook() {
        this.add('amplify:CreateWebHook');
        return this;
    }
    /**
     * Delete an existing Amplify App by appId.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    deleteApp() {
        this.add('amplify:DeleteApp');
        return this;
    }
    /**
     * Deletes a branch for an Amplify App.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    deleteBackendEnvironment() {
        this.add('amplify:DeleteBackendEnvironment');
        return this;
    }
    /**
     * Deletes a branch for an Amplify App.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    deleteBranch() {
        this.add('amplify:DeleteBranch');
        return this;
    }
    /**
     * Deletes a DomainAssociation.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    deleteDomainAssociation() {
        this.add('amplify:DeleteDomainAssociation');
        return this;
    }
    /**
     * Delete a job, for an Amplify branch, part of Amplify App.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    deleteJob() {
        this.add('amplify:DeleteJob');
        return this;
    }
    /**
     * Delete a webhook by id.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    deleteWebHook() {
        this.add('amplify:DeleteWebHook');
        return this;
    }
    /**
     * Generate website access logs for a specific time range via a pre-signed URL.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    generateAccessLogs() {
        this.add('amplify:GenerateAccessLogs');
        return this;
    }
    /**
     * Retrieves an existing Amplify App by appId.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    getApp() {
        this.add('amplify:GetApp');
        return this;
    }
    /**
     * Retrieves artifact info that corresponds to a artifactId.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    getArtifactUrl() {
        this.add('amplify:GetArtifactUrl');
        return this;
    }
    /**
     * Retrieves a backend environment for an Amplify App.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    getBackendEnvironment() {
        this.add('amplify:GetBackendEnvironment');
        return this;
    }
    /**
     * Retrieves a branch for an Amplify App.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    getBranch() {
        this.add('amplify:GetBranch');
        return this;
    }
    /**
     * Retrieves domain info that corresponds to an appId and domainName.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    getDomainAssociation() {
        this.add('amplify:GetDomainAssociation');
        return this;
    }
    /**
     * Get a job for a branch, part of an Amplify App.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    getJob() {
        this.add('amplify:GetJob');
        return this;
    }
    /**
     * Retrieves webhook info that corresponds to a webhookId.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    getWebHook() {
        this.add('amplify:GetWebHook');
        return this;
    }
    /**
     * Lists existing Amplify Apps.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    listApps() {
        this.add('amplify:ListApps');
        return this;
    }
    /**
     * List artifacts with an app, a branch, a job and an artifact type.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    listArtifacts() {
        this.add('amplify:ListArtifacts');
        return this;
    }
    /**
     * Lists backend environments for an Amplify App.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    listBackendEnvironments() {
        this.add('amplify:ListBackendEnvironments');
        return this;
    }
    /**
     * Lists branches for an Amplify App.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    listBranches() {
        this.add('amplify:ListBranches');
        return this;
    }
    /**
     * List domains with an app
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    listDomainAssociations() {
        this.add('amplify:ListDomainAssociations');
        return this;
    }
    /**
     * List Jobs for a branch, part of an Amplify App.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    listJobs() {
        this.add('amplify:ListJobs');
        return this;
    }
    /**
     * List webhooks on an App.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    listWebHooks() {
        this.add('amplify:ListWebHooks');
        return this;
    }
    /**
     * Start a deployment for manual deploy apps. (Apps are not connected to repository)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    startDeployment() {
        this.add('amplify:StartDeployment');
        return this;
    }
    /**
     * Starts a new job for a branch, part of an Amplify App.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    startJob() {
        this.add('amplify:StartJob');
        return this;
    }
    /**
     * Stop a job that is in progress, for an Amplify branch, part of Amplify App.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    stopJob() {
        this.add('amplify:StopJob');
        return this;
    }
    /**
     * This action tags an AWS Amplify Console resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    tagResource() {
        this.add('amplify:TagResource');
        return this;
    }
    /**
     * This action removes a tag from an AWS Amplify Console resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    untagResource() {
        this.add('amplify:UntagResource');
        return this;
    }
    /**
     * Updates an existing Amplify App.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    updateApp() {
        this.add('amplify:UpdateApp');
        return this;
    }
    /**
     * Updates a branch for an Amplify App.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    updateBranch() {
        this.add('amplify:UpdateBranch');
        return this;
    }
    /**
     * Update a DomainAssociation on an App.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    updateDomainAssociation() {
        this.add('amplify:UpdateDomainAssociation');
        return this;
    }
    /**
     * Update a webhook.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     */
    updateWebHook() {
        this.add('amplify:UpdateWebHook');
        return this;
    }
    /**
     * Adds a resource of type apps to the statement
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     *
     * @param appId - Identifier for the appId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onApps(appId, account, region, partition) {
        var arn = 'arn:${Partition}:amplify:${Region}:${Account}:apps/${AppId}';
        arn = arn.replace('${AppId}', appId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type branches to the statement
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     *
     * @param appId - Identifier for the appId.
     * @param branchName - Identifier for the branchName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onBranches(appId, branchName, account, region, partition) {
        var arn = 'arn:${Partition}:amplify:${Region}:${Account}:apps/${AppId}/branches/${BranchName}';
        arn = arn.replace('${AppId}', appId);
        arn = arn.replace('${BranchName}', branchName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type jobs to the statement
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     *
     * @param appId - Identifier for the appId.
     * @param branchName - Identifier for the branchName.
     * @param jobId - Identifier for the jobId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onJobs(appId, branchName, jobId, account, region, partition) {
        var arn = 'arn:${Partition}:amplify:${Region}:${Account}:apps/${AppId}/branches/${BranchName}/jobs/${JobId}';
        arn = arn.replace('${AppId}', appId);
        arn = arn.replace('${BranchName}', branchName);
        arn = arn.replace('${JobId}', jobId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type domains to the statement
     *
     * https://docs.aws.amazon.com/amplify/latest/userguide/welcome.html
     *
     * @param appId - Identifier for the appId.
     * @param domainName - Identifier for the domainName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onDomains(appId, domainName, account, region, partition) {
        var arn = 'arn:${Partition}:amplify:${Region}:${Account}:apps/${AppId}/domains/${DomainName}';
        arn = arn.replace('${AppId}', appId);
        arn = arn.replace('${DomainName}', domainName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Amplify = Amplify;
//# sourceMappingURL=data:application/json;base64,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