"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service appsync
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsappsync.html
 */
class Appsync extends shared_1.PolicyStatement {
    /**
     * Action provider for service appsync
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsappsync.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'appsync';
        this.actions = {
            "CreateApiKey": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateApiKey.html",
                "description": "Creates a unique key that you can distribute to clients who are executing your API.",
                "accessLevel": "Write"
            },
            "CreateDataSource": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateDataSource.html",
                "description": "Creates a DataSource object.",
                "accessLevel": "Write"
            },
            "CreateFunction": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateFunction.html",
                "description": "Create a new Function object.",
                "accessLevel": "Write"
            },
            "CreateGraphqlApi": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateGraphqlApi.html",
                "description": "Creates a GraphqlApi object, which is the top level AppSync resource.",
                "accessLevel": "Tagging",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateResolver": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateResolver.html",
                "description": "Creates a Resolver object. A resolver converts incoming requests into a format that a data source can understand, and converts the data source's responses into GraphQL.",
                "accessLevel": "Write"
            },
            "CreateType": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateType.html",
                "description": "Creates a Type object.",
                "accessLevel": "Write"
            },
            "DeleteApiKey": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteApiKey.html",
                "description": "Deletes an API key.",
                "accessLevel": "Write"
            },
            "DeleteDataSource": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteDataSource.html",
                "description": "Deletes a DataSource object.",
                "accessLevel": "Write"
            },
            "DeleteFunction": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteFunction.html",
                "description": "Deletes a Function object.",
                "accessLevel": "Write"
            },
            "DeleteGraphqlApi": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteGraphqlApi.html",
                "description": "Deletes a GraphqlApi object. This will also clean up every AppSync resource below that API.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "graphqlapi": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "DeleteResolver": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteResolver.html",
                "description": "Deletes a Resolver object.",
                "accessLevel": "Write"
            },
            "DeleteType": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteType.html",
                "description": "Deletes a Type object.",
                "accessLevel": "Write"
            },
            "GetDataSource": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetDataSource.html",
                "description": "Retrieves a DataSource object.",
                "accessLevel": "Read"
            },
            "GetFunction": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetFunction.html",
                "description": "Retrieves a Function object.",
                "accessLevel": "Read"
            },
            "GetGraphqlApi": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetGraphqlApi.html",
                "description": "Retrieves a GraphqlApi object.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "graphqlapi": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "GetIntrospectionSchema": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetIntrospectionSchema.html",
                "description": "Retrieves the introspection schema for a GraphQL API.",
                "accessLevel": "Read"
            },
            "GetResolver": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetResolver.html",
                "description": "Retrieves a Resolver object.",
                "accessLevel": "Read"
            },
            "GetSchemaCreationStatus": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetSchemaCreationStatus.html",
                "description": "Retrieves the current status of a schema creation operation.",
                "accessLevel": "Read"
            },
            "GetType": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetType.html",
                "description": "Retrieves a Type object.",
                "accessLevel": "Read"
            },
            "GraphQL": {
                "url": "https://docs.aws.amazon.com/appsync/latest/devguide/using-your-api.html",
                "description": "Sends a GraphQL query to a GraphQL API.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "field": {
                        "required": true
                    },
                    "graphqlapi": {
                        "required": true
                    }
                }
            },
            "ListApiKeys": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListApiKeys.html",
                "description": "Lists the API keys for a given API.",
                "accessLevel": "List"
            },
            "ListDataSources": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListDataSources.html",
                "description": "Lists the data sources for a given API.",
                "accessLevel": "List"
            },
            "ListFunctions": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListFunctions.html",
                "description": "Lists the functions for a given API.",
                "accessLevel": "List"
            },
            "ListGraphqlApis": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListGraphqlApis.html",
                "description": "Lists your GraphQL APIs.",
                "accessLevel": "List"
            },
            "ListResolvers": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListResolvers.html",
                "description": "Lists the resolvers for a given API and type.",
                "accessLevel": "List"
            },
            "ListResolversByFunction": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListResolversByFunction.html",
                "description": "List the resolvers that are associated with a specific function.",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListTagsForResource.html",
                "description": "List the tags for a resource.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "graphqlapi": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "ListTypes": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListTypes.html",
                "description": "Lists the types for a given API.",
                "accessLevel": "List"
            },
            "StartSchemaCreation": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_StartSchemaCreation.html",
                "description": "Adds a new schema to your GraphQL API. This operation is asynchronous - GetSchemaCreationStatus can show when it has completed.",
                "accessLevel": "Write"
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_TagResource.html",
                "description": "Tag a resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "graphqlapi": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_UntagResource.html",
                "description": "Untag a resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "graphqlapi": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateApiKey": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateApiKey.html",
                "description": "Updates an API key for a given API.",
                "accessLevel": "Write"
            },
            "UpdateDataSource": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateDataSource.html",
                "description": "Updates a DataSource object.",
                "accessLevel": "Write"
            },
            "UpdateFunction": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateFunction.html",
                "description": "Updates an existing Function object.",
                "accessLevel": "Write"
            },
            "UpdateGraphqlApi": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateGraphqlApi.html",
                "description": "Updates a GraphqlApi object.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "graphqlapi": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "UpdateResolver": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateResolver.html",
                "description": "Updates a Resolver object.",
                "accessLevel": "Write"
            },
            "UpdateType": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateType.html",
                "description": "Updates a Type object.",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "datasource": {
                "name": "datasource",
                "url": "https://docs.aws.amazon.com/appsync/latest/devguide/attaching-a-data-source.html",
                "arn": "arn:${Partition}:appsync:${Region}:${Account}:apis/${GraphQLAPIId}/datasources/${DatasourceName}",
                "conditionKeys": []
            },
            "graphqlapi": {
                "name": "graphqlapi",
                "url": "https://docs.aws.amazon.com/appsync/latest/devguide/designing-a-graphql-api.html",
                "arn": "arn:${Partition}:appsync:${Region}:${Account}:apis/${GraphQLAPIId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "field": {
                "name": "field",
                "url": "https://docs.aws.amazon.com/appsync/latest/devguide/configuring-resolvers.html",
                "arn": "arn:${Partition}:appsync:${Region}:${Account}:apis/${GraphQLAPIId}/types/${TypeName}/fields/${FieldName}",
                "conditionKeys": []
            },
            "type": {
                "name": "type",
                "url": "https://docs.aws.amazon.com/appsync/latest/devguide/designing-your-schema.html#adding-a-root-query-type",
                "arn": "arn:${Partition}:appsync:${Region}:${Account}:apis/${GraphQLAPIId}/types/${TypeName}",
                "conditionKeys": []
            },
            "function": {
                "name": "function",
                "url": "https://docs.aws.amazon.com/appsync/latest/devguide/pipeline-resolvers.html",
                "arn": "arn:${Partition}:appsync:${Region}:${Account}:apis/${GraphQLAPIId}/functions/${FunctionId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Creates a unique key that you can distribute to clients who are executing your API.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateApiKey.html
     */
    createApiKey() {
        this.add('appsync:CreateApiKey');
        return this;
    }
    /**
     * Creates a DataSource object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateDataSource.html
     */
    createDataSource() {
        this.add('appsync:CreateDataSource');
        return this;
    }
    /**
     * Create a new Function object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateFunction.html
     */
    createFunction() {
        this.add('appsync:CreateFunction');
        return this;
    }
    /**
     * Creates a GraphqlApi object, which is the top level AppSync resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateGraphqlApi.html
     */
    createGraphqlApi() {
        this.add('appsync:CreateGraphqlApi');
        return this;
    }
    /**
     * Creates a Resolver object. A resolver converts incoming requests into a format that a data source can understand, and converts the data source's responses into GraphQL.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateResolver.html
     */
    createResolver() {
        this.add('appsync:CreateResolver');
        return this;
    }
    /**
     * Creates a Type object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateType.html
     */
    createType() {
        this.add('appsync:CreateType');
        return this;
    }
    /**
     * Deletes an API key.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteApiKey.html
     */
    deleteApiKey() {
        this.add('appsync:DeleteApiKey');
        return this;
    }
    /**
     * Deletes a DataSource object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteDataSource.html
     */
    deleteDataSource() {
        this.add('appsync:DeleteDataSource');
        return this;
    }
    /**
     * Deletes a Function object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteFunction.html
     */
    deleteFunction() {
        this.add('appsync:DeleteFunction');
        return this;
    }
    /**
     * Deletes a GraphqlApi object. This will also clean up every AppSync resource below that API.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteGraphqlApi.html
     */
    deleteGraphqlApi() {
        this.add('appsync:DeleteGraphqlApi');
        return this;
    }
    /**
     * Deletes a Resolver object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteResolver.html
     */
    deleteResolver() {
        this.add('appsync:DeleteResolver');
        return this;
    }
    /**
     * Deletes a Type object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteType.html
     */
    deleteType() {
        this.add('appsync:DeleteType');
        return this;
    }
    /**
     * Retrieves a DataSource object.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetDataSource.html
     */
    getDataSource() {
        this.add('appsync:GetDataSource');
        return this;
    }
    /**
     * Retrieves a Function object.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetFunction.html
     */
    getFunction() {
        this.add('appsync:GetFunction');
        return this;
    }
    /**
     * Retrieves a GraphqlApi object.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetGraphqlApi.html
     */
    getGraphqlApi() {
        this.add('appsync:GetGraphqlApi');
        return this;
    }
    /**
     * Retrieves the introspection schema for a GraphQL API.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetIntrospectionSchema.html
     */
    getIntrospectionSchema() {
        this.add('appsync:GetIntrospectionSchema');
        return this;
    }
    /**
     * Retrieves a Resolver object.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetResolver.html
     */
    getResolver() {
        this.add('appsync:GetResolver');
        return this;
    }
    /**
     * Retrieves the current status of a schema creation operation.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetSchemaCreationStatus.html
     */
    getSchemaCreationStatus() {
        this.add('appsync:GetSchemaCreationStatus');
        return this;
    }
    /**
     * Retrieves a Type object.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetType.html
     */
    getType() {
        this.add('appsync:GetType');
        return this;
    }
    /**
     * Sends a GraphQL query to a GraphQL API.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/devguide/using-your-api.html
     */
    graphQL() {
        this.add('appsync:GraphQL');
        return this;
    }
    /**
     * Lists the API keys for a given API.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListApiKeys.html
     */
    listApiKeys() {
        this.add('appsync:ListApiKeys');
        return this;
    }
    /**
     * Lists the data sources for a given API.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListDataSources.html
     */
    listDataSources() {
        this.add('appsync:ListDataSources');
        return this;
    }
    /**
     * Lists the functions for a given API.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListFunctions.html
     */
    listFunctions() {
        this.add('appsync:ListFunctions');
        return this;
    }
    /**
     * Lists your GraphQL APIs.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListGraphqlApis.html
     */
    listGraphqlApis() {
        this.add('appsync:ListGraphqlApis');
        return this;
    }
    /**
     * Lists the resolvers for a given API and type.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListResolvers.html
     */
    listResolvers() {
        this.add('appsync:ListResolvers');
        return this;
    }
    /**
     * List the resolvers that are associated with a specific function.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListResolversByFunction.html
     */
    listResolversByFunction() {
        this.add('appsync:ListResolversByFunction');
        return this;
    }
    /**
     * List the tags for a resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('appsync:ListTagsForResource');
        return this;
    }
    /**
     * Lists the types for a given API.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListTypes.html
     */
    listTypes() {
        this.add('appsync:ListTypes');
        return this;
    }
    /**
     * Adds a new schema to your GraphQL API. This operation is asynchronous - GetSchemaCreationStatus can show when it has completed.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_StartSchemaCreation.html
     */
    startSchemaCreation() {
        this.add('appsync:StartSchemaCreation');
        return this;
    }
    /**
     * Tag a resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_TagResource.html
     */
    tagResource() {
        this.add('appsync:TagResource');
        return this;
    }
    /**
     * Untag a resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_UntagResource.html
     */
    untagResource() {
        this.add('appsync:UntagResource');
        return this;
    }
    /**
     * Updates an API key for a given API.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateApiKey.html
     */
    updateApiKey() {
        this.add('appsync:UpdateApiKey');
        return this;
    }
    /**
     * Updates a DataSource object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateDataSource.html
     */
    updateDataSource() {
        this.add('appsync:UpdateDataSource');
        return this;
    }
    /**
     * Updates an existing Function object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateFunction.html
     */
    updateFunction() {
        this.add('appsync:UpdateFunction');
        return this;
    }
    /**
     * Updates a GraphqlApi object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateGraphqlApi.html
     */
    updateGraphqlApi() {
        this.add('appsync:UpdateGraphqlApi');
        return this;
    }
    /**
     * Updates a Resolver object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateResolver.html
     */
    updateResolver() {
        this.add('appsync:UpdateResolver');
        return this;
    }
    /**
     * Updates a Type object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateType.html
     */
    updateType() {
        this.add('appsync:UpdateType');
        return this;
    }
    /**
     * Adds a resource of type datasource to the statement
     *
     * https://docs.aws.amazon.com/appsync/latest/devguide/attaching-a-data-source.html
     *
     * @param graphQLAPIId - Identifier for the graphQLAPIId.
     * @param datasourceName - Identifier for the datasourceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDatasource(graphQLAPIId, datasourceName, account, region, partition) {
        var arn = 'arn:${Partition}:appsync:${Region}:${Account}:apis/${GraphQLAPIId}/datasources/${DatasourceName}';
        arn = arn.replace('${GraphQLAPIId}', graphQLAPIId);
        arn = arn.replace('${DatasourceName}', datasourceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type graphqlapi to the statement
     *
     * https://docs.aws.amazon.com/appsync/latest/devguide/designing-a-graphql-api.html
     *
     * @param graphQLAPIId - Identifier for the graphQLAPIId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onGraphqlapi(graphQLAPIId, account, region, partition) {
        var arn = 'arn:${Partition}:appsync:${Region}:${Account}:apis/${GraphQLAPIId}';
        arn = arn.replace('${GraphQLAPIId}', graphQLAPIId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type field to the statement
     *
     * https://docs.aws.amazon.com/appsync/latest/devguide/configuring-resolvers.html
     *
     * @param graphQLAPIId - Identifier for the graphQLAPIId.
     * @param typeName - Identifier for the typeName.
     * @param fieldName - Identifier for the fieldName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onField(graphQLAPIId, typeName, fieldName, account, region, partition) {
        var arn = 'arn:${Partition}:appsync:${Region}:${Account}:apis/${GraphQLAPIId}/types/${TypeName}/fields/${FieldName}';
        arn = arn.replace('${GraphQLAPIId}', graphQLAPIId);
        arn = arn.replace('${TypeName}', typeName);
        arn = arn.replace('${FieldName}', fieldName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type type to the statement
     *
     * https://docs.aws.amazon.com/appsync/latest/devguide/designing-your-schema.html#adding-a-root-query-type
     *
     * @param graphQLAPIId - Identifier for the graphQLAPIId.
     * @param typeName - Identifier for the typeName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onType(graphQLAPIId, typeName, account, region, partition) {
        var arn = 'arn:${Partition}:appsync:${Region}:${Account}:apis/${GraphQLAPIId}/types/${TypeName}';
        arn = arn.replace('${GraphQLAPIId}', graphQLAPIId);
        arn = arn.replace('${TypeName}', typeName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type function to the statement
     *
     * https://docs.aws.amazon.com/appsync/latest/devguide/pipeline-resolvers.html
     *
     * @param graphQLAPIId - Identifier for the graphQLAPIId.
     * @param functionId - Identifier for the functionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onFunction(graphQLAPIId, functionId, account, region, partition) {
        var arn = 'arn:${Partition}:appsync:${Region}:${Account}:apis/${GraphQLAPIId}/functions/${FunctionId}';
        arn = arn.replace('${GraphQLAPIId}', graphQLAPIId);
        arn = arn.replace('${FunctionId}', functionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Appsync = Appsync;
//# sourceMappingURL=data:application/json;base64,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