"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service backup
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsbackup.html
 */
class Backup extends shared_1.PolicyStatement {
    /**
     * Action provider for service backup
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsbackup.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'backup';
        this.actions = {
            "CopyIntoBackupVault": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_CopyIntoBackupVault.html",
                "description": "Copy into a backup vault",
                "accessLevel": "Write"
            },
            "CreateBackupPlan": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_CreateBackupPlan.html",
                "description": "Creates a new backup plan",
                "accessLevel": "Write",
                "resourceTypes": {
                    "backupPlan": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateBackupSelection": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_CreateBackupSelection.html",
                "description": "Creates a new resource assignment in a backup plan.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "backupPlan": {
                        "required": true
                    }
                }
            },
            "CreateBackupVault": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_CreateBackupVault.html",
                "description": "Creates a new backup vault.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "backupVault": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteBackupPlan": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteBackupPlan.html",
                "description": "Deletes a backup plan.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "backupPlan": {
                        "required": true
                    }
                }
            },
            "DeleteBackupSelection": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteBackupSelection.html",
                "description": "Deletes a resource assignment from a backup plan.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "backupPlan": {
                        "required": true
                    }
                }
            },
            "DeleteBackupVault": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteBackupVault.html",
                "description": "Deletes a backup vault.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "backupVault": {
                        "required": true
                    }
                }
            },
            "DeleteBackupVaultAccessPolicy": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteBackupVaultAccessPolicy.html",
                "description": "Deletes backup vault access policy.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "backupVault": {
                        "required": true
                    }
                }
            },
            "DeleteBackupVaultNotifications": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteBackupVaultNotifications.html",
                "description": "Remove notifications from backup vault.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "backupVault": {
                        "required": true
                    }
                }
            },
            "DeleteRecoveryPoint": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteRecoveryPoint.html",
                "description": "Deletes a recovery point from a backup vault.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "recoveryPoint": {
                        "required": true
                    }
                }
            },
            "DescribeBackupJob": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeBackupJob.html",
                "description": "Describes a backup job",
                "accessLevel": "Read"
            },
            "DescribeBackupVault": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeBackupVault.html",
                "description": "Creates a new backup vault with the specified name.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "backupVault": {
                        "required": true
                    }
                }
            },
            "DescribeCopyJob": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeCopyJob.html",
                "description": "Describes a copy job",
                "accessLevel": "Read",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DescribeProtectedResource": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeProtectedResource.html",
                "description": "Describes a protected resource.",
                "accessLevel": "Read"
            },
            "DescribeRecoveryPoint": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeRecoveryPoint.html",
                "description": "Describes a recovery point.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "recoveryPoint": {
                        "required": true
                    }
                }
            },
            "DescribeRestoreJob": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeRestoreJob.html",
                "description": "Describes a restore job.",
                "accessLevel": "Read"
            },
            "ExportBackupPlanTemplate": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ExportBackupPlanTemplate.html",
                "description": "Exports a backup plan as a JSON.",
                "accessLevel": "Read"
            },
            "GetBackupPlan": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupPlan.html",
                "description": "Gets a backup plan.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "backupPlan": {
                        "required": true
                    }
                }
            },
            "GetBackupPlanFromJSON": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupPlanFromJSON.html",
                "description": "Transforms a JSON to a backup plan.",
                "accessLevel": "Read"
            },
            "GetBackupPlanFromTemplate": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupPlanFromTemplate.html",
                "description": "Transforms a template to a backup plan.",
                "accessLevel": "Read"
            },
            "GetBackupSelection": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupSelection.html",
                "description": "Gets a backup plan resource assignment.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "backupPlan": {
                        "required": true
                    }
                }
            },
            "GetBackupVaultAccessPolicy": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupVaultAccessPolicy.html",
                "description": "Gets backup vault access policy.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "backupVault": {
                        "required": true
                    }
                }
            },
            "GetBackupVaultNotifications": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupVaultNotifications.html",
                "description": "Gets backup vault notifications.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "backupVault": {
                        "required": true
                    }
                }
            },
            "GetRecoveryPointRestoreMetadata": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetRecoveryPointRestoreMetadata.html",
                "description": "Gets recovery point restore metadata.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "recoveryPoint": {
                        "required": true
                    }
                }
            },
            "GetSupportedResourceTypes": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetSupportedResourceTypes.html",
                "description": "Gets supported resource types.",
                "accessLevel": "Read"
            },
            "ListBackupJobs": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupJobs.html",
                "description": "Lists backup jobs.",
                "accessLevel": "List"
            },
            "ListBackupPlanTemplates": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupPlanTemplates.html",
                "description": "Lists backup plan templates provided by AWS Backup.",
                "accessLevel": "List"
            },
            "ListBackupPlanVersions": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupPlanVersions.html",
                "description": "Lists backup plan versions.",
                "accessLevel": "List",
                "resourceTypes": {
                    "backupPlan": {
                        "required": true
                    }
                }
            },
            "ListBackupPlans": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupPlans.html",
                "description": "Lists backup plans.",
                "accessLevel": "List"
            },
            "ListBackupSelections": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupSelections.html",
                "description": "Lists resource assignments for a specific backup plan.",
                "accessLevel": "List",
                "resourceTypes": {
                    "backupPlan": {
                        "required": true
                    }
                }
            },
            "ListBackupVaults": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupVaults.html",
                "description": "Lists backup vaults.",
                "accessLevel": "List"
            },
            "ListCopyJobs": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListCopyJobs.html",
                "description": "List copy jobs",
                "accessLevel": "List"
            },
            "ListProtectedResources": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListProtectedResources.html",
                "description": "Lists protected resources by AWS Backup.",
                "accessLevel": "List"
            },
            "ListRecoveryPointsByBackupVault": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListRecoveryPointsByBackupVault.html",
                "description": "Lists recovery points inside a backup vault.",
                "accessLevel": "List",
                "resourceTypes": {
                    "backupVault": {
                        "required": true
                    }
                }
            },
            "ListRecoveryPointsByResource": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListRecoveryPointsByResource.html",
                "description": "Lists recovery points for a resource.",
                "accessLevel": "List"
            },
            "ListRestoreJobs": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListRestoreJobs.html",
                "description": "Lists restore jobs.",
                "accessLevel": "List"
            },
            "ListTags": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListTags.html",
                "description": "Lists tags for a resource.",
                "accessLevel": "List"
            },
            "PutBackupVaultAccessPolicy": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_PutBackupVaultAccessPolicy.html",
                "description": "Adds an access policy to the backup vault.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "backupVault": {
                        "required": true
                    }
                }
            },
            "PutBackupVaultNotifications": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_PutBackupVaultNotifications.html",
                "description": "Adds an SNS topic to the backup vault.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "backupVault": {
                        "required": true
                    }
                }
            },
            "StartBackupJob": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_StartBackupJob.html",
                "description": "Starts a new backup job.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "backupVault": {
                        "required": true
                    }
                }
            },
            "StartCopyJob": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_StartCopyJob.html",
                "description": "Copy a backup from a source region to a destination region.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "recoveryPoint": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "StartRestoreJob": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_StartRestoreJob.html",
                "description": "Starts a new restore job.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "recoveryPoint": {
                        "required": true
                    }
                }
            },
            "StopBackupJob": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_StopBackupJob.html",
                "description": "Stops a backup job.",
                "accessLevel": "Write"
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_TagResource.html",
                "description": "Tags a resource.",
                "accessLevel": "Tagging",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_UntagResource.html",
                "description": "Untags a resource.",
                "accessLevel": "Tagging",
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateBackupPlan": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_UpdateBackupPlan.html",
                "description": "Updates a backup plan.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "backupPlan": {
                        "required": true
                    }
                }
            },
            "UpdateRecoveryPointLifecycle": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_UpdateRecoveryPointLifecycle.html",
                "description": "Updates the lifecycle of the recovery point.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "recoveryPoint": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "backupVault": {
                "name": "backupVault",
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/vaults.html",
                "arn": "arn:${Partition}:backup:${Region}:${Account}:backup-vault:${BackupVaultName}",
                "conditionKeys": []
            },
            "backupPlan": {
                "name": "backupPlan",
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/about-backup-plans.html",
                "arn": "arn:${Partition}:backup:${Region}:${Account}:backup-plan:${BackupPlanId}",
                "conditionKeys": []
            },
            "recoveryPoint": {
                "name": "recoveryPoint",
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/recovery-points.html",
                "arn": "arn:${Partition}:backup:${Region}:${Account}:recovery-point:${RecoveryPointId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Copy into a backup vault
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_CopyIntoBackupVault.html
     */
    copyIntoBackupVault() {
        this.add('backup:CopyIntoBackupVault');
        return this;
    }
    /**
     * Creates a new backup plan
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_CreateBackupPlan.html
     */
    createBackupPlan() {
        this.add('backup:CreateBackupPlan');
        return this;
    }
    /**
     * Creates a new resource assignment in a backup plan.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_CreateBackupSelection.html
     */
    createBackupSelection() {
        this.add('backup:CreateBackupSelection');
        return this;
    }
    /**
     * Creates a new backup vault.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_CreateBackupVault.html
     */
    createBackupVault() {
        this.add('backup:CreateBackupVault');
        return this;
    }
    /**
     * Deletes a backup plan.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteBackupPlan.html
     */
    deleteBackupPlan() {
        this.add('backup:DeleteBackupPlan');
        return this;
    }
    /**
     * Deletes a resource assignment from a backup plan.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteBackupSelection.html
     */
    deleteBackupSelection() {
        this.add('backup:DeleteBackupSelection');
        return this;
    }
    /**
     * Deletes a backup vault.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteBackupVault.html
     */
    deleteBackupVault() {
        this.add('backup:DeleteBackupVault');
        return this;
    }
    /**
     * Deletes backup vault access policy.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteBackupVaultAccessPolicy.html
     */
    deleteBackupVaultAccessPolicy() {
        this.add('backup:DeleteBackupVaultAccessPolicy');
        return this;
    }
    /**
     * Remove notifications from backup vault.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteBackupVaultNotifications.html
     */
    deleteBackupVaultNotifications() {
        this.add('backup:DeleteBackupVaultNotifications');
        return this;
    }
    /**
     * Deletes a recovery point from a backup vault.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteRecoveryPoint.html
     */
    deleteRecoveryPoint() {
        this.add('backup:DeleteRecoveryPoint');
        return this;
    }
    /**
     * Describes a backup job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeBackupJob.html
     */
    describeBackupJob() {
        this.add('backup:DescribeBackupJob');
        return this;
    }
    /**
     * Creates a new backup vault with the specified name.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeBackupVault.html
     */
    describeBackupVault() {
        this.add('backup:DescribeBackupVault');
        return this;
    }
    /**
     * Describes a copy job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeCopyJob.html
     */
    describeCopyJob() {
        this.add('backup:DescribeCopyJob');
        return this;
    }
    /**
     * Describes a protected resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeProtectedResource.html
     */
    describeProtectedResource() {
        this.add('backup:DescribeProtectedResource');
        return this;
    }
    /**
     * Describes a recovery point.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeRecoveryPoint.html
     */
    describeRecoveryPoint() {
        this.add('backup:DescribeRecoveryPoint');
        return this;
    }
    /**
     * Describes a restore job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeRestoreJob.html
     */
    describeRestoreJob() {
        this.add('backup:DescribeRestoreJob');
        return this;
    }
    /**
     * Exports a backup plan as a JSON.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ExportBackupPlanTemplate.html
     */
    exportBackupPlanTemplate() {
        this.add('backup:ExportBackupPlanTemplate');
        return this;
    }
    /**
     * Gets a backup plan.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupPlan.html
     */
    getBackupPlan() {
        this.add('backup:GetBackupPlan');
        return this;
    }
    /**
     * Transforms a JSON to a backup plan.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupPlanFromJSON.html
     */
    getBackupPlanFromJSON() {
        this.add('backup:GetBackupPlanFromJSON');
        return this;
    }
    /**
     * Transforms a template to a backup plan.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupPlanFromTemplate.html
     */
    getBackupPlanFromTemplate() {
        this.add('backup:GetBackupPlanFromTemplate');
        return this;
    }
    /**
     * Gets a backup plan resource assignment.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupSelection.html
     */
    getBackupSelection() {
        this.add('backup:GetBackupSelection');
        return this;
    }
    /**
     * Gets backup vault access policy.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupVaultAccessPolicy.html
     */
    getBackupVaultAccessPolicy() {
        this.add('backup:GetBackupVaultAccessPolicy');
        return this;
    }
    /**
     * Gets backup vault notifications.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupVaultNotifications.html
     */
    getBackupVaultNotifications() {
        this.add('backup:GetBackupVaultNotifications');
        return this;
    }
    /**
     * Gets recovery point restore metadata.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetRecoveryPointRestoreMetadata.html
     */
    getRecoveryPointRestoreMetadata() {
        this.add('backup:GetRecoveryPointRestoreMetadata');
        return this;
    }
    /**
     * Gets supported resource types.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetSupportedResourceTypes.html
     */
    getSupportedResourceTypes() {
        this.add('backup:GetSupportedResourceTypes');
        return this;
    }
    /**
     * Lists backup jobs.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupJobs.html
     */
    listBackupJobs() {
        this.add('backup:ListBackupJobs');
        return this;
    }
    /**
     * Lists backup plan templates provided by AWS Backup.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupPlanTemplates.html
     */
    listBackupPlanTemplates() {
        this.add('backup:ListBackupPlanTemplates');
        return this;
    }
    /**
     * Lists backup plan versions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupPlanVersions.html
     */
    listBackupPlanVersions() {
        this.add('backup:ListBackupPlanVersions');
        return this;
    }
    /**
     * Lists backup plans.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupPlans.html
     */
    listBackupPlans() {
        this.add('backup:ListBackupPlans');
        return this;
    }
    /**
     * Lists resource assignments for a specific backup plan.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupSelections.html
     */
    listBackupSelections() {
        this.add('backup:ListBackupSelections');
        return this;
    }
    /**
     * Lists backup vaults.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupVaults.html
     */
    listBackupVaults() {
        this.add('backup:ListBackupVaults');
        return this;
    }
    /**
     * List copy jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListCopyJobs.html
     */
    listCopyJobs() {
        this.add('backup:ListCopyJobs');
        return this;
    }
    /**
     * Lists protected resources by AWS Backup.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListProtectedResources.html
     */
    listProtectedResources() {
        this.add('backup:ListProtectedResources');
        return this;
    }
    /**
     * Lists recovery points inside a backup vault.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListRecoveryPointsByBackupVault.html
     */
    listRecoveryPointsByBackupVault() {
        this.add('backup:ListRecoveryPointsByBackupVault');
        return this;
    }
    /**
     * Lists recovery points for a resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListRecoveryPointsByResource.html
     */
    listRecoveryPointsByResource() {
        this.add('backup:ListRecoveryPointsByResource');
        return this;
    }
    /**
     * Lists restore jobs.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListRestoreJobs.html
     */
    listRestoreJobs() {
        this.add('backup:ListRestoreJobs');
        return this;
    }
    /**
     * Lists tags for a resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListTags.html
     */
    listTags() {
        this.add('backup:ListTags');
        return this;
    }
    /**
     * Adds an access policy to the backup vault.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_PutBackupVaultAccessPolicy.html
     */
    putBackupVaultAccessPolicy() {
        this.add('backup:PutBackupVaultAccessPolicy');
        return this;
    }
    /**
     * Adds an SNS topic to the backup vault.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_PutBackupVaultNotifications.html
     */
    putBackupVaultNotifications() {
        this.add('backup:PutBackupVaultNotifications');
        return this;
    }
    /**
     * Starts a new backup job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_StartBackupJob.html
     */
    startBackupJob() {
        this.add('backup:StartBackupJob');
        return this;
    }
    /**
     * Copy a backup from a source region to a destination region.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_StartCopyJob.html
     */
    startCopyJob() {
        this.add('backup:StartCopyJob');
        return this;
    }
    /**
     * Starts a new restore job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_StartRestoreJob.html
     */
    startRestoreJob() {
        this.add('backup:StartRestoreJob');
        return this;
    }
    /**
     * Stops a backup job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_StopBackupJob.html
     */
    stopBackupJob() {
        this.add('backup:StopBackupJob');
        return this;
    }
    /**
     * Tags a resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_TagResource.html
     */
    tagResource() {
        this.add('backup:TagResource');
        return this;
    }
    /**
     * Untags a resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_UntagResource.html
     */
    untagResource() {
        this.add('backup:UntagResource');
        return this;
    }
    /**
     * Updates a backup plan.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_UpdateBackupPlan.html
     */
    updateBackupPlan() {
        this.add('backup:UpdateBackupPlan');
        return this;
    }
    /**
     * Updates the lifecycle of the recovery point.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_UpdateRecoveryPointLifecycle.html
     */
    updateRecoveryPointLifecycle() {
        this.add('backup:UpdateRecoveryPointLifecycle');
        return this;
    }
    /**
     * Adds a resource of type backupVault to the statement
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/vaults.html
     *
     * @param backupVaultName - Identifier for the backupVaultName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onBackupVault(backupVaultName, account, region, partition) {
        var arn = 'arn:${Partition}:backup:${Region}:${Account}:backup-vault:${BackupVaultName}';
        arn = arn.replace('${BackupVaultName}', backupVaultName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type backupPlan to the statement
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/about-backup-plans.html
     *
     * @param backupPlanId - Identifier for the backupPlanId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onBackupPlan(backupPlanId, account, region, partition) {
        var arn = 'arn:${Partition}:backup:${Region}:${Account}:backup-plan:${BackupPlanId}';
        arn = arn.replace('${BackupPlanId}', backupPlanId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type recoveryPoint to the statement
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/recovery-points.html
     *
     * @param recoveryPointId - Identifier for the recoveryPointId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onRecoveryPoint(recoveryPointId, account, region, partition) {
        var arn = 'arn:${Partition}:backup:${Region}:${Account}:recovery-point:${RecoveryPointId}';
        arn = arn.replace('${RecoveryPointId}', recoveryPointId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Backup = Backup;
//# sourceMappingURL=data:application/json;base64,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