"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service chatbot
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awschatbot.html
 */
class Chatbot extends shared_1.PolicyStatement {
    /**
     * Action provider for service chatbot
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awschatbot.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'chatbot';
        this.actions = {
            "CreateChimeWebhookConfiguration": {
                "url": "",
                "description": "Creates an AWS Chatbot Chime Webhook Configuration.",
                "accessLevel": "Write"
            },
            "CreateSlackChannelConfiguration": {
                "url": "",
                "description": "Creates an AWS Chatbot Slack Channel Configuration.",
                "accessLevel": "Write"
            },
            "DeleteChimeWebhookConfiguration": {
                "url": "",
                "description": "Deletes an AWS Chatbot Chime Webhook Configuration.",
                "accessLevel": "Write"
            },
            "DeleteSlackChannelConfiguration": {
                "url": "",
                "description": "Deletes an AWS Chatbot Slack Channel Configuration.",
                "accessLevel": "Write"
            },
            "DescribeChimeWebhookConfigurations": {
                "url": "",
                "description": "Lists all AWS Chatbot Chime Webhook Configurations in an AWS Account.",
                "accessLevel": "Read"
            },
            "DescribeSlackChannelConfigurations": {
                "url": "",
                "description": "Lists all AWS Chatbot Slack Channel Configurations in an AWS account.",
                "accessLevel": "Read"
            },
            "DescribeSlackChannels": {
                "url": "",
                "description": "Lists all public Slack channels in the Slack workspace connected to the AWS Account onboarded with AWS Chatbot service.",
                "accessLevel": "Read"
            },
            "DescribeSlackWorkspaces": {
                "url": "",
                "description": "Lists all authorized Slack workspaces connected to the AWS Account onboarded with AWS Chatbot service.",
                "accessLevel": "Read"
            },
            "GetSlackOauthParameters": {
                "url": "",
                "description": "Generate OAuth parameters to request Slack OAuth code to be used by the AWS Chatbot service.",
                "accessLevel": "Read"
            },
            "RedeemSlackOauthCode": {
                "url": "",
                "description": "Redeem previously generated parameters with Slack API, to acquire OAuth tokens to be used by the AWS Chatbot service.",
                "accessLevel": "Write"
            },
            "UpdateChimeWebhookConfiguration": {
                "url": "",
                "description": "Updates an AWS Chatbot Chime Webhook Configuration.",
                "accessLevel": "Write"
            },
            "UpdateSlackChannelConfiguration": {
                "url": "",
                "description": "Updates an AWS Chatbot Slack Channel Configuration.",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "ChatbotConfiguration": {
                "name": "ChatbotConfiguration",
                "url": "",
                "arn": "arn:${Partition}:chatbot::${Account}:${ResourceType}/${ResourceName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Creates an AWS Chatbot Chime Webhook Configuration.
     *
     * Access Level: Write
     */
    createChimeWebhookConfiguration() {
        this.add('chatbot:CreateChimeWebhookConfiguration');
        return this;
    }
    /**
     * Creates an AWS Chatbot Slack Channel Configuration.
     *
     * Access Level: Write
     */
    createSlackChannelConfiguration() {
        this.add('chatbot:CreateSlackChannelConfiguration');
        return this;
    }
    /**
     * Deletes an AWS Chatbot Chime Webhook Configuration.
     *
     * Access Level: Write
     */
    deleteChimeWebhookConfiguration() {
        this.add('chatbot:DeleteChimeWebhookConfiguration');
        return this;
    }
    /**
     * Deletes an AWS Chatbot Slack Channel Configuration.
     *
     * Access Level: Write
     */
    deleteSlackChannelConfiguration() {
        this.add('chatbot:DeleteSlackChannelConfiguration');
        return this;
    }
    /**
     * Lists all AWS Chatbot Chime Webhook Configurations in an AWS Account.
     *
     * Access Level: Read
     */
    describeChimeWebhookConfigurations() {
        this.add('chatbot:DescribeChimeWebhookConfigurations');
        return this;
    }
    /**
     * Lists all AWS Chatbot Slack Channel Configurations in an AWS account.
     *
     * Access Level: Read
     */
    describeSlackChannelConfigurations() {
        this.add('chatbot:DescribeSlackChannelConfigurations');
        return this;
    }
    /**
     * Lists all public Slack channels in the Slack workspace connected to the AWS Account onboarded with AWS Chatbot service.
     *
     * Access Level: Read
     */
    describeSlackChannels() {
        this.add('chatbot:DescribeSlackChannels');
        return this;
    }
    /**
     * Lists all authorized Slack workspaces connected to the AWS Account onboarded with AWS Chatbot service.
     *
     * Access Level: Read
     */
    describeSlackWorkspaces() {
        this.add('chatbot:DescribeSlackWorkspaces');
        return this;
    }
    /**
     * Generate OAuth parameters to request Slack OAuth code to be used by the AWS Chatbot service.
     *
     * Access Level: Read
     */
    getSlackOauthParameters() {
        this.add('chatbot:GetSlackOauthParameters');
        return this;
    }
    /**
     * Redeem previously generated parameters with Slack API, to acquire OAuth tokens to be used by the AWS Chatbot service.
     *
     * Access Level: Write
     */
    redeemSlackOauthCode() {
        this.add('chatbot:RedeemSlackOauthCode');
        return this;
    }
    /**
     * Updates an AWS Chatbot Chime Webhook Configuration.
     *
     * Access Level: Write
     */
    updateChimeWebhookConfiguration() {
        this.add('chatbot:UpdateChimeWebhookConfiguration');
        return this;
    }
    /**
     * Updates an AWS Chatbot Slack Channel Configuration.
     *
     * Access Level: Write
     */
    updateSlackChannelConfiguration() {
        this.add('chatbot:UpdateSlackChannelConfiguration');
        return this;
    }
    /**
     * Adds a resource of type ChatbotConfiguration to the statement
     *
     * @param resourceType - Identifier for the resourceType.
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onChatbotConfiguration(resourceType, resourceName, account, partition) {
        var arn = 'arn:${Partition}:chatbot::${Account}:${ResourceType}/${ResourceName}';
        arn = arn.replace('${ResourceType}', resourceType);
        arn = arn.replace('${ResourceName}', resourceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Chatbot = Chatbot;
//# sourceMappingURL=data:application/json;base64,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