"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service cloud9
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloud9.html
 */
class Cloud9 extends shared_1.PolicyStatement {
    /**
     * Action provider for service cloud9
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloud9.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'cloud9';
        this.actions = {
            "CreateEnvironmentEC2": {
                "url": "https://docs.aws.amazon.com/cloud9/latest/APIReference/API_CreateEnvironmentEC2.html",
                "description": "Grants permission to create an AWS Cloud9 development environment, launches an Amazon Elastic Compute Cloud (Amazon EC2) instance, and then hosts the environment on the instance.",
                "accessLevel": "Write",
                "conditions": [
                    "cloud9:EnvironmentName",
                    "cloud9:InstanceType",
                    "cloud9:SubnetId",
                    "cloud9:UserArn"
                ]
            },
            "CreateEnvironmentMembership": {
                "url": "https://docs.aws.amazon.com/cloud9/latest/APIReference/API_CreateEnvironmentMembership.html",
                "description": "Grants permission to add an environment member to an AWS Cloud9 development environment.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "environment": {
                        "required": true
                    }
                },
                "conditions": [
                    "cloud9:UserArn",
                    "cloud9:EnvironmentId",
                    "cloud9:Permissions"
                ]
            },
            "DeleteEnvironment": {
                "url": "https://docs.aws.amazon.com/cloud9/latest/APIReference/API_DeleteEnvironment.html",
                "description": "Grants permission to delete an AWS Cloud9 development environment. If the environment is hosted on an Amazon Elastic Compute Cloud (Amazon EC2) instance, also terminates the instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "environment": {
                        "required": true
                    }
                }
            },
            "DeleteEnvironmentMembership": {
                "url": "https://docs.aws.amazon.com/cloud9/latest/APIReference/API_DeleteEnvironmentMembership.html",
                "description": "Grants permission to delete an environment member from an AWS Cloud9 development environment.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "environment": {
                        "required": true
                    }
                }
            },
            "DescribeEnvironmentMemberships": {
                "url": "https://docs.aws.amazon.com/cloud9/latest/APIReference/API_DescribeEnvironmentMemberships.html",
                "description": "Grants permission to get information about environment members for an AWS Cloud9 development environment.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "environment": {
                        "required": true
                    }
                },
                "conditions": [
                    "cloud9:UserArn",
                    "cloud9:EnvironmentId"
                ]
            },
            "DescribeEnvironmentStatus": {
                "url": "https://docs.aws.amazon.com/cloud9/latest/APIReference/API_DescribeEnvironmentStatus.html",
                "description": "Grants permission to get status information for an AWS Cloud9 development environment.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "environment": {
                        "required": true
                    }
                }
            },
            "DescribeEnvironments": {
                "url": "https://docs.aws.amazon.com/cloud9/latest/APIReference/API_DescribeEnvironments.html",
                "description": "Grants permission to get information about AWS Cloud9 development environments.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "environment": {
                        "required": true
                    }
                }
            },
            "GetUserSettings": {
                "url": "https://docs.aws.amazon.com/cloud9/latest/user-guide/settings-user.html",
                "description": "Grants permission to get IDE-specific settings of an AWS Cloud9 user.",
                "accessLevel": "Read"
            },
            "ListEnvironments": {
                "url": "https://docs.aws.amazon.com/cloud9/latest/APIReference/API_ListEnvironments.html",
                "description": "Grants permission to get a list of AWS Cloud9 development environment identifiers.",
                "accessLevel": "Read"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/cloud9/latest/APIReference/API_ListTagsForResource.html",
                "description": "Lists tags for a cloud9 environment",
                "accessLevel": "Read",
                "resourceTypes": {
                    "environment": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/cloud9/latest/APIReference/API_TagResource.html",
                "description": "Adds tags to a cloud9 environment",
                "accessLevel": "Write",
                "resourceTypes": {
                    "environment": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/cloud9/latest/APIReference/API_UntagResource.html",
                "description": "Removes tags from a cloud9 environment",
                "accessLevel": "Write",
                "resourceTypes": {
                    "environment": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateEnvironment": {
                "url": "https://docs.aws.amazon.com/cloud9/latest/APIReference/API_UpdateEnvironment.html",
                "description": "Grants permission to change the settings of an existing AWS Cloud9 development environment.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "environment": {
                        "required": true
                    }
                }
            },
            "UpdateEnvironmentMembership": {
                "url": "https://docs.aws.amazon.com/cloud9/latest/APIReference/API_UpdateEnvironmentMembership.html",
                "description": "Grants permission to change the settings of an existing environment member for an AWS Cloud9 development environment.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "environment": {
                        "required": true
                    }
                },
                "conditions": [
                    "cloud9:UserArn",
                    "cloud9:EnvironmentId",
                    "cloud9:Permissions"
                ]
            },
            "UpdateUserSettings": {
                "url": "https://docs.aws.amazon.com/cloud9/latest/user-guide/settings-user.html",
                "description": "Grants permission to update IDE-specific settings of an AWS Cloud9 user.",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "environment": {
                "name": "environment",
                "url": "https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloud9.html#awscloud9-environment",
                "arn": "arn:${Partition}:cloud9:${Region}:${Account}:environment:${ResourceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Grants permission to create an AWS Cloud9 development environment, launches an Amazon Elastic Compute Cloud (Amazon EC2) instance, and then hosts the environment on the instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_CreateEnvironmentEC2.html
     */
    createEnvironmentEC2() {
        this.add('cloud9:CreateEnvironmentEC2');
        return this;
    }
    /**
     * Grants permission to add an environment member to an AWS Cloud9 development environment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_CreateEnvironmentMembership.html
     */
    createEnvironmentMembership() {
        this.add('cloud9:CreateEnvironmentMembership');
        return this;
    }
    /**
     * Grants permission to delete an AWS Cloud9 development environment. If the environment is hosted on an Amazon Elastic Compute Cloud (Amazon EC2) instance, also terminates the instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_DeleteEnvironment.html
     */
    deleteEnvironment() {
        this.add('cloud9:DeleteEnvironment');
        return this;
    }
    /**
     * Grants permission to delete an environment member from an AWS Cloud9 development environment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_DeleteEnvironmentMembership.html
     */
    deleteEnvironmentMembership() {
        this.add('cloud9:DeleteEnvironmentMembership');
        return this;
    }
    /**
     * Grants permission to get information about environment members for an AWS Cloud9 development environment.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_DescribeEnvironmentMemberships.html
     */
    describeEnvironmentMemberships() {
        this.add('cloud9:DescribeEnvironmentMemberships');
        return this;
    }
    /**
     * Grants permission to get status information for an AWS Cloud9 development environment.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_DescribeEnvironmentStatus.html
     */
    describeEnvironmentStatus() {
        this.add('cloud9:DescribeEnvironmentStatus');
        return this;
    }
    /**
     * Grants permission to get information about AWS Cloud9 development environments.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_DescribeEnvironments.html
     */
    describeEnvironments() {
        this.add('cloud9:DescribeEnvironments');
        return this;
    }
    /**
     * Grants permission to get IDE-specific settings of an AWS Cloud9 user.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud9/latest/user-guide/settings-user.html
     */
    getUserSettings() {
        this.add('cloud9:GetUserSettings');
        return this;
    }
    /**
     * Grants permission to get a list of AWS Cloud9 development environment identifiers.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_ListEnvironments.html
     */
    listEnvironments() {
        this.add('cloud9:ListEnvironments');
        return this;
    }
    /**
     * Lists tags for a cloud9 environment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('cloud9:ListTagsForResource');
        return this;
    }
    /**
     * Adds tags to a cloud9 environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_TagResource.html
     */
    tagResource() {
        this.add('cloud9:TagResource');
        return this;
    }
    /**
     * Removes tags from a cloud9 environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_UntagResource.html
     */
    untagResource() {
        this.add('cloud9:UntagResource');
        return this;
    }
    /**
     * Grants permission to change the settings of an existing AWS Cloud9 development environment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_UpdateEnvironment.html
     */
    updateEnvironment() {
        this.add('cloud9:UpdateEnvironment');
        return this;
    }
    /**
     * Grants permission to change the settings of an existing environment member for an AWS Cloud9 development environment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_UpdateEnvironmentMembership.html
     */
    updateEnvironmentMembership() {
        this.add('cloud9:UpdateEnvironmentMembership');
        return this;
    }
    /**
     * Grants permission to update IDE-specific settings of an AWS Cloud9 user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud9/latest/user-guide/settings-user.html
     */
    updateUserSettings() {
        this.add('cloud9:UpdateUserSettings');
        return this;
    }
    /**
     * Adds a resource of type environment to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloud9.html#awscloud9-environment
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onEnvironment(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:cloud9:${Region}:${Account}:environment:${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters access by the AWS Cloud9 environment ID
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloud9.html#awscloud9-cloud9_EnvironmentId
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEnvironmentId(value, operator) {
        return this.if(`cloud9:EnvironmentId`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the AWS Cloud9 environment name
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloud9.html#awscloud9-cloud9_EnvironmentName
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEnvironmentName(value, operator) {
        return this.if(`cloud9:EnvironmentName`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the instance type of the AWS Cloud9 environment's Amazon EC2 instance
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloud9.html#awscloud9-cloud9_InstanceType
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifInstanceType(value, operator) {
        return this.if(`cloud9:InstanceType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the type of AWS Cloud9 permissions
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloud9.html#awscloud9-cloud9_Permissions
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPermissions(value, operator) {
        return this.if(`cloud9:Permissions`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the subnet ID that the AWS Cloud9 environment will be created in
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloud9.html#awscloud9-cloud9_SubnetId
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSubnetId(value, operator) {
        return this.if(`cloud9:SubnetId`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the user ARN specified
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloud9.html#awscloud9-cloud9_UserArn
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifUserArn(value, operator) {
        return this.if(`cloud9:UserArn`, value, operator || 'ArnEquals');
    }
}
exports.Cloud9 = Cloud9;
//# sourceMappingURL=data:application/json;base64,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