"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service cloudformation
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloudformation.html
 */
class Cloudformation extends shared_1.PolicyStatement {
    /**
     * Action provider for service cloudformation
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloudformation.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'cloudformation';
        this.actions = {
            "CancelUpdateStack": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_CancelUpdateStack.html",
                "description": "Cancels an update on the specified stack.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": true
                    }
                }
            },
            "ContinueUpdateRollback": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ContinueUpdateRollback.html",
                "description": "For a specified stack that is in the UPDATE_ROLLBACK_FAILED state, continues rolling it back to the UPDATE_ROLLBACK_COMPLETE state.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": true
                    }
                },
                "conditions": [
                    "cloudformation:RoleArn"
                ]
            },
            "CreateChangeSet": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_CreateChangeSet.html",
                "description": "Creates a list of changes for a stack.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": true
                    }
                },
                "conditions": [
                    "cloudformation:ChangeSetName",
                    "cloudformation:ResourceTypes",
                    "cloudformation:ImportResourceTypes",
                    "cloudformation:RoleArn",
                    "cloudformation:StackPolicyUrl",
                    "cloudformation:TemplateUrl",
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateStack": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_CreateStack.html",
                "description": "Creates a stack as specified in the template.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": true
                    }
                },
                "conditions": [
                    "cloudformation:ResourceTypes",
                    "cloudformation:RoleArn",
                    "cloudformation:StackPolicyUrl",
                    "cloudformation:TemplateUrl",
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateStackInstances": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_CreateStackInstances.html",
                "description": "Creates stack instances for the specified accounts, within the specified regions.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stackset": {
                        "required": true
                    }
                }
            },
            "CreateStackSet": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_CreateStackSet.html",
                "description": "Creates a stackset as specified in the template.",
                "accessLevel": "Write",
                "conditions": [
                    "cloudformation:RoleArn",
                    "cloudformation:TemplateUrl",
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateUploadBucket": {
                "url": "",
                "description": "",
                "accessLevel": "Write"
            },
            "DeleteChangeSet": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DeleteChangeSet.html",
                "description": "Deletes the specified change set. Deleting change sets ensures that no one executes the wrong change set.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": true
                    }
                },
                "conditions": [
                    "cloudformation:ChangeSetName"
                ]
            },
            "DeleteStack": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DeleteStack.html",
                "description": "Deletes a specified stack.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": true
                    }
                },
                "conditions": [
                    "cloudformation:RoleArn"
                ]
            },
            "DeleteStackInstances": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DeleteStackInstances.html",
                "description": "Deletes stack instances for the specified accounts, in the specified regions.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stackset": {
                        "required": true
                    }
                }
            },
            "DeleteStackSet": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DeleteStackSet.html",
                "description": "Deletes a specified stackset.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stackset": {
                        "required": true
                    }
                }
            },
            "DeregisterType": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DeregisterType.html",
                "description": "Deregisters an existing CloudFormation type or type version",
                "accessLevel": "Write"
            },
            "DescribeAccountLimits": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeAccountLimits.html",
                "description": "Retrieves your account's AWS CloudFormation limits.",
                "accessLevel": "Read"
            },
            "DescribeChangeSet": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeChangeSet.html",
                "description": "Returns the description for the specified change set.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "stack": {
                        "required": true
                    }
                },
                "conditions": [
                    "cloudformation:ChangeSetName"
                ]
            },
            "DescribeStackDriftDetectionStatus": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStackDriftDetectionStatus.html",
                "description": "Returns information about a stack drift detection operation.",
                "accessLevel": "Read"
            },
            "DescribeStackEvents": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStackEvents.html",
                "description": "Returns all stack related events for a specified stack.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "stack": {
                        "required": true
                    }
                }
            },
            "DescribeStackInstance": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStackInstance.html",
                "description": "Returns the stack instance that's associated with the specified stack set, AWS account, and region.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "stackset": {
                        "required": true
                    }
                }
            },
            "DescribeStackResource": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStackResource.html",
                "description": "Returns a description of the specified resource in the specified stack.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "stack": {
                        "required": true
                    }
                }
            },
            "DescribeStackResourceDrifts": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStackResourceDrifts.html",
                "description": "Returns drift information for the resources that have been checked for drift in the specified stack.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "stack": {
                        "required": true
                    }
                }
            },
            "DescribeStackResources": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStackResources.html",
                "description": "Returns AWS resource descriptions for running and deleted stacks.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "stack": {
                        "required": true
                    }
                }
            },
            "DescribeStackSet": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStackSet.html",
                "description": "Returns the description of the specified stack set.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "stackset": {
                        "required": true
                    }
                }
            },
            "DescribeStackSetOperation": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStackSetOperation.html",
                "description": "Returns the description of the specified stack set operation.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "stackset": {
                        "required": true
                    }
                }
            },
            "DescribeStacks": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStacks.html",
                "description": "Returns the description for the specified stack.",
                "accessLevel": "List",
                "resourceTypes": {
                    "stack": {
                        "required": true
                    }
                }
            },
            "DescribeType": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeType.html",
                "description": "Returns information about the CloudFormation type requested",
                "accessLevel": "Read"
            },
            "DescribeTypeRegistration": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeTypeRegistration.html",
                "description": "Returns information about the registration process for a CloudFormation type",
                "accessLevel": "Read"
            },
            "DetectStackDrift": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DetectStackDrift.html",
                "description": "Detects whether a stack's actual configuration differs, or has drifted, from it's expected configuration, as defined in the stack template and any values specified as template parameters.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "stack": {
                        "required": true
                    }
                }
            },
            "DetectStackResourceDrift": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DetectStackResourceDrift.html",
                "description": "Returns information about whether a resource's actual configuration differs, or has drifted, from it's expected configuration, as defined in the stack template and any values specified as template parameters.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "stack": {
                        "required": true
                    }
                }
            },
            "DetectStackSetDrift": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DetectStackSetDrift.html",
                "description": "Enables users to detect drift on a stack set and the stack instances that belong to that stack set.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "stackset": {
                        "required": true
                    }
                }
            },
            "EstimateTemplateCost": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_EstimateTemplateCost.html",
                "description": "Returns the estimated monthly cost of a template.",
                "accessLevel": "Read"
            },
            "ExecuteChangeSet": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ExecuteChangeSet.html",
                "description": "Updates a stack using the input information that was provided when the specified change set was created.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": true
                    }
                },
                "conditions": [
                    "cloudformation:ChangeSetName"
                ]
            },
            "GetStackPolicy": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_GetStackPolicy.html",
                "description": "Returns the stack policy for a specified stack.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "stack": {
                        "required": true
                    }
                }
            },
            "GetTemplate": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_GetTemplate.html",
                "description": "Returns the template body for a specified stack.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "stack": {
                        "required": true
                    }
                }
            },
            "GetTemplateSummary": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_GetTemplateSummary.html",
                "description": "Returns information about a new or existing template.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    },
                    "stackset": {
                        "required": false
                    }
                }
            },
            "ListChangeSets": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListChangeSets.html",
                "description": "Returns the ID and status of each active change set for a stack. For example, AWS CloudFormation lists change sets that are in the CREATE_IN_PROGRESS or CREATE_PENDING state.",
                "accessLevel": "List",
                "resourceTypes": {
                    "stack": {
                        "required": true
                    }
                }
            },
            "ListExports": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListExports.html",
                "description": "Lists all exported output values in the account and region in which you call this action.",
                "accessLevel": "List"
            },
            "ListImports": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListImports.html",
                "description": "Lists all stacks that are importing an exported output value.",
                "accessLevel": "List"
            },
            "ListStackInstances": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListStackSets.html",
                "description": "Returns summary information about stack instances that are associated with the specified stack set.",
                "accessLevel": "List",
                "resourceTypes": {
                    "stackset": {
                        "required": true
                    }
                }
            },
            "ListStackResources": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListStackResources.html",
                "description": "Returns descriptions of all resources of the specified stack.",
                "accessLevel": "List",
                "resourceTypes": {
                    "stack": {
                        "required": true
                    }
                }
            },
            "ListStackSetOperationResults": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListStackSetOperationResults.html",
                "description": "Returns summary information about the results of a stack set operation.",
                "accessLevel": "List",
                "resourceTypes": {
                    "stackset": {
                        "required": true
                    }
                }
            },
            "ListStackSetOperations": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListStackSetOperations.html",
                "description": "Returns summary information about operations performed on a stack set.",
                "accessLevel": "List",
                "resourceTypes": {
                    "stackset": {
                        "required": true
                    }
                }
            },
            "ListStackSets": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListStackSets.html",
                "description": "Returns summary information about stack sets that are associated with the user.",
                "accessLevel": "List",
                "resourceTypes": {
                    "stackset": {
                        "required": true
                    }
                }
            },
            "ListStacks": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListStacks.html",
                "description": "Returns the summary information for stacks whose status matches the specified StackStatusFilter.",
                "accessLevel": "List"
            },
            "ListTypeRegistrations": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListTypeRegistrations.html",
                "description": "Lists CloudFormation type registration attempts",
                "accessLevel": "List"
            },
            "ListTypeVersions": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListTypeVersions.html",
                "description": "Lists versions of a particular CloudFormation type",
                "accessLevel": "List"
            },
            "ListTypes": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListTypes.html",
                "description": "Lists available CloudFormation types",
                "accessLevel": "List"
            },
            "RegisterType": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_RegisterType.html",
                "description": "Registers a new CloudFormation type",
                "accessLevel": "Write"
            },
            "SetStackPolicy": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_SetStackPolicy.html",
                "description": "Sets a stack policy for a specified stack.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "stack": {
                        "required": true
                    }
                },
                "conditions": [
                    "cloudformation:StackPolicyUrl"
                ]
            },
            "SetTypeDefaultVersion": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_SetTypeDefaultVersion.html",
                "description": "Sets which version of a CloudFormation type applies to CloudFormation operations",
                "accessLevel": "Write"
            },
            "SignalResource": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_SignalResource.html",
                "description": "Sends a signal to the specified resource with a success or failure status.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": true
                    }
                }
            },
            "StopStackSetOperation": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_StopStackSetOperation.html",
                "description": "Stops an in-progress operation on a stack set and its associated stack instances.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stackset": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "",
                "description": "Tagging cloudformation resources.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    },
                    "stackset": {
                        "required": false
                    }
                }
            },
            "UntagResource": {
                "url": "",
                "description": "Untagging cloudformation resources.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    },
                    "stackset": {
                        "required": false
                    }
                }
            },
            "UpdateStack": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_UpdateStack.html",
                "description": "Updates a stack as specified in the template.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": true
                    }
                },
                "conditions": [
                    "cloudformation:ResourceTypes",
                    "cloudformation:RoleArn",
                    "cloudformation:StackPolicyUrl",
                    "cloudformation:TemplateUrl",
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UpdateStackInstances": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_UpdateStackInstances.html",
                "description": "Updates the parameter values for stack instances for the specified accounts, within the specified regions.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stackset": {
                        "required": true
                    }
                }
            },
            "UpdateStackSet": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_UpdateStackSet.html",
                "description": "Updates a stackset as specified in the template.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stackset": {
                        "required": true
                    }
                },
                "conditions": [
                    "cloudformation:RoleArn",
                    "cloudformation:TemplateUrl",
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UpdateTerminationProtection": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_UpdateTerminationProtection.html",
                "description": "Updates termination protection for the specified stack.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": true
                    }
                }
            },
            "ValidateTemplate": {
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ValidateTemplate.html",
                "description": "Validates a specified template.",
                "accessLevel": "Read"
            }
        };
        this.resourceTypes = {
            "stack": {
                "name": "stack",
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-whatis-concepts.html#w2ab1b5c15b9",
                "arn": "arn:${Partition}:cloudformation:${Region}:${Account}:stack/${StackName}/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "stackset": {
                "name": "stackset",
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-concepts.html#stacksets-concepts-stackset",
                "arn": "arn:${Partition}:cloudformation:${Region}:${Account}:stackset/${StackSetName}:${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "changeset": {
                "name": "changeset",
                "url": "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-whatis-concepts.html#w2ab1b5c15c11",
                "arn": "arn:${Partition}:cloudformation:${Region}:${Account}:changeSet/${ChangeSetName}:${Id}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Cancels an update on the specified stack.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_CancelUpdateStack.html
     */
    cancelUpdateStack() {
        this.add('cloudformation:CancelUpdateStack');
        return this;
    }
    /**
     * For a specified stack that is in the UPDATE_ROLLBACK_FAILED state, continues rolling it back to the UPDATE_ROLLBACK_COMPLETE state.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ContinueUpdateRollback.html
     */
    continueUpdateRollback() {
        this.add('cloudformation:ContinueUpdateRollback');
        return this;
    }
    /**
     * Creates a list of changes for a stack.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_CreateChangeSet.html
     */
    createChangeSet() {
        this.add('cloudformation:CreateChangeSet');
        return this;
    }
    /**
     * Creates a stack as specified in the template.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_CreateStack.html
     */
    createStack() {
        this.add('cloudformation:CreateStack');
        return this;
    }
    /**
     * Creates stack instances for the specified accounts, within the specified regions.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_CreateStackInstances.html
     */
    createStackInstances() {
        this.add('cloudformation:CreateStackInstances');
        return this;
    }
    /**
     * Creates a stackset as specified in the template.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_CreateStackSet.html
     */
    createStackSet() {
        this.add('cloudformation:CreateStackSet');
        return this;
    }
    /**
     *
     *
     * Access Level: Write
     */
    createUploadBucket() {
        this.add('cloudformation:CreateUploadBucket');
        return this;
    }
    /**
     * Deletes the specified change set. Deleting change sets ensures that no one executes the wrong change set.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DeleteChangeSet.html
     */
    deleteChangeSet() {
        this.add('cloudformation:DeleteChangeSet');
        return this;
    }
    /**
     * Deletes a specified stack.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DeleteStack.html
     */
    deleteStack() {
        this.add('cloudformation:DeleteStack');
        return this;
    }
    /**
     * Deletes stack instances for the specified accounts, in the specified regions.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DeleteStackInstances.html
     */
    deleteStackInstances() {
        this.add('cloudformation:DeleteStackInstances');
        return this;
    }
    /**
     * Deletes a specified stackset.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DeleteStackSet.html
     */
    deleteStackSet() {
        this.add('cloudformation:DeleteStackSet');
        return this;
    }
    /**
     * Deregisters an existing CloudFormation type or type version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DeregisterType.html
     */
    deregisterType() {
        this.add('cloudformation:DeregisterType');
        return this;
    }
    /**
     * Retrieves your account's AWS CloudFormation limits.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeAccountLimits.html
     */
    describeAccountLimits() {
        this.add('cloudformation:DescribeAccountLimits');
        return this;
    }
    /**
     * Returns the description for the specified change set.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeChangeSet.html
     */
    describeChangeSet() {
        this.add('cloudformation:DescribeChangeSet');
        return this;
    }
    /**
     * Returns information about a stack drift detection operation.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStackDriftDetectionStatus.html
     */
    describeStackDriftDetectionStatus() {
        this.add('cloudformation:DescribeStackDriftDetectionStatus');
        return this;
    }
    /**
     * Returns all stack related events for a specified stack.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStackEvents.html
     */
    describeStackEvents() {
        this.add('cloudformation:DescribeStackEvents');
        return this;
    }
    /**
     * Returns the stack instance that's associated with the specified stack set, AWS account, and region.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStackInstance.html
     */
    describeStackInstance() {
        this.add('cloudformation:DescribeStackInstance');
        return this;
    }
    /**
     * Returns a description of the specified resource in the specified stack.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStackResource.html
     */
    describeStackResource() {
        this.add('cloudformation:DescribeStackResource');
        return this;
    }
    /**
     * Returns drift information for the resources that have been checked for drift in the specified stack.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStackResourceDrifts.html
     */
    describeStackResourceDrifts() {
        this.add('cloudformation:DescribeStackResourceDrifts');
        return this;
    }
    /**
     * Returns AWS resource descriptions for running and deleted stacks.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStackResources.html
     */
    describeStackResources() {
        this.add('cloudformation:DescribeStackResources');
        return this;
    }
    /**
     * Returns the description of the specified stack set.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStackSet.html
     */
    describeStackSet() {
        this.add('cloudformation:DescribeStackSet');
        return this;
    }
    /**
     * Returns the description of the specified stack set operation.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStackSetOperation.html
     */
    describeStackSetOperation() {
        this.add('cloudformation:DescribeStackSetOperation');
        return this;
    }
    /**
     * Returns the description for the specified stack.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStacks.html
     */
    describeStacks() {
        this.add('cloudformation:DescribeStacks');
        return this;
    }
    /**
     * Returns information about the CloudFormation type requested
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeType.html
     */
    describeType() {
        this.add('cloudformation:DescribeType');
        return this;
    }
    /**
     * Returns information about the registration process for a CloudFormation type
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeTypeRegistration.html
     */
    describeTypeRegistration() {
        this.add('cloudformation:DescribeTypeRegistration');
        return this;
    }
    /**
     * Detects whether a stack's actual configuration differs, or has drifted, from it's expected configuration, as defined in the stack template and any values specified as template parameters.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DetectStackDrift.html
     */
    detectStackDrift() {
        this.add('cloudformation:DetectStackDrift');
        return this;
    }
    /**
     * Returns information about whether a resource's actual configuration differs, or has drifted, from it's expected configuration, as defined in the stack template and any values specified as template parameters.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DetectStackResourceDrift.html
     */
    detectStackResourceDrift() {
        this.add('cloudformation:DetectStackResourceDrift');
        return this;
    }
    /**
     * Enables users to detect drift on a stack set and the stack instances that belong to that stack set.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DetectStackSetDrift.html
     */
    detectStackSetDrift() {
        this.add('cloudformation:DetectStackSetDrift');
        return this;
    }
    /**
     * Returns the estimated monthly cost of a template.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_EstimateTemplateCost.html
     */
    estimateTemplateCost() {
        this.add('cloudformation:EstimateTemplateCost');
        return this;
    }
    /**
     * Updates a stack using the input information that was provided when the specified change set was created.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ExecuteChangeSet.html
     */
    executeChangeSet() {
        this.add('cloudformation:ExecuteChangeSet');
        return this;
    }
    /**
     * Returns the stack policy for a specified stack.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_GetStackPolicy.html
     */
    getStackPolicy() {
        this.add('cloudformation:GetStackPolicy');
        return this;
    }
    /**
     * Returns the template body for a specified stack.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_GetTemplate.html
     */
    getTemplate() {
        this.add('cloudformation:GetTemplate');
        return this;
    }
    /**
     * Returns information about a new or existing template.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_GetTemplateSummary.html
     */
    getTemplateSummary() {
        this.add('cloudformation:GetTemplateSummary');
        return this;
    }
    /**
     * Returns the ID and status of each active change set for a stack. For example, AWS CloudFormation lists change sets that are in the CREATE_IN_PROGRESS or CREATE_PENDING state.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListChangeSets.html
     */
    listChangeSets() {
        this.add('cloudformation:ListChangeSets');
        return this;
    }
    /**
     * Lists all exported output values in the account and region in which you call this action.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListExports.html
     */
    listExports() {
        this.add('cloudformation:ListExports');
        return this;
    }
    /**
     * Lists all stacks that are importing an exported output value.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListImports.html
     */
    listImports() {
        this.add('cloudformation:ListImports');
        return this;
    }
    /**
     * Returns summary information about stack instances that are associated with the specified stack set.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListStackSets.html
     */
    listStackInstances() {
        this.add('cloudformation:ListStackInstances');
        return this;
    }
    /**
     * Returns descriptions of all resources of the specified stack.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListStackResources.html
     */
    listStackResources() {
        this.add('cloudformation:ListStackResources');
        return this;
    }
    /**
     * Returns summary information about the results of a stack set operation.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListStackSetOperationResults.html
     */
    listStackSetOperationResults() {
        this.add('cloudformation:ListStackSetOperationResults');
        return this;
    }
    /**
     * Returns summary information about operations performed on a stack set.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListStackSetOperations.html
     */
    listStackSetOperations() {
        this.add('cloudformation:ListStackSetOperations');
        return this;
    }
    /**
     * Returns summary information about stack sets that are associated with the user.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListStackSets.html
     */
    listStackSets() {
        this.add('cloudformation:ListStackSets');
        return this;
    }
    /**
     * Returns the summary information for stacks whose status matches the specified StackStatusFilter.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListStacks.html
     */
    listStacks() {
        this.add('cloudformation:ListStacks');
        return this;
    }
    /**
     * Lists CloudFormation type registration attempts
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListTypeRegistrations.html
     */
    listTypeRegistrations() {
        this.add('cloudformation:ListTypeRegistrations');
        return this;
    }
    /**
     * Lists versions of a particular CloudFormation type
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListTypeVersions.html
     */
    listTypeVersions() {
        this.add('cloudformation:ListTypeVersions');
        return this;
    }
    /**
     * Lists available CloudFormation types
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListTypes.html
     */
    listTypes() {
        this.add('cloudformation:ListTypes');
        return this;
    }
    /**
     * Registers a new CloudFormation type
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_RegisterType.html
     */
    registerType() {
        this.add('cloudformation:RegisterType');
        return this;
    }
    /**
     * Sets a stack policy for a specified stack.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_SetStackPolicy.html
     */
    setStackPolicy() {
        this.add('cloudformation:SetStackPolicy');
        return this;
    }
    /**
     * Sets which version of a CloudFormation type applies to CloudFormation operations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_SetTypeDefaultVersion.html
     */
    setTypeDefaultVersion() {
        this.add('cloudformation:SetTypeDefaultVersion');
        return this;
    }
    /**
     * Sends a signal to the specified resource with a success or failure status.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_SignalResource.html
     */
    signalResource() {
        this.add('cloudformation:SignalResource');
        return this;
    }
    /**
     * Stops an in-progress operation on a stack set and its associated stack instances.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_StopStackSetOperation.html
     */
    stopStackSetOperation() {
        this.add('cloudformation:StopStackSetOperation');
        return this;
    }
    /**
     * Tagging cloudformation resources.
     *
     * Access Level: Tagging
     */
    tagResource() {
        this.add('cloudformation:TagResource');
        return this;
    }
    /**
     * Untagging cloudformation resources.
     *
     * Access Level: Tagging
     */
    untagResource() {
        this.add('cloudformation:UntagResource');
        return this;
    }
    /**
     * Updates a stack as specified in the template.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_UpdateStack.html
     */
    updateStack() {
        this.add('cloudformation:UpdateStack');
        return this;
    }
    /**
     * Updates the parameter values for stack instances for the specified accounts, within the specified regions.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_UpdateStackInstances.html
     */
    updateStackInstances() {
        this.add('cloudformation:UpdateStackInstances');
        return this;
    }
    /**
     * Updates a stackset as specified in the template.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_UpdateStackSet.html
     */
    updateStackSet() {
        this.add('cloudformation:UpdateStackSet');
        return this;
    }
    /**
     * Updates termination protection for the specified stack.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_UpdateTerminationProtection.html
     */
    updateTerminationProtection() {
        this.add('cloudformation:UpdateTerminationProtection');
        return this;
    }
    /**
     * Validates a specified template.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ValidateTemplate.html
     */
    validateTemplate() {
        this.add('cloudformation:ValidateTemplate');
        return this;
    }
    /**
     * Adds a resource of type stack to the statement
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-whatis-concepts.html#w2ab1b5c15b9
     *
     * @param stackName - Identifier for the stackName.
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onStack(stackName, id, account, region, partition) {
        var arn = 'arn:${Partition}:cloudformation:${Region}:${Account}:stack/${StackName}/${Id}';
        arn = arn.replace('${StackName}', stackName);
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type stackset to the statement
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-concepts.html#stacksets-concepts-stackset
     *
     * @param stackSetName - Identifier for the stackSetName.
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onStackset(stackSetName, id, account, region, partition) {
        var arn = 'arn:${Partition}:cloudformation:${Region}:${Account}:stackset/${StackSetName}:${Id}';
        arn = arn.replace('${StackSetName}', stackSetName);
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type changeset to the statement
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-whatis-concepts.html#w2ab1b5c15c11
     *
     * @param changeSetName - Identifier for the changeSetName.
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onChangeset(changeSetName, id, account, region, partition) {
        var arn = 'arn:${Partition}:cloudformation:${Region}:${Account}:changeSet/${ChangeSetName}:${Id}';
        arn = arn.replace('${ChangeSetName}', changeSetName);
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * An AWS CloudFormation change set name. Use to control which change sets IAM users can execute or delete.
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifChangeSetName(value, operator) {
        return this.if(`cloudformation:ChangeSetName`, value, operator || 'StringLike');
    }
    /**
     * The template resource types, such as `AWS::EC2::Instance`. Use to control which resource types IAM users can work with when they want to import a resource into a stack.
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifImportResourceTypes(value, operator) {
        return this.if(`cloudformation:ImportResourceTypes`, value, operator || 'StringLike');
    }
    /**
     * The template resource types, such as `AWS::EC2::Instance`. Use to control which resource types IAM users can work with when they create or update a stack.
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTypes(value, operator) {
        return this.if(`cloudformation:ResourceTypes`, value, operator || 'StringLike');
    }
    /**
     * The ARN of an IAM service role. Use to control which service role IAM users can use to work with stacks or change sets.
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifRoleArn(value, operator) {
        return this.if(`cloudformation:RoleArn`, value, operator || 'ArnEquals');
    }
    /**
     * An Amazon S3 stack policy URL. Use to control which stack policies IAM users can associate with a stack during a create or update stack action.
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifStackPolicyUrl(value, operator) {
        return this.if(`cloudformation:StackPolicyUrl`, value, operator || 'StringLike');
    }
    /**
     * An Amazon S3 template URL. Use to control which templates IAM users can use when they create or update stacks.
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTemplateUrl(value, operator) {
        return this.if(`cloudformation:TemplateUrl`, value, operator || 'StringLike');
    }
}
exports.Cloudformation = Cloudformation;
//# sourceMappingURL=data:application/json;base64,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