"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service codeartifact
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscodeartifact.html
 */
class Codeartifact extends shared_1.PolicyStatement {
    /**
     * Action provider for service codeartifact
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscodeartifact.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'codeartifact';
        this.actions = {
            "AssociateExternalConnection": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_AssociateExternalConnection.html",
                "description": "Grants permission to add an external connection to a repository",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "AssociateWithDownstreamRepository": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/userguide/repos-upstream.html",
                "description": "Grants permission to associate an existing repository as an upstream repository to another repository",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "CopyPackageVersions": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_CopyPackageVersions.html",
                "description": "Grants permission to copy package versions from one repository to another repository in the same domain.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "CreateDomain": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_CreateDomain.html",
                "description": "Grants permission to create a new domain",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "CreateRepository": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_CreateRepository.html",
                "description": "Grants permission to create a new repository",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "DeleteDomain": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DeleteDomain.html",
                "description": "Grants permission to delete a domain",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DeleteDomainPermissionsPolicy": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DeleteDomainPermissionsPolicy.html",
                "description": "Grants permission to delete the resource policy set on a domain",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DeletePackageVersions": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DeletePackageVersions.html",
                "description": "Grants permission to delete package versions",
                "accessLevel": "Write",
                "resourceTypes": {
                    "package": {
                        "required": true
                    }
                }
            },
            "DeleteRepository": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DeleteRepository.html",
                "description": "Grants permission to delete a repository",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "DeleteRepositoryPermissionsPolicy": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DeleteRepositoryPermissionsPolicy.html",
                "description": "Grants permission to delete the resource policy set on a repository",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "DescribeDomain": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DescribeDomain.html",
                "description": "Grants permission to return information about a domain",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DescribePackageVersion": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DescribePackageVersion.html",
                "description": "Grants permission to return information about a package version",
                "accessLevel": "Read",
                "resourceTypes": {
                    "package": {
                        "required": true
                    }
                }
            },
            "DescribeRepository": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DescribeRepository.html",
                "description": "Grants permission to return detailed information about a repository",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "DisassociateExternalConnection": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DisassociateExternalConnection.html",
                "description": "Grants permission to disassociate an external connection from a repository",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DisposePackageVersions": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DisposePackageVersions.html",
                "description": "Grants permission to set the status of package versions to Disposed and delete their assets",
                "accessLevel": "Write",
                "resourceTypes": {
                    "package": {
                        "required": true
                    }
                }
            },
            "GetAuthorizationToken": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_GetAuthorizationToken.html",
                "description": "Grants permission to generate a temporary authentication token for accessing repositories in a domain",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "GetDomainPermissionsPolicy": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_GetDomainPermissionsPolicy.html",
                "description": "Grants permission to return a domain's resource policy",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "GetPackageVersionAsset": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_GetPackageVersionAsset.html",
                "description": "Grants permission to return an asset (or file) that is part of a package version",
                "accessLevel": "Read",
                "resourceTypes": {
                    "package": {
                        "required": true
                    }
                }
            },
            "GetPackageVersionReadme": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_GetPackageVersionReadme.html",
                "description": "Grants permission to return a package version's readme file",
                "accessLevel": "Read",
                "resourceTypes": {
                    "package": {
                        "required": true
                    }
                }
            },
            "GetRepositoryEndpoint": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_GetRepositoryEndpoint.html",
                "description": "Grants permission to return an endpoint for a repository",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetRepositoryPermissionsPolicy": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_GetRepositoryPermissionsPolicy.html",
                "description": "Grants permission to return a repository's resource policy",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "ListDomains": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListDomains.html",
                "description": "Grants permission to list the domains in the current user's AWS account",
                "accessLevel": "List"
            },
            "ListPackageVersionAssets": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListPackageVersionAssets.html",
                "description": "Grants permission to list a package version's assets",
                "accessLevel": "List",
                "resourceTypes": {
                    "package": {
                        "required": true
                    }
                }
            },
            "ListPackageVersionDependencies": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListPackageVersionDependencies.html",
                "description": "Grants permission to list the direct dependencies of a package version",
                "accessLevel": "List",
                "resourceTypes": {
                    "package": {
                        "required": true
                    }
                }
            },
            "ListPackageVersions": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListPackageVersions.html",
                "description": "Grants permission to list a package's versions",
                "accessLevel": "List",
                "resourceTypes": {
                    "package": {
                        "required": true
                    }
                }
            },
            "ListPackages": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListPackages.html",
                "description": "Grants permission to list the packages in a repository",
                "accessLevel": "List",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "ListRepositories": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListRepositories.html",
                "description": "Grants permission to list the repositories administered by the calling account",
                "accessLevel": "List"
            },
            "ListRepositoriesInDomain": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListRepositoriesInDomain.html",
                "description": "Grants permission to list the repositories in a domain",
                "accessLevel": "List",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "PublishPackageVersion": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/userguide/repo-policies.html",
                "description": "Grants permission to publish assets and metadata to a repository endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "package": {
                        "required": true
                    }
                }
            },
            "PutDomainPermissionsPolicy": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_PutDomainPermissionsPolicy.html",
                "description": "Grants permission to attach a resource policy to a domain",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "PutPackageMetadata": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/userguide/repo-policies.html",
                "description": "Grants permission to add, modify or remove package metadata using a repository endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "package": {
                        "required": true
                    }
                }
            },
            "PutRepositoryPermissionsPolicy": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_PutRepositoryPermissionsPolicy.html",
                "description": "Grants permission to attach a resource policy to a repository",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "ReadFromRepository": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/userguide/repo-policies.html",
                "description": "Grants permission to return package assets and metadata from a repository endpoint",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "UpdatePackageVersionsStatus": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_UpdatePackageVersionsStatus.html",
                "description": "Grants permission to modify the status of one or more versions of a package",
                "accessLevel": "Write",
                "resourceTypes": {
                    "package": {
                        "required": true
                    }
                }
            },
            "UpdateRepository": {
                "url": "https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_UpdateRepository.html",
                "description": "Grants permission to modify the properties of a repository",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "domain": {
                "name": "domain",
                "url": "https://docs.aws.amazon.com/codeartifact/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats",
                "arn": "arn:${Partition}:codeartifact:${Region}:${Account}:domain/${DomainName}",
                "conditionKeys": []
            },
            "repository": {
                "name": "repository",
                "url": "https://docs.aws.amazon.com/codeartifact/latest/userguide/repo-policies.html",
                "arn": "arn:${Partition}:codeartifact:${Region}:${Account}:repository/${DomainName}/${RepositoryName}",
                "conditionKeys": []
            },
            "package": {
                "name": "package",
                "url": "https://docs.aws.amazon.com/codeartifact/latest/userguide/repo-policies.html",
                "arn": "arn:${Partition}:codeartifact:${Region}:${Account}:package/${DomainName}/${RepositoryName}/${PackageFormat}/${PackageNamespace}/${PackageName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to add an external connection to a repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_AssociateExternalConnection.html
     */
    associateExternalConnection() {
        this.add('codeartifact:AssociateExternalConnection');
        return this;
    }
    /**
     * Grants permission to associate an existing repository as an upstream repository to another repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/userguide/repos-upstream.html
     */
    associateWithDownstreamRepository() {
        this.add('codeartifact:AssociateWithDownstreamRepository');
        return this;
    }
    /**
     * Grants permission to copy package versions from one repository to another repository in the same domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_CopyPackageVersions.html
     */
    copyPackageVersions() {
        this.add('codeartifact:CopyPackageVersions');
        return this;
    }
    /**
     * Grants permission to create a new domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_CreateDomain.html
     */
    createDomain() {
        this.add('codeartifact:CreateDomain');
        return this;
    }
    /**
     * Grants permission to create a new repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_CreateRepository.html
     */
    createRepository() {
        this.add('codeartifact:CreateRepository');
        return this;
    }
    /**
     * Grants permission to delete a domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DeleteDomain.html
     */
    deleteDomain() {
        this.add('codeartifact:DeleteDomain');
        return this;
    }
    /**
     * Grants permission to delete the resource policy set on a domain
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DeleteDomainPermissionsPolicy.html
     */
    deleteDomainPermissionsPolicy() {
        this.add('codeartifact:DeleteDomainPermissionsPolicy');
        return this;
    }
    /**
     * Grants permission to delete package versions
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DeletePackageVersions.html
     */
    deletePackageVersions() {
        this.add('codeartifact:DeletePackageVersions');
        return this;
    }
    /**
     * Grants permission to delete a repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DeleteRepository.html
     */
    deleteRepository() {
        this.add('codeartifact:DeleteRepository');
        return this;
    }
    /**
     * Grants permission to delete the resource policy set on a repository
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DeleteRepositoryPermissionsPolicy.html
     */
    deleteRepositoryPermissionsPolicy() {
        this.add('codeartifact:DeleteRepositoryPermissionsPolicy');
        return this;
    }
    /**
     * Grants permission to return information about a domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DescribeDomain.html
     */
    describeDomain() {
        this.add('codeartifact:DescribeDomain');
        return this;
    }
    /**
     * Grants permission to return information about a package version
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DescribePackageVersion.html
     */
    describePackageVersion() {
        this.add('codeartifact:DescribePackageVersion');
        return this;
    }
    /**
     * Grants permission to return detailed information about a repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DescribeRepository.html
     */
    describeRepository() {
        this.add('codeartifact:DescribeRepository');
        return this;
    }
    /**
     * Grants permission to disassociate an external connection from a repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DisassociateExternalConnection.html
     */
    disassociateExternalConnection() {
        this.add('codeartifact:DisassociateExternalConnection');
        return this;
    }
    /**
     * Grants permission to set the status of package versions to Disposed and delete their assets
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DisposePackageVersions.html
     */
    disposePackageVersions() {
        this.add('codeartifact:DisposePackageVersions');
        return this;
    }
    /**
     * Grants permission to generate a temporary authentication token for accessing repositories in a domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_GetAuthorizationToken.html
     */
    getAuthorizationToken() {
        this.add('codeartifact:GetAuthorizationToken');
        return this;
    }
    /**
     * Grants permission to return a domain's resource policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_GetDomainPermissionsPolicy.html
     */
    getDomainPermissionsPolicy() {
        this.add('codeartifact:GetDomainPermissionsPolicy');
        return this;
    }
    /**
     * Grants permission to return an asset (or file) that is part of a package version
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_GetPackageVersionAsset.html
     */
    getPackageVersionAsset() {
        this.add('codeartifact:GetPackageVersionAsset');
        return this;
    }
    /**
     * Grants permission to return a package version's readme file
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_GetPackageVersionReadme.html
     */
    getPackageVersionReadme() {
        this.add('codeartifact:GetPackageVersionReadme');
        return this;
    }
    /**
     * Grants permission to return an endpoint for a repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_GetRepositoryEndpoint.html
     */
    getRepositoryEndpoint() {
        this.add('codeartifact:GetRepositoryEndpoint');
        return this;
    }
    /**
     * Grants permission to return a repository's resource policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_GetRepositoryPermissionsPolicy.html
     */
    getRepositoryPermissionsPolicy() {
        this.add('codeartifact:GetRepositoryPermissionsPolicy');
        return this;
    }
    /**
     * Grants permission to list the domains in the current user's AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListDomains.html
     */
    listDomains() {
        this.add('codeartifact:ListDomains');
        return this;
    }
    /**
     * Grants permission to list a package version's assets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListPackageVersionAssets.html
     */
    listPackageVersionAssets() {
        this.add('codeartifact:ListPackageVersionAssets');
        return this;
    }
    /**
     * Grants permission to list the direct dependencies of a package version
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListPackageVersionDependencies.html
     */
    listPackageVersionDependencies() {
        this.add('codeartifact:ListPackageVersionDependencies');
        return this;
    }
    /**
     * Grants permission to list a package's versions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListPackageVersions.html
     */
    listPackageVersions() {
        this.add('codeartifact:ListPackageVersions');
        return this;
    }
    /**
     * Grants permission to list the packages in a repository
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListPackages.html
     */
    listPackages() {
        this.add('codeartifact:ListPackages');
        return this;
    }
    /**
     * Grants permission to list the repositories administered by the calling account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListRepositories.html
     */
    listRepositories() {
        this.add('codeartifact:ListRepositories');
        return this;
    }
    /**
     * Grants permission to list the repositories in a domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListRepositoriesInDomain.html
     */
    listRepositoriesInDomain() {
        this.add('codeartifact:ListRepositoriesInDomain');
        return this;
    }
    /**
     * Grants permission to publish assets and metadata to a repository endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/userguide/repo-policies.html
     */
    publishPackageVersion() {
        this.add('codeartifact:PublishPackageVersion');
        return this;
    }
    /**
     * Grants permission to attach a resource policy to a domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_PutDomainPermissionsPolicy.html
     */
    putDomainPermissionsPolicy() {
        this.add('codeartifact:PutDomainPermissionsPolicy');
        return this;
    }
    /**
     * Grants permission to add, modify or remove package metadata using a repository endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/userguide/repo-policies.html
     */
    putPackageMetadata() {
        this.add('codeartifact:PutPackageMetadata');
        return this;
    }
    /**
     * Grants permission to attach a resource policy to a repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_PutRepositoryPermissionsPolicy.html
     */
    putRepositoryPermissionsPolicy() {
        this.add('codeartifact:PutRepositoryPermissionsPolicy');
        return this;
    }
    /**
     * Grants permission to return package assets and metadata from a repository endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeartifact/latest/userguide/repo-policies.html
     */
    readFromRepository() {
        this.add('codeartifact:ReadFromRepository');
        return this;
    }
    /**
     * Grants permission to modify the status of one or more versions of a package
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_UpdatePackageVersionsStatus.html
     */
    updatePackageVersionsStatus() {
        this.add('codeartifact:UpdatePackageVersionsStatus');
        return this;
    }
    /**
     * Grants permission to modify the properties of a repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_UpdateRepository.html
     */
    updateRepository() {
        this.add('codeartifact:UpdateRepository');
        return this;
    }
    /**
     * Adds a resource of type domain to the statement
     *
     * https://docs.aws.amazon.com/codeartifact/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats
     *
     * @param domainName - Identifier for the domainName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDomain(domainName, account, region, partition) {
        var arn = 'arn:${Partition}:codeartifact:${Region}:${Account}:domain/${DomainName}';
        arn = arn.replace('${DomainName}', domainName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type repository to the statement
     *
     * https://docs.aws.amazon.com/codeartifact/latest/userguide/repo-policies.html
     *
     * @param domainName - Identifier for the domainName.
     * @param repositoryName - Identifier for the repositoryName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onRepository(domainName, repositoryName, account, region, partition) {
        var arn = 'arn:${Partition}:codeartifact:${Region}:${Account}:repository/${DomainName}/${RepositoryName}';
        arn = arn.replace('${DomainName}', domainName);
        arn = arn.replace('${RepositoryName}', repositoryName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type package to the statement
     *
     * https://docs.aws.amazon.com/codeartifact/latest/userguide/repo-policies.html
     *
     * @param domainName - Identifier for the domainName.
     * @param repositoryName - Identifier for the repositoryName.
     * @param packageFormat - Identifier for the packageFormat.
     * @param packageNamespace - Identifier for the packageNamespace.
     * @param packageName - Identifier for the packageName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onPackage(domainName, repositoryName, packageFormat, packageNamespace, packageName, account, region, partition) {
        var arn = 'arn:${Partition}:codeartifact:${Region}:${Account}:package/${DomainName}/${RepositoryName}/${PackageFormat}/${PackageNamespace}/${PackageName}';
        arn = arn.replace('${DomainName}', domainName);
        arn = arn.replace('${RepositoryName}', repositoryName);
        arn = arn.replace('${PackageFormat}', packageFormat);
        arn = arn.replace('${PackageNamespace}', packageNamespace);
        arn = arn.replace('${PackageName}', packageName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Codeartifact = Codeartifact;
//# sourceMappingURL=data:application/json;base64,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