"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service signer
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscodesigningforamazonfreertos.html
 */
class Signer extends shared_1.PolicyStatement {
    /**
     * Action provider for service signer
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscodesigningforamazonfreertos.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'signer';
        this.actions = {
            "CancelSigningProfile": {
                "url": "https://docs.aws.amazon.com/signer/latest/api/API_CancelSigningProfile.html",
                "description": "Cancels a signing profile.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "signing-profile": {
                        "required": true
                    }
                }
            },
            "DescribeSigningJob": {
                "url": "https://docs.aws.amazon.com/signer/latest/api/API_DescribeSigningJob.html",
                "description": "Describe a signing job.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "signing-job": {
                        "required": true
                    }
                }
            },
            "GetSigningPlatform": {
                "url": "https://docs.aws.amazon.com/signer/latest/api/API_GetSigningPlatform.html",
                "description": "Retrieves a signing platform.",
                "accessLevel": "Read"
            },
            "GetSigningProfile": {
                "url": "https://docs.aws.amazon.com/signer/latest/api/API_GetSigningProfile.html",
                "description": "Retrieves a signing profile.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "signing-profile": {
                        "required": true
                    }
                }
            },
            "ListSigningJobs": {
                "url": "https://docs.aws.amazon.com/signer/latest/api/API_ListSigningJobs.html",
                "description": "List signing jobs.",
                "accessLevel": "List"
            },
            "ListSigningPlatforms": {
                "url": "https://docs.aws.amazon.com/signer/latest/api/API_ListSigningPlatforms.html",
                "description": "List all signing platforms.",
                "accessLevel": "List"
            },
            "ListSigningProfiles": {
                "url": "https://docs.aws.amazon.com/signer/latest/api/API_ListSigningProfiles.html",
                "description": "List all signing profile associated with the account.",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/signer/latest/api/API_ListTagsForResource.html",
                "description": "Lists the tags associated with the Signing Profile resource.",
                "accessLevel": "List",
                "resourceTypes": {
                    "signing-profile": {
                        "required": true
                    }
                }
            },
            "PutSigningProfile": {
                "url": "https://docs.aws.amazon.com/signer/latest/api/API_PutSigningProfile.html",
                "description": "Creates a new signing profile if not exists.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "StartSigningJob": {
                "url": "https://docs.aws.amazon.com/signer/latest/api/API_StartSigningJob.html",
                "description": "Starts a code signing request.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "signing-profile": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/signer/latest/api/API_TagResource.html",
                "description": "Adds one or more tags to an Signing Profile resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "signing-profile": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/signer/latest/api/API_UntagResource.html",
                "description": "Removes one or more tags from an Signing Profile resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "signing-profile": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            }
        };
        this.resourceTypes = {
            "signing-profile": {
                "name": "signing-profile",
                "url": "https://docs.aws.amazon.com/signer/latest/developerguide/Welcome.htmlpermissions.html",
                "arn": "arn:${Partition}:signer:${Region}::/signing-profiles/${ProfileName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "signing-job": {
                "name": "signing-job",
                "url": "https://docs.aws.amazon.com/signer/latest/developerguide/Welcome.htmlpermissions.html",
                "arn": "arn:${Partition}:signer:${Region}::/signing-jobs/${JobId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Cancels a signing profile.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/signer/latest/api/API_CancelSigningProfile.html
     */
    cancelSigningProfile() {
        this.add('signer:CancelSigningProfile');
        return this;
    }
    /**
     * Describe a signing job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/signer/latest/api/API_DescribeSigningJob.html
     */
    describeSigningJob() {
        this.add('signer:DescribeSigningJob');
        return this;
    }
    /**
     * Retrieves a signing platform.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/signer/latest/api/API_GetSigningPlatform.html
     */
    getSigningPlatform() {
        this.add('signer:GetSigningPlatform');
        return this;
    }
    /**
     * Retrieves a signing profile.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/signer/latest/api/API_GetSigningProfile.html
     */
    getSigningProfile() {
        this.add('signer:GetSigningProfile');
        return this;
    }
    /**
     * List signing jobs.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/signer/latest/api/API_ListSigningJobs.html
     */
    listSigningJobs() {
        this.add('signer:ListSigningJobs');
        return this;
    }
    /**
     * List all signing platforms.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/signer/latest/api/API_ListSigningPlatforms.html
     */
    listSigningPlatforms() {
        this.add('signer:ListSigningPlatforms');
        return this;
    }
    /**
     * List all signing profile associated with the account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/signer/latest/api/API_ListSigningProfiles.html
     */
    listSigningProfiles() {
        this.add('signer:ListSigningProfiles');
        return this;
    }
    /**
     * Lists the tags associated with the Signing Profile resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/signer/latest/api/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('signer:ListTagsForResource');
        return this;
    }
    /**
     * Creates a new signing profile if not exists.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/signer/latest/api/API_PutSigningProfile.html
     */
    putSigningProfile() {
        this.add('signer:PutSigningProfile');
        return this;
    }
    /**
     * Starts a code signing request.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/signer/latest/api/API_StartSigningJob.html
     */
    startSigningJob() {
        this.add('signer:StartSigningJob');
        return this;
    }
    /**
     * Adds one or more tags to an Signing Profile resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/signer/latest/api/API_TagResource.html
     */
    tagResource() {
        this.add('signer:TagResource');
        return this;
    }
    /**
     * Removes one or more tags from an Signing Profile resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/signer/latest/api/API_UntagResource.html
     */
    untagResource() {
        this.add('signer:UntagResource');
        return this;
    }
    /**
     * Adds a resource of type signing-profile to the statement
     *
     * https://docs.aws.amazon.com/signer/latest/developerguide/Welcome.htmlpermissions.html
     *
     * @param profileName - Identifier for the profileName.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onSigningProfile(profileName, region, partition) {
        var arn = 'arn:${Partition}:signer:${Region}::/signing-profiles/${ProfileName}';
        arn = arn.replace('${ProfileName}', profileName);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type signing-job to the statement
     *
     * https://docs.aws.amazon.com/signer/latest/developerguide/Welcome.htmlpermissions.html
     *
     * @param jobId - Identifier for the jobId.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSigningJob(jobId, region, partition) {
        var arn = 'arn:${Partition}:signer:${Region}::/signing-jobs/${JobId}';
        arn = arn.replace('${JobId}', jobId);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Signer = Signer;
//# sourceMappingURL=data:application/json;base64,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