"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service dms
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsdatabasemigrationservice.html
 */
class Dms extends shared_1.PolicyStatement {
    /**
     * Action provider for service dms
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsdatabasemigrationservice.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'dms';
        this.actions = {
            "AddTagsToResource": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_AddTagsToResource.html",
                "description": "Adds metadata tags to a DMS resource, including replication instance, endpoint, security group, and migration task",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "Certificate": {
                        "required": false
                    },
                    "Endpoint": {
                        "required": false
                    },
                    "EventSubscription": {
                        "required": false
                    },
                    "ReplicationInstance": {
                        "required": false
                    },
                    "ReplicationSubnetGroup": {
                        "required": false
                    },
                    "ReplicationTask": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "dms:req-tag/${TagKey}"
                ]
            },
            "ApplyPendingMaintenanceAction": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_ApplyPendingMaintenanceAction.html",
                "description": "Applies a pending maintenance action to a resource (for example, to a replication instance).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ReplicationInstance": {
                        "required": true
                    }
                }
            },
            "CreateEndpoint": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateEndpoint.html",
                "description": "Creates an endpoint using the provided settings",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "dms:req-tag/${TagKey}"
                ]
            },
            "CreateEventSubscription": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateEventSubscription.html",
                "description": "Creates an AWS DMS event notification subscription.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "dms:req-tag/${TagKey}"
                ]
            },
            "CreateReplicationInstance": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateReplicationInstance.html",
                "description": "Creates the replication instance using the specified parameters",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "dms:req-tag/${TagKey}"
                ]
            },
            "CreateReplicationSubnetGroup": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateReplicationSubnetGroup.html",
                "description": "Creates a replication subnet group given a list of the subnet IDs in a VPC",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "dms:req-tag/${TagKey}"
                ]
            },
            "CreateReplicationTask": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateReplicationTask.html",
                "description": "Creates a replication task using the specified parameters",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Endpoint": {
                        "required": true
                    },
                    "ReplicationInstance": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "dms:req-tag/${TagKey}"
                ]
            },
            "DeleteCertificate": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteCertificate.html",
                "description": "Deletes the specified certificate",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Certificate": {
                        "required": true
                    }
                }
            },
            "DeleteEndpoint": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteEndpoint.html",
                "description": "Deletes the specified endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Endpoint": {
                        "required": true
                    }
                }
            },
            "DeleteEventSubscription": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteEventSubscription.html",
                "description": "Deletes an AWS DMS event subscription.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "EventSubscription": {
                        "required": true
                    }
                }
            },
            "DeleteReplicationInstance": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteReplicationInstance.html",
                "description": "Deletes the specified replication instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ReplicationInstance": {
                        "required": true
                    }
                }
            },
            "DeleteReplicationSubnetGroup": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteReplicationSubnetGroup.html",
                "description": "Deletes a subnet group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ReplicationSubnetGroup": {
                        "required": true
                    }
                }
            },
            "DeleteReplicationTask": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteReplicationTask.html",
                "description": "Deletes the specified replication task",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ReplicationTask": {
                        "required": true
                    }
                }
            },
            "DescribeAccountAttributes": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeAccountAttributes.html",
                "description": "Lists all of the AWS DMS attributes for a customer account",
                "accessLevel": "Read"
            },
            "DescribeCertificates": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeCertificates.html",
                "description": "Provides a description of the certificate.",
                "accessLevel": "Read"
            },
            "DescribeConnections": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeConnections.html",
                "description": "Describes the status of the connections that have been made between the replication instance and an endpoint",
                "accessLevel": "Read"
            },
            "DescribeEndpointTypes": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeEndpointTypes.html",
                "description": "Returns information about the type of endpoints available",
                "accessLevel": "Read"
            },
            "DescribeEndpoints": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeEndpoints.html",
                "description": "Returns information about the endpoints for your account in the current region",
                "accessLevel": "Read"
            },
            "DescribeEventCategories": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeEventCategories.html",
                "description": "Lists categories for all event source types, or, if specified, for a specified source type.",
                "accessLevel": "Read"
            },
            "DescribeEventSubscriptions": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeEventSubscriptions.html",
                "description": "Lists all the event subscriptions for a customer account.",
                "accessLevel": "Read"
            },
            "DescribeEvents": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeEvents.html",
                "description": "Lists events for a given source identifier and source type.",
                "accessLevel": "Read"
            },
            "DescribeOrderableReplicationInstances": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeOrderableReplicationInstances.html",
                "description": "Returns information about the replication instance types that can be created in the specified region",
                "accessLevel": "Read"
            },
            "DescribeRefreshSchemasStatus": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeRefreshSchemasStatus.html",
                "description": "Returns the status of the RefreshSchemas operation",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Endpoint": {
                        "required": true
                    }
                }
            },
            "DescribeReplicationInstanceTaskLogs": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeReplicationInstanceTaskLogs.html",
                "description": "Returns information about the task logs for the specified task.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ReplicationInstance": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DescribeReplicationInstances": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeReplicationInstances.html",
                "description": "Returns information about replication instances for your account in the current region",
                "accessLevel": "Read"
            },
            "DescribeReplicationSubnetGroups": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeReplicationSubnetGroups.html",
                "description": "Returns information about the replication subnet groups",
                "accessLevel": "Read"
            },
            "DescribeReplicationTaskAssessmentResults": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeReplicationTaskAssessmentResults.html",
                "description": "Returns the task assessment results from Amazon S3. This action always returns the latest results.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ReplicationTask": {
                        "required": false
                    }
                }
            },
            "DescribeReplicationTasks": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeReplicationTasks.html",
                "description": "Returns information about replication tasks for your account in the current region",
                "accessLevel": "Read"
            },
            "DescribeSchemas": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeSchemas.html",
                "description": "Returns information about the schema for the specified endpoint",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Endpoint": {
                        "required": true
                    }
                }
            },
            "DescribeTableStatistics": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeTableStatistics.html",
                "description": "Returns table statistics on the database migration task, including table name, rows inserted, rows updated, and rows deleted",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ReplicationTask": {
                        "required": true
                    }
                }
            },
            "ImportCertificate": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_ImportCertificate.html",
                "description": "Uploads the specified certificate.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_ListTagsForResource.html",
                "description": "Lists all tags for an AWS DMS resource",
                "accessLevel": "List",
                "resourceTypes": {
                    "Certificate": {
                        "required": false
                    },
                    "Endpoint": {
                        "required": false
                    },
                    "EventSubscription": {
                        "required": false
                    },
                    "ReplicationInstance": {
                        "required": false
                    },
                    "ReplicationSubnetGroup": {
                        "required": false
                    },
                    "ReplicationTask": {
                        "required": false
                    }
                }
            },
            "ModifyEndpoint": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyEndpoint.html",
                "description": "Modifies the specified endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Endpoint": {
                        "required": true
                    },
                    "Certificate": {
                        "required": false
                    }
                }
            },
            "ModifyEventSubscription": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyEventSubscription.html",
                "description": "Modifies an existing AWS DMS event notification subscription.",
                "accessLevel": "Write"
            },
            "ModifyReplicationInstance": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyReplicationInstance.html",
                "description": "Modifies the replication instance to apply new settings",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ReplicationInstance": {
                        "required": true
                    }
                }
            },
            "ModifyReplicationSubnetGroup": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyReplicationSubnetGroup.html",
                "description": "Modifies the settings for the specified replication subnet group",
                "accessLevel": "Write"
            },
            "ModifyReplicationTask": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyReplicationTask.html",
                "description": "Modifies the specified replication task.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ReplicationTask": {
                        "required": true
                    }
                }
            },
            "RebootReplicationInstance": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_RebootReplicationInstance.html",
                "description": "Reboots a replication instance. Rebooting results in a momentary outage, until the replication instance becomes available again.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ReplicationInstance": {
                        "required": true
                    }
                }
            },
            "RefreshSchemas": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_RefreshSchemas.html",
                "description": "Populates the schema for the specified endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Endpoint": {
                        "required": true
                    },
                    "ReplicationInstance": {
                        "required": true
                    }
                }
            },
            "ReloadTables": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_ReloadTables.html",
                "description": "Reloads the target database table with the source data.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ReplicationTask": {
                        "required": true
                    }
                }
            },
            "RemoveTagsFromResource": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_RemoveTagsFromResource.html",
                "description": "Removes metadata tags from a DMS resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "Certificate": {
                        "required": false
                    },
                    "Endpoint": {
                        "required": false
                    },
                    "EventSubscription": {
                        "required": false
                    },
                    "ReplicationInstance": {
                        "required": false
                    },
                    "ReplicationSubnetGroup": {
                        "required": false
                    },
                    "ReplicationTask": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "StartReplicationTask": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_StartReplicationTask.html",
                "description": "Starts the replication task",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ReplicationTask": {
                        "required": true
                    }
                }
            },
            "StartReplicationTaskAssessment": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_StartReplicationTaskAssessment.html",
                "description": "Starts the replication task assessment for unsupported data types in the source database.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ReplicationTask": {
                        "required": true
                    }
                }
            },
            "StopReplicationTask": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_StopReplicationTask.html",
                "description": "Stops the replication task",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ReplicationTask": {
                        "required": true
                    }
                }
            },
            "TestConnection": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_TestConnection.html",
                "description": "Tests the connection between the replication instance and the endpoint",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Endpoint": {
                        "required": true
                    },
                    "ReplicationInstance": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "ReplicationInstance": {
                "name": "ReplicationInstance",
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_ReplicationInstance.html",
                "arn": "arn:${Partition}:dms:${Region}:${Account}:rep:${ResourceName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "dms:rep-tag/${TagKey}"
                ]
            },
            "ReplicationTask": {
                "name": "ReplicationTask",
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_ReplicationTask.html",
                "arn": "arn:${Partition}:dms:${Region}:${Account}:task:${ResourceName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "dms:task-tag/${TagKey}"
                ]
            },
            "Endpoint": {
                "name": "Endpoint",
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_Endpoint.html",
                "arn": "arn:${Partition}:dms:${Region}:${Account}:endpoint:${ResourceName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "dms:endpoint-tag/${TagKey}"
                ]
            },
            "Certificate": {
                "name": "Certificate",
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_Certificate.html",
                "arn": "arn:${Partition}:dms:${Region}:${Account}:cert:${ResourceName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "dms:cert-tag/${TagKey}"
                ]
            },
            "EventSubscription": {
                "name": "EventSubscription",
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_EventSubscription.html",
                "arn": "arn:${Partition}:dms:${Region}:${Account}:es:${ResourceName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "dms:es-tag/${TagKey}"
                ]
            },
            "ReplicationSubnetGroup": {
                "name": "ReplicationSubnetGroup",
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_ReplicationSubnetGroup.html",
                "arn": "arn:${Partition}:dms:${Region}:${Account}:subgrp:${ResourceName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "dms:subgrp-tag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Adds metadata tags to a DMS resource, including replication instance, endpoint, security group, and migration task
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_AddTagsToResource.html
     */
    addTagsToResource() {
        this.add('dms:AddTagsToResource');
        return this;
    }
    /**
     * Applies a pending maintenance action to a resource (for example, to a replication instance).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ApplyPendingMaintenanceAction.html
     */
    applyPendingMaintenanceAction() {
        this.add('dms:ApplyPendingMaintenanceAction');
        return this;
    }
    /**
     * Creates an endpoint using the provided settings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateEndpoint.html
     */
    createEndpoint() {
        this.add('dms:CreateEndpoint');
        return this;
    }
    /**
     * Creates an AWS DMS event notification subscription.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateEventSubscription.html
     */
    createEventSubscription() {
        this.add('dms:CreateEventSubscription');
        return this;
    }
    /**
     * Creates the replication instance using the specified parameters
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateReplicationInstance.html
     */
    createReplicationInstance() {
        this.add('dms:CreateReplicationInstance');
        return this;
    }
    /**
     * Creates a replication subnet group given a list of the subnet IDs in a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateReplicationSubnetGroup.html
     */
    createReplicationSubnetGroup() {
        this.add('dms:CreateReplicationSubnetGroup');
        return this;
    }
    /**
     * Creates a replication task using the specified parameters
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateReplicationTask.html
     */
    createReplicationTask() {
        this.add('dms:CreateReplicationTask');
        return this;
    }
    /**
     * Deletes the specified certificate
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteCertificate.html
     */
    deleteCertificate() {
        this.add('dms:DeleteCertificate');
        return this;
    }
    /**
     * Deletes the specified endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteEndpoint.html
     */
    deleteEndpoint() {
        this.add('dms:DeleteEndpoint');
        return this;
    }
    /**
     * Deletes an AWS DMS event subscription.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteEventSubscription.html
     */
    deleteEventSubscription() {
        this.add('dms:DeleteEventSubscription');
        return this;
    }
    /**
     * Deletes the specified replication instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteReplicationInstance.html
     */
    deleteReplicationInstance() {
        this.add('dms:DeleteReplicationInstance');
        return this;
    }
    /**
     * Deletes a subnet group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteReplicationSubnetGroup.html
     */
    deleteReplicationSubnetGroup() {
        this.add('dms:DeleteReplicationSubnetGroup');
        return this;
    }
    /**
     * Deletes the specified replication task
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteReplicationTask.html
     */
    deleteReplicationTask() {
        this.add('dms:DeleteReplicationTask');
        return this;
    }
    /**
     * Lists all of the AWS DMS attributes for a customer account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeAccountAttributes.html
     */
    describeAccountAttributes() {
        this.add('dms:DescribeAccountAttributes');
        return this;
    }
    /**
     * Provides a description of the certificate.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeCertificates.html
     */
    describeCertificates() {
        this.add('dms:DescribeCertificates');
        return this;
    }
    /**
     * Describes the status of the connections that have been made between the replication instance and an endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeConnections.html
     */
    describeConnections() {
        this.add('dms:DescribeConnections');
        return this;
    }
    /**
     * Returns information about the type of endpoints available
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeEndpointTypes.html
     */
    describeEndpointTypes() {
        this.add('dms:DescribeEndpointTypes');
        return this;
    }
    /**
     * Returns information about the endpoints for your account in the current region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeEndpoints.html
     */
    describeEndpoints() {
        this.add('dms:DescribeEndpoints');
        return this;
    }
    /**
     * Lists categories for all event source types, or, if specified, for a specified source type.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeEventCategories.html
     */
    describeEventCategories() {
        this.add('dms:DescribeEventCategories');
        return this;
    }
    /**
     * Lists all the event subscriptions for a customer account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeEventSubscriptions.html
     */
    describeEventSubscriptions() {
        this.add('dms:DescribeEventSubscriptions');
        return this;
    }
    /**
     * Lists events for a given source identifier and source type.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeEvents.html
     */
    describeEvents() {
        this.add('dms:DescribeEvents');
        return this;
    }
    /**
     * Returns information about the replication instance types that can be created in the specified region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeOrderableReplicationInstances.html
     */
    describeOrderableReplicationInstances() {
        this.add('dms:DescribeOrderableReplicationInstances');
        return this;
    }
    /**
     * Returns the status of the RefreshSchemas operation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeRefreshSchemasStatus.html
     */
    describeRefreshSchemasStatus() {
        this.add('dms:DescribeRefreshSchemasStatus');
        return this;
    }
    /**
     * Returns information about the task logs for the specified task.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeReplicationInstanceTaskLogs.html
     */
    describeReplicationInstanceTaskLogs() {
        this.add('dms:DescribeReplicationInstanceTaskLogs');
        return this;
    }
    /**
     * Returns information about replication instances for your account in the current region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeReplicationInstances.html
     */
    describeReplicationInstances() {
        this.add('dms:DescribeReplicationInstances');
        return this;
    }
    /**
     * Returns information about the replication subnet groups
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeReplicationSubnetGroups.html
     */
    describeReplicationSubnetGroups() {
        this.add('dms:DescribeReplicationSubnetGroups');
        return this;
    }
    /**
     * Returns the task assessment results from Amazon S3. This action always returns the latest results.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeReplicationTaskAssessmentResults.html
     */
    describeReplicationTaskAssessmentResults() {
        this.add('dms:DescribeReplicationTaskAssessmentResults');
        return this;
    }
    /**
     * Returns information about replication tasks for your account in the current region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeReplicationTasks.html
     */
    describeReplicationTasks() {
        this.add('dms:DescribeReplicationTasks');
        return this;
    }
    /**
     * Returns information about the schema for the specified endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeSchemas.html
     */
    describeSchemas() {
        this.add('dms:DescribeSchemas');
        return this;
    }
    /**
     * Returns table statistics on the database migration task, including table name, rows inserted, rows updated, and rows deleted
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeTableStatistics.html
     */
    describeTableStatistics() {
        this.add('dms:DescribeTableStatistics');
        return this;
    }
    /**
     * Uploads the specified certificate.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ImportCertificate.html
     */
    importCertificate() {
        this.add('dms:ImportCertificate');
        return this;
    }
    /**
     * Lists all tags for an AWS DMS resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('dms:ListTagsForResource');
        return this;
    }
    /**
     * Modifies the specified endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyEndpoint.html
     */
    modifyEndpoint() {
        this.add('dms:ModifyEndpoint');
        return this;
    }
    /**
     * Modifies an existing AWS DMS event notification subscription.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyEventSubscription.html
     */
    modifyEventSubscription() {
        this.add('dms:ModifyEventSubscription');
        return this;
    }
    /**
     * Modifies the replication instance to apply new settings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyReplicationInstance.html
     */
    modifyReplicationInstance() {
        this.add('dms:ModifyReplicationInstance');
        return this;
    }
    /**
     * Modifies the settings for the specified replication subnet group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyReplicationSubnetGroup.html
     */
    modifyReplicationSubnetGroup() {
        this.add('dms:ModifyReplicationSubnetGroup');
        return this;
    }
    /**
     * Modifies the specified replication task.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyReplicationTask.html
     */
    modifyReplicationTask() {
        this.add('dms:ModifyReplicationTask');
        return this;
    }
    /**
     * Reboots a replication instance. Rebooting results in a momentary outage, until the replication instance becomes available again.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_RebootReplicationInstance.html
     */
    rebootReplicationInstance() {
        this.add('dms:RebootReplicationInstance');
        return this;
    }
    /**
     * Populates the schema for the specified endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_RefreshSchemas.html
     */
    refreshSchemas() {
        this.add('dms:RefreshSchemas');
        return this;
    }
    /**
     * Reloads the target database table with the source data.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ReloadTables.html
     */
    reloadTables() {
        this.add('dms:ReloadTables');
        return this;
    }
    /**
     * Removes metadata tags from a DMS resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_RemoveTagsFromResource.html
     */
    removeTagsFromResource() {
        this.add('dms:RemoveTagsFromResource');
        return this;
    }
    /**
     * Starts the replication task
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_StartReplicationTask.html
     */
    startReplicationTask() {
        this.add('dms:StartReplicationTask');
        return this;
    }
    /**
     * Starts the replication task assessment for unsupported data types in the source database.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_StartReplicationTaskAssessment.html
     */
    startReplicationTaskAssessment() {
        this.add('dms:StartReplicationTaskAssessment');
        return this;
    }
    /**
     * Stops the replication task
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_StopReplicationTask.html
     */
    stopReplicationTask() {
        this.add('dms:StopReplicationTask');
        return this;
    }
    /**
     * Tests the connection between the replication instance and the endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_TestConnection.html
     */
    testConnection() {
        this.add('dms:TestConnection');
        return this;
    }
    /**
     * Adds a resource of type ReplicationInstance to the statement
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ReplicationInstance.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     *  - dms:rep-tag/${TagKey}
     */
    onReplicationInstance(resourceName, account, region, partition) {
        var arn = 'arn:${Partition}:dms:${Region}:${Account}:rep:${ResourceName}';
        arn = arn.replace('${ResourceName}', resourceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type ReplicationTask to the statement
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ReplicationTask.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     *  - dms:task-tag/${TagKey}
     */
    onReplicationTask(resourceName, account, region, partition) {
        var arn = 'arn:${Partition}:dms:${Region}:${Account}:task:${ResourceName}';
        arn = arn.replace('${ResourceName}', resourceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Endpoint to the statement
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_Endpoint.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     *  - dms:endpoint-tag/${TagKey}
     */
    onEndpoint(resourceName, account, region, partition) {
        var arn = 'arn:${Partition}:dms:${Region}:${Account}:endpoint:${ResourceName}';
        arn = arn.replace('${ResourceName}', resourceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Certificate to the statement
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_Certificate.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     *  - dms:cert-tag/${TagKey}
     */
    onCertificate(resourceName, account, region, partition) {
        var arn = 'arn:${Partition}:dms:${Region}:${Account}:cert:${ResourceName}';
        arn = arn.replace('${ResourceName}', resourceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type EventSubscription to the statement
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_EventSubscription.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     *  - dms:es-tag/${TagKey}
     */
    onEventSubscription(resourceName, account, region, partition) {
        var arn = 'arn:${Partition}:dms:${Region}:${Account}:es:${ResourceName}';
        arn = arn.replace('${ResourceName}', resourceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type ReplicationSubnetGroup to the statement
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ReplicationSubnetGroup.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     *  - dms:subgrp-tag/${TagKey}
     */
    onReplicationSubnetGroup(resourceName, account, region, partition) {
        var arn = 'arn:${Partition}:dms:${Region}:${Account}:subgrp:${ResourceName}';
        arn = arn.replace('${ResourceName}', resourceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters actions based on the presence of tag keys in the request for Certificate
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCertTag(tagKey, value, operator) {
        return this.if(`dms:cert-tag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the presence of tag keys in the request for Endpoint
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEndpointTag(tagKey, value, operator) {
        return this.if(`dms:endpoint-tag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the presence of tag keys in the request for EventSubscription
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEsTag(tagKey, value, operator) {
        return this.if(`dms:es-tag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the presence of tag keys in the request for ReplicationInstance
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRepTag(tagKey, value, operator) {
        return this.if(`dms:rep-tag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the presence of tag key-value pairs in the request
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifReqTag(tagKey, value, operator) {
        return this.if(`dms:req-tag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the presence of tag keys in the request for ReplicationSubnetGroup
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSubgrpTag(tagKey, value, operator) {
        return this.if(`dms:subgrp-tag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the presence of tag keys in the request for ReplicationTask
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTaskTag(tagKey, value, operator) {
        return this.if(`dms:task-tag/${tagKey}`, value, operator || 'StringLike');
    }
}
exports.Dms = Dms;
//# sourceMappingURL=data:application/json;base64,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