"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service deepracer
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsdeepracer.html
 */
class Deepracer extends shared_1.PolicyStatement {
    /**
     * Action provider for service deepracer
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsdeepracer.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'deepracer';
        this.actions = {
            "CloneReinforcementLearningModel": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-console-train-evaluate-models.html#deepracer-clone-trained-model",
                "description": "Grants permission to clone existing DeepRacer models",
                "accessLevel": "Write",
                "resourceTypes": {
                    "reinforcement_learning_model": {
                        "required": true
                    },
                    "track": {
                        "required": true
                    }
                }
            },
            "CreateAccountResources": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html",
                "description": "Grants permission to create resources needed by DeepRacer on behalf of the user",
                "accessLevel": "Write"
            },
            "CreateLeaderboardSubmission": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html",
                "description": "Grants permission to submit DeepRacer models to be evaluated for leaderboards",
                "accessLevel": "Write",
                "resourceTypes": {
                    "leaderboard": {
                        "required": true
                    },
                    "reinforcement_learning_model": {
                        "required": true
                    }
                }
            },
            "CreateReinforcementLearningModel": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html",
                "description": "Grants permission to create reinforcement learning models for DeepRacer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "track": {
                        "required": true
                    }
                }
            },
            "DeleteAccountResources": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html",
                "description": "Grants permission to delete resources created by DeepRacer on behalf of the user",
                "accessLevel": "Write"
            },
            "DeleteModel": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html",
                "description": "Grants permission to delete DeepRacer models",
                "accessLevel": "Write",
                "resourceTypes": {
                    "reinforcement_learning_model": {
                        "required": true
                    }
                }
            },
            "GetAccountResources": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html",
                "description": "Grants permission to retrieve the resources created by DeepRacer on behalf of the user",
                "accessLevel": "Read"
            },
            "GetAlias": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html",
                "description": "Grants permission to retrieve the user's alias for submitting DeepRacer models to leaderboards",
                "accessLevel": "Read"
            },
            "GetEvaluation": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-test-in-simulator.html",
                "description": "Grants permission to retrieve information about existing DeepRacer models' evaluation jobs",
                "accessLevel": "Read",
                "resourceTypes": {
                    "evaluation_job": {
                        "required": true
                    }
                }
            },
            "GetLatestUserSubmission": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html",
                "description": "Grants permission to retrieve information about how the latest submitted DeepRacer model for a user performed on a leaderboard",
                "accessLevel": "Read",
                "resourceTypes": {
                    "leaderboard": {
                        "required": true
                    }
                }
            },
            "GetLeaderboard": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html",
                "description": "Grants permission to retrieve information about leaderboards",
                "accessLevel": "Read",
                "resourceTypes": {
                    "leaderboard": {
                        "required": true
                    }
                }
            },
            "GetModel": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html",
                "description": "Grants permission to retrieve information about existing DeepRacer models",
                "accessLevel": "Read",
                "resourceTypes": {
                    "reinforcement_learning_model": {
                        "required": true
                    }
                }
            },
            "GetRankedUserSubmission": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html",
                "description": "Grants permission to retrieve information about the performance of a user's DeepRacer model that got placed on a leaderboard",
                "accessLevel": "Read",
                "resourceTypes": {
                    "leaderboard": {
                        "required": true
                    }
                }
            },
            "GetTrack": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-console-train-evaluate-models.html",
                "description": "Grants permission to retrieve information about DeepRacer tracks",
                "accessLevel": "Read",
                "resourceTypes": {
                    "track": {
                        "required": true
                    }
                }
            },
            "GetTrainingJob": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html",
                "description": "Grants permission to retrieve information about existing DeepRacer models' training job",
                "accessLevel": "Read",
                "resourceTypes": {
                    "training_job": {
                        "required": true
                    }
                }
            },
            "ListEvaluations": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-test-in-simulator.html",
                "description": "Grants permission to list DeepRacer models' evaluation jobs",
                "accessLevel": "List",
                "resourceTypes": {
                    "reinforcement_learning_model": {
                        "required": true
                    }
                }
            },
            "ListLeaderboardSubmissions": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html",
                "description": "Grants permission to list all the submissions of DeepRacer models of a user on a leaderboard",
                "accessLevel": "List",
                "resourceTypes": {
                    "leaderboard": {
                        "required": true
                    }
                }
            },
            "ListLeaderboards": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html",
                "description": "Grants permission to list all the available leaderboards",
                "accessLevel": "List"
            },
            "ListModels": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html",
                "description": "Grants permission to list all existing DeepRacer models",
                "accessLevel": "List"
            },
            "ListTracks": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-console-train-evaluate-models.html",
                "description": "Grants permission to list all DeepRacer tracks",
                "accessLevel": "List"
            },
            "ListTrainingJobs": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html",
                "description": "Grants permission to list DeepRacer models' training jobs",
                "accessLevel": "List",
                "resourceTypes": {
                    "reinforcement_learning_model": {
                        "required": true
                    }
                }
            },
            "SetAlias": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html",
                "description": "Grants permission to set the user's alias for submitting DeepRacer models to leaderboards",
                "accessLevel": "Write"
            },
            "StartEvaluation": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-test-in-simulator.html",
                "description": "Grants permission to evaluate DeepRacer models in a simulated environment",
                "accessLevel": "Write",
                "resourceTypes": {
                    "reinforcement_learning_model": {
                        "required": true
                    },
                    "track": {
                        "required": true
                    }
                }
            },
            "StopEvaluation": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-test-in-simulator.html",
                "description": "Grants permission to stop DeepRacer model evaluations",
                "accessLevel": "Write",
                "resourceTypes": {
                    "evaluation_job": {
                        "required": true
                    }
                }
            },
            "StopTrainingReinforcementLearningModel": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html",
                "description": "Grants permission to stop training DeepRacer models",
                "accessLevel": "Write",
                "resourceTypes": {
                    "reinforcement_learning_model": {
                        "required": true
                    }
                }
            },
            "TestRewardFunction": {
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-console-train-evaluate-models.html#deepracer-train-models-define-reward-function",
                "description": "Grants permission to test reward functions for correctness",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "reinforcement_learning_model": {
                "name": "reinforcement_learning_model",
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html",
                "arn": "arn:${Partition}:deepracer:${Region}:${Account}:model/reinforcement_learning/${ResourceId}",
                "conditionKeys": []
            },
            "training_job": {
                "name": "training_job",
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html",
                "arn": "arn:${Partition}:deepracer:${Region}:${Account}:training_job/${ResourceId}",
                "conditionKeys": []
            },
            "evaluation_job": {
                "name": "evaluation_job",
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-test-in-simulator.html",
                "arn": "arn:${Partition}:deepracer:${Region}:${Account}:evaluation_job/${ResourceId}",
                "conditionKeys": []
            },
            "leaderboard_evaluation_job": {
                "name": "leaderboard_evaluation_job",
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html",
                "arn": "arn:${Partition}:deepracer:${Region}:${Account}:leaderboard_evaluation_job/${ResourceId}",
                "conditionKeys": []
            },
            "track": {
                "name": "track",
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-console-train-evaluate-models.html",
                "arn": "arn:${Partition}:deepracer:${Region}::track/${ResourceId}",
                "conditionKeys": []
            },
            "leaderboard": {
                "name": "leaderboard",
                "url": "https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html",
                "arn": "arn:${Partition}:deepracer:${Region}::leaderboard/${ResourceId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to clone existing DeepRacer models
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-console-train-evaluate-models.html#deepracer-clone-trained-model
     */
    cloneReinforcementLearningModel() {
        this.add('deepracer:CloneReinforcementLearningModel');
        return this;
    }
    /**
     * Grants permission to create resources needed by DeepRacer on behalf of the user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     */
    createAccountResources() {
        this.add('deepracer:CreateAccountResources');
        return this;
    }
    /**
     * Grants permission to submit DeepRacer models to be evaluated for leaderboards
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html
     */
    createLeaderboardSubmission() {
        this.add('deepracer:CreateLeaderboardSubmission');
        return this;
    }
    /**
     * Grants permission to create reinforcement learning models for DeepRacer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     */
    createReinforcementLearningModel() {
        this.add('deepracer:CreateReinforcementLearningModel');
        return this;
    }
    /**
     * Grants permission to delete resources created by DeepRacer on behalf of the user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     */
    deleteAccountResources() {
        this.add('deepracer:DeleteAccountResources');
        return this;
    }
    /**
     * Grants permission to delete DeepRacer models
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     */
    deleteModel() {
        this.add('deepracer:DeleteModel');
        return this;
    }
    /**
     * Grants permission to retrieve the resources created by DeepRacer on behalf of the user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     */
    getAccountResources() {
        this.add('deepracer:GetAccountResources');
        return this;
    }
    /**
     * Grants permission to retrieve the user's alias for submitting DeepRacer models to leaderboards
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html
     */
    getAlias() {
        this.add('deepracer:GetAlias');
        return this;
    }
    /**
     * Grants permission to retrieve information about existing DeepRacer models' evaluation jobs
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-test-in-simulator.html
     */
    getEvaluation() {
        this.add('deepracer:GetEvaluation');
        return this;
    }
    /**
     * Grants permission to retrieve information about how the latest submitted DeepRacer model for a user performed on a leaderboard
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html
     */
    getLatestUserSubmission() {
        this.add('deepracer:GetLatestUserSubmission');
        return this;
    }
    /**
     * Grants permission to retrieve information about leaderboards
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html
     */
    getLeaderboard() {
        this.add('deepracer:GetLeaderboard');
        return this;
    }
    /**
     * Grants permission to retrieve information about existing DeepRacer models
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     */
    getModel() {
        this.add('deepracer:GetModel');
        return this;
    }
    /**
     * Grants permission to retrieve information about the performance of a user's DeepRacer model that got placed on a leaderboard
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html
     */
    getRankedUserSubmission() {
        this.add('deepracer:GetRankedUserSubmission');
        return this;
    }
    /**
     * Grants permission to retrieve information about DeepRacer tracks
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-console-train-evaluate-models.html
     */
    getTrack() {
        this.add('deepracer:GetTrack');
        return this;
    }
    /**
     * Grants permission to retrieve information about existing DeepRacer models' training job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     */
    getTrainingJob() {
        this.add('deepracer:GetTrainingJob');
        return this;
    }
    /**
     * Grants permission to list DeepRacer models' evaluation jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-test-in-simulator.html
     */
    listEvaluations() {
        this.add('deepracer:ListEvaluations');
        return this;
    }
    /**
     * Grants permission to list all the submissions of DeepRacer models of a user on a leaderboard
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html
     */
    listLeaderboardSubmissions() {
        this.add('deepracer:ListLeaderboardSubmissions');
        return this;
    }
    /**
     * Grants permission to list all the available leaderboards
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html
     */
    listLeaderboards() {
        this.add('deepracer:ListLeaderboards');
        return this;
    }
    /**
     * Grants permission to list all existing DeepRacer models
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     */
    listModels() {
        this.add('deepracer:ListModels');
        return this;
    }
    /**
     * Grants permission to list all DeepRacer tracks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-console-train-evaluate-models.html
     */
    listTracks() {
        this.add('deepracer:ListTracks');
        return this;
    }
    /**
     * Grants permission to list DeepRacer models' training jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     */
    listTrainingJobs() {
        this.add('deepracer:ListTrainingJobs');
        return this;
    }
    /**
     * Grants permission to set the user's alias for submitting DeepRacer models to leaderboards
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html
     */
    setAlias() {
        this.add('deepracer:SetAlias');
        return this;
    }
    /**
     * Grants permission to evaluate DeepRacer models in a simulated environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-test-in-simulator.html
     */
    startEvaluation() {
        this.add('deepracer:StartEvaluation');
        return this;
    }
    /**
     * Grants permission to stop DeepRacer model evaluations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-test-in-simulator.html
     */
    stopEvaluation() {
        this.add('deepracer:StopEvaluation');
        return this;
    }
    /**
     * Grants permission to stop training DeepRacer models
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     */
    stopTrainingReinforcementLearningModel() {
        this.add('deepracer:StopTrainingReinforcementLearningModel');
        return this;
    }
    /**
     * Grants permission to test reward functions for correctness
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-console-train-evaluate-models.html#deepracer-train-models-define-reward-function
     */
    testRewardFunction() {
        this.add('deepracer:TestRewardFunction');
        return this;
    }
    /**
     * Adds a resource of type reinforcement_learning_model to the statement
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onReinforcementLearningModel(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:deepracer:${Region}:${Account}:model/reinforcement_learning/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type training_job to the statement
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTrainingJob(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:deepracer:${Region}:${Account}:training_job/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type evaluation_job to the statement
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-test-in-simulator.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onEvaluationJob(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:deepracer:${Region}:${Account}:evaluation_job/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type leaderboard_evaluation_job to the statement
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onLeaderboardEvaluationJob(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:deepracer:${Region}:${Account}:leaderboard_evaluation_job/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type track to the statement
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-console-train-evaluate-models.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTrack(resourceId, region, partition) {
        var arn = 'arn:${Partition}:deepracer:${Region}::track/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type leaderboard to the statement
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onLeaderboard(resourceId, region, partition) {
        var arn = 'arn:${Partition}:deepracer:${Region}::leaderboard/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Deepracer = Deepracer;
//# sourceMappingURL=data:application/json;base64,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