"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service ds
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsdirectoryservice.html
 */
class Ds extends shared_1.PolicyStatement {
    /**
     * Action provider for service ds
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsdirectoryservice.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'ds';
        this.actions = {
            "AcceptSharedDirectory": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_AcceptSharedDirectory.html",
                "description": "Accepts a directory sharing request that was sent from the directory owner account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "AddIpRoutes": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_AddIpRoutes.html",
                "description": "Adds a CIDR address block to correctly route traffic to and from your Microsoft AD on Amazon Web Services",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "AddTagsToResource": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_AddTagsToResource.html",
                "description": "Adds or overwrites one or more tags for the specified Amazon Directory Services directory.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "AuthorizeApplication": {
                "url": "",
                "description": "Authorizes an application for your AWS Directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "CancelSchemaExtension": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CancelSchemaExtension.html",
                "description": "Cancels an in-progress schema extension to a Microsoft AD directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "CheckAlias": {
                "url": "",
                "description": "Verifies that the alias is available for use.",
                "accessLevel": "Read"
            },
            "ConnectDirectory": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ConnectDirectory.html",
                "description": "Creates an AD Connector to connect to an on-premises directory.",
                "accessLevel": "Tagging",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateAlias": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateAlias.html",
                "description": "Creates an alias for a directory and assigns the alias to the directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "CreateComputer": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateComputer.html",
                "description": "Creates a computer account in the specified directory, and joins the computer to the directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "CreateConditionalForwarder": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateConditionalForwarder.html",
                "description": "Creates a conditional forwarder associated with your AWS directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "CreateDirectory": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateDirectory.html",
                "description": "Creates a Simple AD directory.",
                "accessLevel": "Tagging",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateIdentityPoolDirectory": {
                "url": "",
                "description": "Creates a IdentityPool Directory in the AWS cloud.",
                "accessLevel": "Tagging",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateLogSubscription": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateLogSubscription.html",
                "description": "Creates a subscription to forward real time Directory Service domain controller security logs to the specified CloudWatch log group in your AWS account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "CreateMicrosoftAD": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateMicrosoftAD.html",
                "description": "Creates a Microsoft AD in the AWS cloud.",
                "accessLevel": "Tagging",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateSnapshot": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateSnapshot.html",
                "description": "Creates a snapshot of a Simple AD or Microsoft AD directory in the AWS cloud.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "CreateTrust": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateTrust.html",
                "description": "Initiates the creation of the AWS side of a trust relationship between a Microsoft AD in the AWS cloud and an external domain.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DeleteConditionalForwarder": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeleteConditionalForwarder.html",
                "description": "Deletes a conditional forwarder that has been set up for your AWS directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DeleteDirectory": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeleteDirectory.html",
                "description": "Deletes an AWS Directory Service directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DeleteLogSubscription": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeleteLogSubscription.html",
                "description": "Deletes the specified log subscription.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DeleteSnapshot": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeleteSnapshot.html",
                "description": "Deletes a directory snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DeleteTrust": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/DeleteTrust.html",
                "description": "Deletes an existing trust relationship between your Microsoft AD in the AWS cloud and an external domain.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DeregisterCertificate": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeregisterCertificate.html",
                "description": "Deletes from the system the certificate that was registered for a secured LDAP connection.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DeregisterEventTopic": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeregisterEventTopic.html",
                "description": "Removes the specified directory as a publisher to the specified SNS topic.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DescribeCertificate": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeCertificate.html",
                "description": "Displays information about the certificate registered for a secured LDAP connection.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DescribeConditionalForwarders": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeConditionalForwarders.html",
                "description": "Obtains information about the conditional forwarders for this account.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DescribeDirectories": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeDirectories.html",
                "description": "Obtains information about the directories that belong to this account.",
                "accessLevel": "List"
            },
            "DescribeDomainControllers": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeDomainControllers.html",
                "description": "Provides information about any domain controllers in your directory.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DescribeEventTopics": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeEventTopics.html",
                "description": "Obtains information about which SNS topics receive status messages from the specified directory.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DescribeLDAPSSettings": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeLDAPSSettings.html",
                "description": "Describes the status of LDAP security for the specified directory.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DescribeSharedDirectories": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeSharedDirectories.html",
                "description": "Returns the shared directories in your account.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DescribeSnapshots": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeSnapshots.html",
                "description": "Obtains information about the directory snapshots that belong to this account.",
                "accessLevel": "Read"
            },
            "DescribeTrusts": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeTrusts.html",
                "description": "Obtains information about the trust relationships for this account.",
                "accessLevel": "Read"
            },
            "DisableLDAPS": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DisableLDAPS.html",
                "description": "Deactivates LDAP secure calls for the specified directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DisableRadius": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DisableRadius.html",
                "description": "Disables multi-factor authentication (MFA) with the Remote Authentication Dial In User Service (RADIUS) server for an AD Connector directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DisableSso": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DisableSso.html",
                "description": "Disables single-sign on for a directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "EnableLDAPS": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_EnableLDAPS.html",
                "description": "Activates the switch for the specific directory to always use LDAP secure calls.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "EnableRadius": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_EnableRadius.html",
                "description": "Enables multi-factor authentication (MFA) with the Remote Authentication Dial In User Service (RADIUS) server for an AD Connector directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "EnableSso": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_EnableSso.html",
                "description": "Enables single-sign on for a directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "GetAuthorizedApplicationDetails": {
                "url": "",
                "description": "",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "GetDirectoryLimits": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_GetDirectoryLimits.html",
                "description": "Obtains directory limit information for the current region.",
                "accessLevel": "Read"
            },
            "GetSnapshotLimits": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_GetSnapshotLimits.html",
                "description": "Obtains the manual snapshot limits for a directory.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "ListAuthorizedApplications": {
                "url": "",
                "description": "Obtains the aws applications authorized for a directory.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "ListCertificates": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListCertificates.html",
                "description": "For the specified directory, lists all the certificates registered for a secured LDAP connection.",
                "accessLevel": "List",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "ListIpRoutes": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListIpRoutes.html",
                "description": "Lists the address blocks that you have added to a directory.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "ListLogSubscriptions": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListLogSubscriptions.html",
                "description": "Lists the active log subscriptions for the AWS account.",
                "accessLevel": "Read"
            },
            "ListSchemaExtensions": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListSchemaExtensions.html",
                "description": "Lists all schema extensions applied to a Microsoft AD Directory.",
                "accessLevel": "List",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListTagsForResource.html",
                "description": "Lists all tags on an Amazon Directory Services directory.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "RegisterCertificate": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RegisterCertificate.html",
                "description": "Registers a certificate for secured LDAP connection.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "RegisterEventTopic": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RegisterEventTopic.html",
                "description": "Associates a directory with an SNS topic.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "RejectSharedDirectory": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RejectSharedDirectory.html",
                "description": "Rejects a directory sharing request that was sent from the directory owner account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "RemoveIpRoutes": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RemoveIpRoutes.html",
                "description": "Removes IP address blocks from a directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "RemoveTagsFromResource": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RemoveTagsFromResource.html",
                "description": "Removes tags from an Amazon Directory Services directory.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "ResetUserPassword": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ResetUserPassword.html",
                "description": "Resets the password for any user in your AWS Managed Microsoft AD or Simple AD directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "RestoreFromSnapshot": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RestoreFromSnapshot.html",
                "description": "Restores a directory using an existing directory snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "ShareDirectory": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ShareDirectory.html",
                "description": "Shares a specified directory in your AWS account (directory owner) with another AWS account (directory consumer). With this operation you can use your directory from any AWS account and from any Amazon VPC within an AWS Region.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "StartSchemaExtension": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_StartSchemaExtension.html",
                "description": "Applies a schema extension to a Microsoft AD directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "UnauthorizeApplication": {
                "url": "",
                "description": "Unauthorizes an application from your AWS Directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "UnshareDirectory": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UnshareDirectory.html",
                "description": "Stops the directory sharing between the directory owner and consumer accounts.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "UpdateConditionalForwarder": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UpdateConditionalForwarder.html",
                "description": "Updates a conditional forwarder that has been set up for your AWS directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "UpdateNumberOfDomainControllers": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UpdateNumberOfDomainControllers.html",
                "description": "Adds or removes domain controllers to or from the directory. Based on the difference between current value and new value (provided through this API call), domain controllers will be added or removed. It may take up to 45 minutes for any new domain controllers to become fully active once the requested number of domain controllers is updated. During this time, you cannot make another update request.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "UpdateRadius": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UpdateRadius.html",
                "description": "Updates the Remote Authentication Dial In User Service (RADIUS) server information for an AD Connector directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "UpdateTrust": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UpdateTrust.html",
                "description": "Updates the trust that has been set up between your AWS Managed Microsoft AD directory and an on-premises Active Directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "VerifyTrust": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_VerifyTrust.html",
                "description": "Verifies a trust relationship between your Microsoft AD in the AWS cloud and an external domain.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "directory": {
                "name": "directory",
                "url": "",
                "arn": "arn:${Partition}:ds:${Region}:${Account}:directory/${DirectoryId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Accepts a directory sharing request that was sent from the directory owner account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_AcceptSharedDirectory.html
     */
    acceptSharedDirectory() {
        this.add('ds:AcceptSharedDirectory');
        return this;
    }
    /**
     * Adds a CIDR address block to correctly route traffic to and from your Microsoft AD on Amazon Web Services
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_AddIpRoutes.html
     */
    addIpRoutes() {
        this.add('ds:AddIpRoutes');
        return this;
    }
    /**
     * Adds or overwrites one or more tags for the specified Amazon Directory Services directory.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_AddTagsToResource.html
     */
    addTagsToResource() {
        this.add('ds:AddTagsToResource');
        return this;
    }
    /**
     * Authorizes an application for your AWS Directory.
     *
     * Access Level: Write
     */
    authorizeApplication() {
        this.add('ds:AuthorizeApplication');
        return this;
    }
    /**
     * Cancels an in-progress schema extension to a Microsoft AD directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CancelSchemaExtension.html
     */
    cancelSchemaExtension() {
        this.add('ds:CancelSchemaExtension');
        return this;
    }
    /**
     * Verifies that the alias is available for use.
     *
     * Access Level: Read
     */
    checkAlias() {
        this.add('ds:CheckAlias');
        return this;
    }
    /**
     * Creates an AD Connector to connect to an on-premises directory.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ConnectDirectory.html
     */
    connectDirectory() {
        this.add('ds:ConnectDirectory');
        return this;
    }
    /**
     * Creates an alias for a directory and assigns the alias to the directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateAlias.html
     */
    createAlias() {
        this.add('ds:CreateAlias');
        return this;
    }
    /**
     * Creates a computer account in the specified directory, and joins the computer to the directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateComputer.html
     */
    createComputer() {
        this.add('ds:CreateComputer');
        return this;
    }
    /**
     * Creates a conditional forwarder associated with your AWS directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateConditionalForwarder.html
     */
    createConditionalForwarder() {
        this.add('ds:CreateConditionalForwarder');
        return this;
    }
    /**
     * Creates a Simple AD directory.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateDirectory.html
     */
    createDirectory() {
        this.add('ds:CreateDirectory');
        return this;
    }
    /**
     * Creates a IdentityPool Directory in the AWS cloud.
     *
     * Access Level: Tagging
     */
    createIdentityPoolDirectory() {
        this.add('ds:CreateIdentityPoolDirectory');
        return this;
    }
    /**
     * Creates a subscription to forward real time Directory Service domain controller security logs to the specified CloudWatch log group in your AWS account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateLogSubscription.html
     */
    createLogSubscription() {
        this.add('ds:CreateLogSubscription');
        return this;
    }
    /**
     * Creates a Microsoft AD in the AWS cloud.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateMicrosoftAD.html
     */
    createMicrosoftAD() {
        this.add('ds:CreateMicrosoftAD');
        return this;
    }
    /**
     * Creates a snapshot of a Simple AD or Microsoft AD directory in the AWS cloud.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateSnapshot.html
     */
    createSnapshot() {
        this.add('ds:CreateSnapshot');
        return this;
    }
    /**
     * Initiates the creation of the AWS side of a trust relationship between a Microsoft AD in the AWS cloud and an external domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateTrust.html
     */
    createTrust() {
        this.add('ds:CreateTrust');
        return this;
    }
    /**
     * Deletes a conditional forwarder that has been set up for your AWS directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeleteConditionalForwarder.html
     */
    deleteConditionalForwarder() {
        this.add('ds:DeleteConditionalForwarder');
        return this;
    }
    /**
     * Deletes an AWS Directory Service directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeleteDirectory.html
     */
    deleteDirectory() {
        this.add('ds:DeleteDirectory');
        return this;
    }
    /**
     * Deletes the specified log subscription.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeleteLogSubscription.html
     */
    deleteLogSubscription() {
        this.add('ds:DeleteLogSubscription');
        return this;
    }
    /**
     * Deletes a directory snapshot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeleteSnapshot.html
     */
    deleteSnapshot() {
        this.add('ds:DeleteSnapshot');
        return this;
    }
    /**
     * Deletes an existing trust relationship between your Microsoft AD in the AWS cloud and an external domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/DeleteTrust.html
     */
    deleteTrust() {
        this.add('ds:DeleteTrust');
        return this;
    }
    /**
     * Deletes from the system the certificate that was registered for a secured LDAP connection.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeregisterCertificate.html
     */
    deregisterCertificate() {
        this.add('ds:DeregisterCertificate');
        return this;
    }
    /**
     * Removes the specified directory as a publisher to the specified SNS topic.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeregisterEventTopic.html
     */
    deregisterEventTopic() {
        this.add('ds:DeregisterEventTopic');
        return this;
    }
    /**
     * Displays information about the certificate registered for a secured LDAP connection.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeCertificate.html
     */
    describeCertificate() {
        this.add('ds:DescribeCertificate');
        return this;
    }
    /**
     * Obtains information about the conditional forwarders for this account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeConditionalForwarders.html
     */
    describeConditionalForwarders() {
        this.add('ds:DescribeConditionalForwarders');
        return this;
    }
    /**
     * Obtains information about the directories that belong to this account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeDirectories.html
     */
    describeDirectories() {
        this.add('ds:DescribeDirectories');
        return this;
    }
    /**
     * Provides information about any domain controllers in your directory.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeDomainControllers.html
     */
    describeDomainControllers() {
        this.add('ds:DescribeDomainControllers');
        return this;
    }
    /**
     * Obtains information about which SNS topics receive status messages from the specified directory.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeEventTopics.html
     */
    describeEventTopics() {
        this.add('ds:DescribeEventTopics');
        return this;
    }
    /**
     * Describes the status of LDAP security for the specified directory.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeLDAPSSettings.html
     */
    describeLDAPSSettings() {
        this.add('ds:DescribeLDAPSSettings');
        return this;
    }
    /**
     * Returns the shared directories in your account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeSharedDirectories.html
     */
    describeSharedDirectories() {
        this.add('ds:DescribeSharedDirectories');
        return this;
    }
    /**
     * Obtains information about the directory snapshots that belong to this account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeSnapshots.html
     */
    describeSnapshots() {
        this.add('ds:DescribeSnapshots');
        return this;
    }
    /**
     * Obtains information about the trust relationships for this account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeTrusts.html
     */
    describeTrusts() {
        this.add('ds:DescribeTrusts');
        return this;
    }
    /**
     * Deactivates LDAP secure calls for the specified directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DisableLDAPS.html
     */
    disableLDAPS() {
        this.add('ds:DisableLDAPS');
        return this;
    }
    /**
     * Disables multi-factor authentication (MFA) with the Remote Authentication Dial In User Service (RADIUS) server for an AD Connector directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DisableRadius.html
     */
    disableRadius() {
        this.add('ds:DisableRadius');
        return this;
    }
    /**
     * Disables single-sign on for a directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DisableSso.html
     */
    disableSso() {
        this.add('ds:DisableSso');
        return this;
    }
    /**
     * Activates the switch for the specific directory to always use LDAP secure calls.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_EnableLDAPS.html
     */
    enableLDAPS() {
        this.add('ds:EnableLDAPS');
        return this;
    }
    /**
     * Enables multi-factor authentication (MFA) with the Remote Authentication Dial In User Service (RADIUS) server for an AD Connector directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_EnableRadius.html
     */
    enableRadius() {
        this.add('ds:EnableRadius');
        return this;
    }
    /**
     * Enables single-sign on for a directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_EnableSso.html
     */
    enableSso() {
        this.add('ds:EnableSso');
        return this;
    }
    /**
     *
     *
     * Access Level: Read
     */
    getAuthorizedApplicationDetails() {
        this.add('ds:GetAuthorizedApplicationDetails');
        return this;
    }
    /**
     * Obtains directory limit information for the current region.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_GetDirectoryLimits.html
     */
    getDirectoryLimits() {
        this.add('ds:GetDirectoryLimits');
        return this;
    }
    /**
     * Obtains the manual snapshot limits for a directory.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_GetSnapshotLimits.html
     */
    getSnapshotLimits() {
        this.add('ds:GetSnapshotLimits');
        return this;
    }
    /**
     * Obtains the aws applications authorized for a directory.
     *
     * Access Level: Read
     */
    listAuthorizedApplications() {
        this.add('ds:ListAuthorizedApplications');
        return this;
    }
    /**
     * For the specified directory, lists all the certificates registered for a secured LDAP connection.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListCertificates.html
     */
    listCertificates() {
        this.add('ds:ListCertificates');
        return this;
    }
    /**
     * Lists the address blocks that you have added to a directory.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListIpRoutes.html
     */
    listIpRoutes() {
        this.add('ds:ListIpRoutes');
        return this;
    }
    /**
     * Lists the active log subscriptions for the AWS account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListLogSubscriptions.html
     */
    listLogSubscriptions() {
        this.add('ds:ListLogSubscriptions');
        return this;
    }
    /**
     * Lists all schema extensions applied to a Microsoft AD Directory.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListSchemaExtensions.html
     */
    listSchemaExtensions() {
        this.add('ds:ListSchemaExtensions');
        return this;
    }
    /**
     * Lists all tags on an Amazon Directory Services directory.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('ds:ListTagsForResource');
        return this;
    }
    /**
     * Registers a certificate for secured LDAP connection.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RegisterCertificate.html
     */
    registerCertificate() {
        this.add('ds:RegisterCertificate');
        return this;
    }
    /**
     * Associates a directory with an SNS topic.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RegisterEventTopic.html
     */
    registerEventTopic() {
        this.add('ds:RegisterEventTopic');
        return this;
    }
    /**
     * Rejects a directory sharing request that was sent from the directory owner account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RejectSharedDirectory.html
     */
    rejectSharedDirectory() {
        this.add('ds:RejectSharedDirectory');
        return this;
    }
    /**
     * Removes IP address blocks from a directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RemoveIpRoutes.html
     */
    removeIpRoutes() {
        this.add('ds:RemoveIpRoutes');
        return this;
    }
    /**
     * Removes tags from an Amazon Directory Services directory.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RemoveTagsFromResource.html
     */
    removeTagsFromResource() {
        this.add('ds:RemoveTagsFromResource');
        return this;
    }
    /**
     * Resets the password for any user in your AWS Managed Microsoft AD or Simple AD directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ResetUserPassword.html
     */
    resetUserPassword() {
        this.add('ds:ResetUserPassword');
        return this;
    }
    /**
     * Restores a directory using an existing directory snapshot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RestoreFromSnapshot.html
     */
    restoreFromSnapshot() {
        this.add('ds:RestoreFromSnapshot');
        return this;
    }
    /**
     * Shares a specified directory in your AWS account (directory owner) with another AWS account (directory consumer). With this operation you can use your directory from any AWS account and from any Amazon VPC within an AWS Region.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ShareDirectory.html
     */
    shareDirectory() {
        this.add('ds:ShareDirectory');
        return this;
    }
    /**
     * Applies a schema extension to a Microsoft AD directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_StartSchemaExtension.html
     */
    startSchemaExtension() {
        this.add('ds:StartSchemaExtension');
        return this;
    }
    /**
     * Unauthorizes an application from your AWS Directory.
     *
     * Access Level: Write
     */
    unauthorizeApplication() {
        this.add('ds:UnauthorizeApplication');
        return this;
    }
    /**
     * Stops the directory sharing between the directory owner and consumer accounts.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UnshareDirectory.html
     */
    unshareDirectory() {
        this.add('ds:UnshareDirectory');
        return this;
    }
    /**
     * Updates a conditional forwarder that has been set up for your AWS directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UpdateConditionalForwarder.html
     */
    updateConditionalForwarder() {
        this.add('ds:UpdateConditionalForwarder');
        return this;
    }
    /**
     * Adds or removes domain controllers to or from the directory. Based on the difference between current value and new value (provided through this API call), domain controllers will be added or removed. It may take up to 45 minutes for any new domain controllers to become fully active once the requested number of domain controllers is updated. During this time, you cannot make another update request.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UpdateNumberOfDomainControllers.html
     */
    updateNumberOfDomainControllers() {
        this.add('ds:UpdateNumberOfDomainControllers');
        return this;
    }
    /**
     * Updates the Remote Authentication Dial In User Service (RADIUS) server information for an AD Connector directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UpdateRadius.html
     */
    updateRadius() {
        this.add('ds:UpdateRadius');
        return this;
    }
    /**
     * Updates the trust that has been set up between your AWS Managed Microsoft AD directory and an on-premises Active Directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UpdateTrust.html
     */
    updateTrust() {
        this.add('ds:UpdateTrust');
        return this;
    }
    /**
     * Verifies a trust relationship between your Microsoft AD in the AWS cloud and an external domain.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_VerifyTrust.html
     */
    verifyTrust() {
        this.add('ds:VerifyTrust');
        return this;
    }
    /**
     * Adds a resource of type directory to the statement
     *
     * @param directoryId - Identifier for the directoryId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onDirectory(directoryId, account, region, partition) {
        var arn = 'arn:${Partition}:ds:${Region}:${Account}:directory/${DirectoryId}';
        arn = arn.replace('${DirectoryId}', directoryId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Ds = Ds;
//# sourceMappingURL=data:application/json;base64,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