"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service mediaconnect
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awselementalmediaconnect.html
 */
class Mediaconnect extends shared_1.PolicyStatement {
    /**
     * Action provider for service mediaconnect
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awselementalmediaconnect.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'mediaconnect';
        this.actions = {
            "AddFlowOutputs": {
                "url": "https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn-outputs.html",
                "description": "Grants permission to add outputs to any flow.",
                "accessLevel": "Write"
            },
            "CreateFlow": {
                "url": "https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows.html",
                "description": "Grants permission to create flows.",
                "accessLevel": "Write"
            },
            "DeleteFlow": {
                "url": "https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn.html",
                "description": "Grants permission to delete flows.",
                "accessLevel": "Write"
            },
            "DescribeFlow": {
                "url": "https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn.html",
                "description": "Grants permission to display the details of a flow including the flow ARN, name, and Availability Zone, as well as details about the source, outputs, and entitlements.",
                "accessLevel": "Read"
            },
            "GrantFlowEntitlements": {
                "url": "https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn-entitlements.html",
                "description": "Grants permission to grant entitlements on any flow.",
                "accessLevel": "Write"
            },
            "ListEntitlements": {
                "url": "https://docs.aws.amazon.com/mediaconnect/latest/api/v1-entitlements.html",
                "description": "Grants permission to display a list of all entitlements that have been granted to the account.",
                "accessLevel": "List"
            },
            "ListFlows": {
                "url": "https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows.html",
                "description": "Grants permission to display a list of flows that are associated with this account.",
                "accessLevel": "List"
            },
            "RemoveFlowOutput": {
                "url": "https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn-outputs-outputarn.html",
                "description": "Grants permission to remove outputs from any flow.",
                "accessLevel": "Write"
            },
            "RevokeFlowEntitlement": {
                "url": "https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn-entitlements-entitlementarn.html",
                "description": "Grants permission to revoke entitlements on any flow.",
                "accessLevel": "Write"
            },
            "StartFlow": {
                "url": "https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-start-flowarn.html",
                "description": "Grants permission to start flows.",
                "accessLevel": "Write"
            },
            "StopFlow": {
                "url": "https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-stop-flowarn.html",
                "description": "Grants permission to stop flows.",
                "accessLevel": "Write"
            },
            "UpdateFlowEntitlement": {
                "url": "https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn-entitlements-entitlementarn.html",
                "description": "Grants permission to update entitlements on any flow.",
                "accessLevel": "Write"
            },
            "UpdateFlowOutput": {
                "url": "https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn-outputs-outputarn.html",
                "description": "Grants permission to update outputs on any flow.",
                "accessLevel": "Write"
            },
            "UpdateFlowSource": {
                "url": "https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn-source-sourcearn.html",
                "description": "Grants permission to update the source of any flow.",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "Entitlement": {
                "name": "Entitlement",
                "url": "https://docs.aws.amazon.com/mediaconnect/latest/ug/entitlements.html",
                "arn": "arn:${Partition}:mediaconnect:${Region}:${Account}:entitlement:${FlowId}:${EntitlementName}",
                "conditionKeys": []
            },
            "Flow": {
                "name": "Flow",
                "url": "https://docs.aws.amazon.com/mediaconnect/latest/ug/flows.html",
                "arn": "arn:${Partition}:mediaconnect:${Region}:${Account}:flow:${FlowId}:${FlowName}",
                "conditionKeys": []
            },
            "Output": {
                "name": "Output",
                "url": "https://docs.aws.amazon.com/mediaconnect/latest/ug/outputs.html",
                "arn": "arn:${Partition}:mediaconnect:${Region}:${Account}:output:${OutputId}:${OutputName}",
                "conditionKeys": []
            },
            "Source": {
                "name": "Source",
                "url": "https://docs.aws.amazon.com/mediaconnect/latest/ug/sources.html",
                "arn": "arn:${Partition}:mediaconnect:${Region}:${Account}:source:${SourceId}:${SourceName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to add outputs to any flow.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn-outputs.html
     */
    addFlowOutputs() {
        this.add('mediaconnect:AddFlowOutputs');
        return this;
    }
    /**
     * Grants permission to create flows.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows.html
     */
    createFlow() {
        this.add('mediaconnect:CreateFlow');
        return this;
    }
    /**
     * Grants permission to delete flows.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn.html
     */
    deleteFlow() {
        this.add('mediaconnect:DeleteFlow');
        return this;
    }
    /**
     * Grants permission to display the details of a flow including the flow ARN, name, and Availability Zone, as well as details about the source, outputs, and entitlements.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn.html
     */
    describeFlow() {
        this.add('mediaconnect:DescribeFlow');
        return this;
    }
    /**
     * Grants permission to grant entitlements on any flow.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn-entitlements.html
     */
    grantFlowEntitlements() {
        this.add('mediaconnect:GrantFlowEntitlements');
        return this;
    }
    /**
     * Grants permission to display a list of all entitlements that have been granted to the account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-entitlements.html
     */
    listEntitlements() {
        this.add('mediaconnect:ListEntitlements');
        return this;
    }
    /**
     * Grants permission to display a list of flows that are associated with this account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows.html
     */
    listFlows() {
        this.add('mediaconnect:ListFlows');
        return this;
    }
    /**
     * Grants permission to remove outputs from any flow.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn-outputs-outputarn.html
     */
    removeFlowOutput() {
        this.add('mediaconnect:RemoveFlowOutput');
        return this;
    }
    /**
     * Grants permission to revoke entitlements on any flow.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn-entitlements-entitlementarn.html
     */
    revokeFlowEntitlement() {
        this.add('mediaconnect:RevokeFlowEntitlement');
        return this;
    }
    /**
     * Grants permission to start flows.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-start-flowarn.html
     */
    startFlow() {
        this.add('mediaconnect:StartFlow');
        return this;
    }
    /**
     * Grants permission to stop flows.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-stop-flowarn.html
     */
    stopFlow() {
        this.add('mediaconnect:StopFlow');
        return this;
    }
    /**
     * Grants permission to update entitlements on any flow.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn-entitlements-entitlementarn.html
     */
    updateFlowEntitlement() {
        this.add('mediaconnect:UpdateFlowEntitlement');
        return this;
    }
    /**
     * Grants permission to update outputs on any flow.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn-outputs-outputarn.html
     */
    updateFlowOutput() {
        this.add('mediaconnect:UpdateFlowOutput');
        return this;
    }
    /**
     * Grants permission to update the source of any flow.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn-source-sourcearn.html
     */
    updateFlowSource() {
        this.add('mediaconnect:UpdateFlowSource');
        return this;
    }
    /**
     * Adds a resource of type Entitlement to the statement
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/ug/entitlements.html
     *
     * @param flowId - Identifier for the flowId.
     * @param entitlementName - Identifier for the entitlementName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onEntitlement(flowId, entitlementName, account, region, partition) {
        var arn = 'arn:${Partition}:mediaconnect:${Region}:${Account}:entitlement:${FlowId}:${EntitlementName}';
        arn = arn.replace('${FlowId}', flowId);
        arn = arn.replace('${EntitlementName}', entitlementName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Flow to the statement
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/ug/flows.html
     *
     * @param flowId - Identifier for the flowId.
     * @param flowName - Identifier for the flowName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onFlow(flowId, flowName, account, region, partition) {
        var arn = 'arn:${Partition}:mediaconnect:${Region}:${Account}:flow:${FlowId}:${FlowName}';
        arn = arn.replace('${FlowId}', flowId);
        arn = arn.replace('${FlowName}', flowName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Output to the statement
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/ug/outputs.html
     *
     * @param outputId - Identifier for the outputId.
     * @param outputName - Identifier for the outputName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onOutput(outputId, outputName, account, region, partition) {
        var arn = 'arn:${Partition}:mediaconnect:${Region}:${Account}:output:${OutputId}:${OutputName}';
        arn = arn.replace('${OutputId}', outputId);
        arn = arn.replace('${OutputName}', outputName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Source to the statement
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/ug/sources.html
     *
     * @param sourceId - Identifier for the sourceId.
     * @param sourceName - Identifier for the sourceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSource(sourceId, sourceName, account, region, partition) {
        var arn = 'arn:${Partition}:mediaconnect:${Region}:${Account}:source:${SourceId}:${SourceName}';
        arn = arn.replace('${SourceId}', sourceId);
        arn = arn.replace('${SourceName}', sourceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Mediaconnect = Mediaconnect;
//# sourceMappingURL=data:application/json;base64,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