"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service mediastore
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awselementalmediastore.html
 */
class Mediastore extends shared_1.PolicyStatement {
    /**
     * Action provider for service mediastore
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awselementalmediastore.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'mediastore';
        this.actions = {
            "CreateContainer": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_CreateContainer.html",
                "description": "Grants permission to create containers.",
                "accessLevel": "Write"
            },
            "DeleteContainer": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_DeleteContainer.html",
                "description": "Grants permission to delete any container in the current account.",
                "accessLevel": "Write"
            },
            "DeleteContainerPolicy": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_DeleteContainerPolicy.html",
                "description": "Grants permission to delete the access policy of any container in the current account.",
                "accessLevel": "Permissions management"
            },
            "DeleteCorsPolicy": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_DeleteCorsPolicy.html",
                "description": "Grants permission to delete the CORS policy from any container in the current account.",
                "accessLevel": "Write"
            },
            "DeleteLifecyclePolicy": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_DeleteLifecyclePolicy.html",
                "description": "Grants permission to delete the lifecycle policy from any container in the current account.",
                "accessLevel": "Write"
            },
            "DeleteMetricPolicy": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_DeleteMetricPolicy.html",
                "description": "Grants permission to delete the metric policy from any container in the current account.",
                "accessLevel": "Write"
            },
            "DeleteObject": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_objstore_DeleteObject.html",
                "description": "Grants permission to delete objects.",
                "accessLevel": "Write"
            },
            "DescribeContainer": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_DescribeContainer.html",
                "description": "Grants permission to retrieve details on any container in the current account.",
                "accessLevel": "List"
            },
            "DescribeObject": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_objstore_DescribeObject.html",
                "description": "Grants permission to retrieve object metadata.",
                "accessLevel": "List"
            },
            "GetContainerPolicy": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_GetContainerPolicy.html",
                "description": "Grants permission to retrieve the access policy of any container in the current account.",
                "accessLevel": "Read"
            },
            "GetCorsPolicy": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_GetCorsPolicy.html",
                "description": "Grants permission to retrieve the CORS policy of any container in the current account.",
                "accessLevel": "Read"
            },
            "GetLifecyclePolicy": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_GetLifecyclePolicy.html",
                "description": "Grants permission to retrieve the lifecycle policy that is assigned to any container in the current account.",
                "accessLevel": "Read"
            },
            "GetMetricPolicy": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_GetMetricPolicy.html",
                "description": "Grants permission to retrieve the metric policy that is assigned to any container in the current account.",
                "accessLevel": "Read"
            },
            "GetObject": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_objstore_GetObject.html",
                "description": "Grants permission to retrieve objects.",
                "accessLevel": "Read"
            },
            "ListContainers": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_ListContainers.html",
                "description": "Grants permission to retrieve a list of containers in the current account.",
                "accessLevel": "List"
            },
            "ListItems": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_objstore_ListItems.html",
                "description": "Grants permission to retrieve a list of objects and folders in the current account.",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_ListTagsForResource.html",
                "description": "Grants permission to list tags on any container in the current account.",
                "accessLevel": "Read"
            },
            "PutContainerPolicy": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_PutContainerPolicy.html",
                "description": "Grants permission to create or replace the access policy of any container in the current account.",
                "accessLevel": "Permissions management"
            },
            "PutCorsPolicy": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_PutCorsPolicy.html",
                "description": "Grants permission to add or modify the CORS policy of any container in the current account.",
                "accessLevel": "Write"
            },
            "PutLifecyclePolicy": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_PutLifecyclePolicy.html",
                "description": "Grants permission to add or modify the lifecycle policy that is assigned to any container in the current account.",
                "accessLevel": "Write"
            },
            "PutMetricPolicy": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_PutMetricPolicy.html",
                "description": "Grants permission to add or modify the metric policy that is assigned to any container in the current account.",
                "accessLevel": "Write"
            },
            "PutObject": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_objstore_PutObject.html",
                "description": "Grants permission to upload objects.",
                "accessLevel": "Write"
            },
            "StartAccessLogging": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_StartAccessLogging.html",
                "description": "Grants permission to enable access logging on any container in the current account.",
                "accessLevel": "Write"
            },
            "StopAccessLogging": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_StopAccessLogging.html",
                "description": "Grants permission to disable access logging on any container in the current account.",
                "accessLevel": "Write"
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_TagResource.html",
                "description": "Grants permission to add tags to any container in the current account.",
                "accessLevel": "Tagging"
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_UntagResource.html",
                "description": "Grants permission to remove tags from any container in the current account.",
                "accessLevel": "Tagging"
            }
        };
        this.resourceTypes = {
            "container": {
                "name": "container",
                "url": "https://docs.aws.amazon.com/mediastore/latest/ug/containers.html",
                "arn": "arn:${Partition}:mediastore:${Region}:${Account}:container/${ContainerName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to create containers.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_CreateContainer.html
     */
    createContainer() {
        this.add('mediastore:CreateContainer');
        return this;
    }
    /**
     * Grants permission to delete any container in the current account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_DeleteContainer.html
     */
    deleteContainer() {
        this.add('mediastore:DeleteContainer');
        return this;
    }
    /**
     * Grants permission to delete the access policy of any container in the current account.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_DeleteContainerPolicy.html
     */
    deleteContainerPolicy() {
        this.add('mediastore:DeleteContainerPolicy');
        return this;
    }
    /**
     * Grants permission to delete the CORS policy from any container in the current account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_DeleteCorsPolicy.html
     */
    deleteCorsPolicy() {
        this.add('mediastore:DeleteCorsPolicy');
        return this;
    }
    /**
     * Grants permission to delete the lifecycle policy from any container in the current account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_DeleteLifecyclePolicy.html
     */
    deleteLifecyclePolicy() {
        this.add('mediastore:DeleteLifecyclePolicy');
        return this;
    }
    /**
     * Grants permission to delete the metric policy from any container in the current account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_DeleteMetricPolicy.html
     */
    deleteMetricPolicy() {
        this.add('mediastore:DeleteMetricPolicy');
        return this;
    }
    /**
     * Grants permission to delete objects.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_objstore_DeleteObject.html
     */
    deleteObject() {
        this.add('mediastore:DeleteObject');
        return this;
    }
    /**
     * Grants permission to retrieve details on any container in the current account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_DescribeContainer.html
     */
    describeContainer() {
        this.add('mediastore:DescribeContainer');
        return this;
    }
    /**
     * Grants permission to retrieve object metadata.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_objstore_DescribeObject.html
     */
    describeObject() {
        this.add('mediastore:DescribeObject');
        return this;
    }
    /**
     * Grants permission to retrieve the access policy of any container in the current account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_GetContainerPolicy.html
     */
    getContainerPolicy() {
        this.add('mediastore:GetContainerPolicy');
        return this;
    }
    /**
     * Grants permission to retrieve the CORS policy of any container in the current account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_GetCorsPolicy.html
     */
    getCorsPolicy() {
        this.add('mediastore:GetCorsPolicy');
        return this;
    }
    /**
     * Grants permission to retrieve the lifecycle policy that is assigned to any container in the current account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_GetLifecyclePolicy.html
     */
    getLifecyclePolicy() {
        this.add('mediastore:GetLifecyclePolicy');
        return this;
    }
    /**
     * Grants permission to retrieve the metric policy that is assigned to any container in the current account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_GetMetricPolicy.html
     */
    getMetricPolicy() {
        this.add('mediastore:GetMetricPolicy');
        return this;
    }
    /**
     * Grants permission to retrieve objects.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_objstore_GetObject.html
     */
    getObject() {
        this.add('mediastore:GetObject');
        return this;
    }
    /**
     * Grants permission to retrieve a list of containers in the current account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_ListContainers.html
     */
    listContainers() {
        this.add('mediastore:ListContainers');
        return this;
    }
    /**
     * Grants permission to retrieve a list of objects and folders in the current account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_objstore_ListItems.html
     */
    listItems() {
        this.add('mediastore:ListItems');
        return this;
    }
    /**
     * Grants permission to list tags on any container in the current account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('mediastore:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to create or replace the access policy of any container in the current account.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_PutContainerPolicy.html
     */
    putContainerPolicy() {
        this.add('mediastore:PutContainerPolicy');
        return this;
    }
    /**
     * Grants permission to add or modify the CORS policy of any container in the current account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_PutCorsPolicy.html
     */
    putCorsPolicy() {
        this.add('mediastore:PutCorsPolicy');
        return this;
    }
    /**
     * Grants permission to add or modify the lifecycle policy that is assigned to any container in the current account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_PutLifecyclePolicy.html
     */
    putLifecyclePolicy() {
        this.add('mediastore:PutLifecyclePolicy');
        return this;
    }
    /**
     * Grants permission to add or modify the metric policy that is assigned to any container in the current account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_PutMetricPolicy.html
     */
    putMetricPolicy() {
        this.add('mediastore:PutMetricPolicy');
        return this;
    }
    /**
     * Grants permission to upload objects.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_objstore_PutObject.html
     */
    putObject() {
        this.add('mediastore:PutObject');
        return this;
    }
    /**
     * Grants permission to enable access logging on any container in the current account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_StartAccessLogging.html
     */
    startAccessLogging() {
        this.add('mediastore:StartAccessLogging');
        return this;
    }
    /**
     * Grants permission to disable access logging on any container in the current account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_StopAccessLogging.html
     */
    stopAccessLogging() {
        this.add('mediastore:StopAccessLogging');
        return this;
    }
    /**
     * Grants permission to add tags to any container in the current account.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_TagResource.html
     */
    tagResource() {
        this.add('mediastore:TagResource');
        return this;
    }
    /**
     * Grants permission to remove tags from any container in the current account.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_UntagResource.html
     */
    untagResource() {
        this.add('mediastore:UntagResource');
        return this;
    }
    /**
     * Adds a resource of type container to the statement
     *
     * https://docs.aws.amazon.com/mediastore/latest/ug/containers.html
     *
     * @param containerName - Identifier for the containerName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onContainer(containerName, account, region, partition) {
        var arn = 'arn:${Partition}:mediastore:${Region}:${Account}:container/${ContainerName}';
        arn = arn.replace('${ContainerName}', containerName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Mediastore = Mediastore;
//# sourceMappingURL=data:application/json;base64,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