"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service globalaccelerator
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsglobalaccelerator.html
 */
class Globalaccelerator extends shared_1.PolicyStatement {
    /**
     * Action provider for service globalaccelerator
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsglobalaccelerator.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'globalaccelerator';
        this.actions = {
            "AdvertiseByoipCidr": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_AdvertiseByoipCidr.html",
                "description": "Advertises an IPv4 address range that is provisioned for use with your accelerator through bring your own IP addresses (BYOIP).",
                "accessLevel": "Write"
            },
            "CreateAccelerator": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_CreateAccelerator.html",
                "description": "Create an accelerator.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateEndpointGroup": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_CreateEndpointGroup.html",
                "description": "Add an endpoint group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "listener": {
                        "required": true
                    }
                }
            },
            "CreateListener": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_CreateListener.html",
                "description": "Add a listener.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "accelerator": {
                        "required": true
                    }
                }
            },
            "DeleteAccelerator": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_DeleteAccelerator.html",
                "description": "Delete the accelerator.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "accelerator": {
                        "required": true
                    }
                }
            },
            "DeleteEndpointGroup": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_DeleteEndpointGroup.html",
                "description": "Delete the endpoint group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "endpointgroup": {
                        "required": true
                    }
                }
            },
            "DeleteListener": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_DeleteListener.html",
                "description": "Delete the listener.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "listener": {
                        "required": true
                    }
                }
            },
            "DeprovisionByoipCidr": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_DeprovisionByoipCidr.html",
                "description": "Releases the specified address range that you provisioned for use with your accelerator through bring your own IP addresses (BYOIP) and deletes the corresponding address pool.",
                "accessLevel": "Write"
            },
            "DescribeAccelerator": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_DescribeAccelerator.html",
                "description": "Describe the accelerator.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "accelerator": {
                        "required": true
                    }
                }
            },
            "DescribeAcceleratorAttributes": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_DescribeAcceleratorAttributes.html",
                "description": "Describe the accelerator Attributes.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "accelerator": {
                        "required": true
                    }
                }
            },
            "DescribeEndpointGroup": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_DescribeEndpointGroup.html",
                "description": "Describe the endpoint group.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "endpointgroup": {
                        "required": true
                    }
                }
            },
            "DescribeListener": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_DescribeListener.html",
                "description": "Describe the listener.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "listener": {
                        "required": true
                    }
                }
            },
            "ListAccelerators": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_ListAccelerators.html",
                "description": "List the accelerators.",
                "accessLevel": "List"
            },
            "ListByoipCidrs": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_ListByoipCidrs.html",
                "description": "List the byoip cidrs.",
                "accessLevel": "List"
            },
            "ListEndpointGroups": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_ListEndpointGroups.html",
                "description": "List the endpoint groups.",
                "accessLevel": "List",
                "resourceTypes": {
                    "listener": {
                        "required": true
                    }
                }
            },
            "ListListeners": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_ListListeners.html",
                "description": "List the listeners.",
                "accessLevel": "List",
                "resourceTypes": {
                    "accelerator": {
                        "required": true
                    }
                }
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_ListTagsForResource.html",
                "description": "List tags for a globalaccelerator resource.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "accelerator": {
                        "required": false
                    }
                }
            },
            "ProvisionByoipCidr": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_ProvisionByoipCidr.html",
                "description": "Provisions an address range for use with your accelerator through bring your own IP addresses (BYOIP) and creates a corresponding address pool.",
                "accessLevel": "Write"
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_TagResource.html",
                "description": "Add tags to globalaccelerator resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "accelerator": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_UntagResource.html",
                "description": "Remove tags from globalaccelerator resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "accelerator": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateAccelerator": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_UpdateAccelerator.html",
                "description": "Update the accelerator.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "accelerator": {
                        "required": true
                    }
                }
            },
            "UpdateAcceleratorAttributes": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_UpdateAcceleratorAttributes.html",
                "description": "Update the accelerator attributes.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "accelerator": {
                        "required": true
                    }
                }
            },
            "UpdateEndpointGroup": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_UpdateEndpointGroup.html",
                "description": "Update the endpoint group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "endpointgroup": {
                        "required": true
                    }
                }
            },
            "UpdateListener": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_UpdateListener.html",
                "description": "Update the listener.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "listener": {
                        "required": true
                    }
                }
            },
            "WithdrawByoipCidr": {
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_WithdrawByoipCidr.html",
                "description": "Stops advertising an IPv4 address range that is provisioned as an address pool.",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "accelerator": {
                "name": "accelerator",
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_Accelerator.html",
                "arn": "arn:${Partition}:globalaccelerator::${Account}:accelerator/${AcceleratorId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "listener": {
                "name": "listener",
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_Listener.html",
                "arn": "arn:${Partition}:globalaccelerator::${Account}:accelerator/${AcceleratorId}/listener/${ListenerId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "endpointgroup": {
                "name": "endpointgroup",
                "url": "https://docs.aws.amazon.com/global-accelerator/latest/api/API_EndpointGroup.html",
                "arn": "arn:${Partition}:globalaccelerator::${Account}:accelerator/${AcceleratorId}/listener/${ListenerId}/endpoint-group/${EndpointGroupId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Advertises an IPv4 address range that is provisioned for use with your accelerator through bring your own IP addresses (BYOIP).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_AdvertiseByoipCidr.html
     */
    advertiseByoipCidr() {
        this.add('globalaccelerator:AdvertiseByoipCidr');
        return this;
    }
    /**
     * Create an accelerator.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_CreateAccelerator.html
     */
    createAccelerator() {
        this.add('globalaccelerator:CreateAccelerator');
        return this;
    }
    /**
     * Add an endpoint group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_CreateEndpointGroup.html
     */
    createEndpointGroup() {
        this.add('globalaccelerator:CreateEndpointGroup');
        return this;
    }
    /**
     * Add a listener.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_CreateListener.html
     */
    createListener() {
        this.add('globalaccelerator:CreateListener');
        return this;
    }
    /**
     * Delete the accelerator.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_DeleteAccelerator.html
     */
    deleteAccelerator() {
        this.add('globalaccelerator:DeleteAccelerator');
        return this;
    }
    /**
     * Delete the endpoint group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_DeleteEndpointGroup.html
     */
    deleteEndpointGroup() {
        this.add('globalaccelerator:DeleteEndpointGroup');
        return this;
    }
    /**
     * Delete the listener.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_DeleteListener.html
     */
    deleteListener() {
        this.add('globalaccelerator:DeleteListener');
        return this;
    }
    /**
     * Releases the specified address range that you provisioned for use with your accelerator through bring your own IP addresses (BYOIP) and deletes the corresponding address pool.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_DeprovisionByoipCidr.html
     */
    deprovisionByoipCidr() {
        this.add('globalaccelerator:DeprovisionByoipCidr');
        return this;
    }
    /**
     * Describe the accelerator.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_DescribeAccelerator.html
     */
    describeAccelerator() {
        this.add('globalaccelerator:DescribeAccelerator');
        return this;
    }
    /**
     * Describe the accelerator Attributes.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_DescribeAcceleratorAttributes.html
     */
    describeAcceleratorAttributes() {
        this.add('globalaccelerator:DescribeAcceleratorAttributes');
        return this;
    }
    /**
     * Describe the endpoint group.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_DescribeEndpointGroup.html
     */
    describeEndpointGroup() {
        this.add('globalaccelerator:DescribeEndpointGroup');
        return this;
    }
    /**
     * Describe the listener.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_DescribeListener.html
     */
    describeListener() {
        this.add('globalaccelerator:DescribeListener');
        return this;
    }
    /**
     * List the accelerators.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_ListAccelerators.html
     */
    listAccelerators() {
        this.add('globalaccelerator:ListAccelerators');
        return this;
    }
    /**
     * List the byoip cidrs.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_ListByoipCidrs.html
     */
    listByoipCidrs() {
        this.add('globalaccelerator:ListByoipCidrs');
        return this;
    }
    /**
     * List the endpoint groups.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_ListEndpointGroups.html
     */
    listEndpointGroups() {
        this.add('globalaccelerator:ListEndpointGroups');
        return this;
    }
    /**
     * List the listeners.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_ListListeners.html
     */
    listListeners() {
        this.add('globalaccelerator:ListListeners');
        return this;
    }
    /**
     * List tags for a globalaccelerator resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('globalaccelerator:ListTagsForResource');
        return this;
    }
    /**
     * Provisions an address range for use with your accelerator through bring your own IP addresses (BYOIP) and creates a corresponding address pool.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_ProvisionByoipCidr.html
     */
    provisionByoipCidr() {
        this.add('globalaccelerator:ProvisionByoipCidr');
        return this;
    }
    /**
     * Add tags to globalaccelerator resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_TagResource.html
     */
    tagResource() {
        this.add('globalaccelerator:TagResource');
        return this;
    }
    /**
     * Remove tags from globalaccelerator resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_UntagResource.html
     */
    untagResource() {
        this.add('globalaccelerator:UntagResource');
        return this;
    }
    /**
     * Update the accelerator.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_UpdateAccelerator.html
     */
    updateAccelerator() {
        this.add('globalaccelerator:UpdateAccelerator');
        return this;
    }
    /**
     * Update the accelerator attributes.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_UpdateAcceleratorAttributes.html
     */
    updateAcceleratorAttributes() {
        this.add('globalaccelerator:UpdateAcceleratorAttributes');
        return this;
    }
    /**
     * Update the endpoint group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_UpdateEndpointGroup.html
     */
    updateEndpointGroup() {
        this.add('globalaccelerator:UpdateEndpointGroup');
        return this;
    }
    /**
     * Update the listener.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_UpdateListener.html
     */
    updateListener() {
        this.add('globalaccelerator:UpdateListener');
        return this;
    }
    /**
     * Stops advertising an IPv4 address range that is provisioned as an address pool.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_WithdrawByoipCidr.html
     */
    withdrawByoipCidr() {
        this.add('globalaccelerator:WithdrawByoipCidr');
        return this;
    }
    /**
     * Adds a resource of type accelerator to the statement
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_Accelerator.html
     *
     * @param acceleratorId - Identifier for the acceleratorId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onAccelerator(acceleratorId, account, partition) {
        var arn = 'arn:${Partition}:globalaccelerator::${Account}:accelerator/${AcceleratorId}';
        arn = arn.replace('${AcceleratorId}', acceleratorId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type listener to the statement
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_Listener.html
     *
     * @param acceleratorId - Identifier for the acceleratorId.
     * @param listenerId - Identifier for the listenerId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onListener(acceleratorId, listenerId, account, partition) {
        var arn = 'arn:${Partition}:globalaccelerator::${Account}:accelerator/${AcceleratorId}/listener/${ListenerId}';
        arn = arn.replace('${AcceleratorId}', acceleratorId);
        arn = arn.replace('${ListenerId}', listenerId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type endpointgroup to the statement
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_EndpointGroup.html
     *
     * @param acceleratorId - Identifier for the acceleratorId.
     * @param listenerId - Identifier for the listenerId.
     * @param endpointGroupId - Identifier for the endpointGroupId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onEndpointgroup(acceleratorId, listenerId, endpointGroupId, account, partition) {
        var arn = 'arn:${Partition}:globalaccelerator::${Account}:accelerator/${AcceleratorId}/listener/${ListenerId}/endpoint-group/${EndpointGroupId}';
        arn = arn.replace('${AcceleratorId}', acceleratorId);
        arn = arn.replace('${ListenerId}', listenerId);
        arn = arn.replace('${EndpointGroupId}', endpointGroupId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Globalaccelerator = Globalaccelerator;
//# sourceMappingURL=data:application/json;base64,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