"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service iot1click
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsiot1-click.html
 */
class Iot1click extends shared_1.PolicyStatement {
    /**
     * Action provider for service iot1click
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsiot1-click.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'iot1click';
        this.actions = {
            "AssociateDeviceWithPlacement": {
                "url": "https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_AssociateDeviceWithPlacement.html",
                "description": "Associate a device to a placement",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "ClaimDevicesByClaimCode": {
                "url": "https://docs.aws.amazon.com/iot-1-click/1.0/devices-apireference/claims-claimcode.html",
                "description": "Claim a batch of devices with a claim code.",
                "accessLevel": "Read"
            },
            "CreatePlacement": {
                "url": "https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_CreatePlacement.html",
                "description": "Create a new placement in a project",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "CreateProject": {
                "url": "https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_CreateProject.html",
                "description": "Create a new project",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeletePlacement": {
                "url": "https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_DeletePlacement.html",
                "description": "Delete a placement from a project",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "DeleteProject": {
                "url": "https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_DeleteProject.html",
                "description": "Delete a project",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "DescribeDevice": {
                "url": "https://docs.aws.amazon.com/iot-1-click/1.0/devices-apireference/devices-deviceid.html",
                "description": "Describe a device",
                "accessLevel": "Read",
                "resourceTypes": {
                    "device": {
                        "required": true
                    }
                }
            },
            "DescribePlacement": {
                "url": "https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_DescribePlacement.html",
                "description": "Describe a placement",
                "accessLevel": "Read",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "DescribeProject": {
                "url": "https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_DescribeProject.html",
                "description": "Describe a project",
                "accessLevel": "Read",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "DisassociateDeviceFromPlacement": {
                "url": "https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_DissacociateDeviceFromPlacement.html",
                "description": "Disassociate a device from a placement",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "FinalizeDeviceClaim": {
                "url": "https://docs.aws.amazon.com/iot-1-click/1.0/devices-apireference/devices-deviceid-finalize-claim.html",
                "description": "Finalize a device claim",
                "accessLevel": "Read",
                "resourceTypes": {
                    "device": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "GetDeviceMethods": {
                "url": "https://docs.aws.amazon.com/iot-1-click/1.0/devices-apireference/devices-deviceid-methods.html",
                "description": "Get available methods of a device",
                "accessLevel": "Read",
                "resourceTypes": {
                    "device": {
                        "required": true
                    }
                }
            },
            "GetDevicesInPlacement": {
                "url": "https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_GetDevicesInPlacement.html",
                "description": "Get devices associated to a placement",
                "accessLevel": "Read",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "InitiateDeviceClaim": {
                "url": "https://docs.aws.amazon.com/iot-1-click/1.0/devices-apireference/devices-deviceid-initiate-claim.html",
                "description": "Initialize a device claim",
                "accessLevel": "Read",
                "resourceTypes": {
                    "device": {
                        "required": true
                    }
                }
            },
            "InvokeDeviceMethod": {
                "url": "https://docs.aws.amazon.com/iot-1-click/1.0/devices-apireference/devices-deviceid-methods.html",
                "description": "Invoke a device method",
                "accessLevel": "Write",
                "resourceTypes": {
                    "device": {
                        "required": true
                    }
                }
            },
            "ListDeviceEvents": {
                "url": "https://docs.aws.amazon.com/iot-1-click/1.0/devices-apireference/devices-deviceid-events.html",
                "description": "List past events published by a device",
                "accessLevel": "Read",
                "resourceTypes": {
                    "device": {
                        "required": true
                    }
                }
            },
            "ListDevices": {
                "url": "https://docs.aws.amazon.com/iot-1-click/1.0/devices-apireference/devices.html",
                "description": "List all devices",
                "accessLevel": "List"
            },
            "ListPlacements": {
                "url": "https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_ListPlacements.html",
                "description": "List placements in a project",
                "accessLevel": "Read",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "ListProjects": {
                "url": "https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_ListProjects.html",
                "description": "List all projects",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_ListTagsForResource.html",
                "description": "Lists the tags (metadata) which you have assigned to the resource.",
                "accessLevel": "List",
                "resourceTypes": {
                    "device": {
                        "required": false
                    },
                    "project": {
                        "required": false
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_TagResource.html",
                "description": "Adds to or modifies the tags of the given resource. Tags are metadata which can be used to manage a resource.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "device": {
                        "required": false
                    },
                    "project": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UnclaimDevice": {
                "url": "https://docs.aws.amazon.com/iot-1-click/1.0/devices-apireference/devices-deviceid-unclaim.html",
                "description": "Unclaim a device",
                "accessLevel": "Read",
                "resourceTypes": {
                    "device": {
                        "required": true
                    }
                }
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_UntagResource.html",
                "description": "Removes the given tags (metadata) from the resource.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "device": {
                        "required": false
                    },
                    "project": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateDeviceState": {
                "url": "https://docs.aws.amazon.com/iot-1-click/1.0/devices-apireference/devices-deviceid-state.html",
                "description": "Update device state",
                "accessLevel": "Write",
                "resourceTypes": {
                    "device": {
                        "required": true
                    }
                }
            },
            "UpdatePlacement": {
                "url": "https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_UpdatePlacement.html",
                "description": "Update a placement",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "UpdateProject": {
                "url": "https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_UpdateProject.html",
                "description": "Update a project",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "device": {
                "name": "device",
                "url": "https://docs.aws.amazon.com/iot-1-click/1.0/devices-apireference/resources.html",
                "arn": "arn:${Partition}:iot1click:${Region}:${Account}:devices/${DeviceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "project": {
                "name": "project",
                "url": "https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_Operations.html",
                "arn": "arn:${Partition}:iot1click:${Region}:${Account}:projects/${ProjectName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Associate a device to a placement
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_AssociateDeviceWithPlacement.html
     */
    associateDeviceWithPlacement() {
        this.add('iot1click:AssociateDeviceWithPlacement');
        return this;
    }
    /**
     * Claim a batch of devices with a claim code.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-1-click/1.0/devices-apireference/claims-claimcode.html
     */
    claimDevicesByClaimCode() {
        this.add('iot1click:ClaimDevicesByClaimCode');
        return this;
    }
    /**
     * Create a new placement in a project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_CreatePlacement.html
     */
    createPlacement() {
        this.add('iot1click:CreatePlacement');
        return this;
    }
    /**
     * Create a new project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_CreateProject.html
     */
    createProject() {
        this.add('iot1click:CreateProject');
        return this;
    }
    /**
     * Delete a placement from a project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_DeletePlacement.html
     */
    deletePlacement() {
        this.add('iot1click:DeletePlacement');
        return this;
    }
    /**
     * Delete a project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_DeleteProject.html
     */
    deleteProject() {
        this.add('iot1click:DeleteProject');
        return this;
    }
    /**
     * Describe a device
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-1-click/1.0/devices-apireference/devices-deviceid.html
     */
    describeDevice() {
        this.add('iot1click:DescribeDevice');
        return this;
    }
    /**
     * Describe a placement
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_DescribePlacement.html
     */
    describePlacement() {
        this.add('iot1click:DescribePlacement');
        return this;
    }
    /**
     * Describe a project
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_DescribeProject.html
     */
    describeProject() {
        this.add('iot1click:DescribeProject');
        return this;
    }
    /**
     * Disassociate a device from a placement
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_DissacociateDeviceFromPlacement.html
     */
    disassociateDeviceFromPlacement() {
        this.add('iot1click:DisassociateDeviceFromPlacement');
        return this;
    }
    /**
     * Finalize a device claim
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-1-click/1.0/devices-apireference/devices-deviceid-finalize-claim.html
     */
    finalizeDeviceClaim() {
        this.add('iot1click:FinalizeDeviceClaim');
        return this;
    }
    /**
     * Get available methods of a device
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-1-click/1.0/devices-apireference/devices-deviceid-methods.html
     */
    getDeviceMethods() {
        this.add('iot1click:GetDeviceMethods');
        return this;
    }
    /**
     * Get devices associated to a placement
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_GetDevicesInPlacement.html
     */
    getDevicesInPlacement() {
        this.add('iot1click:GetDevicesInPlacement');
        return this;
    }
    /**
     * Initialize a device claim
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-1-click/1.0/devices-apireference/devices-deviceid-initiate-claim.html
     */
    initiateDeviceClaim() {
        this.add('iot1click:InitiateDeviceClaim');
        return this;
    }
    /**
     * Invoke a device method
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-1-click/1.0/devices-apireference/devices-deviceid-methods.html
     */
    invokeDeviceMethod() {
        this.add('iot1click:InvokeDeviceMethod');
        return this;
    }
    /**
     * List past events published by a device
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-1-click/1.0/devices-apireference/devices-deviceid-events.html
     */
    listDeviceEvents() {
        this.add('iot1click:ListDeviceEvents');
        return this;
    }
    /**
     * List all devices
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-1-click/1.0/devices-apireference/devices.html
     */
    listDevices() {
        this.add('iot1click:ListDevices');
        return this;
    }
    /**
     * List placements in a project
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_ListPlacements.html
     */
    listPlacements() {
        this.add('iot1click:ListPlacements');
        return this;
    }
    /**
     * List all projects
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_ListProjects.html
     */
    listProjects() {
        this.add('iot1click:ListProjects');
        return this;
    }
    /**
     * Lists the tags (metadata) which you have assigned to the resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('iot1click:ListTagsForResource');
        return this;
    }
    /**
     * Adds to or modifies the tags of the given resource. Tags are metadata which can be used to manage a resource.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_TagResource.html
     */
    tagResource() {
        this.add('iot1click:TagResource');
        return this;
    }
    /**
     * Unclaim a device
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-1-click/1.0/devices-apireference/devices-deviceid-unclaim.html
     */
    unclaimDevice() {
        this.add('iot1click:UnclaimDevice');
        return this;
    }
    /**
     * Removes the given tags (metadata) from the resource.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_UntagResource.html
     */
    untagResource() {
        this.add('iot1click:UntagResource');
        return this;
    }
    /**
     * Update device state
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-1-click/1.0/devices-apireference/devices-deviceid-state.html
     */
    updateDeviceState() {
        this.add('iot1click:UpdateDeviceState');
        return this;
    }
    /**
     * Update a placement
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_UpdatePlacement.html
     */
    updatePlacement() {
        this.add('iot1click:UpdatePlacement');
        return this;
    }
    /**
     * Update a project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_UpdateProject.html
     */
    updateProject() {
        this.add('iot1click:UpdateProject');
        return this;
    }
    /**
     * Adds a resource of type device to the statement
     *
     * https://docs.aws.amazon.com/iot-1-click/1.0/devices-apireference/resources.html
     *
     * @param deviceId - Identifier for the deviceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onDevice(deviceId, account, region, partition) {
        var arn = 'arn:${Partition}:iot1click:${Region}:${Account}:devices/${DeviceId}';
        arn = arn.replace('${DeviceId}', deviceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type project to the statement
     *
     * https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_Operations.html
     *
     * @param projectName - Identifier for the projectName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onProject(projectName, account, region, partition) {
        var arn = 'arn:${Partition}:iot1click:${Region}:${Account}:projects/${ProjectName}';
        arn = arn.replace('${ProjectName}', projectName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Iot1click = Iot1click;
//# sourceMappingURL=data:application/json;base64,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