"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service mobilehub
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsmobilehub.html
 */
class Mobilehub extends shared_1.PolicyStatement {
    /**
     * Action provider for service mobilehub
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsmobilehub.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'mobilehub';
        this.actions = {
            "CreateProject": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Create a project",
                "accessLevel": "Write"
            },
            "CreateServiceRole": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Enable AWS Mobile Hub in the account by creating the required service role",
                "accessLevel": "Write"
            },
            "DeleteProject": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Delete the specified project",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "DeleteProjectSnapshot": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Delete a saved snapshot of project configuration",
                "accessLevel": "Write"
            },
            "DeployToStage": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Deploy changes to the specified stage",
                "accessLevel": "Write"
            },
            "DescribeBundle": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Describe the download bundle",
                "accessLevel": "Read"
            },
            "ExportBundle": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Export the download bundle",
                "accessLevel": "Read"
            },
            "ExportProject": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Export the project configuration",
                "accessLevel": "Read",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "GenerateProjectParameters": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Generate project parameters required for code generation",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "GetProject": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Get project configuration and resources",
                "accessLevel": "Read",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "GetProjectSnapshot": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Fetch the previously exported project configuration snapshot",
                "accessLevel": "Read"
            },
            "ImportProject": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Create a new project from the previously exported project configuration",
                "accessLevel": "Write"
            },
            "InstallBundle": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Install a bundle in the project deployments S3 bucket",
                "accessLevel": "Write"
            },
            "ListAvailableConnectors": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "List the available SaaS (Software as a Service) connectors",
                "accessLevel": "List"
            },
            "ListAvailableFeatures": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "List available features",
                "accessLevel": "List"
            },
            "ListAvailableRegions": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "List available regions for projects",
                "accessLevel": "List"
            },
            "ListBundles": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "List the available download bundles",
                "accessLevel": "List"
            },
            "ListProjectSnapshots": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "List saved snapshots of project configuration",
                "accessLevel": "List"
            },
            "ListProjects": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "List projects",
                "accessLevel": "List"
            },
            "SynchronizeProject": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Synchronize state of resources into project",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "UpdateProject": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Update project",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "ValidateProject": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Validate a mobile hub project.",
                "accessLevel": "Read"
            },
            "VerifyServiceRole": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Verify AWS Mobile Hub is enabled in the account",
                "accessLevel": "Read"
            }
        };
        this.resourceTypes = {
            "project": {
                "name": "project",
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/reference-mobile-hub-iam-managed-policies.html",
                "arn": "arn:${Partition}:mobilehub:${Region}:${Account}:project/${ProjectId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Create a project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    createProject() {
        this.add('mobilehub:CreateProject');
        return this;
    }
    /**
     * Enable AWS Mobile Hub in the account by creating the required service role
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    createServiceRole() {
        this.add('mobilehub:CreateServiceRole');
        return this;
    }
    /**
     * Delete the specified project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    deleteProject() {
        this.add('mobilehub:DeleteProject');
        return this;
    }
    /**
     * Delete a saved snapshot of project configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    deleteProjectSnapshot() {
        this.add('mobilehub:DeleteProjectSnapshot');
        return this;
    }
    /**
     * Deploy changes to the specified stage
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    deployToStage() {
        this.add('mobilehub:DeployToStage');
        return this;
    }
    /**
     * Describe the download bundle
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    describeBundle() {
        this.add('mobilehub:DescribeBundle');
        return this;
    }
    /**
     * Export the download bundle
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    exportBundle() {
        this.add('mobilehub:ExportBundle');
        return this;
    }
    /**
     * Export the project configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    exportProject() {
        this.add('mobilehub:ExportProject');
        return this;
    }
    /**
     * Generate project parameters required for code generation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    generateProjectParameters() {
        this.add('mobilehub:GenerateProjectParameters');
        return this;
    }
    /**
     * Get project configuration and resources
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    getProject() {
        this.add('mobilehub:GetProject');
        return this;
    }
    /**
     * Fetch the previously exported project configuration snapshot
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    getProjectSnapshot() {
        this.add('mobilehub:GetProjectSnapshot');
        return this;
    }
    /**
     * Create a new project from the previously exported project configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    importProject() {
        this.add('mobilehub:ImportProject');
        return this;
    }
    /**
     * Install a bundle in the project deployments S3 bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    installBundle() {
        this.add('mobilehub:InstallBundle');
        return this;
    }
    /**
     * List the available SaaS (Software as a Service) connectors
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    listAvailableConnectors() {
        this.add('mobilehub:ListAvailableConnectors');
        return this;
    }
    /**
     * List available features
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    listAvailableFeatures() {
        this.add('mobilehub:ListAvailableFeatures');
        return this;
    }
    /**
     * List available regions for projects
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    listAvailableRegions() {
        this.add('mobilehub:ListAvailableRegions');
        return this;
    }
    /**
     * List the available download bundles
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    listBundles() {
        this.add('mobilehub:ListBundles');
        return this;
    }
    /**
     * List saved snapshots of project configuration
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    listProjectSnapshots() {
        this.add('mobilehub:ListProjectSnapshots');
        return this;
    }
    /**
     * List projects
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    listProjects() {
        this.add('mobilehub:ListProjects');
        return this;
    }
    /**
     * Synchronize state of resources into project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    synchronizeProject() {
        this.add('mobilehub:SynchronizeProject');
        return this;
    }
    /**
     * Update project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    updateProject() {
        this.add('mobilehub:UpdateProject');
        return this;
    }
    /**
     * Validate a mobile hub project.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    validateProject() {
        this.add('mobilehub:ValidateProject');
        return this;
    }
    /**
     * Verify AWS Mobile Hub is enabled in the account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    verifyServiceRole() {
        this.add('mobilehub:VerifyServiceRole');
        return this;
    }
    /**
     * Adds a resource of type project to the statement
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/reference-mobile-hub-iam-managed-policies.html
     *
     * @param projectId - Identifier for the projectId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onProject(projectId, account, region, partition) {
        var arn = 'arn:${Partition}:mobilehub:${Region}:${Account}:project/${ProjectId}';
        arn = arn.replace('${ProjectId}', projectId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Mobilehub = Mobilehub;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXdzbW9iaWxlaHViLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiYXdzbW9iaWxlaHViLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUEsc0NBQW9FO0FBR3BFOzs7O0dBSUc7QUFDSCxNQUFhLFNBQVUsU0FBUSx3QkFBZTtJQThKNUM7Ozs7T0FJRztJQUNILFlBQWEsS0FBNEI7UUFDdkMsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBbktSLGtCQUFhLEdBQUcsV0FBVyxDQUFDO1FBQzVCLFlBQU8sR0FBWTtZQUN4QixlQUFlLEVBQUU7Z0JBQ2YsS0FBSyxFQUFFLG9GQUFvRjtnQkFDM0YsYUFBYSxFQUFFLGtCQUFrQjtnQkFDakMsYUFBYSxFQUFFLE9BQU87YUFDdkI7WUFDRCxtQkFBbUIsRUFBRTtnQkFDbkIsS0FBSyxFQUFFLG9GQUFvRjtnQkFDM0YsYUFBYSxFQUFFLDRFQUE0RTtnQkFDM0YsYUFBYSxFQUFFLE9BQU87YUFDdkI7WUFDRCxlQUFlLEVBQUU7Z0JBQ2YsS0FBSyxFQUFFLG9GQUFvRjtnQkFDM0YsYUFBYSxFQUFFLDhCQUE4QjtnQkFDN0MsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixTQUFTLEVBQUU7d0JBQ1QsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCx1QkFBdUIsRUFBRTtnQkFDdkIsS0FBSyxFQUFFLG9GQUFvRjtnQkFDM0YsYUFBYSxFQUFFLGtEQUFrRDtnQkFDakUsYUFBYSxFQUFFLE9BQU87YUFDdkI7WUFDRCxlQUFlLEVBQUU7Z0JBQ2YsS0FBSyxFQUFFLG9GQUFvRjtnQkFDM0YsYUFBYSxFQUFFLHVDQUF1QztnQkFDdEQsYUFBYSxFQUFFLE9BQU87YUFDdkI7WUFDRCxnQkFBZ0IsRUFBRTtnQkFDaEIsS0FBSyxFQUFFLG9GQUFvRjtnQkFDM0YsYUFBYSxFQUFFLDhCQUE4QjtnQkFDN0MsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxjQUFjLEVBQUU7Z0JBQ2QsS0FBSyxFQUFFLG9GQUFvRjtnQkFDM0YsYUFBYSxFQUFFLDRCQUE0QjtnQkFDM0MsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxlQUFlLEVBQUU7Z0JBQ2YsS0FBSyxFQUFFLG9GQUFvRjtnQkFDM0YsYUFBYSxFQUFFLGtDQUFrQztnQkFDakQsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixTQUFTLEVBQUU7d0JBQ1QsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCwyQkFBMkIsRUFBRTtnQkFDM0IsS0FBSyxFQUFFLG9GQUFvRjtnQkFDM0YsYUFBYSxFQUFFLDBEQUEwRDtnQkFDekUsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixTQUFTLEVBQUU7d0JBQ1QsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxZQUFZLEVBQUU7Z0JBQ1osS0FBSyxFQUFFLG9GQUFvRjtnQkFDM0YsYUFBYSxFQUFFLHlDQUF5QztnQkFDeEQsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixTQUFTLEVBQUU7d0JBQ1QsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxvQkFBb0IsRUFBRTtnQkFDcEIsS0FBSyxFQUFFLG9GQUFvRjtnQkFDM0YsYUFBYSxFQUFFLDhEQUE4RDtnQkFDN0UsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxlQUFlLEVBQUU7Z0JBQ2YsS0FBSyxFQUFFLG9GQUFvRjtnQkFDM0YsYUFBYSxFQUFFLHlFQUF5RTtnQkFDeEYsYUFBYSxFQUFFLE9BQU87YUFDdkI7WUFDRCxlQUFlLEVBQUU7Z0JBQ2YsS0FBSyxFQUFFLG9GQUFvRjtnQkFDM0YsYUFBYSxFQUFFLHVEQUF1RDtnQkFDdEUsYUFBYSxFQUFFLE9BQU87YUFDdkI7WUFDRCx5QkFBeUIsRUFBRTtnQkFDekIsS0FBSyxFQUFFLG9GQUFvRjtnQkFDM0YsYUFBYSxFQUFFLDREQUE0RDtnQkFDM0UsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCx1QkFBdUIsRUFBRTtnQkFDdkIsS0FBSyxFQUFFLG9GQUFvRjtnQkFDM0YsYUFBYSxFQUFFLHlCQUF5QjtnQkFDeEMsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxzQkFBc0IsRUFBRTtnQkFDdEIsS0FBSyxFQUFFLG9GQUFvRjtnQkFDM0YsYUFBYSxFQUFFLHFDQUFxQztnQkFDcEQsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxhQUFhLEVBQUU7Z0JBQ2IsS0FBSyxFQUFFLG9GQUFvRjtnQkFDM0YsYUFBYSxFQUFFLHFDQUFxQztnQkFDcEQsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxzQkFBc0IsRUFBRTtnQkFDdEIsS0FBSyxFQUFFLG9GQUFvRjtnQkFDM0YsYUFBYSxFQUFFLCtDQUErQztnQkFDOUQsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxjQUFjLEVBQUU7Z0JBQ2QsS0FBSyxFQUFFLG9GQUFvRjtnQkFDM0YsYUFBYSxFQUFFLGVBQWU7Z0JBQzlCLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0Qsb0JBQW9CLEVBQUU7Z0JBQ3BCLEtBQUssRUFBRSxvRkFBb0Y7Z0JBQzNGLGFBQWEsRUFBRSw2Q0FBNkM7Z0JBQzVELGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsU0FBUyxFQUFFO3dCQUNULFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsZUFBZSxFQUFFO2dCQUNmLEtBQUssRUFBRSxvRkFBb0Y7Z0JBQzNGLGFBQWEsRUFBRSxnQkFBZ0I7Z0JBQy9CLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsU0FBUyxFQUFFO3dCQUNULFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsaUJBQWlCLEVBQUU7Z0JBQ2pCLEtBQUssRUFBRSxvRkFBb0Y7Z0JBQzNGLGFBQWEsRUFBRSxnQ0FBZ0M7Z0JBQy9DLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0QsbUJBQW1CLEVBQUU7Z0JBQ25CLEtBQUssRUFBRSxvRkFBb0Y7Z0JBQzNGLGFBQWEsRUFBRSxpREFBaUQ7Z0JBQ2hFLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1NBQ0YsQ0FBQztRQUNLLGtCQUFhLEdBQWtCO1lBQ3BDLFNBQVMsRUFBRTtnQkFDVCxNQUFNLEVBQUUsU0FBUztnQkFDakIsS0FBSyxFQUFFLDZHQUE2RztnQkFDcEgsS0FBSyxFQUFFLHNFQUFzRTtnQkFDN0UsZUFBZSxFQUFFLEVBQUU7YUFDcEI7U0FDRixDQUFDO0lBU0YsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGFBQWE7UUFDbEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO1FBQ3BDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGlCQUFpQjtRQUN0QixJQUFJLENBQUMsR0FBRyxDQUFDLDZCQUE2QixDQUFDLENBQUM7UUFDeEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsR0FBRyxDQUFDLHlCQUF5QixDQUFDLENBQUM7UUFDcEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLElBQUksQ0FBQyxHQUFHLENBQUMsaUNBQWlDLENBQUMsQ0FBQztRQUM1QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxhQUFhO1FBQ2xCLElBQUksQ0FBQyxHQUFHLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxjQUFjO1FBQ25CLElBQUksQ0FBQyxHQUFHLENBQUMsMEJBQTBCLENBQUMsQ0FBQztRQUNyQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxZQUFZO1FBQ2pCLElBQUksQ0FBQyxHQUFHLENBQUMsd0JBQXdCLENBQUMsQ0FBQztRQUNuQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxhQUFhO1FBQ2xCLElBQUksQ0FBQyxHQUFHLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx5QkFBeUI7UUFDOUIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxxQ0FBcUMsQ0FBQyxDQUFDO1FBQ2hELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFVBQVU7UUFDZixJQUFJLENBQUMsR0FBRyxDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFDakMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLElBQUksQ0FBQyxHQUFHLENBQUMsOEJBQThCLENBQUMsQ0FBQztRQUN6QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxhQUFhO1FBQ2xCLElBQUksQ0FBQyxHQUFHLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxhQUFhO1FBQ2xCLElBQUksQ0FBQyxHQUFHLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx1QkFBdUI7UUFDNUIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxtQ0FBbUMsQ0FBQyxDQUFDO1FBQzlDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixJQUFJLENBQUMsR0FBRyxDQUFDLGlDQUFpQyxDQUFDLENBQUM7UUFDNUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0JBQW9CO1FBQ3pCLElBQUksQ0FBQyxHQUFHLENBQUMsZ0NBQWdDLENBQUMsQ0FBQztRQUMzQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxXQUFXO1FBQ2hCLElBQUksQ0FBQyxHQUFHLENBQUMsdUJBQXVCLENBQUMsQ0FBQztRQUNsQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQkFBb0I7UUFDekIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO1FBQzNDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFlBQVk7UUFDakIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO1FBQ25DLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixJQUFJLENBQUMsR0FBRyxDQUFDLDhCQUE4QixDQUFDLENBQUM7UUFDekMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsR0FBRyxDQUFDLHlCQUF5QixDQUFDLENBQUM7UUFDcEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZUFBZTtRQUNwQixJQUFJLENBQUMsR0FBRyxDQUFDLDJCQUEyQixDQUFDLENBQUM7UUFDdEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksaUJBQWlCO1FBQ3RCLElBQUksQ0FBQyxHQUFHLENBQUMsNkJBQTZCLENBQUMsQ0FBQztRQUN4QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxTQUFTLENBQUMsU0FBaUIsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUN2RixJQUFJLEdBQUcsR0FBRyxzRUFBc0UsQ0FBQztRQUNqRixHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFDN0MsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsWUFBWSxFQUFFLE9BQU8sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUNoRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxXQUFXLEVBQUUsTUFBTSxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQzlDLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGNBQWMsRUFBRSxTQUFTLElBQUksS0FBSyxDQUFDLENBQUM7UUFDdEQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ3RCLENBQUM7Q0FDRjtBQTdjRCw4QkE2Y0MiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBBY3Rpb25zLCBQb2xpY3lTdGF0ZW1lbnQsIFJlc291cmNlVHlwZXMgfSBmcm9tIFwiLi4vc2hhcmVkXCI7XG5pbXBvcnQgeyBQb2xpY3lTdGF0ZW1lbnRQcm9wcyB9IGZyb20gXCJAYXdzLWNkay9hd3MtaWFtXCI7XG5cbi8qKlxuICogQWN0aW9uIHByb3ZpZGVyIGZvciBzZXJ2aWNlIG1vYmlsZWh1YlxuICpcbiAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9saXN0X2F3c21vYmlsZWh1Yi5odG1sXG4gKi9cbmV4cG9ydCBjbGFzcyBNb2JpbGVodWIgZXh0ZW5kcyBQb2xpY3lTdGF0ZW1lbnQge1xuICBwdWJsaWMgc2VydmljZVByZWZpeCA9ICdtb2JpbGVodWInO1xuICBwdWJsaWMgYWN0aW9uczogQWN0aW9ucyA9IHtcbiAgICBcIkNyZWF0ZVByb2plY3RcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbW9iaWxlLWh1Yi9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvbWFuYWdlZC1wb2xpY2llcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQ3JlYXRlIGEgcHJvamVjdFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCJcbiAgICB9LFxuICAgIFwiQ3JlYXRlU2VydmljZVJvbGVcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbW9iaWxlLWh1Yi9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvbWFuYWdlZC1wb2xpY2llcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRW5hYmxlIEFXUyBNb2JpbGUgSHViIGluIHRoZSBhY2NvdW50IGJ5IGNyZWF0aW5nIHRoZSByZXF1aXJlZCBzZXJ2aWNlIHJvbGVcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiXG4gICAgfSxcbiAgICBcIkRlbGV0ZVByb2plY3RcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbW9iaWxlLWh1Yi9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvbWFuYWdlZC1wb2xpY2llcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVsZXRlIHRoZSBzcGVjaWZpZWQgcHJvamVjdFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInByb2plY3RcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlbGV0ZVByb2plY3RTbmFwc2hvdFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tb2JpbGUtaHViL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9tYW5hZ2VkLXBvbGljaWVzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJEZWxldGUgYSBzYXZlZCBzbmFwc2hvdCBvZiBwcm9qZWN0IGNvbmZpZ3VyYXRpb25cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiXG4gICAgfSxcbiAgICBcIkRlcGxveVRvU3RhZ2VcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbW9iaWxlLWh1Yi9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvbWFuYWdlZC1wb2xpY2llcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVwbG95IGNoYW5nZXMgdG8gdGhlIHNwZWNpZmllZCBzdGFnZVwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCJcbiAgICB9LFxuICAgIFwiRGVzY3JpYmVCdW5kbGVcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbW9iaWxlLWh1Yi9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvbWFuYWdlZC1wb2xpY2llcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVzY3JpYmUgdGhlIGRvd25sb2FkIGJ1bmRsZVwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIlxuICAgIH0sXG4gICAgXCJFeHBvcnRCdW5kbGVcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbW9iaWxlLWh1Yi9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvbWFuYWdlZC1wb2xpY2llcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRXhwb3J0IHRoZSBkb3dubG9hZCBidW5kbGVcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCJcbiAgICB9LFxuICAgIFwiRXhwb3J0UHJvamVjdFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tb2JpbGUtaHViL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9tYW5hZ2VkLXBvbGljaWVzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJFeHBvcnQgdGhlIHByb2plY3QgY29uZmlndXJhdGlvblwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwicHJvamVjdFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiR2VuZXJhdGVQcm9qZWN0UGFyYW1ldGVyc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tb2JpbGUtaHViL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9tYW5hZ2VkLXBvbGljaWVzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHZW5lcmF0ZSBwcm9qZWN0IHBhcmFtZXRlcnMgcmVxdWlyZWQgZm9yIGNvZGUgZ2VuZXJhdGlvblwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInByb2plY3RcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkdldFByb2plY3RcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbW9iaWxlLWh1Yi9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvbWFuYWdlZC1wb2xpY2llcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR2V0IHByb2plY3QgY29uZmlndXJhdGlvbiBhbmQgcmVzb3VyY2VzXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJwcm9qZWN0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJHZXRQcm9qZWN0U25hcHNob3RcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbW9iaWxlLWh1Yi9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvbWFuYWdlZC1wb2xpY2llcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRmV0Y2ggdGhlIHByZXZpb3VzbHkgZXhwb3J0ZWQgcHJvamVjdCBjb25maWd1cmF0aW9uIHNuYXBzaG90XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiXG4gICAgfSxcbiAgICBcIkltcG9ydFByb2plY3RcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbW9iaWxlLWh1Yi9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvbWFuYWdlZC1wb2xpY2llcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQ3JlYXRlIGEgbmV3IHByb2plY3QgZnJvbSB0aGUgcHJldmlvdXNseSBleHBvcnRlZCBwcm9qZWN0IGNvbmZpZ3VyYXRpb25cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiXG4gICAgfSxcbiAgICBcIkluc3RhbGxCdW5kbGVcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbW9iaWxlLWh1Yi9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvbWFuYWdlZC1wb2xpY2llcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiSW5zdGFsbCBhIGJ1bmRsZSBpbiB0aGUgcHJvamVjdCBkZXBsb3ltZW50cyBTMyBidWNrZXRcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiXG4gICAgfSxcbiAgICBcIkxpc3RBdmFpbGFibGVDb25uZWN0b3JzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21vYmlsZS1odWIvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL21hbmFnZWQtcG9saWNpZXMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkxpc3QgdGhlIGF2YWlsYWJsZSBTYWFTIChTb2Z0d2FyZSBhcyBhIFNlcnZpY2UpIGNvbm5lY3RvcnNcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCJcbiAgICB9LFxuICAgIFwiTGlzdEF2YWlsYWJsZUZlYXR1cmVzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21vYmlsZS1odWIvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL21hbmFnZWQtcG9saWNpZXMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkxpc3QgYXZhaWxhYmxlIGZlYXR1cmVzXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiXG4gICAgfSxcbiAgICBcIkxpc3RBdmFpbGFibGVSZWdpb25zXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21vYmlsZS1odWIvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL21hbmFnZWQtcG9saWNpZXMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkxpc3QgYXZhaWxhYmxlIHJlZ2lvbnMgZm9yIHByb2plY3RzXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiXG4gICAgfSxcbiAgICBcIkxpc3RCdW5kbGVzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21vYmlsZS1odWIvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL21hbmFnZWQtcG9saWNpZXMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkxpc3QgdGhlIGF2YWlsYWJsZSBkb3dubG9hZCBidW5kbGVzXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiXG4gICAgfSxcbiAgICBcIkxpc3RQcm9qZWN0U25hcHNob3RzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21vYmlsZS1odWIvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL21hbmFnZWQtcG9saWNpZXMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkxpc3Qgc2F2ZWQgc25hcHNob3RzIG9mIHByb2plY3QgY29uZmlndXJhdGlvblwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIlxuICAgIH0sXG4gICAgXCJMaXN0UHJvamVjdHNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbW9iaWxlLWh1Yi9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvbWFuYWdlZC1wb2xpY2llcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiTGlzdCBwcm9qZWN0c1wiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIlxuICAgIH0sXG4gICAgXCJTeW5jaHJvbml6ZVByb2plY3RcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbW9iaWxlLWh1Yi9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvbWFuYWdlZC1wb2xpY2llcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiU3luY2hyb25pemUgc3RhdGUgb2YgcmVzb3VyY2VzIGludG8gcHJvamVjdFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInByb2plY3RcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIlVwZGF0ZVByb2plY3RcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbW9iaWxlLWh1Yi9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvbWFuYWdlZC1wb2xpY2llcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiVXBkYXRlIHByb2plY3RcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJwcm9qZWN0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJWYWxpZGF0ZVByb2plY3RcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbW9iaWxlLWh1Yi9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvbWFuYWdlZC1wb2xpY2llcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiVmFsaWRhdGUgYSBtb2JpbGUgaHViIHByb2plY3QuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiXG4gICAgfSxcbiAgICBcIlZlcmlmeVNlcnZpY2VSb2xlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21vYmlsZS1odWIvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL21hbmFnZWQtcG9saWNpZXMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlZlcmlmeSBBV1MgTW9iaWxlIEh1YiBpcyBlbmFibGVkIGluIHRoZSBhY2NvdW50XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiXG4gICAgfVxuICB9O1xuICBwdWJsaWMgcmVzb3VyY2VUeXBlczogUmVzb3VyY2VUeXBlcyA9IHtcbiAgICBcInByb2plY3RcIjoge1xuICAgICAgXCJuYW1lXCI6IFwicHJvamVjdFwiLFxuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbW9iaWxlLWh1Yi9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvcmVmZXJlbmNlLW1vYmlsZS1odWItaWFtLW1hbmFnZWQtcG9saWNpZXMuaHRtbFwiLFxuICAgICAgXCJhcm5cIjogXCJhcm46JHtQYXJ0aXRpb259Om1vYmlsZWh1Yjoke1JlZ2lvbn06JHtBY2NvdW50fTpwcm9qZWN0LyR7UHJvamVjdElkfVwiLFxuICAgICAgXCJjb25kaXRpb25LZXlzXCI6IFtdXG4gICAgfVxuICB9O1xuXG4gIC8qKlxuICAgKiBBY3Rpb24gcHJvdmlkZXIgZm9yIHNlcnZpY2UgbW9iaWxlaHViXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9saXN0X2F3c21vYmlsZWh1Yi5odG1sXG4gICAqL1xuICBjb25zdHJ1Y3RvciAocHJvcHM/OiBQb2xpY3lTdGF0ZW1lbnRQcm9wcykge1xuICAgIHN1cGVyKHByb3BzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGUgYSBwcm9qZWN0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21vYmlsZS1odWIvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL21hbmFnZWQtcG9saWNpZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIGNyZWF0ZVByb2plY3QoKSB7XG4gICAgdGhpcy5hZGQoJ21vYmlsZWh1YjpDcmVhdGVQcm9qZWN0Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRW5hYmxlIEFXUyBNb2JpbGUgSHViIGluIHRoZSBhY2NvdW50IGJ5IGNyZWF0aW5nIHRoZSByZXF1aXJlZCBzZXJ2aWNlIHJvbGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbW9iaWxlLWh1Yi9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvbWFuYWdlZC1wb2xpY2llcy5odG1sXG4gICAqL1xuICBwdWJsaWMgY3JlYXRlU2VydmljZVJvbGUoKSB7XG4gICAgdGhpcy5hZGQoJ21vYmlsZWh1YjpDcmVhdGVTZXJ2aWNlUm9sZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlbGV0ZSB0aGUgc3BlY2lmaWVkIHByb2plY3RcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbW9iaWxlLWh1Yi9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvbWFuYWdlZC1wb2xpY2llcy5odG1sXG4gICAqL1xuICBwdWJsaWMgZGVsZXRlUHJvamVjdCgpIHtcbiAgICB0aGlzLmFkZCgnbW9iaWxlaHViOkRlbGV0ZVByb2plY3QnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWxldGUgYSBzYXZlZCBzbmFwc2hvdCBvZiBwcm9qZWN0IGNvbmZpZ3VyYXRpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbW9iaWxlLWh1Yi9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvbWFuYWdlZC1wb2xpY2llcy5odG1sXG4gICAqL1xuICBwdWJsaWMgZGVsZXRlUHJvamVjdFNuYXBzaG90KCkge1xuICAgIHRoaXMuYWRkKCdtb2JpbGVodWI6RGVsZXRlUHJvamVjdFNuYXBzaG90Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVwbG95IGNoYW5nZXMgdG8gdGhlIHNwZWNpZmllZCBzdGFnZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tb2JpbGUtaHViL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9tYW5hZ2VkLXBvbGljaWVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyBkZXBsb3lUb1N0YWdlKCkge1xuICAgIHRoaXMuYWRkKCdtb2JpbGVodWI6RGVwbG95VG9TdGFnZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlc2NyaWJlIHRoZSBkb3dubG9hZCBidW5kbGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tb2JpbGUtaHViL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9tYW5hZ2VkLXBvbGljaWVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyBkZXNjcmliZUJ1bmRsZSgpIHtcbiAgICB0aGlzLmFkZCgnbW9iaWxlaHViOkRlc2NyaWJlQnVuZGxlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRXhwb3J0IHRoZSBkb3dubG9hZCBidW5kbGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tb2JpbGUtaHViL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9tYW5hZ2VkLXBvbGljaWVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyBleHBvcnRCdW5kbGUoKSB7XG4gICAgdGhpcy5hZGQoJ21vYmlsZWh1YjpFeHBvcnRCdW5kbGUnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBFeHBvcnQgdGhlIHByb2plY3QgY29uZmlndXJhdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21vYmlsZS1odWIvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL21hbmFnZWQtcG9saWNpZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIGV4cG9ydFByb2plY3QoKSB7XG4gICAgdGhpcy5hZGQoJ21vYmlsZWh1YjpFeHBvcnRQcm9qZWN0Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR2VuZXJhdGUgcHJvamVjdCBwYXJhbWV0ZXJzIHJlcXVpcmVkIGZvciBjb2RlIGdlbmVyYXRpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbW9iaWxlLWh1Yi9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvbWFuYWdlZC1wb2xpY2llcy5odG1sXG4gICAqL1xuICBwdWJsaWMgZ2VuZXJhdGVQcm9qZWN0UGFyYW1ldGVycygpIHtcbiAgICB0aGlzLmFkZCgnbW9iaWxlaHViOkdlbmVyYXRlUHJvamVjdFBhcmFtZXRlcnMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHZXQgcHJvamVjdCBjb25maWd1cmF0aW9uIGFuZCByZXNvdXJjZXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tb2JpbGUtaHViL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9tYW5hZ2VkLXBvbGljaWVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyBnZXRQcm9qZWN0KCkge1xuICAgIHRoaXMuYWRkKCdtb2JpbGVodWI6R2V0UHJvamVjdCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEZldGNoIHRoZSBwcmV2aW91c2x5IGV4cG9ydGVkIHByb2plY3QgY29uZmlndXJhdGlvbiBzbmFwc2hvdFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21vYmlsZS1odWIvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL21hbmFnZWQtcG9saWNpZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIGdldFByb2plY3RTbmFwc2hvdCgpIHtcbiAgICB0aGlzLmFkZCgnbW9iaWxlaHViOkdldFByb2plY3RTbmFwc2hvdCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZSBhIG5ldyBwcm9qZWN0IGZyb20gdGhlIHByZXZpb3VzbHkgZXhwb3J0ZWQgcHJvamVjdCBjb25maWd1cmF0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21vYmlsZS1odWIvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL21hbmFnZWQtcG9saWNpZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIGltcG9ydFByb2plY3QoKSB7XG4gICAgdGhpcy5hZGQoJ21vYmlsZWh1YjpJbXBvcnRQcm9qZWN0Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogSW5zdGFsbCBhIGJ1bmRsZSBpbiB0aGUgcHJvamVjdCBkZXBsb3ltZW50cyBTMyBidWNrZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbW9iaWxlLWh1Yi9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvbWFuYWdlZC1wb2xpY2llcy5odG1sXG4gICAqL1xuICBwdWJsaWMgaW5zdGFsbEJ1bmRsZSgpIHtcbiAgICB0aGlzLmFkZCgnbW9iaWxlaHViOkluc3RhbGxCdW5kbGUnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBMaXN0IHRoZSBhdmFpbGFibGUgU2FhUyAoU29mdHdhcmUgYXMgYSBTZXJ2aWNlKSBjb25uZWN0b3JzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbW9iaWxlLWh1Yi9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvbWFuYWdlZC1wb2xpY2llcy5odG1sXG4gICAqL1xuICBwdWJsaWMgbGlzdEF2YWlsYWJsZUNvbm5lY3RvcnMoKSB7XG4gICAgdGhpcy5hZGQoJ21vYmlsZWh1YjpMaXN0QXZhaWxhYmxlQ29ubmVjdG9ycycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIExpc3QgYXZhaWxhYmxlIGZlYXR1cmVzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbW9iaWxlLWh1Yi9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvbWFuYWdlZC1wb2xpY2llcy5odG1sXG4gICAqL1xuICBwdWJsaWMgbGlzdEF2YWlsYWJsZUZlYXR1cmVzKCkge1xuICAgIHRoaXMuYWRkKCdtb2JpbGVodWI6TGlzdEF2YWlsYWJsZUZlYXR1cmVzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogTGlzdCBhdmFpbGFibGUgcmVnaW9ucyBmb3IgcHJvamVjdHNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tb2JpbGUtaHViL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9tYW5hZ2VkLXBvbGljaWVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyBsaXN0QXZhaWxhYmxlUmVnaW9ucygpIHtcbiAgICB0aGlzLmFkZCgnbW9iaWxlaHViOkxpc3RBdmFpbGFibGVSZWdpb25zJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogTGlzdCB0aGUgYXZhaWxhYmxlIGRvd25sb2FkIGJ1bmRsZXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tb2JpbGUtaHViL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9tYW5hZ2VkLXBvbGljaWVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyBsaXN0QnVuZGxlcygpIHtcbiAgICB0aGlzLmFkZCgnbW9iaWxlaHViOkxpc3RCdW5kbGVzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogTGlzdCBzYXZlZCBzbmFwc2hvdHMgb2YgcHJvamVjdCBjb25maWd1cmF0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbW9iaWxlLWh1Yi9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvbWFuYWdlZC1wb2xpY2llcy5odG1sXG4gICAqL1xuICBwdWJsaWMgbGlzdFByb2plY3RTbmFwc2hvdHMoKSB7XG4gICAgdGhpcy5hZGQoJ21vYmlsZWh1YjpMaXN0UHJvamVjdFNuYXBzaG90cycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIExpc3QgcHJvamVjdHNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tb2JpbGUtaHViL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9tYW5hZ2VkLXBvbGljaWVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyBsaXN0UHJvamVjdHMoKSB7XG4gICAgdGhpcy5hZGQoJ21vYmlsZWh1YjpMaXN0UHJvamVjdHMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBTeW5jaHJvbml6ZSBzdGF0ZSBvZiByZXNvdXJjZXMgaW50byBwcm9qZWN0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21vYmlsZS1odWIvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL21hbmFnZWQtcG9saWNpZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHN5bmNocm9uaXplUHJvamVjdCgpIHtcbiAgICB0aGlzLmFkZCgnbW9iaWxlaHViOlN5bmNocm9uaXplUHJvamVjdCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFVwZGF0ZSBwcm9qZWN0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21vYmlsZS1odWIvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL21hbmFnZWQtcG9saWNpZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHVwZGF0ZVByb2plY3QoKSB7XG4gICAgdGhpcy5hZGQoJ21vYmlsZWh1YjpVcGRhdGVQcm9qZWN0Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVmFsaWRhdGUgYSBtb2JpbGUgaHViIHByb2plY3QuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbW9iaWxlLWh1Yi9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvbWFuYWdlZC1wb2xpY2llcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdmFsaWRhdGVQcm9qZWN0KCkge1xuICAgIHRoaXMuYWRkKCdtb2JpbGVodWI6VmFsaWRhdGVQcm9qZWN0Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVmVyaWZ5IEFXUyBNb2JpbGUgSHViIGlzIGVuYWJsZWQgaW4gdGhlIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tb2JpbGUtaHViL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9tYW5hZ2VkLXBvbGljaWVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB2ZXJpZnlTZXJ2aWNlUm9sZSgpIHtcbiAgICB0aGlzLmFkZCgnbW9iaWxlaHViOlZlcmlmeVNlcnZpY2VSb2xlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgcHJvamVjdCB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tb2JpbGUtaHViL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9yZWZlcmVuY2UtbW9iaWxlLWh1Yi1pYW0tbWFuYWdlZC1wb2xpY2llcy5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBwcm9qZWN0SWQgLSBJZGVudGlmaWVyIGZvciB0aGUgcHJvamVjdElkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIHJlZ2lvbnMuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLlxuICAgKi9cbiAgcHVibGljIG9uUHJvamVjdChwcm9qZWN0SWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06bW9iaWxlaHViOiR7UmVnaW9ufToke0FjY291bnR9OnByb2plY3QvJHtQcm9qZWN0SWR9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQcm9qZWN0SWR9JywgcHJvamVjdElkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZWdpb259JywgcmVnaW9uIHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGFydGl0aW9ufScsIHBhcnRpdGlvbiB8fCAnYXdzJyk7XG4gICAgcmV0dXJuIHRoaXMub24oYXJuKTtcbiAgfVxufVxuIl19