"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service ram
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsresourceaccessmanager.html
 */
class Ram extends shared_1.PolicyStatement {
    /**
     * Action provider for service ram
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsresourceaccessmanager.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'ram';
        this.actions = {
            "AcceptResourceShareInvitation": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_AcceptResourceShareInvitation.html",
                "description": "Accept the specified resource share invitation",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resource-share-invitation": {
                        "required": true
                    }
                },
                "conditions": [
                    "ram:ShareOwnerAccountId"
                ]
            },
            "AssociateResourceShare": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_AssociateResourceShare.html",
                "description": "Associates resource(s) and/or principal(s) to a resource share",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resource-share": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}",
                    "ram:ResourceShareName",
                    "ram:AllowsExternalPrincipals",
                    "ram:Principal",
                    "ram:RequestedResourceType",
                    "ram:ResourceArn"
                ]
            },
            "AssociateResourceSharePermission": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_AssociateResourceSharePermission.html",
                "description": "Associate a Permission with a Resource Share",
                "accessLevel": "Write",
                "resourceTypes": {
                    "permission": {
                        "required": true
                    },
                    "resource-share": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}",
                    "ram:AllowsExternalPrincipals",
                    "ram:ResourceShareName",
                    "ram:PermissionArn"
                ]
            },
            "CreateResourceShare": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_CreateResourceShare.html",
                "description": "Create resource share with provided resource(s) and/or principal(s)",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "ram:RequestedResourceType",
                    "ram:ResourceArn",
                    "ram:RequestedAllowsExternalPrincipals",
                    "ram:Principal"
                ]
            },
            "DeleteResourceShare": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_DeleteResourceShare.html",
                "description": "Delete resource share",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resource-share": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}",
                    "ram:ResourceShareName",
                    "ram:AllowsExternalPrincipals"
                ]
            },
            "DisassociateResourceShare": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_DisassociateResourceShare.html",
                "description": "Disassociates resource(s) and/or principal(s) from a resource share",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resource-share": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}",
                    "ram:ResourceShareName",
                    "ram:AllowsExternalPrincipals",
                    "ram:Principal",
                    "ram:RequestedResourceType",
                    "ram:ResourceArn"
                ]
            },
            "DisassociateResourceSharePermission": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_DisassociateResourceSharePermission.html",
                "description": "Disassociate a Permission from a Resource Share",
                "accessLevel": "Write",
                "resourceTypes": {
                    "permission": {
                        "required": true
                    },
                    "resource-share": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}",
                    "ram:AllowsExternalPrincipals",
                    "ram:ResourceShareName",
                    "ram:PermissionArn"
                ]
            },
            "EnableSharingWithAwsOrganization": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_EnableSharingWithAwsOrganization.html",
                "description": "Grants permission to access customer's organization and create a SLR in the customer's account",
                "accessLevel": "Write"
            },
            "GetPermission": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_GetPermission.html",
                "description": "Gets the contents of an AWS RAM permission",
                "accessLevel": "Read",
                "resourceTypes": {
                    "permission": {
                        "required": true
                    }
                },
                "conditions": [
                    "ram:PermissionArn"
                ]
            },
            "GetResourcePolicies": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_GetResourcePolicies.html",
                "description": "Gets the policies for the specified resources that you own and have shared",
                "accessLevel": "Read"
            },
            "GetResourceShareAssociations": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_GetResourceShareAssociations.html",
                "description": "Get a set of resource share associations from a provided list or with a specified status of the specified type",
                "accessLevel": "Read"
            },
            "GetResourceShareInvitations": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_GetResourceShareInvitations.html",
                "description": "Get resource share invitations by the specified invitation arn or those for the resource share",
                "accessLevel": "Read"
            },
            "GetResourceShares": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_GetResourceShares.html",
                "description": "Get a set of resource shares from a provided list or with a specified status",
                "accessLevel": "Read"
            },
            "ListPendingInvitationResources": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_ListPendingInvitationResources.html",
                "description": "Lists the resources in a resource share that is shared with you but that the invitation is still pending for",
                "accessLevel": "Read",
                "resourceTypes": {
                    "resource-share-invitation": {
                        "required": true
                    }
                }
            },
            "ListPermissions": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_ListPermissions.html",
                "description": "Lists the AWS RAM permissions",
                "accessLevel": "List"
            },
            "ListPrincipals": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_ListPrincipals.html",
                "description": "Lists the principals that you have shared resources with or that have shared resources with you",
                "accessLevel": "List"
            },
            "ListResourceSharePermissions": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_ListResourceSharePermissions.html",
                "description": "List the Permissions associated with a Resource Share",
                "accessLevel": "List",
                "resourceTypes": {
                    "resource-share": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}",
                    "ram:ResourceShareName",
                    "ram:AllowsExternalPrincipals"
                ]
            },
            "ListResources": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_ListResources.html",
                "description": "Lists the resources that you added to a resource shares or the resources that are shared with you",
                "accessLevel": "List"
            },
            "RejectResourceShareInvitation": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_RejectResourceShareInvitation.html",
                "description": "Reject the specified resource share invitation",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resource-share-invitation": {
                        "required": true
                    }
                },
                "conditions": [
                    "ram:ShareOwnerAccountId"
                ]
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_TagResource.html",
                "description": "Tag the specified resources share",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resource-share": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_UntagResource.html",
                "description": "Untag the specified resource share",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resource-share": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UpdateResourceShare": {
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_UpdateResourceShare.html",
                "description": "Update attributes of the resource share",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resource-share": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}",
                    "ram:ResourceShareName",
                    "ram:AllowsExternalPrincipals",
                    "ram:RequestedAllowsExternalPrincipals"
                ]
            }
        };
        this.resourceTypes = {
            "resource-share": {
                "name": "resource-share",
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_ResourceShare.html",
                "arn": "arn:${Partition}:ram:${Region}:${Account}:resource-share/${ResourcePath}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "ram:AllowsExternalPrincipals",
                    "ram:ResourceShareName"
                ]
            },
            "resource-share-invitation": {
                "name": "resource-share-invitation",
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_ResourceShareInvitation.html",
                "arn": "arn:${Partition}:ram:${Region}:${Account}:resource-share-invitation/${ResourcePath}",
                "conditionKeys": []
            },
            "permission": {
                "name": "permission",
                "url": "https://docs.aws.amazon.com/ram/latest/APIReference/API_ResourceSharePermissionDetail.html",
                "arn": "arn:${Partition}:ram::${Account}:permission/${ResourcePath}",
                "conditionKeys": [
                    "ram:PermissionArn"
                ]
            }
        };
    }
    /**
     * Accept the specified resource share invitation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_AcceptResourceShareInvitation.html
     */
    acceptResourceShareInvitation() {
        this.add('ram:AcceptResourceShareInvitation');
        return this;
    }
    /**
     * Associates resource(s) and/or principal(s) to a resource share
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_AssociateResourceShare.html
     */
    associateResourceShare() {
        this.add('ram:AssociateResourceShare');
        return this;
    }
    /**
     * Associate a Permission with a Resource Share
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_AssociateResourceSharePermission.html
     */
    associateResourceSharePermission() {
        this.add('ram:AssociateResourceSharePermission');
        return this;
    }
    /**
     * Create resource share with provided resource(s) and/or principal(s)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_CreateResourceShare.html
     */
    createResourceShare() {
        this.add('ram:CreateResourceShare');
        return this;
    }
    /**
     * Delete resource share
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_DeleteResourceShare.html
     */
    deleteResourceShare() {
        this.add('ram:DeleteResourceShare');
        return this;
    }
    /**
     * Disassociates resource(s) and/or principal(s) from a resource share
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_DisassociateResourceShare.html
     */
    disassociateResourceShare() {
        this.add('ram:DisassociateResourceShare');
        return this;
    }
    /**
     * Disassociate a Permission from a Resource Share
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_DisassociateResourceSharePermission.html
     */
    disassociateResourceSharePermission() {
        this.add('ram:DisassociateResourceSharePermission');
        return this;
    }
    /**
     * Grants permission to access customer's organization and create a SLR in the customer's account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_EnableSharingWithAwsOrganization.html
     */
    enableSharingWithAwsOrganization() {
        this.add('ram:EnableSharingWithAwsOrganization');
        return this;
    }
    /**
     * Gets the contents of an AWS RAM permission
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_GetPermission.html
     */
    getPermission() {
        this.add('ram:GetPermission');
        return this;
    }
    /**
     * Gets the policies for the specified resources that you own and have shared
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_GetResourcePolicies.html
     */
    getResourcePolicies() {
        this.add('ram:GetResourcePolicies');
        return this;
    }
    /**
     * Get a set of resource share associations from a provided list or with a specified status of the specified type
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_GetResourceShareAssociations.html
     */
    getResourceShareAssociations() {
        this.add('ram:GetResourceShareAssociations');
        return this;
    }
    /**
     * Get resource share invitations by the specified invitation arn or those for the resource share
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_GetResourceShareInvitations.html
     */
    getResourceShareInvitations() {
        this.add('ram:GetResourceShareInvitations');
        return this;
    }
    /**
     * Get a set of resource shares from a provided list or with a specified status
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_GetResourceShares.html
     */
    getResourceShares() {
        this.add('ram:GetResourceShares');
        return this;
    }
    /**
     * Lists the resources in a resource share that is shared with you but that the invitation is still pending for
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ListPendingInvitationResources.html
     */
    listPendingInvitationResources() {
        this.add('ram:ListPendingInvitationResources');
        return this;
    }
    /**
     * Lists the AWS RAM permissions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ListPermissions.html
     */
    listPermissions() {
        this.add('ram:ListPermissions');
        return this;
    }
    /**
     * Lists the principals that you have shared resources with or that have shared resources with you
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ListPrincipals.html
     */
    listPrincipals() {
        this.add('ram:ListPrincipals');
        return this;
    }
    /**
     * List the Permissions associated with a Resource Share
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ListResourceSharePermissions.html
     */
    listResourceSharePermissions() {
        this.add('ram:ListResourceSharePermissions');
        return this;
    }
    /**
     * Lists the resources that you added to a resource shares or the resources that are shared with you
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ListResources.html
     */
    listResources() {
        this.add('ram:ListResources');
        return this;
    }
    /**
     * Reject the specified resource share invitation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_RejectResourceShareInvitation.html
     */
    rejectResourceShareInvitation() {
        this.add('ram:RejectResourceShareInvitation');
        return this;
    }
    /**
     * Tag the specified resources share
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_TagResource.html
     */
    tagResource() {
        this.add('ram:TagResource');
        return this;
    }
    /**
     * Untag the specified resource share
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_UntagResource.html
     */
    untagResource() {
        this.add('ram:UntagResource');
        return this;
    }
    /**
     * Update attributes of the resource share
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_UpdateResourceShare.html
     */
    updateResourceShare() {
        this.add('ram:UpdateResourceShare');
        return this;
    }
    /**
     * Adds a resource of type resource-share to the statement
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ResourceShare.html
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     *  - ram:AllowsExternalPrincipals
     *  - ram:ResourceShareName
     */
    onResourceShare(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:ram:${Region}:${Account}:resource-share/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type resource-share-invitation to the statement
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ResourceShareInvitation.html
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onResourceShareInvitation(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:ram:${Region}:${Account}:resource-share-invitation/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type permission to the statement
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ResourceSharePermissionDetail.html
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - ram:PermissionArn
     */
    onPermission(resourcePath, account, partition) {
        var arn = 'arn:${Partition}:ram::${Account}:permission/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Indicates that the action can only be performed on resource shares that allow or deny sharing with external principals. For example, specify true if the action can only be performed on resource shares that allow sharing with external principals. External principals are AWS accounts that are outside of its AWS organization
     *
     * https://docs.aws.amazon.com/ram/latest/userguide/iam-policies.html#iam-policies-condition
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifAllowsExternalPrincipals(value) {
        return this.if(`ram:AllowsExternalPrincipals`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Indicates that the action can only be performed on a resource using the specified Permission ARN.
     *
     * https://docs.aws.amazon.com/ram/latest/userguide/iam-policies.html#iam-policies-condition
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifPermissionArn(value, operator) {
        return this.if(`ram:PermissionArn`, value, operator || 'ArnEquals');
    }
    /**
     * Principals with the specified format can be associated to or disassociated from a resource share
     *
     * https://docs.aws.amazon.com/ram/latest/userguide/iam-policies.html#iam-policies-condition
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPrincipal(value, operator) {
        return this.if(`ram:Principal`, value, operator || 'StringLike');
    }
    /**
     * The request must have the specified value for 'allowExternalPrincipals'. External principals are AWS accounts that are outside of its AWS Organization
     *
     * https://docs.aws.amazon.com/ram/latest/userguide/iam-policies.html#iam-policies-condition
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifRequestedAllowsExternalPrincipals(value) {
        return this.if(`ram:RequestedAllowsExternalPrincipals`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Indicates that the action can only be performed on the specified resource type
     *
     * https://docs.aws.amazon.com/ram/latest/userguide/iam-policies.html#iam-policies-condition
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestedResourceType(value, operator) {
        return this.if(`ram:RequestedResourceType`, value, operator || 'StringLike');
    }
    /**
     * Indicates that the action can only be performed on a resource with the specified ARN.
     *
     * https://docs.aws.amazon.com/ram/latest/userguide/iam-policies.html#iam-policies-condition
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifResourceArn(value, operator) {
        return this.if(`ram:ResourceArn`, value, operator || 'ArnEquals');
    }
    /**
     * Indicates that the action can only be performed on a resource share with the specified name.
     *
     * https://docs.aws.amazon.com/ram/latest/userguide/iam-policies.html#iam-policies-condition
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceShareName(value, operator) {
        return this.if(`ram:ResourceShareName`, value, operator || 'StringLike');
    }
    /**
     * Indicates that the action can only be performed on resource shares owned by a specific account. For example, you can use this condition key to specify which resource share invitations can be accepted or rejected based on the resource share owner’s account ID.
     *
     * https://docs.aws.amazon.com/ram/latest/userguide/iam-policies.html#iam-policies-condition
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifShareOwnerAccountId(value, operator) {
        return this.if(`ram:ShareOwnerAccountId`, value, operator || 'StringLike');
    }
}
exports.Ram = Ram;
//# sourceMappingURL=data:application/json;base64,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