"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service transfer
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awstransferforsftp.html
 */
class Transfer extends shared_1.PolicyStatement {
    /**
     * Action provider for service transfer
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awstransferforsftp.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'transfer';
        this.actions = {
            "CreateServer": {
                "url": "https://docs.aws.amazon.com/transfer/latest/userguide/API_CreateServer.html",
                "description": "Enables the caller to create a server.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "CreateUser": {
                "url": "https://docs.aws.amazon.com/transfer/latest/userguide/API_CreateUser.html",
                "description": "Enables the caller to add a user associated with a server.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "server": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "DeleteServer": {
                "url": "https://docs.aws.amazon.com/transfer/latest/userguide/API_DeleteServer.html",
                "description": "Enables the caller to delete a server.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "server": {
                        "required": true
                    }
                }
            },
            "DeleteSshPublicKey": {
                "url": "https://docs.aws.amazon.com/transfer/latest/userguide/API_DeleteSshPublicKey.html",
                "description": "Enables the caller to delete an SSH public key from a user.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "DeleteUser": {
                "url": "https://docs.aws.amazon.com/transfer/latest/userguide/API_DeleteUser.html",
                "description": "Enables the caller to delete a user associated with a server.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "DescribeServer": {
                "url": "https://docs.aws.amazon.com/transfer/latest/userguide/API_DescribeServer.html",
                "description": "Enables the caller to describe a server.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "server": {
                        "required": true
                    }
                }
            },
            "DescribeUser": {
                "url": "https://docs.aws.amazon.com/transfer/latest/userguide/API_DescribeUser.html",
                "description": "Enables the caller to describe a user associated with a server.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "ImportSshPublicKey": {
                "url": "https://docs.aws.amazon.com/transfer/latest/userguide/API_ImportSshPublicKey.html",
                "description": "Enables the caller to add an SSH public key to a user.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "ListServers": {
                "url": "https://docs.aws.amazon.com/transfer/latest/userguide/API_ListServers.html",
                "description": "Enables the caller to list servers",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/transfer/latest/userguide/API_ListTagsForResource.html",
                "description": "Enables the caller to list tags for a server or a user.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "server": {
                        "required": false
                    },
                    "user": {
                        "required": false
                    }
                }
            },
            "ListUsers": {
                "url": "https://docs.aws.amazon.com/transfer/latest/userguide/API_ListUsers.html",
                "description": "Enables the caller to list users associated with a server.",
                "accessLevel": "List",
                "resourceTypes": {
                    "server": {
                        "required": true
                    }
                }
            },
            "StartServer": {
                "url": "https://docs.aws.amazon.com/transfer/latest/userguide/API_StartServer.html",
                "description": "Enables the caller to start a server.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "server": {
                        "required": true
                    }
                }
            },
            "StopServer": {
                "url": "https://docs.aws.amazon.com/transfer/latest/userguide/API_StopServer.html",
                "description": "Enables the caller to stop a server.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "server": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/transfer/latest/userguide/API_TagResource.html",
                "description": "Enables the caller to tag a server or a user.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "server": {
                        "required": false
                    },
                    "user": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "TestIdentityProvider": {
                "url": "https://docs.aws.amazon.com/transfer/latest/userguide/API_TestIdentityProvider.html",
                "description": "Enables the caller to test a server's custom identity provider.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/transfer/latest/userguide/API_UntagResource.html",
                "description": "Enables the caller to untag a server or a user.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "server": {
                        "required": false
                    },
                    "user": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateServer": {
                "url": "https://docs.aws.amazon.com/transfer/latest/userguide/API_UpdateServer.html",
                "description": "Enables the caller to update the configuration of a server",
                "accessLevel": "Write",
                "resourceTypes": {
                    "server": {
                        "required": true
                    }
                }
            },
            "UpdateUser": {
                "url": "https://docs.aws.amazon.com/transfer/latest/userguide/API_UpdateUser.html",
                "description": "Enables the caller to update the configuration of a user",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "user": {
                "name": "user",
                "url": "https://docs.aws.amazon.com/transfer/latest/userguide/users.html",
                "arn": "arn:${Partition}:transfer:${Region}:${Account}:user/${ServerId}/${Username}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "server": {
                "name": "server",
                "url": "https://docs.aws.amazon.com/transfer/latest/userguide/configuring-servers.html",
                "arn": "arn:${Partition}:transfer:${Region}:${Account}:server/${ServerId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Enables the caller to create a server.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_CreateServer.html
     */
    createServer() {
        this.add('transfer:CreateServer');
        return this;
    }
    /**
     * Enables the caller to add a user associated with a server.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_CreateUser.html
     */
    createUser() {
        this.add('transfer:CreateUser');
        return this;
    }
    /**
     * Enables the caller to delete a server.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_DeleteServer.html
     */
    deleteServer() {
        this.add('transfer:DeleteServer');
        return this;
    }
    /**
     * Enables the caller to delete an SSH public key from a user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_DeleteSshPublicKey.html
     */
    deleteSshPublicKey() {
        this.add('transfer:DeleteSshPublicKey');
        return this;
    }
    /**
     * Enables the caller to delete a user associated with a server.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_DeleteUser.html
     */
    deleteUser() {
        this.add('transfer:DeleteUser');
        return this;
    }
    /**
     * Enables the caller to describe a server.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_DescribeServer.html
     */
    describeServer() {
        this.add('transfer:DescribeServer');
        return this;
    }
    /**
     * Enables the caller to describe a user associated with a server.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_DescribeUser.html
     */
    describeUser() {
        this.add('transfer:DescribeUser');
        return this;
    }
    /**
     * Enables the caller to add an SSH public key to a user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_ImportSshPublicKey.html
     */
    importSshPublicKey() {
        this.add('transfer:ImportSshPublicKey');
        return this;
    }
    /**
     * Enables the caller to list servers
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_ListServers.html
     */
    listServers() {
        this.add('transfer:ListServers');
        return this;
    }
    /**
     * Enables the caller to list tags for a server or a user.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('transfer:ListTagsForResource');
        return this;
    }
    /**
     * Enables the caller to list users associated with a server.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_ListUsers.html
     */
    listUsers() {
        this.add('transfer:ListUsers');
        return this;
    }
    /**
     * Enables the caller to start a server.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_StartServer.html
     */
    startServer() {
        this.add('transfer:StartServer');
        return this;
    }
    /**
     * Enables the caller to stop a server.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_StopServer.html
     */
    stopServer() {
        this.add('transfer:StopServer');
        return this;
    }
    /**
     * Enables the caller to tag a server or a user.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_TagResource.html
     */
    tagResource() {
        this.add('transfer:TagResource');
        return this;
    }
    /**
     * Enables the caller to test a server's custom identity provider.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_TestIdentityProvider.html
     */
    testIdentityProvider() {
        this.add('transfer:TestIdentityProvider');
        return this;
    }
    /**
     * Enables the caller to untag a server or a user.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_UntagResource.html
     */
    untagResource() {
        this.add('transfer:UntagResource');
        return this;
    }
    /**
     * Enables the caller to update the configuration of a server
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_UpdateServer.html
     */
    updateServer() {
        this.add('transfer:UpdateServer');
        return this;
    }
    /**
     * Enables the caller to update the configuration of a user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_UpdateUser.html
     */
    updateUser() {
        this.add('transfer:UpdateUser');
        return this;
    }
    /**
     * Adds a resource of type user to the statement
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/users.html
     *
     * @param serverId - Identifier for the serverId.
     * @param username - Identifier for the username.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onUser(serverId, username, account, region, partition) {
        var arn = 'arn:${Partition}:transfer:${Region}:${Account}:user/${ServerId}/${Username}';
        arn = arn.replace('${ServerId}', serverId);
        arn = arn.replace('${Username}', username);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type server to the statement
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/configuring-servers.html
     *
     * @param serverId - Identifier for the serverId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onServer(serverId, account, region, partition) {
        var arn = 'arn:${Partition}:transfer:${Region}:${Account}:server/${ServerId}';
        arn = arn.replace('${ServerId}', serverId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Transfer = Transfer;
//# sourceMappingURL=data:application/json;base64,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