"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service trustedadvisor
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awstrustedadvisor.html
 */
class Trustedadvisor extends shared_1.PolicyStatement {
    /**
     * Action provider for service trustedadvisor
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awstrustedadvisor.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'trustedadvisor';
        this.actions = {
            "DescribeAccount": {
                "url": "",
                "description": "View support plan and various TA preferences.",
                "accessLevel": "Read"
            },
            "DescribeAccountAccess": {
                "url": "",
                "description": "Resolve whether Account has disabled Trusted Advisor",
                "accessLevel": "Read"
            },
            "DescribeCheckItems": {
                "url": "",
                "description": "View details for the check items",
                "accessLevel": "Read",
                "resourceTypes": {
                    "checks": {
                        "required": true
                    }
                }
            },
            "DescribeCheckRefreshStatuses": {
                "url": "",
                "description": "Describe check refresh statuses",
                "accessLevel": "Read",
                "resourceTypes": {
                    "checks": {
                        "required": true
                    }
                }
            },
            "DescribeCheckSummaries": {
                "url": "",
                "description": "Describes the check's summaries",
                "accessLevel": "Read",
                "resourceTypes": {
                    "checks": {
                        "required": true
                    }
                }
            },
            "DescribeChecks": {
                "url": "",
                "description": "List valid Trusted Advisor checks and details.",
                "accessLevel": "Read"
            },
            "DescribeNotificationPreferences": {
                "url": "",
                "description": "Describes the notification preferences for the account",
                "accessLevel": "Read"
            },
            "ExcludeCheckItems": {
                "url": "",
                "description": "Exclude recommendations for checks for a given customer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "checks": {
                        "required": true
                    }
                }
            },
            "IncludeCheckItems": {
                "url": "",
                "description": "Include recommendations for checks for a given customer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "checks": {
                        "required": true
                    }
                }
            },
            "RefreshCheck": {
                "url": "",
                "description": "Enqueue a refresh for the specified check",
                "accessLevel": "Write",
                "resourceTypes": {
                    "checks": {
                        "required": true
                    }
                }
            },
            "SetAccountAccess": {
                "url": "",
                "description": "Toggle whether TrustedAdvisor is enabled/disabled for the account",
                "accessLevel": "Write"
            },
            "UpdateNotificationPreferences": {
                "url": "",
                "description": "Update notification preferences",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "checks": {
                "name": "checks",
                "url": "https://docs.aws.amazon.com/awssupport/latest/APIReference/API_TrustedAdvisorCheckDescription.html",
                "arn": "arn:${Partition}:trustedadvisor:${Region}:${Account}:checks/${CategoryCode}/${CheckId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * View support plan and various TA preferences.
     *
     * Access Level: Read
     */
    describeAccount() {
        this.add('trustedadvisor:DescribeAccount');
        return this;
    }
    /**
     * Resolve whether Account has disabled Trusted Advisor
     *
     * Access Level: Read
     */
    describeAccountAccess() {
        this.add('trustedadvisor:DescribeAccountAccess');
        return this;
    }
    /**
     * View details for the check items
     *
     * Access Level: Read
     */
    describeCheckItems() {
        this.add('trustedadvisor:DescribeCheckItems');
        return this;
    }
    /**
     * Describe check refresh statuses
     *
     * Access Level: Read
     */
    describeCheckRefreshStatuses() {
        this.add('trustedadvisor:DescribeCheckRefreshStatuses');
        return this;
    }
    /**
     * Describes the check's summaries
     *
     * Access Level: Read
     */
    describeCheckSummaries() {
        this.add('trustedadvisor:DescribeCheckSummaries');
        return this;
    }
    /**
     * List valid Trusted Advisor checks and details.
     *
     * Access Level: Read
     */
    describeChecks() {
        this.add('trustedadvisor:DescribeChecks');
        return this;
    }
    /**
     * Describes the notification preferences for the account
     *
     * Access Level: Read
     */
    describeNotificationPreferences() {
        this.add('trustedadvisor:DescribeNotificationPreferences');
        return this;
    }
    /**
     * Exclude recommendations for checks for a given customer
     *
     * Access Level: Write
     */
    excludeCheckItems() {
        this.add('trustedadvisor:ExcludeCheckItems');
        return this;
    }
    /**
     * Include recommendations for checks for a given customer
     *
     * Access Level: Write
     */
    includeCheckItems() {
        this.add('trustedadvisor:IncludeCheckItems');
        return this;
    }
    /**
     * Enqueue a refresh for the specified check
     *
     * Access Level: Write
     */
    refreshCheck() {
        this.add('trustedadvisor:RefreshCheck');
        return this;
    }
    /**
     * Toggle whether TrustedAdvisor is enabled/disabled for the account
     *
     * Access Level: Write
     */
    setAccountAccess() {
        this.add('trustedadvisor:SetAccountAccess');
        return this;
    }
    /**
     * Update notification preferences
     *
     * Access Level: Write
     */
    updateNotificationPreferences() {
        this.add('trustedadvisor:UpdateNotificationPreferences');
        return this;
    }
    /**
     * Adds a resource of type checks to the statement
     *
     * https://docs.aws.amazon.com/awssupport/latest/APIReference/API_TrustedAdvisorCheckDescription.html
     *
     * @param categoryCode - Identifier for the categoryCode.
     * @param checkId - Identifier for the checkId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onChecks(categoryCode, checkId, account, region, partition) {
        var arn = 'arn:${Partition}:trustedadvisor:${Region}:${Account}:checks/${CategoryCode}/${CheckId}';
        arn = arn.replace('${CategoryCode}', categoryCode);
        arn = arn.replace('${CheckId}', checkId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Trustedadvisor = Trustedadvisor;
//# sourceMappingURL=data:application/json;base64,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