"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service xray
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsx-ray.html
 */
class Xray extends shared_1.PolicyStatement {
    /**
     * Action provider for service xray
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsx-ray.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'xray';
        this.actions = {
            "BatchGetTraces": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_BatchGetTraces.html",
                "description": "Retrieves a list of traces specified by ID. Each trace is a collection of segment documents that originates from a single request. Use GetTraceSummaries to get a list of trace IDs.",
                "accessLevel": "Read"
            },
            "CreateGroup": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_CreateGroup.html",
                "description": "Creates a group resource with a name and a filter expression.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "CreateSamplingRule": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_CreateSamplingRule.html",
                "description": "Creates a rule to control sampling behavior for instrumented applications.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "sampling-rule": {
                        "required": true
                    }
                }
            },
            "DeleteGroup": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_DeleteGroup.html",
                "description": "Deletes a group resource.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "DeleteSamplingRule": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_DeleteSamplingRule.html",
                "description": "Deletes a sampling rule.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "sampling-rule": {
                        "required": true
                    }
                }
            },
            "GetEncryptionConfig": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_GetEncryptionConfig.html",
                "description": "Retrieves the current encryption configuration for X-Ray data.",
                "accessLevel": "Permissions management"
            },
            "GetGroup": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_GetGroup.html",
                "description": "Retrieves group resource details.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "GetGroups": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_GetGroups.html",
                "description": "Retrieves all active group details.",
                "accessLevel": "Read"
            },
            "GetSamplingRules": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_GetSamplingRules.html",
                "description": "Retrieves all sampling rules.",
                "accessLevel": "Read"
            },
            "GetSamplingStatisticSummaries": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_GetSamplingStatisticSummaries.html",
                "description": "Retrieves information about recent sampling results for all sampling rules.",
                "accessLevel": "Read"
            },
            "GetSamplingTargets": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_GetSamplingTargets.html",
                "description": "Requests a sampling quota for rules that the service is using to sample requests.",
                "accessLevel": "Read"
            },
            "GetServiceGraph": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_GetServiceGraph.html",
                "description": "Retrieves a document that describes services that process incoming requests, and downstream services that they call as a result.",
                "accessLevel": "Read"
            },
            "GetTimeSeriesServiceStatistics": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_GetTimeSeriesServiceStatistics.html",
                "description": "Get an aggregation of service statistics defined by a specific time range bucketed into time intervals.",
                "accessLevel": "Read"
            },
            "GetTraceGraph": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_GetTraceGraph.html",
                "description": "Retrieves a service graph for one or more specific trace IDs.",
                "accessLevel": "Read"
            },
            "GetTraceSummaries": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_GetTraceSummaries.html",
                "description": "Retrieves IDs and metadata for traces available for a specified time frame using an optional filter. To get the full traces, pass the trace IDs to BatchGetTraces.",
                "accessLevel": "Read"
            },
            "PutEncryptionConfig": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_PutEncryptionConfig.html",
                "description": "Updates the encryption configuration for X-Ray data.",
                "accessLevel": "Permissions management"
            },
            "PutTelemetryRecords": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_PutTelemetryRecords.html",
                "description": "Used by the AWS X-Ray daemon to send telemetry to the service.",
                "accessLevel": "Write"
            },
            "PutTraceSegments": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_PutTraceSegments.html",
                "description": "Uploads segment documents to AWS X-Ray. The X-Ray SDK generates segment documents and sends them to the X-Ray daemon, which uploads them in batches.",
                "accessLevel": "Write"
            },
            "UpdateGroup": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_UpdateGroup.html",
                "description": "Updates a group resource.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "UpdateSamplingRule": {
                "url": "https://docs.aws.amazon.com/xray/latest/api/API_UpdateSamplingRule.html",
                "description": "Modifies a sampling rule's configuration.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "sampling-rule": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "group": {
                "name": "group",
                "url": "https://docs.aws.amazon.com/xray/latest/devguide/xray-concepts.html#xray-concepts-groups",
                "arn": "arn:${Partition}:xray:${Region}:${Account}:group/${GroupName}/${Id}",
                "conditionKeys": []
            },
            "sampling-rule": {
                "name": "sampling-rule",
                "url": "https://docs.aws.amazon.com/xray/latest/devguide/xray-concepts.html#xray-concepts-sampling",
                "arn": "arn:${Partition}:xray:${Region}:${Account}:sampling-rule/${SamplingRuleName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Retrieves a list of traces specified by ID. Each trace is a collection of segment documents that originates from a single request. Use GetTraceSummaries to get a list of trace IDs.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_BatchGetTraces.html
     */
    batchGetTraces() {
        this.add('xray:BatchGetTraces');
        return this;
    }
    /**
     * Creates a group resource with a name and a filter expression.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_CreateGroup.html
     */
    createGroup() {
        this.add('xray:CreateGroup');
        return this;
    }
    /**
     * Creates a rule to control sampling behavior for instrumented applications.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_CreateSamplingRule.html
     */
    createSamplingRule() {
        this.add('xray:CreateSamplingRule');
        return this;
    }
    /**
     * Deletes a group resource.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_DeleteGroup.html
     */
    deleteGroup() {
        this.add('xray:DeleteGroup');
        return this;
    }
    /**
     * Deletes a sampling rule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_DeleteSamplingRule.html
     */
    deleteSamplingRule() {
        this.add('xray:DeleteSamplingRule');
        return this;
    }
    /**
     * Retrieves the current encryption configuration for X-Ray data.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetEncryptionConfig.html
     */
    getEncryptionConfig() {
        this.add('xray:GetEncryptionConfig');
        return this;
    }
    /**
     * Retrieves group resource details.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetGroup.html
     */
    getGroup() {
        this.add('xray:GetGroup');
        return this;
    }
    /**
     * Retrieves all active group details.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetGroups.html
     */
    getGroups() {
        this.add('xray:GetGroups');
        return this;
    }
    /**
     * Retrieves all sampling rules.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetSamplingRules.html
     */
    getSamplingRules() {
        this.add('xray:GetSamplingRules');
        return this;
    }
    /**
     * Retrieves information about recent sampling results for all sampling rules.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetSamplingStatisticSummaries.html
     */
    getSamplingStatisticSummaries() {
        this.add('xray:GetSamplingStatisticSummaries');
        return this;
    }
    /**
     * Requests a sampling quota for rules that the service is using to sample requests.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetSamplingTargets.html
     */
    getSamplingTargets() {
        this.add('xray:GetSamplingTargets');
        return this;
    }
    /**
     * Retrieves a document that describes services that process incoming requests, and downstream services that they call as a result.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetServiceGraph.html
     */
    getServiceGraph() {
        this.add('xray:GetServiceGraph');
        return this;
    }
    /**
     * Get an aggregation of service statistics defined by a specific time range bucketed into time intervals.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetTimeSeriesServiceStatistics.html
     */
    getTimeSeriesServiceStatistics() {
        this.add('xray:GetTimeSeriesServiceStatistics');
        return this;
    }
    /**
     * Retrieves a service graph for one or more specific trace IDs.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetTraceGraph.html
     */
    getTraceGraph() {
        this.add('xray:GetTraceGraph');
        return this;
    }
    /**
     * Retrieves IDs and metadata for traces available for a specified time frame using an optional filter. To get the full traces, pass the trace IDs to BatchGetTraces.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetTraceSummaries.html
     */
    getTraceSummaries() {
        this.add('xray:GetTraceSummaries');
        return this;
    }
    /**
     * Updates the encryption configuration for X-Ray data.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_PutEncryptionConfig.html
     */
    putEncryptionConfig() {
        this.add('xray:PutEncryptionConfig');
        return this;
    }
    /**
     * Used by the AWS X-Ray daemon to send telemetry to the service.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_PutTelemetryRecords.html
     */
    putTelemetryRecords() {
        this.add('xray:PutTelemetryRecords');
        return this;
    }
    /**
     * Uploads segment documents to AWS X-Ray. The X-Ray SDK generates segment documents and sends them to the X-Ray daemon, which uploads them in batches.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_PutTraceSegments.html
     */
    putTraceSegments() {
        this.add('xray:PutTraceSegments');
        return this;
    }
    /**
     * Updates a group resource.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_UpdateGroup.html
     */
    updateGroup() {
        this.add('xray:UpdateGroup');
        return this;
    }
    /**
     * Modifies a sampling rule's configuration.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_UpdateSamplingRule.html
     */
    updateSamplingRule() {
        this.add('xray:UpdateSamplingRule');
        return this;
    }
    /**
     * Adds a resource of type group to the statement
     *
     * https://docs.aws.amazon.com/xray/latest/devguide/xray-concepts.html#xray-concepts-groups
     *
     * @param groupName - Identifier for the groupName.
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onGroup(groupName, id, account, region, partition) {
        var arn = 'arn:${Partition}:xray:${Region}:${Account}:group/${GroupName}/${Id}';
        arn = arn.replace('${GroupName}', groupName);
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type sampling-rule to the statement
     *
     * https://docs.aws.amazon.com/xray/latest/devguide/xray-concepts.html#xray-concepts-sampling
     *
     * @param samplingRuleName - Identifier for the samplingRuleName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSamplingRule(samplingRuleName, account, region, partition) {
        var arn = 'arn:${Partition}:xray:${Region}:${Account}:sampling-rule/${SamplingRuleName}';
        arn = arn.replace('${SamplingRuleName}', samplingRuleName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Xray = Xray;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXdzeC1yYXkuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJhd3N4LXJheS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUFBLHNDQUFvRTtBQUdwRTs7OztHQUlHO0FBQ0gsTUFBYSxJQUFLLFNBQVEsd0JBQWU7SUEwSnZDOzs7O09BSUc7SUFDSCxZQUFhLEtBQTRCO1FBQ3ZDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQztRQS9KUixrQkFBYSxHQUFHLE1BQU0sQ0FBQztRQUN2QixZQUFPLEdBQVk7WUFDeEIsZ0JBQWdCLEVBQUU7Z0JBQ2hCLEtBQUssRUFBRSxxRUFBcUU7Z0JBQzVFLGFBQWEsRUFBRSxzTEFBc0w7Z0JBQ3JNLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0QsYUFBYSxFQUFFO2dCQUNiLEtBQUssRUFBRSxrRUFBa0U7Z0JBQ3pFLGFBQWEsRUFBRSwrREFBK0Q7Z0JBQzlFLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0Qsb0JBQW9CLEVBQUU7Z0JBQ3BCLEtBQUssRUFBRSx5RUFBeUU7Z0JBQ2hGLGFBQWEsRUFBRSw0RUFBNEU7Z0JBQzNGLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsZUFBZSxFQUFFO3dCQUNmLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsYUFBYSxFQUFFO2dCQUNiLEtBQUssRUFBRSxrRUFBa0U7Z0JBQ3pFLGFBQWEsRUFBRSwyQkFBMkI7Z0JBQzFDLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0Qsb0JBQW9CLEVBQUU7Z0JBQ3BCLEtBQUssRUFBRSx5RUFBeUU7Z0JBQ2hGLGFBQWEsRUFBRSwwQkFBMEI7Z0JBQ3pDLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsZUFBZSxFQUFFO3dCQUNmLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QscUJBQXFCLEVBQUU7Z0JBQ3JCLEtBQUssRUFBRSwwRUFBMEU7Z0JBQ2pGLGFBQWEsRUFBRSxnRUFBZ0U7Z0JBQy9FLGFBQWEsRUFBRSx3QkFBd0I7YUFDeEM7WUFDRCxVQUFVLEVBQUU7Z0JBQ1YsS0FBSyxFQUFFLCtEQUErRDtnQkFDdEUsYUFBYSxFQUFFLG1DQUFtQztnQkFDbEQsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxXQUFXLEVBQUU7Z0JBQ1gsS0FBSyxFQUFFLGdFQUFnRTtnQkFDdkUsYUFBYSxFQUFFLHFDQUFxQztnQkFDcEQsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxrQkFBa0IsRUFBRTtnQkFDbEIsS0FBSyxFQUFFLHVFQUF1RTtnQkFDOUUsYUFBYSxFQUFFLCtCQUErQjtnQkFDOUMsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCwrQkFBK0IsRUFBRTtnQkFDL0IsS0FBSyxFQUFFLG9GQUFvRjtnQkFDM0YsYUFBYSxFQUFFLDZFQUE2RTtnQkFDNUYsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxvQkFBb0IsRUFBRTtnQkFDcEIsS0FBSyxFQUFFLHlFQUF5RTtnQkFDaEYsYUFBYSxFQUFFLG1GQUFtRjtnQkFDbEcsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxpQkFBaUIsRUFBRTtnQkFDakIsS0FBSyxFQUFFLHNFQUFzRTtnQkFDN0UsYUFBYSxFQUFFLGtJQUFrSTtnQkFDakosYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxnQ0FBZ0MsRUFBRTtnQkFDaEMsS0FBSyxFQUFFLHFGQUFxRjtnQkFDNUYsYUFBYSxFQUFFLHlHQUF5RztnQkFDeEgsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxlQUFlLEVBQUU7Z0JBQ2YsS0FBSyxFQUFFLG9FQUFvRTtnQkFDM0UsYUFBYSxFQUFFLCtEQUErRDtnQkFDOUUsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxtQkFBbUIsRUFBRTtnQkFDbkIsS0FBSyxFQUFFLHdFQUF3RTtnQkFDL0UsYUFBYSxFQUFFLG9LQUFvSztnQkFDbkwsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxxQkFBcUIsRUFBRTtnQkFDckIsS0FBSyxFQUFFLDBFQUEwRTtnQkFDakYsYUFBYSxFQUFFLHNEQUFzRDtnQkFDckUsYUFBYSxFQUFFLHdCQUF3QjthQUN4QztZQUNELHFCQUFxQixFQUFFO2dCQUNyQixLQUFLLEVBQUUsMEVBQTBFO2dCQUNqRixhQUFhLEVBQUUsZ0VBQWdFO2dCQUMvRSxhQUFhLEVBQUUsT0FBTzthQUN2QjtZQUNELGtCQUFrQixFQUFFO2dCQUNsQixLQUFLLEVBQUUsdUVBQXVFO2dCQUM5RSxhQUFhLEVBQUUsc0pBQXNKO2dCQUNySyxhQUFhLEVBQUUsT0FBTzthQUN2QjtZQUNELGFBQWEsRUFBRTtnQkFDYixLQUFLLEVBQUUsa0VBQWtFO2dCQUN6RSxhQUFhLEVBQUUsMkJBQTJCO2dCQUMxQyxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELG9CQUFvQixFQUFFO2dCQUNwQixLQUFLLEVBQUUseUVBQXlFO2dCQUNoRixhQUFhLEVBQUUsMkNBQTJDO2dCQUMxRCxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLGVBQWUsRUFBRTt3QkFDZixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtTQUNGLENBQUM7UUFDSyxrQkFBYSxHQUFrQjtZQUNwQyxPQUFPLEVBQUU7Z0JBQ1AsTUFBTSxFQUFFLE9BQU87Z0JBQ2YsS0FBSyxFQUFFLDBGQUEwRjtnQkFDakcsS0FBSyxFQUFFLHFFQUFxRTtnQkFDNUUsZUFBZSxFQUFFLEVBQUU7YUFDcEI7WUFDRCxlQUFlLEVBQUU7Z0JBQ2YsTUFBTSxFQUFFLGVBQWU7Z0JBQ3ZCLEtBQUssRUFBRSw0RkFBNEY7Z0JBQ25HLEtBQUssRUFBRSw4RUFBOEU7Z0JBQ3JGLGVBQWUsRUFBRSxFQUFFO2FBQ3BCO1NBQ0YsQ0FBQztJQVNGLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxjQUFjO1FBQ25CLElBQUksQ0FBQyxHQUFHLENBQUMscUJBQXFCLENBQUMsQ0FBQztRQUNoQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxXQUFXO1FBQ2hCLElBQUksQ0FBQyxHQUFHLENBQUMsa0JBQWtCLENBQUMsQ0FBQztRQUM3QixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO1FBQ3BDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFdBQVc7UUFDaEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO1FBQzdCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixJQUFJLENBQUMsR0FBRyxDQUFDLHlCQUF5QixDQUFDLENBQUM7UUFDcEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLElBQUksQ0FBQyxHQUFHLENBQUMsMEJBQTBCLENBQUMsQ0FBQztRQUNyQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxRQUFRO1FBQ2IsSUFBSSxDQUFDLEdBQUcsQ0FBQyxlQUFlLENBQUMsQ0FBQztRQUMxQixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxTQUFTO1FBQ2QsSUFBSSxDQUFDLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO1FBQzNCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixJQUFJLENBQUMsR0FBRyxDQUFDLHVCQUF1QixDQUFDLENBQUM7UUFDbEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksNkJBQTZCO1FBQ2xDLElBQUksQ0FBQyxHQUFHLENBQUMsb0NBQW9DLENBQUMsQ0FBQztRQUMvQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO1FBQ3BDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGVBQWU7UUFDcEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO1FBQ2pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDhCQUE4QjtRQUNuQyxJQUFJLENBQUMsR0FBRyxDQUFDLHFDQUFxQyxDQUFDLENBQUM7UUFDaEQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsR0FBRyxDQUFDLG9CQUFvQixDQUFDLENBQUM7UUFDL0IsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksaUJBQWlCO1FBQ3RCLElBQUksQ0FBQyxHQUFHLENBQUMsd0JBQXdCLENBQUMsQ0FBQztRQUNuQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsSUFBSSxDQUFDLEdBQUcsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO1FBQ3JDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQjtRQUN4QixJQUFJLENBQUMsR0FBRyxDQUFDLDBCQUEwQixDQUFDLENBQUM7UUFDckMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0JBQWdCO1FBQ3JCLElBQUksQ0FBQyxHQUFHLENBQUMsdUJBQXVCLENBQUMsQ0FBQztRQUNsQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxXQUFXO1FBQ2hCLElBQUksQ0FBQyxHQUFHLENBQUMsa0JBQWtCLENBQUMsQ0FBQztRQUM3QixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO1FBQ3BDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxPQUFPLENBQUMsU0FBaUIsRUFBRSxFQUFVLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDakcsSUFBSSxHQUFHLEdBQUcscUVBQXFFLENBQUM7UUFDaEYsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQzdDLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLE9BQU8sRUFBRSxFQUFFLENBQUMsQ0FBQztRQUMvQixHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsT0FBTyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQ2hELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxNQUFNLElBQUksR0FBRyxDQUFDLENBQUM7UUFDOUMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLGNBQWMsQ0FBQyxnQkFBd0IsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNuRyxJQUFJLEdBQUcsR0FBRyw4RUFBOEUsQ0FBQztRQUN6RixHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxxQkFBcUIsRUFBRSxnQkFBZ0IsQ0FBQyxDQUFDO1FBQzNELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0NBQ0Y7QUExYkQsb0JBMGJDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQWN0aW9ucywgUG9saWN5U3RhdGVtZW50LCBSZXNvdXJjZVR5cGVzIH0gZnJvbSBcIi4uL3NoYXJlZFwiO1xuaW1wb3J0IHsgUG9saWN5U3RhdGVtZW50UHJvcHMgfSBmcm9tIFwiQGF3cy1jZGsvYXdzLWlhbVwiO1xuXG4vKipcbiAqIEFjdGlvbiBwcm92aWRlciBmb3Igc2VydmljZSB4cmF5XG4gKlxuICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL2xpc3RfYXdzeC1yYXkuaHRtbFxuICovXG5leHBvcnQgY2xhc3MgWHJheSBleHRlbmRzIFBvbGljeVN0YXRlbWVudCB7XG4gIHB1YmxpYyBzZXJ2aWNlUHJlZml4ID0gJ3hyYXknO1xuICBwdWJsaWMgYWN0aW9uczogQWN0aW9ucyA9IHtcbiAgICBcIkJhdGNoR2V0VHJhY2VzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3hyYXkvbGF0ZXN0L2FwaS9BUElfQmF0Y2hHZXRUcmFjZXMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHJpZXZlcyBhIGxpc3Qgb2YgdHJhY2VzIHNwZWNpZmllZCBieSBJRC4gRWFjaCB0cmFjZSBpcyBhIGNvbGxlY3Rpb24gb2Ygc2VnbWVudCBkb2N1bWVudHMgdGhhdCBvcmlnaW5hdGVzIGZyb20gYSBzaW5nbGUgcmVxdWVzdC4gVXNlIEdldFRyYWNlU3VtbWFyaWVzIHRvIGdldCBhIGxpc3Qgb2YgdHJhY2UgSURzLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIlxuICAgIH0sXG4gICAgXCJDcmVhdGVHcm91cFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS94cmF5L2xhdGVzdC9hcGkvQVBJX0NyZWF0ZUdyb3VwLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJDcmVhdGVzIGEgZ3JvdXAgcmVzb3VyY2Ugd2l0aCBhIG5hbWUgYW5kIGEgZmlsdGVyIGV4cHJlc3Npb24uXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiZ3JvdXBcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkNyZWF0ZVNhbXBsaW5nUnVsZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS94cmF5L2xhdGVzdC9hcGkvQVBJX0NyZWF0ZVNhbXBsaW5nUnVsZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQ3JlYXRlcyBhIHJ1bGUgdG8gY29udHJvbCBzYW1wbGluZyBiZWhhdmlvciBmb3IgaW5zdHJ1bWVudGVkIGFwcGxpY2F0aW9ucy5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzYW1wbGluZy1ydWxlXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZWxldGVHcm91cFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS94cmF5L2xhdGVzdC9hcGkvQVBJX0RlbGV0ZUdyb3VwLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJEZWxldGVzIGEgZ3JvdXAgcmVzb3VyY2UuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiZ3JvdXBcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlbGV0ZVNhbXBsaW5nUnVsZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS94cmF5L2xhdGVzdC9hcGkvQVBJX0RlbGV0ZVNhbXBsaW5nUnVsZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVsZXRlcyBhIHNhbXBsaW5nIHJ1bGUuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic2FtcGxpbmctcnVsZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiR2V0RW5jcnlwdGlvbkNvbmZpZ1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS94cmF5L2xhdGVzdC9hcGkvQVBJX0dldEVuY3J5cHRpb25Db25maWcuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHJpZXZlcyB0aGUgY3VycmVudCBlbmNyeXB0aW9uIGNvbmZpZ3VyYXRpb24gZm9yIFgtUmF5IGRhdGEuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUGVybWlzc2lvbnMgbWFuYWdlbWVudFwiXG4gICAgfSxcbiAgICBcIkdldEdyb3VwXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3hyYXkvbGF0ZXN0L2FwaS9BUElfR2V0R3JvdXAuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHJpZXZlcyBncm91cCByZXNvdXJjZSBkZXRhaWxzLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiZ3JvdXBcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkdldEdyb3Vwc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS94cmF5L2xhdGVzdC9hcGkvQVBJX0dldEdyb3Vwcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0cmlldmVzIGFsbCBhY3RpdmUgZ3JvdXAgZGV0YWlscy5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCJcbiAgICB9LFxuICAgIFwiR2V0U2FtcGxpbmdSdWxlc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS94cmF5L2xhdGVzdC9hcGkvQVBJX0dldFNhbXBsaW5nUnVsZXMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHJpZXZlcyBhbGwgc2FtcGxpbmcgcnVsZXMuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiXG4gICAgfSxcbiAgICBcIkdldFNhbXBsaW5nU3RhdGlzdGljU3VtbWFyaWVzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3hyYXkvbGF0ZXN0L2FwaS9BUElfR2V0U2FtcGxpbmdTdGF0aXN0aWNTdW1tYXJpZXMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHJpZXZlcyBpbmZvcm1hdGlvbiBhYm91dCByZWNlbnQgc2FtcGxpbmcgcmVzdWx0cyBmb3IgYWxsIHNhbXBsaW5nIHJ1bGVzLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIlxuICAgIH0sXG4gICAgXCJHZXRTYW1wbGluZ1RhcmdldHNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20veHJheS9sYXRlc3QvYXBpL0FQSV9HZXRTYW1wbGluZ1RhcmdldHMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJlcXVlc3RzIGEgc2FtcGxpbmcgcXVvdGEgZm9yIHJ1bGVzIHRoYXQgdGhlIHNlcnZpY2UgaXMgdXNpbmcgdG8gc2FtcGxlIHJlcXVlc3RzLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIlxuICAgIH0sXG4gICAgXCJHZXRTZXJ2aWNlR3JhcGhcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20veHJheS9sYXRlc3QvYXBpL0FQSV9HZXRTZXJ2aWNlR3JhcGguaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHJpZXZlcyBhIGRvY3VtZW50IHRoYXQgZGVzY3JpYmVzIHNlcnZpY2VzIHRoYXQgcHJvY2VzcyBpbmNvbWluZyByZXF1ZXN0cywgYW5kIGRvd25zdHJlYW0gc2VydmljZXMgdGhhdCB0aGV5IGNhbGwgYXMgYSByZXN1bHQuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiXG4gICAgfSxcbiAgICBcIkdldFRpbWVTZXJpZXNTZXJ2aWNlU3RhdGlzdGljc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS94cmF5L2xhdGVzdC9hcGkvQVBJX0dldFRpbWVTZXJpZXNTZXJ2aWNlU3RhdGlzdGljcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR2V0IGFuIGFnZ3JlZ2F0aW9uIG9mIHNlcnZpY2Ugc3RhdGlzdGljcyBkZWZpbmVkIGJ5IGEgc3BlY2lmaWMgdGltZSByYW5nZSBidWNrZXRlZCBpbnRvIHRpbWUgaW50ZXJ2YWxzLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIlxuICAgIH0sXG4gICAgXCJHZXRUcmFjZUdyYXBoXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3hyYXkvbGF0ZXN0L2FwaS9BUElfR2V0VHJhY2VHcmFwaC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0cmlldmVzIGEgc2VydmljZSBncmFwaCBmb3Igb25lIG9yIG1vcmUgc3BlY2lmaWMgdHJhY2UgSURzLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIlxuICAgIH0sXG4gICAgXCJHZXRUcmFjZVN1bW1hcmllc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS94cmF5L2xhdGVzdC9hcGkvQVBJX0dldFRyYWNlU3VtbWFyaWVzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZXRyaWV2ZXMgSURzIGFuZCBtZXRhZGF0YSBmb3IgdHJhY2VzIGF2YWlsYWJsZSBmb3IgYSBzcGVjaWZpZWQgdGltZSBmcmFtZSB1c2luZyBhbiBvcHRpb25hbCBmaWx0ZXIuIFRvIGdldCB0aGUgZnVsbCB0cmFjZXMsIHBhc3MgdGhlIHRyYWNlIElEcyB0byBCYXRjaEdldFRyYWNlcy5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCJcbiAgICB9LFxuICAgIFwiUHV0RW5jcnlwdGlvbkNvbmZpZ1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS94cmF5L2xhdGVzdC9hcGkvQVBJX1B1dEVuY3J5cHRpb25Db25maWcuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlVwZGF0ZXMgdGhlIGVuY3J5cHRpb24gY29uZmlndXJhdGlvbiBmb3IgWC1SYXkgZGF0YS5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJQZXJtaXNzaW9ucyBtYW5hZ2VtZW50XCJcbiAgICB9LFxuICAgIFwiUHV0VGVsZW1ldHJ5UmVjb3Jkc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS94cmF5L2xhdGVzdC9hcGkvQVBJX1B1dFRlbGVtZXRyeVJlY29yZHMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlVzZWQgYnkgdGhlIEFXUyBYLVJheSBkYWVtb24gdG8gc2VuZCB0ZWxlbWV0cnkgdG8gdGhlIHNlcnZpY2UuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIlxuICAgIH0sXG4gICAgXCJQdXRUcmFjZVNlZ21lbnRzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3hyYXkvbGF0ZXN0L2FwaS9BUElfUHV0VHJhY2VTZWdtZW50cy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiVXBsb2FkcyBzZWdtZW50IGRvY3VtZW50cyB0byBBV1MgWC1SYXkuIFRoZSBYLVJheSBTREsgZ2VuZXJhdGVzIHNlZ21lbnQgZG9jdW1lbnRzIGFuZCBzZW5kcyB0aGVtIHRvIHRoZSBYLVJheSBkYWVtb24sIHdoaWNoIHVwbG9hZHMgdGhlbSBpbiBiYXRjaGVzLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCJcbiAgICB9LFxuICAgIFwiVXBkYXRlR3JvdXBcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20veHJheS9sYXRlc3QvYXBpL0FQSV9VcGRhdGVHcm91cC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiVXBkYXRlcyBhIGdyb3VwIHJlc291cmNlLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImdyb3VwXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJVcGRhdGVTYW1wbGluZ1J1bGVcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20veHJheS9sYXRlc3QvYXBpL0FQSV9VcGRhdGVTYW1wbGluZ1J1bGUuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIk1vZGlmaWVzIGEgc2FtcGxpbmcgcnVsZSdzIGNvbmZpZ3VyYXRpb24uXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic2FtcGxpbmctcnVsZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9XG4gIH07XG4gIHB1YmxpYyByZXNvdXJjZVR5cGVzOiBSZXNvdXJjZVR5cGVzID0ge1xuICAgIFwiZ3JvdXBcIjoge1xuICAgICAgXCJuYW1lXCI6IFwiZ3JvdXBcIixcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3hyYXkvbGF0ZXN0L2Rldmd1aWRlL3hyYXktY29uY2VwdHMuaHRtbCN4cmF5LWNvbmNlcHRzLWdyb3Vwc1wiLFxuICAgICAgXCJhcm5cIjogXCJhcm46JHtQYXJ0aXRpb259OnhyYXk6JHtSZWdpb259OiR7QWNjb3VudH06Z3JvdXAvJHtHcm91cE5hbWV9LyR7SWR9XCIsXG4gICAgICBcImNvbmRpdGlvbktleXNcIjogW11cbiAgICB9LFxuICAgIFwic2FtcGxpbmctcnVsZVwiOiB7XG4gICAgICBcIm5hbWVcIjogXCJzYW1wbGluZy1ydWxlXCIsXG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS94cmF5L2xhdGVzdC9kZXZndWlkZS94cmF5LWNvbmNlcHRzLmh0bWwjeHJheS1jb25jZXB0cy1zYW1wbGluZ1wiLFxuICAgICAgXCJhcm5cIjogXCJhcm46JHtQYXJ0aXRpb259OnhyYXk6JHtSZWdpb259OiR7QWNjb3VudH06c2FtcGxpbmctcnVsZS8ke1NhbXBsaW5nUnVsZU5hbWV9XCIsXG4gICAgICBcImNvbmRpdGlvbktleXNcIjogW11cbiAgICB9XG4gIH07XG5cbiAgLyoqXG4gICAqIEFjdGlvbiBwcm92aWRlciBmb3Igc2VydmljZSB4cmF5XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9saXN0X2F3c3gtcmF5Lmh0bWxcbiAgICovXG4gIGNvbnN0cnVjdG9yIChwcm9wcz86IFBvbGljeVN0YXRlbWVudFByb3BzKSB7XG4gICAgc3VwZXIocHJvcHMpO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHJpZXZlcyBhIGxpc3Qgb2YgdHJhY2VzIHNwZWNpZmllZCBieSBJRC4gRWFjaCB0cmFjZSBpcyBhIGNvbGxlY3Rpb24gb2Ygc2VnbWVudCBkb2N1bWVudHMgdGhhdCBvcmlnaW5hdGVzIGZyb20gYSBzaW5nbGUgcmVxdWVzdC4gVXNlIEdldFRyYWNlU3VtbWFyaWVzIHRvIGdldCBhIGxpc3Qgb2YgdHJhY2UgSURzLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3hyYXkvbGF0ZXN0L2FwaS9BUElfQmF0Y2hHZXRUcmFjZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIGJhdGNoR2V0VHJhY2VzKCkge1xuICAgIHRoaXMuYWRkKCd4cmF5OkJhdGNoR2V0VHJhY2VzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBhIGdyb3VwIHJlc291cmNlIHdpdGggYSBuYW1lIGFuZCBhIGZpbHRlciBleHByZXNzaW9uLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS94cmF5L2xhdGVzdC9hcGkvQVBJX0NyZWF0ZUdyb3VwLmh0bWxcbiAgICovXG4gIHB1YmxpYyBjcmVhdGVHcm91cCgpIHtcbiAgICB0aGlzLmFkZCgneHJheTpDcmVhdGVHcm91cCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgYSBydWxlIHRvIGNvbnRyb2wgc2FtcGxpbmcgYmVoYXZpb3IgZm9yIGluc3RydW1lbnRlZCBhcHBsaWNhdGlvbnMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3hyYXkvbGF0ZXN0L2FwaS9BUElfQ3JlYXRlU2FtcGxpbmdSdWxlLmh0bWxcbiAgICovXG4gIHB1YmxpYyBjcmVhdGVTYW1wbGluZ1J1bGUoKSB7XG4gICAgdGhpcy5hZGQoJ3hyYXk6Q3JlYXRlU2FtcGxpbmdSdWxlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVsZXRlcyBhIGdyb3VwIHJlc291cmNlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS94cmF5L2xhdGVzdC9hcGkvQVBJX0RlbGV0ZUdyb3VwLmh0bWxcbiAgICovXG4gIHB1YmxpYyBkZWxldGVHcm91cCgpIHtcbiAgICB0aGlzLmFkZCgneHJheTpEZWxldGVHcm91cCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlbGV0ZXMgYSBzYW1wbGluZyBydWxlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS94cmF5L2xhdGVzdC9hcGkvQVBJX0RlbGV0ZVNhbXBsaW5nUnVsZS5odG1sXG4gICAqL1xuICBwdWJsaWMgZGVsZXRlU2FtcGxpbmdSdWxlKCkge1xuICAgIHRoaXMuYWRkKCd4cmF5OkRlbGV0ZVNhbXBsaW5nUnVsZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHJpZXZlcyB0aGUgY3VycmVudCBlbmNyeXB0aW9uIGNvbmZpZ3VyYXRpb24gZm9yIFgtUmF5IGRhdGEuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUGVybWlzc2lvbnMgbWFuYWdlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20veHJheS9sYXRlc3QvYXBpL0FQSV9HZXRFbmNyeXB0aW9uQ29uZmlnLmh0bWxcbiAgICovXG4gIHB1YmxpYyBnZXRFbmNyeXB0aW9uQ29uZmlnKCkge1xuICAgIHRoaXMuYWRkKCd4cmF5OkdldEVuY3J5cHRpb25Db25maWcnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXRyaWV2ZXMgZ3JvdXAgcmVzb3VyY2UgZGV0YWlscy5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS94cmF5L2xhdGVzdC9hcGkvQVBJX0dldEdyb3VwLmh0bWxcbiAgICovXG4gIHB1YmxpYyBnZXRHcm91cCgpIHtcbiAgICB0aGlzLmFkZCgneHJheTpHZXRHcm91cCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHJpZXZlcyBhbGwgYWN0aXZlIGdyb3VwIGRldGFpbHMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20veHJheS9sYXRlc3QvYXBpL0FQSV9HZXRHcm91cHMuaHRtbFxuICAgKi9cbiAgcHVibGljIGdldEdyb3VwcygpIHtcbiAgICB0aGlzLmFkZCgneHJheTpHZXRHcm91cHMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXRyaWV2ZXMgYWxsIHNhbXBsaW5nIHJ1bGVzLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3hyYXkvbGF0ZXN0L2FwaS9BUElfR2V0U2FtcGxpbmdSdWxlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgZ2V0U2FtcGxpbmdSdWxlcygpIHtcbiAgICB0aGlzLmFkZCgneHJheTpHZXRTYW1wbGluZ1J1bGVzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0cmlldmVzIGluZm9ybWF0aW9uIGFib3V0IHJlY2VudCBzYW1wbGluZyByZXN1bHRzIGZvciBhbGwgc2FtcGxpbmcgcnVsZXMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20veHJheS9sYXRlc3QvYXBpL0FQSV9HZXRTYW1wbGluZ1N0YXRpc3RpY1N1bW1hcmllcy5odG1sXG4gICAqL1xuICBwdWJsaWMgZ2V0U2FtcGxpbmdTdGF0aXN0aWNTdW1tYXJpZXMoKSB7XG4gICAgdGhpcy5hZGQoJ3hyYXk6R2V0U2FtcGxpbmdTdGF0aXN0aWNTdW1tYXJpZXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXF1ZXN0cyBhIHNhbXBsaW5nIHF1b3RhIGZvciBydWxlcyB0aGF0IHRoZSBzZXJ2aWNlIGlzIHVzaW5nIHRvIHNhbXBsZSByZXF1ZXN0cy5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS94cmF5L2xhdGVzdC9hcGkvQVBJX0dldFNhbXBsaW5nVGFyZ2V0cy5odG1sXG4gICAqL1xuICBwdWJsaWMgZ2V0U2FtcGxpbmdUYXJnZXRzKCkge1xuICAgIHRoaXMuYWRkKCd4cmF5OkdldFNhbXBsaW5nVGFyZ2V0cycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHJpZXZlcyBhIGRvY3VtZW50IHRoYXQgZGVzY3JpYmVzIHNlcnZpY2VzIHRoYXQgcHJvY2VzcyBpbmNvbWluZyByZXF1ZXN0cywgYW5kIGRvd25zdHJlYW0gc2VydmljZXMgdGhhdCB0aGV5IGNhbGwgYXMgYSByZXN1bHQuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20veHJheS9sYXRlc3QvYXBpL0FQSV9HZXRTZXJ2aWNlR3JhcGguaHRtbFxuICAgKi9cbiAgcHVibGljIGdldFNlcnZpY2VHcmFwaCgpIHtcbiAgICB0aGlzLmFkZCgneHJheTpHZXRTZXJ2aWNlR3JhcGgnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHZXQgYW4gYWdncmVnYXRpb24gb2Ygc2VydmljZSBzdGF0aXN0aWNzIGRlZmluZWQgYnkgYSBzcGVjaWZpYyB0aW1lIHJhbmdlIGJ1Y2tldGVkIGludG8gdGltZSBpbnRlcnZhbHMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20veHJheS9sYXRlc3QvYXBpL0FQSV9HZXRUaW1lU2VyaWVzU2VydmljZVN0YXRpc3RpY3MuaHRtbFxuICAgKi9cbiAgcHVibGljIGdldFRpbWVTZXJpZXNTZXJ2aWNlU3RhdGlzdGljcygpIHtcbiAgICB0aGlzLmFkZCgneHJheTpHZXRUaW1lU2VyaWVzU2VydmljZVN0YXRpc3RpY3MnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXRyaWV2ZXMgYSBzZXJ2aWNlIGdyYXBoIGZvciBvbmUgb3IgbW9yZSBzcGVjaWZpYyB0cmFjZSBJRHMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20veHJheS9sYXRlc3QvYXBpL0FQSV9HZXRUcmFjZUdyYXBoLmh0bWxcbiAgICovXG4gIHB1YmxpYyBnZXRUcmFjZUdyYXBoKCkge1xuICAgIHRoaXMuYWRkKCd4cmF5OkdldFRyYWNlR3JhcGgnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXRyaWV2ZXMgSURzIGFuZCBtZXRhZGF0YSBmb3IgdHJhY2VzIGF2YWlsYWJsZSBmb3IgYSBzcGVjaWZpZWQgdGltZSBmcmFtZSB1c2luZyBhbiBvcHRpb25hbCBmaWx0ZXIuIFRvIGdldCB0aGUgZnVsbCB0cmFjZXMsIHBhc3MgdGhlIHRyYWNlIElEcyB0byBCYXRjaEdldFRyYWNlcy5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS94cmF5L2xhdGVzdC9hcGkvQVBJX0dldFRyYWNlU3VtbWFyaWVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyBnZXRUcmFjZVN1bW1hcmllcygpIHtcbiAgICB0aGlzLmFkZCgneHJheTpHZXRUcmFjZVN1bW1hcmllcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFVwZGF0ZXMgdGhlIGVuY3J5cHRpb24gY29uZmlndXJhdGlvbiBmb3IgWC1SYXkgZGF0YS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBQZXJtaXNzaW9ucyBtYW5hZ2VtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS94cmF5L2xhdGVzdC9hcGkvQVBJX1B1dEVuY3J5cHRpb25Db25maWcuaHRtbFxuICAgKi9cbiAgcHVibGljIHB1dEVuY3J5cHRpb25Db25maWcoKSB7XG4gICAgdGhpcy5hZGQoJ3hyYXk6UHV0RW5jcnlwdGlvbkNvbmZpZycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFVzZWQgYnkgdGhlIEFXUyBYLVJheSBkYWVtb24gdG8gc2VuZCB0ZWxlbWV0cnkgdG8gdGhlIHNlcnZpY2UuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3hyYXkvbGF0ZXN0L2FwaS9BUElfUHV0VGVsZW1ldHJ5UmVjb3Jkcy5odG1sXG4gICAqL1xuICBwdWJsaWMgcHV0VGVsZW1ldHJ5UmVjb3JkcygpIHtcbiAgICB0aGlzLmFkZCgneHJheTpQdXRUZWxlbWV0cnlSZWNvcmRzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVXBsb2FkcyBzZWdtZW50IGRvY3VtZW50cyB0byBBV1MgWC1SYXkuIFRoZSBYLVJheSBTREsgZ2VuZXJhdGVzIHNlZ21lbnQgZG9jdW1lbnRzIGFuZCBzZW5kcyB0aGVtIHRvIHRoZSBYLVJheSBkYWVtb24sIHdoaWNoIHVwbG9hZHMgdGhlbSBpbiBiYXRjaGVzLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS94cmF5L2xhdGVzdC9hcGkvQVBJX1B1dFRyYWNlU2VnbWVudHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHB1dFRyYWNlU2VnbWVudHMoKSB7XG4gICAgdGhpcy5hZGQoJ3hyYXk6UHV0VHJhY2VTZWdtZW50cycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFVwZGF0ZXMgYSBncm91cCByZXNvdXJjZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20veHJheS9sYXRlc3QvYXBpL0FQSV9VcGRhdGVHcm91cC5odG1sXG4gICAqL1xuICBwdWJsaWMgdXBkYXRlR3JvdXAoKSB7XG4gICAgdGhpcy5hZGQoJ3hyYXk6VXBkYXRlR3JvdXAnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBNb2RpZmllcyBhIHNhbXBsaW5nIHJ1bGUncyBjb25maWd1cmF0aW9uLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS94cmF5L2xhdGVzdC9hcGkvQVBJX1VwZGF0ZVNhbXBsaW5nUnVsZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdXBkYXRlU2FtcGxpbmdSdWxlKCkge1xuICAgIHRoaXMuYWRkKCd4cmF5OlVwZGF0ZVNhbXBsaW5nUnVsZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGdyb3VwIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3hyYXkvbGF0ZXN0L2Rldmd1aWRlL3hyYXktY29uY2VwdHMuaHRtbCN4cmF5LWNvbmNlcHRzLWdyb3Vwc1xuICAgKlxuICAgKiBAcGFyYW0gZ3JvdXBOYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIGdyb3VwTmFtZS5cbiAgICogQHBhcmFtIGlkIC0gSWRlbnRpZmllciBmb3IgdGhlIGlkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIHJlZ2lvbnMuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLlxuICAgKi9cbiAgcHVibGljIG9uR3JvdXAoZ3JvdXBOYW1lOiBzdHJpbmcsIGlkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OnhyYXk6JHtSZWdpb259OiR7QWNjb3VudH06Z3JvdXAvJHtHcm91cE5hbWV9LyR7SWR9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtHcm91cE5hbWV9JywgZ3JvdXBOYW1lKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtJZH0nLCBpZCk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QWNjb3VudH0nLCBhY2NvdW50IHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVnaW9ufScsIHJlZ2lvbiB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgc2FtcGxpbmctcnVsZSB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS94cmF5L2xhdGVzdC9kZXZndWlkZS94cmF5LWNvbmNlcHRzLmh0bWwjeHJheS1jb25jZXB0cy1zYW1wbGluZ1xuICAgKlxuICAgKiBAcGFyYW0gc2FtcGxpbmdSdWxlTmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBzYW1wbGluZ1J1bGVOYW1lLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIHJlZ2lvbnMuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLlxuICAgKi9cbiAgcHVibGljIG9uU2FtcGxpbmdSdWxlKHNhbXBsaW5nUnVsZU5hbWU6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06eHJheToke1JlZ2lvbn06JHtBY2NvdW50fTpzYW1wbGluZy1ydWxlLyR7U2FtcGxpbmdSdWxlTmFtZX0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1NhbXBsaW5nUnVsZU5hbWV9Jywgc2FtcGxpbmdSdWxlTmFtZSk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QWNjb3VudH0nLCBhY2NvdW50IHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVnaW9ufScsIHJlZ2lvbiB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cbn1cbiJdfQ==