"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service servicequotas
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_servicequotas.html
 */
class Servicequotas extends shared_1.PolicyStatement {
    /**
     * Action provider for service servicequotas
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_servicequotas.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'servicequotas';
        this.actions = {
            "AssociateServiceQuotaTemplate": {
                "url": "https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_AssociateServiceQuotaTemplate.html",
                "description": "Grants permission to associate the Service Quotas template with your organization",
                "accessLevel": "Write"
            },
            "DeleteServiceQuotaIncreaseRequestFromTemplate": {
                "url": "https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_DeleteServiceQuotaIncreaseRequestFromTemplate.html",
                "description": "Grants permission to remove the specified service quota from the service quota template",
                "accessLevel": "Write"
            },
            "DisassociateServiceQuotaTemplate": {
                "url": "https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_DisassociateServiceQuotaTemplate.html",
                "description": "Grants permission to disassociate the Service Quotas template from your organization",
                "accessLevel": "Write"
            },
            "GetAWSDefaultServiceQuota": {
                "url": "https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_GetAWSDefaultServiceQuota.html",
                "description": "Grants permission to return the details for the specified service quota, including the AWS default value",
                "accessLevel": "Read"
            },
            "GetAssociationForServiceQuotaTemplate": {
                "url": "https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_GetAssociationForServiceQuotaTemplate.html",
                "description": "Grants permission to retrieve the ServiceQuotaTemplateAssociationStatus value, which tells you if the Service Quotas template is associated with an organization",
                "accessLevel": "Read"
            },
            "GetRequestedServiceQuotaChange": {
                "url": "https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_GetRequestedServiceQuotaChange.html",
                "description": "Grants permission to retrieve the details for a particular service quota increase request",
                "accessLevel": "Read"
            },
            "GetServiceQuota": {
                "url": "https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_GetServiceQuota.html",
                "description": "Grants permission to return the details for the specified service quota, including the applied value",
                "accessLevel": "Read"
            },
            "GetServiceQuotaIncreaseRequestFromTemplate": {
                "url": "https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_GetServiceQuotaIncreaseRequestFromTemplate.html",
                "description": "Grants permission to retrieve the details for a service quota increase request from the service quota template",
                "accessLevel": "Read"
            },
            "ListAWSDefaultServiceQuotas": {
                "url": "https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_ListAWSDefaultServiceQuotas.html",
                "description": "Grants permission to list all default service quotas for the specified AWS service",
                "accessLevel": "Read"
            },
            "ListRequestedServiceQuotaChangeHistory": {
                "url": "https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_ListRequestedServiceQuotaChangeHistory.html",
                "description": "Grants permission to request a list of the changes to quotas for a service",
                "accessLevel": "Read"
            },
            "ListRequestedServiceQuotaChangeHistoryByQuota": {
                "url": "https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_ListRequestedServiceQuotaChangeHistoryByQuota.html",
                "description": "Grants permission to request a list of the changes to specific service quotas",
                "accessLevel": "Read"
            },
            "ListServiceQuotaIncreaseRequestsInTemplate": {
                "url": "https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_ListServiceQuotaIncreaseRequestsInTemplate",
                "description": "Grants permission to return a list of the service quota increase requests from the service quota template",
                "accessLevel": "Read"
            },
            "ListServiceQuotas": {
                "url": "https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_ListServiceQuotas.html",
                "description": "Grants permission to list all service quotas for the specified AWS service, in that account, in that Region",
                "accessLevel": "Read"
            },
            "ListServices": {
                "url": "https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_ListServices.html",
                "description": "Grants permission to list the AWS services available in Service Quotas",
                "accessLevel": "Read"
            },
            "PutServiceQuotaIncreaseRequestIntoTemplate": {
                "url": "https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_PutServiceQuotaIncreaseRequestIntoTemplate.html",
                "description": "Grants permission to define and add a quota to the service quota template",
                "accessLevel": "Write",
                "resourceTypes": {
                    "quota": {
                        "required": false
                    }
                },
                "conditions": [
                    "servicequotas:service"
                ]
            },
            "RequestServiceQuotaIncrease": {
                "url": "https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_RequestServiceQuotaIncrease.html",
                "description": "Grants permission to submit the request for a service quota increase",
                "accessLevel": "Write",
                "resourceTypes": {
                    "quota": {
                        "required": false
                    }
                },
                "conditions": [
                    "servicequotas:service"
                ]
            }
        };
        this.resourceTypes = {
            "quota": {
                "name": "quota",
                "url": "https://docs.aws.amazon.com/servicequotas/latest/userguide/quota-as-resource.html",
                "arn": "arn:${Partition}:servicequotas:${Region}:${Account}:${ServiceCode}/${QuotaCode}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to associate the Service Quotas template with your organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_AssociateServiceQuotaTemplate.html
     */
    associateServiceQuotaTemplate() {
        this.add('servicequotas:AssociateServiceQuotaTemplate');
        return this;
    }
    /**
     * Grants permission to remove the specified service quota from the service quota template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_DeleteServiceQuotaIncreaseRequestFromTemplate.html
     */
    deleteServiceQuotaIncreaseRequestFromTemplate() {
        this.add('servicequotas:DeleteServiceQuotaIncreaseRequestFromTemplate');
        return this;
    }
    /**
     * Grants permission to disassociate the Service Quotas template from your organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_DisassociateServiceQuotaTemplate.html
     */
    disassociateServiceQuotaTemplate() {
        this.add('servicequotas:DisassociateServiceQuotaTemplate');
        return this;
    }
    /**
     * Grants permission to return the details for the specified service quota, including the AWS default value
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_GetAWSDefaultServiceQuota.html
     */
    getAWSDefaultServiceQuota() {
        this.add('servicequotas:GetAWSDefaultServiceQuota');
        return this;
    }
    /**
     * Grants permission to retrieve the ServiceQuotaTemplateAssociationStatus value, which tells you if the Service Quotas template is associated with an organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_GetAssociationForServiceQuotaTemplate.html
     */
    getAssociationForServiceQuotaTemplate() {
        this.add('servicequotas:GetAssociationForServiceQuotaTemplate');
        return this;
    }
    /**
     * Grants permission to retrieve the details for a particular service quota increase request
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_GetRequestedServiceQuotaChange.html
     */
    getRequestedServiceQuotaChange() {
        this.add('servicequotas:GetRequestedServiceQuotaChange');
        return this;
    }
    /**
     * Grants permission to return the details for the specified service quota, including the applied value
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_GetServiceQuota.html
     */
    getServiceQuota() {
        this.add('servicequotas:GetServiceQuota');
        return this;
    }
    /**
     * Grants permission to retrieve the details for a service quota increase request from the service quota template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_GetServiceQuotaIncreaseRequestFromTemplate.html
     */
    getServiceQuotaIncreaseRequestFromTemplate() {
        this.add('servicequotas:GetServiceQuotaIncreaseRequestFromTemplate');
        return this;
    }
    /**
     * Grants permission to list all default service quotas for the specified AWS service
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_ListAWSDefaultServiceQuotas.html
     */
    listAWSDefaultServiceQuotas() {
        this.add('servicequotas:ListAWSDefaultServiceQuotas');
        return this;
    }
    /**
     * Grants permission to request a list of the changes to quotas for a service
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_ListRequestedServiceQuotaChangeHistory.html
     */
    listRequestedServiceQuotaChangeHistory() {
        this.add('servicequotas:ListRequestedServiceQuotaChangeHistory');
        return this;
    }
    /**
     * Grants permission to request a list of the changes to specific service quotas
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_ListRequestedServiceQuotaChangeHistoryByQuota.html
     */
    listRequestedServiceQuotaChangeHistoryByQuota() {
        this.add('servicequotas:ListRequestedServiceQuotaChangeHistoryByQuota');
        return this;
    }
    /**
     * Grants permission to return a list of the service quota increase requests from the service quota template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_ListServiceQuotaIncreaseRequestsInTemplate
     */
    listServiceQuotaIncreaseRequestsInTemplate() {
        this.add('servicequotas:ListServiceQuotaIncreaseRequestsInTemplate');
        return this;
    }
    /**
     * Grants permission to list all service quotas for the specified AWS service, in that account, in that Region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_ListServiceQuotas.html
     */
    listServiceQuotas() {
        this.add('servicequotas:ListServiceQuotas');
        return this;
    }
    /**
     * Grants permission to list the AWS services available in Service Quotas
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_ListServices.html
     */
    listServices() {
        this.add('servicequotas:ListServices');
        return this;
    }
    /**
     * Grants permission to define and add a quota to the service quota template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_PutServiceQuotaIncreaseRequestIntoTemplate.html
     */
    putServiceQuotaIncreaseRequestIntoTemplate() {
        this.add('servicequotas:PutServiceQuotaIncreaseRequestIntoTemplate');
        return this;
    }
    /**
     * Grants permission to submit the request for a service quota increase
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_RequestServiceQuotaIncrease.html
     */
    requestServiceQuotaIncrease() {
        this.add('servicequotas:RequestServiceQuotaIncrease');
        return this;
    }
    /**
     * Adds a resource of type quota to the statement
     *
     * https://docs.aws.amazon.com/servicequotas/latest/userguide/quota-as-resource.html
     *
     * @param serviceCode - Identifier for the serviceCode.
     * @param quotaCode - Identifier for the quotaCode.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onQuota(serviceCode, quotaCode, account, region, partition) {
        var arn = 'arn:${Partition}:servicequotas:${Region}:${Account}:${ServiceCode}/${QuotaCode}';
        arn = arn.replace('${ServiceCode}', serviceCode);
        arn = arn.replace('${QuotaCode}', quotaCode);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters or restricts access to a specified AWS service
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifService(value, operator) {
        return this.if(`servicequotas:service`, value, operator || 'StringLike');
    }
}
exports.Servicequotas = Servicequotas;
//# sourceMappingURL=data:application/json;base64,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