/**
 * Modifiers can be used to extend the functionality of any `Operator`, except the `null` operator
 */
export declare enum OperatorModifier {
    /**
     * You can add `IfExists` to the end of any condition operator name except the `Null` condition. For example, `StringLikeIfExists`. You do this to say "If the policy key is present in the context of the request, process the key as specified in the policy. If the key is not present, evaluate the condition element as true." Other condition elements in the statement can still result in a nonmatch, but not a missing key when checked with `...IfExists`.
     */
    IF_EXISTS = "IfExists",
    /**
     * Tests whether the value of every member of the request set is a subset of the condition key set. The condition returns true if every key value in the request matches at least one value in the policy. It also returns true if there are no keys in the request, or if the key values resolve to a null data set, such as an empty string.
     */
    FOR_ALL_VALUES = "ForAllValues:",
    /**
     * Tests whether at least one member of the set of request values matches at least one member of the set of condition key values. The condition returns true if any one of the key values in the request matches any one of the condition values in the policy. For no matching key or a null dataset, the condition returns false.
     */
    FOR_ANY_VALUES = "ForAnyValues:"
}
/**
 * Use condition operators in the `Condition` element to match the condition key and value in the policy against values in the request context. For more information about the `Condition` element, see [IAM JSON Policy Elements: Condition](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html).
 *
 * The condition operator that you can use in a policy depends on the condition key you choose. You can choose a global condition key or a service-specific condition key. To learn which condition operator you can use for a global condition key, see [AWS Global Condition Context Keys](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html). To learn which condition operator you can use for a service-specific condition key, see [Actions, Resources, and Condition Keys for AWS Services](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_actions-resources-contextkeys.html) and choose the service that you want to view.
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html
 */
export declare class Operator {
    /**
     * Exact match, case sensitive.
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static stringEquals(...modifiers: OperatorModifier[]): string;
    /**
     * Negated exact match, case sensitive.
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static stringNotEquals(...modifiers: OperatorModifier[]): string;
    /**
     * Exact match, ignore case.
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static stringEqualsIgnoreCase(...modifiers: OperatorModifier[]): string;
    /**
     * Exact exact match, ignore case.
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static stringNotEqualsIgnoreCase(...modifiers: OperatorModifier[]): string;
    /**
     * Case-sensitive match.
     *
     * The values can include a multi-character match wildcard (`*`) or a single-character match wildcard (`?`) anywhere in the string.
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static stringLike(...modifiers: OperatorModifier[]): string;
    /**
     * Negated case-sensitive matching.
     *
     * The values can include a multi-character match wildcard (`*`) or a single-character match wildcard (`?`) anywhere in the string.
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static stringNotLike(...modifiers: OperatorModifier[]): string;
    /**
     * Exact match.
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static numericEquals(...modifiers: OperatorModifier[]): string;
    /**
     * Negated exact match.
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static numericNotEquals(...modifiers: OperatorModifier[]): string;
    /**
     * Match numbers lower than value: `<`
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static numericLessThan(...modifiers: OperatorModifier[]): string;
    /**
     * Match numbers lower or equal to value: `<=`
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static numericLessThanEquals(...modifiers: OperatorModifier[]): string;
    /**
     * Match numbers higher than value: `>`
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static numericGreaterThan(...modifiers: OperatorModifier[]): string;
    /**
     * Match numbers higher or equal value: `>=`
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static numericGreaterThanEquals(...modifiers: OperatorModifier[]): string;
    /**
     * Match a specific date: `=`
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static dateEquals(...modifiers: OperatorModifier[]): string;
    /**
     * Negated match for a specific date: `!=`
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static dateNotEquals(...modifiers: OperatorModifier[]): string;
    /**
     * Match before a specific date and time: `<`
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static dateLessThan(...modifiers: OperatorModifier[]): string;
    /**
     * Match at or before a specific date and time: `<=`
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static dateLessThanEquals(...modifiers: OperatorModifier[]): string;
    /**
     * Match after a specific a date and time: `>`
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static dateGreaterThan(...modifiers: OperatorModifier[]): string;
    /**
     * Match at or after a specific date and time: `>=`
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static dateGreaterThanEquals(...modifiers: OperatorModifier[]): string;
    /**
     * Boolean match
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static bool(...modifiers: OperatorModifier[]): string;
    /**
     * The BinaryEquals condition operator lets you construct Condition elements that test key values that are in binary format. It compares the value of the specified key byte for byte against a [base-64](https://en.wikipedia.org/wiki/Base64) encoded representation of the binary value in the policy.
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static binaryEquals(...modifiers: OperatorModifier[]): string;
    /**
     * Match an IP address or range: `=`
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static ipAddress(...modifiers: OperatorModifier[]): string;
    /**
     * All IP addresses except the specified IP address or range `!=`
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static notIpAddress(...modifiers: OperatorModifier[]): string;
    /**
     * Match of the ARN, case sensitive.
     *
     * Each of the six colon-delimited components of the ARN is checked separately and each can include a multi-character match wildcard (`*`) or a single-character match wildcard (`?`).
     *
     * `ArnEquals` and `ArnLike` behave identical.
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static arnEquals(...modifiers: OperatorModifier[]): string;
    /**
     * Negated match of the ARN, case sensitive.
     *
     * Each of the six colon-delimited components of the ARN is checked separately and each can include a multi-character match wildcard (`*`) or a single-character match wildcard (`?`).
     *
     * `ArnNotEquals` and `ArnNotLike` behave identical.
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static arnNotEquals(...modifiers: OperatorModifier[]): string;
    /**
     * Match of the ARN, case sensitive.
     *
     * Each of the six colon-delimited components of the ARN is checked separately and each can include a multi-character match wildcard (`*`) or a single-character match wildcard (`?`).
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static arnLike(...modifiers: OperatorModifier[]): string;
    /**
     * Negated match of the ARN, case sensitive.
     *
     * Each of the six colon-delimited components of the ARN is checked separately and each can include a multi-character match wildcard (`*`) or a single-character match wildcard (`?`).
     *
     * `ArnNotEquals` and `ArnNotLike` behave identical.
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static arnNotLike(...modifiers: OperatorModifier[]): string;
    /**
     * Check if a key is present at the time of authorization. In the policy statement, use either true (the key doesn't exist — it is null) or false (the key exists and its value is not null).
     */
    static null(): string;
}
