import { IResolvable, IResolveContext } from '@aws-cdk/core';
import { IRule } from './rule-ref';
/**
 * The input to send to the event target.
 *
 * @stability stable
 */
export declare abstract class RuleTargetInput {
    /**
     * Pass text to the event target.
     *
     * May contain strings returned by `EventField.from()` to substitute in parts of the
     * matched event.
     *
     * The Rule Target input value will be a single string: the string you pass
     * here.  Do not use this method to pass a complex value like a JSON object to
     * a Rule Target.  Use `RuleTargetInput.fromObject()` instead.
     *
     * @stability stable
     */
    static fromText(text: string): RuleTargetInput;
    /**
     * Pass text to the event target, splitting on newlines.
     *
     * This is only useful when passing to a target that does not
     * take a single argument.
     *
     * May contain strings returned by `EventField.from()` to substitute in parts
     * of the matched event.
     *
     * @stability stable
     */
    static fromMultilineText(text: string): RuleTargetInput;
    /**
     * Pass a JSON object to the event target.
     *
     * May contain strings returned by `EventField.from()` to substitute in parts of the
     * matched event.
     *
     * @stability stable
     */
    static fromObject(obj: any): RuleTargetInput;
    /**
     * Take the event target input from a path in the event JSON.
     *
     * @stability stable
     */
    static fromEventPath(path: string): RuleTargetInput;
    /**
     * @stability stable
     */
    protected constructor();
    /**
     * Return the input properties for this input object.
     *
     * @stability stable
     */
    abstract bind(rule: IRule): RuleTargetInputProperties;
}
/**
 * The input properties for an event target.
 *
 * @stability stable
 */
export interface RuleTargetInputProperties {
    /**
     * Literal input to the target service (must be valid JSON).
     *
     * @default - input for the event target. If the input contains a paths map
     * values wil be extracted from event and inserted into the `inputTemplate`.
     * @stability stable
     */
    readonly input?: string;
    /**
     * JsonPath to take input from the input event.
     *
     * @default - None. The entire matched event is passed as input
     * @stability stable
     */
    readonly inputPath?: string;
    /**
     * Input template to insert paths map into.
     *
     * @default - None.
     * @stability stable
     */
    readonly inputTemplate?: string;
    /**
     * Paths map to extract values from event and insert into `inputTemplate`.
     *
     * @default - No values extracted from event.
     * @stability stable
     */
    readonly inputPathsMap?: {
        [key: string]: string;
    };
}
/**
 * Represents a field in the event pattern.
 *
 * @stability stable
 */
export declare class EventField implements IResolvable {
    readonly path: string;
    /**
     * Extract the event ID from the event.
     *
     * @stability stable
     */
    static get eventId(): string;
    /**
     * Extract the detail type from the event.
     *
     * @stability stable
     */
    static get detailType(): string;
    /**
     * Extract the source from the event.
     *
     * @stability stable
     */
    static get source(): string;
    /**
     * Extract the account from the event.
     *
     * @stability stable
     */
    static get account(): string;
    /**
     * Extract the time from the event.
     *
     * @stability stable
     */
    static get time(): string;
    /**
     * Extract the region from the event.
     *
     * @stability stable
     */
    static get region(): string;
    /**
     * Extract a custom JSON path from the event.
     *
     * @stability stable
     */
    static fromPath(path: string): string;
    /**
     * Human readable display hint about the event pattern.
     *
     * @stability stable
     */
    readonly displayHint: string;
    /**
     * The creation stack of this resolvable which will be appended to errors thrown during resolution.
     *
     * This may return an array with a single informational element indicating how
     * to get this property populated, if it was skipped for performance reasons.
     *
     * @stability stable
     */
    readonly creationStack: string[];
    /**
     *
     * @param path the path to a field in the event pattern
     */
    private constructor();
    /**
     * Produce the Token's value at resolution time.
     *
     * @stability stable
     */
    resolve(_ctx: IResolveContext): any;
    /**
     * Return a string representation of this resolvable object.
     *
     * Returns a reversible string representation.
     *
     * @stability stable
     */
    toString(): string;
    /**
     * Convert the path to the field in the event pattern to JSON.
     *
     * @stability stable
     */
    toJSON(): string;
}
