"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EventField = exports.RuleTargetInput = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
/**
 * The input to send to the event target.
 *
 * @stability stable
 */
class RuleTargetInput {
    /**
     * @stability stable
     */
    constructor() {
    }
    /**
     * Pass text to the event target.
     *
     * May contain strings returned by `EventField.from()` to substitute in parts of the
     * matched event.
     *
     * The Rule Target input value will be a single string: the string you pass
     * here.  Do not use this method to pass a complex value like a JSON object to
     * a Rule Target.  Use `RuleTargetInput.fromObject()` instead.
     *
     * @stability stable
     */
    static fromText(text) {
        return new FieldAwareEventInput(text, InputType.Text);
    }
    /**
     * Pass text to the event target, splitting on newlines.
     *
     * This is only useful when passing to a target that does not
     * take a single argument.
     *
     * May contain strings returned by `EventField.from()` to substitute in parts
     * of the matched event.
     *
     * @stability stable
     */
    static fromMultilineText(text) {
        return new FieldAwareEventInput(text, InputType.Multiline);
    }
    /**
     * Pass a JSON object to the event target.
     *
     * May contain strings returned by `EventField.from()` to substitute in parts of the
     * matched event.
     *
     * @stability stable
     */
    static fromObject(obj) {
        return new FieldAwareEventInput(obj, InputType.Object);
    }
    /**
     * Take the event target input from a path in the event JSON.
     *
     * @stability stable
     */
    static fromEventPath(path) {
        return new LiteralEventInput({ inputPath: path });
    }
}
exports.RuleTargetInput = RuleTargetInput;
_a = JSII_RTTI_SYMBOL_1;
RuleTargetInput[_a] = { fqn: "@aws-cdk/aws-events.RuleTargetInput", version: "1.144.0" };
/**
 * Event Input that is directly derived from the construct
 */
class LiteralEventInput extends RuleTargetInput {
    constructor(props) {
        super();
        this.props = props;
    }
    /**
     * Return the input properties for this input object
     */
    bind(_rule) {
        return this.props;
    }
}
/**
 * Input object that can contain field replacements
 *
 * Evaluation is done in the bind() method because token resolution
 * requires access to the construct tree.
 *
 * Multiple tokens that use the same path will use the same substitution
 * key.
 *
 * One weird exception: if we're in object context, we MUST skip the quotes
 * around the placeholder. I assume this is so once a trivial string replace is
 * done later on by EventBridge, numbers are still numbers.
 *
 * So in string context:
 *
 *    "this is a string with a <field>"
 *
 * But in object context:
 *
 *    "{ \"this is the\": <field> }"
 *
 * To achieve the latter, we postprocess the JSON string to remove the surrounding
 * quotes by using a string replace.
 */
class FieldAwareEventInput extends RuleTargetInput {
    constructor(input, inputType) {
        super();
        this.input = input;
        this.inputType = inputType;
    }
    bind(rule) {
        let fieldCounter = 0;
        const pathToKey = new Map();
        const inputPathsMap = {};
        function keyForField(f) {
            const existing = pathToKey.get(f.path);
            if (existing !== undefined) {
                return existing;
            }
            fieldCounter += 1;
            const key = f.displayHint || `f${fieldCounter}`;
            pathToKey.set(f.path, key);
            return key;
        }
        class EventFieldReplacer extends core_1.DefaultTokenResolver {
            constructor() {
                super(new core_1.StringConcat());
            }
            resolveToken(t, _context) {
                if (!isEventField(t)) {
                    return core_1.Token.asString(t);
                }
                const key = keyForField(t);
                if (inputPathsMap[key] && inputPathsMap[key] !== t.path) {
                    throw new Error(`Single key '${key}' is used for two different JSON paths: '${t.path}' and '${inputPathsMap[key]}'`);
                }
                inputPathsMap[key] = t.path;
                return `<${key}>`;
            }
        }
        const stack = core_1.Stack.of(rule);
        let resolved;
        if (this.inputType === InputType.Multiline) {
            // JSONify individual lines
            resolved = core_1.Tokenization.resolve(this.input, {
                scope: rule,
                resolver: new EventFieldReplacer(),
            });
            resolved = resolved.split('\n').map(stack.toJsonString).join('\n');
        }
        else {
            resolved = stack.toJsonString(core_1.Tokenization.resolve(this.input, {
                scope: rule,
                resolver: new EventFieldReplacer(),
            }));
        }
        const keys = Object.keys(inputPathsMap);
        if (keys.length === 0) {
            // Nothing special, just return 'input'
            return { input: resolved };
        }
        return {
            inputTemplate: this.unquoteKeyPlaceholders(resolved, keys),
            inputPathsMap,
        };
    }
    /**
     * Removing surrounding quotes from any object placeholders
     * when key is the lone value.
     *
     * Those have been put there by JSON.stringify(), but we need to
     * remove them.
     *
     * Do not remove quotes when the key is part of a larger string.
     *
     * Valid: { "data": "Some string with \"quotes\"<key>" } // key will be string
     * Valid: { "data": <key> } // Key could be number, bool, obj, or string
     */
    unquoteKeyPlaceholders(sub, keys) {
        if (this.inputType !== InputType.Object) {
            return sub;
        }
        return core_1.Lazy.uncachedString({ produce: (ctx) => core_1.Token.asString(deepUnquote(ctx.resolve(sub))) });
        function deepUnquote(resolved) {
            if (Array.isArray(resolved)) {
                return resolved.map(deepUnquote);
            }
            else if (typeof (resolved) === 'object' && resolved !== null) {
                for (const [key, value] of Object.entries(resolved)) {
                    resolved[key] = deepUnquote(value);
                }
                return resolved;
            }
            else if (typeof (resolved) === 'string') {
                return keys.reduce((r, key) => r.replace(new RegExp(`(?<!\\\\)\"\<${key}\>\"`, 'g'), `<${key}>`), resolved);
            }
            return resolved;
        }
    }
}
/**
 * Represents a field in the event pattern.
 *
 * @stability stable
 */
class EventField {
    /**
     *
     * @param path the path to a field in the event pattern
     */
    constructor(path) {
        this.path = path;
        this.displayHint = this.path.replace(/^[^a-zA-Z0-9_-]+/, '').replace(/[^a-zA-Z0-9_-]/g, '-');
        Object.defineProperty(this, EVENT_FIELD_SYMBOL, { value: true });
        this.creationStack = core_1.captureStackTrace();
    }
    /**
     * Extract the event ID from the event.
     *
     * @stability stable
     */
    static get eventId() {
        return this.fromPath('$.id');
    }
    /**
     * Extract the detail type from the event.
     *
     * @stability stable
     */
    static get detailType() {
        return this.fromPath('$.detail-type');
    }
    /**
     * Extract the source from the event.
     *
     * @stability stable
     */
    static get source() {
        return this.fromPath('$.source');
    }
    /**
     * Extract the account from the event.
     *
     * @stability stable
     */
    static get account() {
        return this.fromPath('$.account');
    }
    /**
     * Extract the time from the event.
     *
     * @stability stable
     */
    static get time() {
        return this.fromPath('$.time');
    }
    /**
     * Extract the region from the event.
     *
     * @stability stable
     */
    static get region() {
        return this.fromPath('$.region');
    }
    /**
     * Extract a custom JSON path from the event.
     *
     * @stability stable
     */
    static fromPath(path) {
        return new EventField(path).toString();
    }
    /**
     * Produce the Token's value at resolution time.
     *
     * @stability stable
     */
    resolve(_ctx) {
        return this.path;
    }
    /**
     * Return a string representation of this resolvable object.
     *
     * Returns a reversible string representation.
     *
     * @stability stable
     */
    toString() {
        return core_1.Token.asString(this, { displayHint: this.displayHint });
    }
    /**
     * Convert the path to the field in the event pattern to JSON.
     *
     * @stability stable
     */
    toJSON() {
        return `<path:${this.path}>`;
    }
}
exports.EventField = EventField;
_b = JSII_RTTI_SYMBOL_1;
EventField[_b] = { fqn: "@aws-cdk/aws-events.EventField", version: "1.144.0" };
var InputType;
(function (InputType) {
    InputType[InputType["Object"] = 0] = "Object";
    InputType[InputType["Text"] = 1] = "Text";
    InputType[InputType["Multiline"] = 2] = "Multiline";
})(InputType || (InputType = {}));
function isEventField(x) {
    return EVENT_FIELD_SYMBOL in x;
}
const EVENT_FIELD_SYMBOL = Symbol.for('@aws-cdk/aws-events.EventField');
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5wdXQuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJpbnB1dC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBLHdDQUd1Qjs7Ozs7O0FBSXZCLE1BQXNCLGVBQWU7Ozs7SUFxQm5DO0tBQ0M7Ozs7Ozs7Ozs7Ozs7SUFwQk0sTUFBTSxDQUFDLFFBQVEsQ0FBQyxJQUFZO1FBQ2pDLE9BQU8sSUFBSSxvQkFBb0IsQ0FBQyxJQUFJLEVBQUUsU0FBUyxDQUFDLElBQUksQ0FBQyxDQUFDO0tBQ3ZEOzs7Ozs7Ozs7Ozs7SUFHTSxNQUFNLENBQUMsaUJBQWlCLENBQUMsSUFBWTtRQUMxQyxPQUFPLElBQUksb0JBQW9CLENBQUMsSUFBSSxFQUFFLFNBQVMsQ0FBQyxTQUFTLENBQUMsQ0FBQztLQUM1RDs7Ozs7Ozs7O0lBR00sTUFBTSxDQUFDLFVBQVUsQ0FBQyxHQUFRO1FBQy9CLE9BQU8sSUFBSSxvQkFBb0IsQ0FBQyxHQUFHLEVBQUUsU0FBUyxDQUFDLE1BQU0sQ0FBQyxDQUFDO0tBQ3hEOzs7Ozs7SUFHTSxNQUFNLENBQUMsYUFBYSxDQUFDLElBQVk7UUFDdEMsT0FBTyxJQUFJLGlCQUFpQixDQUFDLEVBQUUsU0FBUyxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7S0FDbkQ7O0FBbkJILDBDQTBCQzs7O0FBaUJEOztHQUVHO0FBQ0gsTUFBTSxpQkFBa0IsU0FBUSxlQUFlO0lBQzdDLFlBQTZCLEtBQWdDO1FBQzNELEtBQUssRUFBRSxDQUFDO1FBRG1CLFVBQUssR0FBTCxLQUFLLENBQTJCO0tBRTVEO0lBRUQ7O09BRUc7SUFDSSxJQUFJLENBQUMsS0FBWTtRQUN0QixPQUFPLElBQUksQ0FBQyxLQUFLLENBQUM7S0FDbkI7Q0FDRjtBQUVEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztHQXVCRztBQUNILE1BQU0sb0JBQXFCLFNBQVEsZUFBZTtJQUNoRCxZQUE2QixLQUFVLEVBQW1CLFNBQW9CO1FBQzVFLEtBQUssRUFBRSxDQUFDO1FBRG1CLFVBQUssR0FBTCxLQUFLLENBQUs7UUFBbUIsY0FBUyxHQUFULFNBQVMsQ0FBVztLQUU3RTtJQUVNLElBQUksQ0FBQyxJQUFXO1FBQ3JCLElBQUksWUFBWSxHQUFHLENBQUMsQ0FBQztRQUNyQixNQUFNLFNBQVMsR0FBRyxJQUFJLEdBQUcsRUFBa0IsQ0FBQztRQUM1QyxNQUFNLGFBQWEsR0FBNEIsRUFBRSxDQUFDO1FBRWxELFNBQVMsV0FBVyxDQUFDLENBQWE7WUFDaEMsTUFBTSxRQUFRLEdBQUcsU0FBUyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUM7WUFDdkMsSUFBSSxRQUFRLEtBQUssU0FBUyxFQUFFO2dCQUFFLE9BQU8sUUFBUSxDQUFDO2FBQUU7WUFFaEQsWUFBWSxJQUFJLENBQUMsQ0FBQztZQUNsQixNQUFNLEdBQUcsR0FBRyxDQUFDLENBQUMsV0FBVyxJQUFJLElBQUksWUFBWSxFQUFFLENBQUM7WUFDaEQsU0FBUyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsSUFBSSxFQUFFLEdBQUcsQ0FBQyxDQUFDO1lBQzNCLE9BQU8sR0FBRyxDQUFDO1FBQ2IsQ0FBQztRQUVELE1BQU0sa0JBQW1CLFNBQVEsMkJBQW9CO1lBQ25EO2dCQUNFLEtBQUssQ0FBQyxJQUFJLG1CQUFZLEVBQUUsQ0FBQyxDQUFDO1lBQzVCLENBQUM7WUFFTSxZQUFZLENBQUMsQ0FBUSxFQUFFLFFBQXlCO2dCQUNyRCxJQUFJLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxFQUFFO29CQUFFLE9BQU8sWUFBSyxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQztpQkFBRTtnQkFFbkQsTUFBTSxHQUFHLEdBQUcsV0FBVyxDQUFDLENBQUMsQ0FBQyxDQUFDO2dCQUMzQixJQUFJLGFBQWEsQ0FBQyxHQUFHLENBQUMsSUFBSSxhQUFhLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxDQUFDLElBQUksRUFBRTtvQkFDdkQsTUFBTSxJQUFJLEtBQUssQ0FBQyxlQUFlLEdBQUcsNENBQTRDLENBQUMsQ0FBQyxJQUFJLFVBQVUsYUFBYSxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsQ0FBQztpQkFDdEg7Z0JBQ0QsYUFBYSxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsQ0FBQyxJQUFJLENBQUM7Z0JBRTVCLE9BQU8sSUFBSSxHQUFHLEdBQUcsQ0FBQztZQUNwQixDQUFDO1NBQ0Y7UUFFRCxNQUFNLEtBQUssR0FBRyxZQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBRTdCLElBQUksUUFBZ0IsQ0FBQztRQUNyQixJQUFJLElBQUksQ0FBQyxTQUFTLEtBQUssU0FBUyxDQUFDLFNBQVMsRUFBRTtZQUMxQywyQkFBMkI7WUFDM0IsUUFBUSxHQUFHLG1CQUFZLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUU7Z0JBQzFDLEtBQUssRUFBRSxJQUFJO2dCQUNYLFFBQVEsRUFBRSxJQUFJLGtCQUFrQixFQUFFO2FBQ25DLENBQUMsQ0FBQztZQUNILFFBQVEsR0FBRyxRQUFRLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1NBQ3BFO2FBQU07WUFDTCxRQUFRLEdBQUcsS0FBSyxDQUFDLFlBQVksQ0FBQyxtQkFBWSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFO2dCQUM3RCxLQUFLLEVBQUUsSUFBSTtnQkFDWCxRQUFRLEVBQUUsSUFBSSxrQkFBa0IsRUFBRTthQUNuQyxDQUFDLENBQUMsQ0FBQztTQUNMO1FBRUQsTUFBTSxJQUFJLEdBQUcsTUFBTSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQztRQUV4QyxJQUFJLElBQUksQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQ3JCLHVDQUF1QztZQUN2QyxPQUFPLEVBQUUsS0FBSyxFQUFFLFFBQVEsRUFBRSxDQUFDO1NBQzVCO1FBRUQsT0FBTztZQUNMLGFBQWEsRUFBRSxJQUFJLENBQUMsc0JBQXNCLENBQUMsUUFBUSxFQUFFLElBQUksQ0FBQztZQUMxRCxhQUFhO1NBQ2QsQ0FBQztLQUNIO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSyxzQkFBc0IsQ0FBQyxHQUFXLEVBQUUsSUFBYztRQUN4RCxJQUFJLElBQUksQ0FBQyxTQUFTLEtBQUssU0FBUyxDQUFDLE1BQU0sRUFBRTtZQUFFLE9BQU8sR0FBRyxDQUFDO1NBQUU7UUFFeEQsT0FBTyxXQUFJLENBQUMsY0FBYyxDQUFDLEVBQUUsT0FBTyxFQUFFLENBQUMsR0FBb0IsRUFBRSxFQUFFLENBQUMsWUFBSyxDQUFDLFFBQVEsQ0FBQyxXQUFXLENBQUMsR0FBRyxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDO1FBRWpILFNBQVMsV0FBVyxDQUFDLFFBQWE7WUFDaEMsSUFBSSxLQUFLLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxFQUFFO2dCQUMzQixPQUFPLFFBQVEsQ0FBQyxHQUFHLENBQUMsV0FBVyxDQUFDLENBQUM7YUFDbEM7aUJBQU0sSUFBSSxPQUFNLENBQUMsUUFBUSxDQUFDLEtBQUssUUFBUSxJQUFJLFFBQVEsS0FBSyxJQUFJLEVBQUU7Z0JBQzdELEtBQUssTUFBTSxDQUFDLEdBQUcsRUFBRSxLQUFLLENBQUMsSUFBSSxNQUFNLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxFQUFFO29CQUNuRCxRQUFRLENBQUMsR0FBRyxDQUFDLEdBQUcsV0FBVyxDQUFDLEtBQUssQ0FBQyxDQUFDO2lCQUNwQztnQkFDRCxPQUFPLFFBQVEsQ0FBQzthQUNqQjtpQkFBTSxJQUFJLE9BQU0sQ0FBQyxRQUFRLENBQUMsS0FBSyxRQUFRLEVBQUU7Z0JBQ3hDLE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsRUFBRSxHQUFHLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsSUFBSSxNQUFNLENBQUMsZ0JBQWdCLEdBQUcsTUFBTSxFQUFFLEdBQUcsQ0FBQyxFQUFFLElBQUksR0FBRyxHQUFHLENBQUMsRUFBRSxRQUFRLENBQUMsQ0FBQzthQUM3RztZQUNELE9BQU8sUUFBUSxDQUFDO1FBQ2xCLENBQUM7S0FDRjtDQUNGOzs7Ozs7QUFHRCxNQUFhLFVBQVU7SUF3Q3JCOzs7T0FHRztJQUNILFlBQW9DLElBQVk7UUFBWixTQUFJLEdBQUosSUFBSSxDQUFRO1FBQzlDLElBQUksQ0FBQyxXQUFXLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsa0JBQWtCLEVBQUUsRUFBRSxDQUFDLENBQUMsT0FBTyxDQUFDLGlCQUFpQixFQUFFLEdBQUcsQ0FBQyxDQUFDO1FBQzdGLE1BQU0sQ0FBQyxjQUFjLENBQUMsSUFBSSxFQUFFLGtCQUFrQixFQUFFLEVBQUUsS0FBSyxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7UUFDakUsSUFBSSxDQUFDLGFBQWEsR0FBRyx3QkFBaUIsRUFBRSxDQUFDO0tBQzFDOzs7Ozs7SUE5Q00sTUFBTSxLQUFLLE9BQU87UUFDdkIsT0FBTyxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxDQUFDO0tBQzlCOzs7Ozs7SUFHTSxNQUFNLEtBQUssVUFBVTtRQUMxQixPQUFPLElBQUksQ0FBQyxRQUFRLENBQUMsZUFBZSxDQUFDLENBQUM7S0FDdkM7Ozs7OztJQUdNLE1BQU0sS0FBSyxNQUFNO1FBQ3RCLE9BQU8sSUFBSSxDQUFDLFFBQVEsQ0FBQyxVQUFVLENBQUMsQ0FBQztLQUNsQzs7Ozs7O0lBR00sTUFBTSxLQUFLLE9BQU87UUFDdkIsT0FBTyxJQUFJLENBQUMsUUFBUSxDQUFDLFdBQVcsQ0FBQyxDQUFDO0tBQ25DOzs7Ozs7SUFHTSxNQUFNLEtBQUssSUFBSTtRQUNwQixPQUFPLElBQUksQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLENBQUM7S0FDaEM7Ozs7OztJQUdNLE1BQU0sS0FBSyxNQUFNO1FBQ3RCLE9BQU8sSUFBSSxDQUFDLFFBQVEsQ0FBQyxVQUFVLENBQUMsQ0FBQztLQUNsQzs7Ozs7O0lBR00sTUFBTSxDQUFDLFFBQVEsQ0FBQyxJQUFZO1FBQ2pDLE9BQU8sSUFBSSxVQUFVLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxFQUFFLENBQUM7S0FDeEM7Ozs7OztJQWdCTSxPQUFPLENBQUMsSUFBcUI7UUFDbEMsT0FBTyxJQUFJLENBQUMsSUFBSSxDQUFDO0tBQ2xCOzs7Ozs7OztJQUVNLFFBQVE7UUFDYixPQUFPLFlBQUssQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLEVBQUUsV0FBVyxFQUFFLElBQUksQ0FBQyxXQUFXLEVBQUUsQ0FBQyxDQUFDO0tBQ2hFOzs7Ozs7SUFHTSxNQUFNO1FBQ1gsT0FBTyxTQUFTLElBQUksQ0FBQyxJQUFJLEdBQUcsQ0FBQztLQUM5Qjs7QUE3REgsZ0NBOERDOzs7QUFFRCxJQUFLLFNBSUo7QUFKRCxXQUFLLFNBQVM7SUFDWiw2Q0FBTSxDQUFBO0lBQ04seUNBQUksQ0FBQTtJQUNKLG1EQUFTLENBQUE7QUFDWCxDQUFDLEVBSkksU0FBUyxLQUFULFNBQVMsUUFJYjtBQUVELFNBQVMsWUFBWSxDQUFDLENBQU07SUFDMUIsT0FBTyxrQkFBa0IsSUFBSSxDQUFDLENBQUM7QUFDakMsQ0FBQztBQUVELE1BQU0sa0JBQWtCLEdBQUcsTUFBTSxDQUFDLEdBQUcsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHtcbiAgY2FwdHVyZVN0YWNrVHJhY2UsIERlZmF1bHRUb2tlblJlc29sdmVyLCBJUmVzb2x2YWJsZSxcbiAgSVJlc29sdmVDb250ZXh0LCBMYXp5LCBTdGFjaywgU3RyaW5nQ29uY2F0LCBUb2tlbiwgVG9rZW5pemF0aW9uLFxufSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IElSdWxlIH0gZnJvbSAnLi9ydWxlLXJlZic7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGFic3RyYWN0IGNsYXNzIFJ1bGVUYXJnZXRJbnB1dCB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBmcm9tVGV4dCh0ZXh0OiBzdHJpbmcpOiBSdWxlVGFyZ2V0SW5wdXQge1xuICAgIHJldHVybiBuZXcgRmllbGRBd2FyZUV2ZW50SW5wdXQodGV4dCwgSW5wdXRUeXBlLlRleHQpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBmcm9tTXVsdGlsaW5lVGV4dCh0ZXh0OiBzdHJpbmcpOiBSdWxlVGFyZ2V0SW5wdXQge1xuICAgIHJldHVybiBuZXcgRmllbGRBd2FyZUV2ZW50SW5wdXQodGV4dCwgSW5wdXRUeXBlLk11bHRpbGluZSk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBmcm9tT2JqZWN0KG9iajogYW55KTogUnVsZVRhcmdldElucHV0IHtcbiAgICByZXR1cm4gbmV3IEZpZWxkQXdhcmVFdmVudElucHV0KG9iaiwgSW5wdXRUeXBlLk9iamVjdCk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBmcm9tRXZlbnRQYXRoKHBhdGg6IHN0cmluZyk6IFJ1bGVUYXJnZXRJbnB1dCB7XG4gICAgcmV0dXJuIG5ldyBMaXRlcmFsRXZlbnRJbnB1dCh7IGlucHV0UGF0aDogcGF0aCB9KTtcbiAgfVxuXG4gIHByb3RlY3RlZCBjb25zdHJ1Y3RvcigpIHtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGFic3RyYWN0IGJpbmQocnVsZTogSVJ1bGUpOiBSdWxlVGFyZ2V0SW5wdXRQcm9wZXJ0aWVzO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgUnVsZVRhcmdldElucHV0UHJvcGVydGllcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgaW5wdXQ/OiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGlucHV0UGF0aD86IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgaW5wdXRUZW1wbGF0ZT86IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgaW5wdXRQYXRoc01hcD86IHsgW2tleTogc3RyaW5nXTogc3RyaW5nIH07XG59XG5cbi8qKlxuICogRXZlbnQgSW5wdXQgdGhhdCBpcyBkaXJlY3RseSBkZXJpdmVkIGZyb20gdGhlIGNvbnN0cnVjdFxuICovXG5jbGFzcyBMaXRlcmFsRXZlbnRJbnB1dCBleHRlbmRzIFJ1bGVUYXJnZXRJbnB1dCB7XG4gIGNvbnN0cnVjdG9yKHByaXZhdGUgcmVhZG9ubHkgcHJvcHM6IFJ1bGVUYXJnZXRJbnB1dFByb3BlcnRpZXMpIHtcbiAgICBzdXBlcigpO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybiB0aGUgaW5wdXQgcHJvcGVydGllcyBmb3IgdGhpcyBpbnB1dCBvYmplY3RcbiAgICovXG4gIHB1YmxpYyBiaW5kKF9ydWxlOiBJUnVsZSk6IFJ1bGVUYXJnZXRJbnB1dFByb3BlcnRpZXMge1xuICAgIHJldHVybiB0aGlzLnByb3BzO1xuICB9XG59XG5cbi8qKlxuICogSW5wdXQgb2JqZWN0IHRoYXQgY2FuIGNvbnRhaW4gZmllbGQgcmVwbGFjZW1lbnRzXG4gKlxuICogRXZhbHVhdGlvbiBpcyBkb25lIGluIHRoZSBiaW5kKCkgbWV0aG9kIGJlY2F1c2UgdG9rZW4gcmVzb2x1dGlvblxuICogcmVxdWlyZXMgYWNjZXNzIHRvIHRoZSBjb25zdHJ1Y3QgdHJlZS5cbiAqXG4gKiBNdWx0aXBsZSB0b2tlbnMgdGhhdCB1c2UgdGhlIHNhbWUgcGF0aCB3aWxsIHVzZSB0aGUgc2FtZSBzdWJzdGl0dXRpb25cbiAqIGtleS5cbiAqXG4gKiBPbmUgd2VpcmQgZXhjZXB0aW9uOiBpZiB3ZSdyZSBpbiBvYmplY3QgY29udGV4dCwgd2UgTVVTVCBza2lwIHRoZSBxdW90ZXNcbiAqIGFyb3VuZCB0aGUgcGxhY2Vob2xkZXIuIEkgYXNzdW1lIHRoaXMgaXMgc28gb25jZSBhIHRyaXZpYWwgc3RyaW5nIHJlcGxhY2UgaXNcbiAqIGRvbmUgbGF0ZXIgb24gYnkgRXZlbnRCcmlkZ2UsIG51bWJlcnMgYXJlIHN0aWxsIG51bWJlcnMuXG4gKlxuICogU28gaW4gc3RyaW5nIGNvbnRleHQ6XG4gKlxuICogICAgXCJ0aGlzIGlzIGEgc3RyaW5nIHdpdGggYSA8ZmllbGQ+XCJcbiAqXG4gKiBCdXQgaW4gb2JqZWN0IGNvbnRleHQ6XG4gKlxuICogICAgXCJ7IFxcXCJ0aGlzIGlzIHRoZVxcXCI6IDxmaWVsZD4gfVwiXG4gKlxuICogVG8gYWNoaWV2ZSB0aGUgbGF0dGVyLCB3ZSBwb3N0cHJvY2VzcyB0aGUgSlNPTiBzdHJpbmcgdG8gcmVtb3ZlIHRoZSBzdXJyb3VuZGluZ1xuICogcXVvdGVzIGJ5IHVzaW5nIGEgc3RyaW5nIHJlcGxhY2UuXG4gKi9cbmNsYXNzIEZpZWxkQXdhcmVFdmVudElucHV0IGV4dGVuZHMgUnVsZVRhcmdldElucHV0IHtcbiAgY29uc3RydWN0b3IocHJpdmF0ZSByZWFkb25seSBpbnB1dDogYW55LCBwcml2YXRlIHJlYWRvbmx5IGlucHV0VHlwZTogSW5wdXRUeXBlKSB7XG4gICAgc3VwZXIoKTtcbiAgfVxuXG4gIHB1YmxpYyBiaW5kKHJ1bGU6IElSdWxlKTogUnVsZVRhcmdldElucHV0UHJvcGVydGllcyB7XG4gICAgbGV0IGZpZWxkQ291bnRlciA9IDA7XG4gICAgY29uc3QgcGF0aFRvS2V5ID0gbmV3IE1hcDxzdHJpbmcsIHN0cmluZz4oKTtcbiAgICBjb25zdCBpbnB1dFBhdGhzTWFwOiB7W2tleTogc3RyaW5nXTogc3RyaW5nfSA9IHt9O1xuXG4gICAgZnVuY3Rpb24ga2V5Rm9yRmllbGQoZjogRXZlbnRGaWVsZCkge1xuICAgICAgY29uc3QgZXhpc3RpbmcgPSBwYXRoVG9LZXkuZ2V0KGYucGF0aCk7XG4gICAgICBpZiAoZXhpc3RpbmcgIT09IHVuZGVmaW5lZCkgeyByZXR1cm4gZXhpc3Rpbmc7IH1cblxuICAgICAgZmllbGRDb3VudGVyICs9IDE7XG4gICAgICBjb25zdCBrZXkgPSBmLmRpc3BsYXlIaW50IHx8IGBmJHtmaWVsZENvdW50ZXJ9YDtcbiAgICAgIHBhdGhUb0tleS5zZXQoZi5wYXRoLCBrZXkpO1xuICAgICAgcmV0dXJuIGtleTtcbiAgICB9XG5cbiAgICBjbGFzcyBFdmVudEZpZWxkUmVwbGFjZXIgZXh0ZW5kcyBEZWZhdWx0VG9rZW5SZXNvbHZlciB7XG4gICAgICBjb25zdHJ1Y3RvcigpIHtcbiAgICAgICAgc3VwZXIobmV3IFN0cmluZ0NvbmNhdCgpKTtcbiAgICAgIH1cblxuICAgICAgcHVibGljIHJlc29sdmVUb2tlbih0OiBUb2tlbiwgX2NvbnRleHQ6IElSZXNvbHZlQ29udGV4dCkge1xuICAgICAgICBpZiAoIWlzRXZlbnRGaWVsZCh0KSkgeyByZXR1cm4gVG9rZW4uYXNTdHJpbmcodCk7IH1cblxuICAgICAgICBjb25zdCBrZXkgPSBrZXlGb3JGaWVsZCh0KTtcbiAgICAgICAgaWYgKGlucHV0UGF0aHNNYXBba2V5XSAmJiBpbnB1dFBhdGhzTWFwW2tleV0gIT09IHQucGF0aCkge1xuICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgU2luZ2xlIGtleSAnJHtrZXl9JyBpcyB1c2VkIGZvciB0d28gZGlmZmVyZW50IEpTT04gcGF0aHM6ICcke3QucGF0aH0nIGFuZCAnJHtpbnB1dFBhdGhzTWFwW2tleV19J2ApO1xuICAgICAgICB9XG4gICAgICAgIGlucHV0UGF0aHNNYXBba2V5XSA9IHQucGF0aDtcblxuICAgICAgICByZXR1cm4gYDwke2tleX0+YDtcbiAgICAgIH1cbiAgICB9XG5cbiAgICBjb25zdCBzdGFjayA9IFN0YWNrLm9mKHJ1bGUpO1xuXG4gICAgbGV0IHJlc29sdmVkOiBzdHJpbmc7XG4gICAgaWYgKHRoaXMuaW5wdXRUeXBlID09PSBJbnB1dFR5cGUuTXVsdGlsaW5lKSB7XG4gICAgICAvLyBKU09OaWZ5IGluZGl2aWR1YWwgbGluZXNcbiAgICAgIHJlc29sdmVkID0gVG9rZW5pemF0aW9uLnJlc29sdmUodGhpcy5pbnB1dCwge1xuICAgICAgICBzY29wZTogcnVsZSxcbiAgICAgICAgcmVzb2x2ZXI6IG5ldyBFdmVudEZpZWxkUmVwbGFjZXIoKSxcbiAgICAgIH0pO1xuICAgICAgcmVzb2x2ZWQgPSByZXNvbHZlZC5zcGxpdCgnXFxuJykubWFwKHN0YWNrLnRvSnNvblN0cmluZykuam9pbignXFxuJyk7XG4gICAgfSBlbHNlIHtcbiAgICAgIHJlc29sdmVkID0gc3RhY2sudG9Kc29uU3RyaW5nKFRva2VuaXphdGlvbi5yZXNvbHZlKHRoaXMuaW5wdXQsIHtcbiAgICAgICAgc2NvcGU6IHJ1bGUsXG4gICAgICAgIHJlc29sdmVyOiBuZXcgRXZlbnRGaWVsZFJlcGxhY2VyKCksXG4gICAgICB9KSk7XG4gICAgfVxuXG4gICAgY29uc3Qga2V5cyA9IE9iamVjdC5rZXlzKGlucHV0UGF0aHNNYXApO1xuXG4gICAgaWYgKGtleXMubGVuZ3RoID09PSAwKSB7XG4gICAgICAvLyBOb3RoaW5nIHNwZWNpYWwsIGp1c3QgcmV0dXJuICdpbnB1dCdcbiAgICAgIHJldHVybiB7IGlucHV0OiByZXNvbHZlZCB9O1xuICAgIH1cblxuICAgIHJldHVybiB7XG4gICAgICBpbnB1dFRlbXBsYXRlOiB0aGlzLnVucXVvdGVLZXlQbGFjZWhvbGRlcnMocmVzb2x2ZWQsIGtleXMpLFxuICAgICAgaW5wdXRQYXRoc01hcCxcbiAgICB9O1xuICB9XG5cbiAgLyoqXG4gICAqIFJlbW92aW5nIHN1cnJvdW5kaW5nIHF1b3RlcyBmcm9tIGFueSBvYmplY3QgcGxhY2Vob2xkZXJzXG4gICAqIHdoZW4ga2V5IGlzIHRoZSBsb25lIHZhbHVlLlxuICAgKlxuICAgKiBUaG9zZSBoYXZlIGJlZW4gcHV0IHRoZXJlIGJ5IEpTT04uc3RyaW5naWZ5KCksIGJ1dCB3ZSBuZWVkIHRvXG4gICAqIHJlbW92ZSB0aGVtLlxuICAgKlxuICAgKiBEbyBub3QgcmVtb3ZlIHF1b3RlcyB3aGVuIHRoZSBrZXkgaXMgcGFydCBvZiBhIGxhcmdlciBzdHJpbmcuXG4gICAqXG4gICAqIFZhbGlkOiB7IFwiZGF0YVwiOiBcIlNvbWUgc3RyaW5nIHdpdGggXFxcInF1b3Rlc1xcXCI8a2V5PlwiIH0gLy8ga2V5IHdpbGwgYmUgc3RyaW5nXG4gICAqIFZhbGlkOiB7IFwiZGF0YVwiOiA8a2V5PiB9IC8vIEtleSBjb3VsZCBiZSBudW1iZXIsIGJvb2wsIG9iaiwgb3Igc3RyaW5nXG4gICAqL1xuICBwcml2YXRlIHVucXVvdGVLZXlQbGFjZWhvbGRlcnMoc3ViOiBzdHJpbmcsIGtleXM6IHN0cmluZ1tdKSB7XG4gICAgaWYgKHRoaXMuaW5wdXRUeXBlICE9PSBJbnB1dFR5cGUuT2JqZWN0KSB7IHJldHVybiBzdWI7IH1cblxuICAgIHJldHVybiBMYXp5LnVuY2FjaGVkU3RyaW5nKHsgcHJvZHVjZTogKGN0eDogSVJlc29sdmVDb250ZXh0KSA9PiBUb2tlbi5hc1N0cmluZyhkZWVwVW5xdW90ZShjdHgucmVzb2x2ZShzdWIpKSkgfSk7XG5cbiAgICBmdW5jdGlvbiBkZWVwVW5xdW90ZShyZXNvbHZlZDogYW55KTogYW55IHtcbiAgICAgIGlmIChBcnJheS5pc0FycmF5KHJlc29sdmVkKSkge1xuICAgICAgICByZXR1cm4gcmVzb2x2ZWQubWFwKGRlZXBVbnF1b3RlKTtcbiAgICAgIH0gZWxzZSBpZiAodHlwZW9mKHJlc29sdmVkKSA9PT0gJ29iamVjdCcgJiYgcmVzb2x2ZWQgIT09IG51bGwpIHtcbiAgICAgICAgZm9yIChjb25zdCBba2V5LCB2YWx1ZV0gb2YgT2JqZWN0LmVudHJpZXMocmVzb2x2ZWQpKSB7XG4gICAgICAgICAgcmVzb2x2ZWRba2V5XSA9IGRlZXBVbnF1b3RlKHZhbHVlKTtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gcmVzb2x2ZWQ7XG4gICAgICB9IGVsc2UgaWYgKHR5cGVvZihyZXNvbHZlZCkgPT09ICdzdHJpbmcnKSB7XG4gICAgICAgIHJldHVybiBrZXlzLnJlZHVjZSgociwga2V5KSA9PiByLnJlcGxhY2UobmV3IFJlZ0V4cChgKD88IVxcXFxcXFxcKVxcXCJcXDwke2tleX1cXD5cXFwiYCwgJ2cnKSwgYDwke2tleX0+YCksIHJlc29sdmVkKTtcbiAgICAgIH1cbiAgICAgIHJldHVybiByZXNvbHZlZDtcbiAgICB9XG4gIH1cbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBFdmVudEZpZWxkIGltcGxlbWVudHMgSVJlc29sdmFibGUge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGdldCBldmVudElkKCk6IHN0cmluZyB7XG4gICAgcmV0dXJuIHRoaXMuZnJvbVBhdGgoJyQuaWQnKTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgZ2V0IGRldGFpbFR5cGUoKTogc3RyaW5nIHtcbiAgICByZXR1cm4gdGhpcy5mcm9tUGF0aCgnJC5kZXRhaWwtdHlwZScpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgZ2V0IHNvdXJjZSgpOiBzdHJpbmcge1xuICAgIHJldHVybiB0aGlzLmZyb21QYXRoKCckLnNvdXJjZScpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGdldCBhY2NvdW50KCk6IHN0cmluZyB7XG4gICAgcmV0dXJuIHRoaXMuZnJvbVBhdGgoJyQuYWNjb3VudCcpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGdldCB0aW1lKCk6IHN0cmluZyB7XG4gICAgcmV0dXJuIHRoaXMuZnJvbVBhdGgoJyQudGltZScpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgZ2V0IHJlZ2lvbigpOiBzdHJpbmcge1xuICAgIHJldHVybiB0aGlzLmZyb21QYXRoKCckLnJlZ2lvbicpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBmcm9tUGF0aChwYXRoOiBzdHJpbmcpOiBzdHJpbmcge1xuICAgIHJldHVybiBuZXcgRXZlbnRGaWVsZChwYXRoKS50b1N0cmluZygpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBkaXNwbGF5SGludDogc3RyaW5nO1xuICBwdWJsaWMgcmVhZG9ubHkgY3JlYXRpb25TdGFjazogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqXG4gICAqIEBwYXJhbSBwYXRoIHRoZSBwYXRoIHRvIGEgZmllbGQgaW4gdGhlIGV2ZW50IHBhdHRlcm5cbiAgICovXG4gIHByaXZhdGUgY29uc3RydWN0b3IocHVibGljIHJlYWRvbmx5IHBhdGg6IHN0cmluZykge1xuICAgIHRoaXMuZGlzcGxheUhpbnQgPSB0aGlzLnBhdGgucmVwbGFjZSgvXlteYS16QS1aMC05Xy1dKy8sICcnKS5yZXBsYWNlKC9bXmEtekEtWjAtOV8tXS9nLCAnLScpO1xuICAgIE9iamVjdC5kZWZpbmVQcm9wZXJ0eSh0aGlzLCBFVkVOVF9GSUVMRF9TWU1CT0wsIHsgdmFsdWU6IHRydWUgfSk7XG4gICAgdGhpcy5jcmVhdGlvblN0YWNrID0gY2FwdHVyZVN0YWNrVHJhY2UoKTtcbiAgfVxuXG4gIHB1YmxpYyByZXNvbHZlKF9jdHg6IElSZXNvbHZlQ29udGV4dCk6IGFueSB7XG4gICAgcmV0dXJuIHRoaXMucGF0aDtcbiAgfVxuXG4gIHB1YmxpYyB0b1N0cmluZygpIHtcbiAgICByZXR1cm4gVG9rZW4uYXNTdHJpbmcodGhpcywgeyBkaXNwbGF5SGludDogdGhpcy5kaXNwbGF5SGludCB9KTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHRvSlNPTigpIHtcbiAgICByZXR1cm4gYDxwYXRoOiR7dGhpcy5wYXRofT5gO1xuICB9XG59XG5cbmVudW0gSW5wdXRUeXBlIHtcbiAgT2JqZWN0LFxuICBUZXh0LFxuICBNdWx0aWxpbmUsXG59XG5cbmZ1bmN0aW9uIGlzRXZlbnRGaWVsZCh4OiBhbnkpOiB4IGlzIEV2ZW50RmllbGQge1xuICByZXR1cm4gRVZFTlRfRklFTERfU1lNQk9MIGluIHg7XG59XG5cbmNvbnN0IEVWRU5UX0ZJRUxEX1NZTUJPTCA9IFN5bWJvbC5mb3IoJ0Bhd3MtY2RrL2F3cy1ldmVudHMuRXZlbnRGaWVsZCcpO1xuIl19