"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NxMonorepoProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const fs = require("fs");
const path = require("path");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const python_1 = require("projen/lib/python");
const typescript_1 = require("projen/lib/typescript");
const syncpack_options_1 = require("./syncpack-options");
const NX_MONOREPO_PLUGIN_PATH = ".nx/plugins/nx-monorepo-plugin.js";
/**
 * This project type will bootstrap a NX based monorepo with support for polygot
 * builds, build caching, dependency graph visualization and much more.
 *
 * @pjid nx-monorepo
 */
class NxMonorepoProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        super({
            ...options,
            github: options.github ?? false,
            package: options.package ?? false,
            prettier: options.prettier ?? true,
            projenrcTs: true,
            release: options.release ?? false,
            jest: options.jest ?? false,
            defaultReleaseBranch: options.defaultReleaseBranch ?? "mainline",
            sampleCode: false,
            eslintOptions: options.eslintOptions ?? {
                dirs: ["."],
                ignorePatterns: ["packages/**/*.*"],
            },
            tsconfig: options.tsconfig ?? {
                compilerOptions: {
                    rootDir: ".",
                },
                include: ["**/*.ts"],
            },
        });
        this.nxConfig = options.nxConfig;
        this.workspaceConfig = options.workspaceConfig;
        this.workspacePackages = options.workspaceConfig?.additionalPackages ?? [];
        this.implicitDependencies = this.nxConfig?.implicitDependencies || {};
        // Never publish a monorepo root package.
        this.package.addField("private", true);
        // No need to compile or test a monorepo root package.
        this.compileTask.reset();
        this.testTask.reset();
        this.addDevDeps("@nrwl/cli", "@nrwl/workspace");
        this.addDeps("aws-cdk-lib", "constructs", "cdk-nag"); // Needed as this can be bundled in aws-prototyping-sdk
        this.package.addPackageResolutions("@types/babel__traverse@7.18.2");
        if (options.monorepoUpgradeDeps !== false) {
            this.addDevDeps("npm-check-updates", "syncpack");
            const upgradeDepsTask = this.addTask(options.monorepoUpgradeDepsOptions?.taskName || "upgrade-deps");
            upgradeDepsTask.exec("npx npm-check-updates --deep --rejectVersion 0.0.0 -u");
            upgradeDepsTask.exec("npx syncpack fix-mismatches");
            upgradeDepsTask.exec(`${this.package.packageManager} install`);
            upgradeDepsTask.exec("npx projen");
            new projen_1.JsonFile(this, ".syncpackrc.json", {
                obj: options.monorepoUpgradeDepsOptions?.syncpackConfig || syncpack_options_1.DEFAULT_CONFIG,
                readonly: true,
            });
        }
        options.nxConfig?.nxCloudReadOnlyAccessToken &&
            this.addDevDeps("@nrwl/nx-cloud");
        new projen_1.IgnoreFile(this, ".nxignore").exclude("test-reports", "target", ".env", ".pytest_cache", ...(this.nxConfig?.nxIgnore || []));
        new projen_1.TextFile(this, NX_MONOREPO_PLUGIN_PATH, {
            readonly: true,
            lines: fs.readFileSync(getPluginPath()).toString("utf-8").split("\n"),
        });
        this.nxJson = new projen_1.JsonFile(this, "nx.json", {
            obj: {
                extends: "@nrwl/workspace/presets/npm.json",
                plugins: [`./${NX_MONOREPO_PLUGIN_PATH}`],
                npmScope: "monorepo",
                tasksRunnerOptions: {
                    default: {
                        runner: options.nxConfig?.nxCloudReadOnlyAccessToken
                            ? "@nrwl/nx-cloud"
                            : "@nrwl/workspace/tasks-runners/default",
                        options: {
                            useDaemonProcess: false,
                            cacheableOperations: options.nxConfig?.cacheableOperations || [
                                "build",
                                "test",
                            ],
                            accessToken: options.nxConfig?.nxCloudReadOnlyAccessToken,
                        },
                    },
                },
                namedInputs: options.nxConfig?.namedInputs,
                targetDefaults: options.nxConfig?.targetDefaults,
                implicitDependencies: this.implicitDependencies,
                targetDependencies: {
                    build: [
                        {
                            target: "build",
                            projects: "dependencies",
                        },
                    ],
                    ...(this.nxConfig?.targetDependencies || {}),
                },
                affected: {
                    defaultBase: this.nxConfig?.affectedBranch || "mainline",
                },
            },
        });
    }
    /**
     * Create an implicit dependency between two Project's. This is typically
     * used in polygot repos where a Typescript project wants a build dependency
     * on a Python project as an example.
     *
     * @param dependent project you want to have the dependency.
     * @param dependee project you wish to depend on.
     */
    addImplicitDependency(dependent, dependee) {
        if (this.implicitDependencies[dependent.name]) {
            this.implicitDependencies[dependent.name].push(dependee.name);
        }
        else {
            this.implicitDependencies[dependent.name] = [dependee.name];
        }
    }
    /**
     * Add one or more additional package globs to the workspace.
     * @param packageGlobs paths to the package to include in the workspace (for example packages/my-package)
     */
    addWorkspacePackages(...packageGlobs) {
        // Any subprojects that were added since the last call to this method need to be added first, in order to ensure
        // we add the workspace packages in a sane order.
        const relativeSubProjectWorkspacePackages = this.instantiationOrderSubProjects.map((project) => path.relative(this.outdir, project.outdir));
        const existingWorkspacePackages = new Set(this.workspacePackages);
        this.workspacePackages.push(...relativeSubProjectWorkspacePackages.filter((pkg) => !existingWorkspacePackages.has(pkg)));
        // Add the additional packages next
        this.workspacePackages.push(...packageGlobs);
    }
    // Remove this hack once subProjects is made public in Projen
    get instantiationOrderSubProjects() {
        // @ts-ignore
        const subProjects = this.subprojects || [];
        return subProjects;
    }
    get subProjects() {
        return this.instantiationOrderSubProjects.sort((a, b) => a.name.localeCompare(b.name));
    }
    /**
     * @inheritDoc
     */
    synth() {
        this.validateSubProjects();
        this.updateWorkspace();
        this.wirePythonDependencies();
        this.synthesizeNonNodePackageJson();
        // Prevent sub NodeProject packages from `postSynthesis` which will cause individual/extraneous installs.
        // The workspace package install will handle all the sub NodeProject packages automatically.
        const subProjectPackages = [];
        this.subProjects.forEach((subProject) => {
            if (isNodeProject(subProject)) {
                const subNodeProject = subProject;
                subProjectPackages.push(subNodeProject.package);
                // @ts-ignore - `installDependencies` is private
                subNodeProject.package.installDependencies = () => { };
            }
        });
        super.synth();
        // Force workspace install deps if any node subproject package has change, unless the workspace changed
        if (
        // @ts-ignore - `file` is private
        this.package.file.changed !== true &&
            // @ts-ignore - `file` is private
            subProjectPackages.find((pkg) => pkg.file.changed === true)) {
            // @ts-ignore - `installDependencies` is private
            this.package.installDependencies();
        }
    }
    /**
     * Ensures subprojects don't have a default task and that all packages use the same package manager.
     */
    validateSubProjects() {
        this.subProjects.forEach((subProject) => {
            // Disable default task on subprojects as this isn't supported in a monorepo
            subProject.defaultTask?.reset();
            if (isNodeProject(subProject) &&
                subProject.package.packageManager !== this.package.packageManager) {
                throw new Error(`${subProject.name} packageManager does not match the monorepo packageManager: ${this.package.packageManager}.`);
            }
        });
    }
    /**
     * For non-node projects, a package.json is required in order to be discovered by NX.
     */
    synthesizeNonNodePackageJson() {
        this.subProjects
            .filter((subProject) => !isNodeProject(subProject))
            .filter((subProject) => !subProject.tryFindFile("package.json"))
            .forEach((subProject) => {
            // generate a package.json if not found
            const manifest = {
                name: subProject.name,
                private: true,
                __pdk__: true,
                scripts: subProject.tasks.all.reduce((p, c) => ({
                    [c.name]: `npx projen ${c.name}`,
                    ...p,
                }), {}),
                version: "0.0.0",
            };
            new projen_1.JsonFile(subProject, "package.json", {
                obj: manifest,
                readonly: true,
            });
        });
    }
    /**
     * Add a submodule entry to the appropriate workspace file.
     */
    updateWorkspace() {
        // A final call to addWorkspacePackages will update the list of workspace packages with any subprojects that have
        // not yet been added, in the correct order
        this.addWorkspacePackages();
        let noHoist = this.workspaceConfig?.noHoist;
        // Automatically add all sub-project "bundledDependencies" to workspace "hohoist", otherwise they are not bundled in npm package
        if (this.workspaceConfig?.disableNoHoistBundled !== true) {
            const noHoistBundled = this.subProjects.flatMap((sub) => {
                if (sub instanceof javascript_1.NodeProject) {
                    return sub.deps.all
                        .filter((dep) => dep.type === projen_1.DependencyType.BUNDLED)
                        .flatMap((dep) => [
                        `${sub.name}/${dep.name}`,
                        `${sub.name}/${dep.name}/*`,
                    ]);
                }
                return [];
            });
            if (noHoistBundled.length) {
                noHoist = [...(noHoist || []), ...noHoistBundled];
            }
        }
        // Add workspaces for each subproject
        if (this.package.packageManager === javascript_1.NodePackageManager.PNPM) {
            new projen_1.YamlFile(this, "pnpm-workspace.yaml", {
                readonly: true,
                obj: {
                    packages: this.workspacePackages,
                },
            });
        }
        else {
            this.package.addField("workspaces", {
                packages: this.workspacePackages,
                nohoist: noHoist,
            });
        }
    }
    /**
     * Updates the install task for python projects so that they are run serially and in dependency order such that python
     * projects within the monorepo can declare dependencies on one another.
     * @private
     */
    wirePythonDependencies() {
        // Find any python projects
        const pythonProjects = this.subProjects.filter((project) => project instanceof python_1.PythonProject);
        if (pythonProjects.length === 0) {
            // Nothing to do for no python projects
            return;
        }
        // Move all install commands to install-py so that they are not installed in parallel by the monorepo package manager.
        // eg yarn install will run the install task for all packages in parallel which can lead to conflicts for python.
        pythonProjects.forEach((pythonProject) => {
            const installPyTask = pythonProject.tasks.tryFind("install-py") ??
                pythonProject.addTask("install-py");
            const installTask = pythonProject.tasks.tryFind("install");
            (installTask?.steps || []).forEach((step) => {
                this.copyStepIntoTask(step, installPyTask, pythonProject);
            });
            installTask?.reset();
        });
        // Add an install task to the monorepo to include running the install-py command serially to avoid conflicting writes
        // to a shared virtual env. This is also managed by nx so that installs occur in dependency order.
        const monorepoInstallTask = this.tasks.tryFind("install") ?? this.addTask("install");
        monorepoInstallTask.exec(`npx nx run-many --target install-py --projects ${pythonProjects
            .map((project) => project.name)
            .join(",")} --parallel=1`);
        // Update the nx.json to ensure that install-py follows dependency order
        this.nxJson.addOverride("targetDependencies.install-py", [
            {
                target: "install-py",
                projects: "dependencies",
            },
        ]);
    }
    /**
     * Copies the given step into the given task. Step and Task must be from the given Project
     * @private
     */
    copyStepIntoTask(step, task, project) {
        if (step.exec) {
            task.exec(step.exec, { name: step.name, cwd: step.cwd });
        }
        else if (step.say) {
            task.say(step.say, { name: step.name, cwd: step.cwd });
        }
        else if (step.spawn) {
            const stepToSpawn = project.tasks.tryFind(step.spawn);
            if (stepToSpawn) {
                task.spawn(stepToSpawn, { name: step.name, cwd: step.cwd });
            }
        }
        else if (step.builtin) {
            task.builtin(step.builtin);
        }
    }
}
exports.NxMonorepoProject = NxMonorepoProject;
_a = JSII_RTTI_SYMBOL_1;
NxMonorepoProject[_a] = { fqn: "aws-prototyping-sdk.nx_monorepo.NxMonorepoProject", version: "0.14.5" };
/**
 * Determines if the passed in project is of type NodeProject.
 *
 * @param project Project instance.
 * @returns true if the project instance is of type NodeProject.
 */
function isNodeProject(project) {
    return project instanceof javascript_1.NodeProject || project.package;
}
function getPluginPath() {
    return path.join(__dirname, "plugin/nx-monorepo-plugin.js");
}
//# sourceMappingURL=data:application/json;base64,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