"use strict";
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cdk_nag_1 = require("cdk-nag");
const aws_prototyping_1 = require("../src/packs/aws-prototyping");
const expectedErrors = [
    "AwsPrototyping-AutoScalingLaunchConfigPublicIpDisabled",
    "AwsPrototyping-Cloud9InstanceNoIngressSystemsManager",
    "AwsPrototyping-CloudFrontDistributionGeoRestrictions",
    "AwsPrototyping-CloudFrontDistributionS3OriginAccessIdentity",
    "AwsPrototyping-CodeBuildProjectEnvVarAwsCred",
    "AwsPrototyping-CognitoUserPoolNoUnauthenticatedLogins",
    "AwsPrototyping-CognitoUserPoolStrongPasswordPolicy",
    "AwsPrototyping-DMSReplicationNotPublic",
    "AwsPrototyping-DocumentDBCredentialsInSecretsManager",
    "AwsPrototyping-EC2InstancesInVPC",
    "AwsPrototyping-EC2RestrictedCommonPorts",
    "AwsPrototyping-EC2RestrictedInbound",
    "AwsPrototyping-EC2RestrictedSSH",
    "AwsPrototyping-ECROpenAccess",
    "AwsPrototyping-EKSClusterNoEndpointPublicAccess",
    "AwsPrototyping-ElastiCacheClusterInVPC",
    "AwsPrototyping-ElasticBeanstalkManagedUpdatesEnabled",
    "AwsPrototyping-EventBusOpenAccess",
    "AwsPrototyping-IAMPolicyNoStatementsWithAdminAccess",
    "AwsPrototyping-LambdaFunctionPublicAccessProhibited",
    "AwsPrototyping-LambdaFunctionUrlAuth",
    "AwsPrototyping-NeptuneClusterAutomaticMinorVersionUpgrade",
    "AwsPrototyping-OpenSearchInVPCOnly",
    "AwsPrototyping-OpenSearchNoUnsignedOrAnonymousAccess",
    "AwsPrototyping-RDSAutomaticMinorVersionUpgradeEnabled",
    "AwsPrototyping-RDSInstancePublicAccess",
    "AwsPrototyping-RDSRestrictedInbound",
    "AwsPrototyping-RedshiftClusterInVPC",
    "AwsPrototyping-RedshiftClusterPublicAccess",
    "AwsPrototyping-RedshiftClusterVersionUpgrade",
    "AwsPrototyping-S3BucketLevelPublicAccessProhibited",
    "AwsPrototyping-S3BucketPublicReadProhibited",
    "AwsPrototyping-S3BucketPublicWriteProhibited",
    "AwsPrototyping-S3WebBucketOAIAccess",
    "AwsPrototyping-VPCSubnetAutoAssignPublicIpDisabled",
];
const expectedWarnings = [
    "AwsPrototyping-APIGWAssociatedWithWAF",
    "AwsPrototyping-APIGWAuthorization",
    "AwsPrototyping-APIGWRequestValidation",
    "AwsPrototyping-AthenaWorkgroupEncryptedQueryResults",
    "AwsPrototyping-CloudFrontDistributionHttpsViewerNoOutdatedSSL",
    "AwsPrototyping-CloudFrontDistributionNoOutdatedSSL",
    "AwsPrototyping-CloudFrontDistributionWAFIntegration",
    "AwsPrototyping-CodeBuildProjectKMSEncryptedArtifacts",
    "AwsPrototyping-CodeBuildProjectManagedImages",
    "AwsPrototyping-CodeBuildProjectPrivilegedModeDisabled",
    "AwsPrototyping-CognitoUserPoolAdvancedSecurityModeEnforced",
    "AwsPrototyping-CognitoUserPoolAPIGWAuthorizer",
    "AwsPrototyping-CognitoUserPoolMFA",
    "AwsPrototyping-DocumentDBClusterEncryptionAtRest",
    "AwsPrototyping-DocumentDBClusterNonDefaultPort",
    "AwsPrototyping-DAXEncrypted",
    "AwsPrototyping-EC2EBSVolumeEncrypted",
    "AwsPrototyping-EC2InstanceNoPublicIp",
    "AwsPrototyping-EC2InstanceProfileAttached",
    "AwsPrototyping-EC2SecurityGroupDescription",
    "AwsPrototyping-ECSTaskDefinitionContainerLogging",
    "AwsPrototyping-ECSTaskDefinitionNoEnvironmentVariables",
    "AwsPrototyping-EFSEncrypted",
    "AwsPrototyping-EKSClusterControlPlaneLogs",
    "AwsPrototyping-ElastiCacheClusterNonDefaultPort",
    "AwsPrototyping-ElastiCacheRedisClusterEncryption",
    "AwsPrototyping-ElastiCacheRedisClusterRedisAuth",
    "AwsPrototyping-ElasticBeanstalkEC2InstanceLogsToS3",
    "AwsPrototyping-ElasticBeanstalkVPCSpecified",
    "AwsPrototyping-ALBHttpToHttpsRedirection",
    "AwsPrototyping-ELBTlsHttpsListenersOnly",
    "AwsPrototyping-EMRAuthEC2KeyPairOrKerberos",
    "AwsPrototyping-EMREncryptionInTransit",
    "AwsPrototyping-EMRLocalDiskEncryption",
    "AwsPrototyping-GlueEncryptedCloudWatchLogs",
    "AwsPrototyping-GlueJobBookmarkEncrypted",
    "AwsPrototyping-IAMNoManagedPolicies",
    "AwsPrototyping-IAMNoWildcardPermissions",
    "AwsPrototyping-IAMPolicyNoStatementsWithFullAccess",
    "AwsPrototyping-KinesisDataFirehoseSSE",
    "AwsPrototyping-KinesisDataStreamDefaultKeyWhenSSE",
    "AwsPrototyping-KinesisDataStreamSSE",
    "AwsPrototyping-KMSBackingKeyRotationEnabled",
    "AwsPrototyping-LambdaLatestVersion",
    "AwsPrototyping-MediaStoreContainerCORSPolicy",
    "AwsPrototyping-MediaStoreContainerHasContainerPolicy",
    "AwsPrototyping-MediaStoreContainerSSLRequestsOnly",
    "AwsPrototyping-MSKBrokerToBrokerTLS",
    "AwsPrototyping-MSKClientToBrokerTLS",
    "AwsPrototyping-NeptuneClusterEncryptionAtRest",
    "AwsPrototyping-NeptuneClusterIAMAuth",
    "AwsPrototyping-OpenSearchAllowlistedIPs",
    "AwsPrototyping-OpenSearchEncryptedAtRest",
    "AwsPrototyping-OpenSearchNodeToNodeEncryption",
    "AwsPrototyping-QuicksightSSLConnections",
    "AwsPrototyping-AuroraMySQLPostgresIAMAuth",
    "AwsPrototyping-RDSNonDefaultPort",
    "AwsPrototyping-RDSStorageEncrypted",
    "AwsPrototyping-RedshiftClusterEncryptionAtRest",
    "AwsPrototyping-RedshiftClusterNonDefaultPort",
    "AwsPrototyping-RedshiftClusterNonDefaultUsername",
    "AwsPrototyping-RedshiftRequireTlsSSL",
    "AwsPrototyping-S3BucketLoggingEnabled",
    "AwsPrototyping-S3BucketServerSideEncryptionEnabled",
    "AwsPrototyping-S3BucketSSLRequestsOnly",
    "AwsPrototyping-SageMakerNotebookInstanceKMSKeyConfigured",
    "AwsPrototyping-SageMakerNotebookInVPC",
    "AwsPrototyping-SageMakerNotebookNoDirectInternetAccess",
    "AwsPrototyping-SecretsManagerRotationEnabled",
    "AwsPrototyping-SNSEncryptedKMS",
    "AwsPrototyping-SNSTopicSSLPublishOnly",
    "AwsPrototyping-SQSQueueSSE",
    "AwsPrototyping-SQSQueueSSLRequestsOnly",
    "AwsPrototyping-TimestreamDatabaseCustomerManagedKey",
    "AwsPrototyping-VPCNoNACLs",
];
describe("Check NagPack Details", () => {
    describe("AwsPrototyping-CDK", () => {
        class AwsPrototypingChecksExtended extends aws_prototyping_1.AwsPrototypingChecks {
            constructor() {
                super(...arguments);
                this.actualWarnings = new Array();
                this.actualErrors = new Array();
            }
            applyRule(params) {
                const ruleSuffix = params.ruleSuffixOverride
                    ? params.ruleSuffixOverride
                    : params.rule.name;
                const ruleId = `${pack.readPackName}-${ruleSuffix}`;
                if (params.level === cdk_nag_1.NagMessageLevel.WARN) {
                    this.actualWarnings.push(ruleId);
                }
                else {
                    this.actualErrors.push(ruleId);
                }
            }
        }
        const pack = new AwsPrototypingChecksExtended();
        test("Pack Name is correct", () => {
            expect(pack.readPackName).toStrictEqual("AwsPrototyping");
        });
        test("Pack contains expected warning and error rules", () => {
            jest.spyOn(pack, "applyRule");
            const stack = new aws_cdk_lib_1.Stack();
            aws_cdk_lib_1.Aspects.of(stack).add(pack);
            new aws_cdk_lib_1.CfnResource(stack, "rTestResource", { type: "foo" });
            assert_1.SynthUtils.synthesize(stack).messages;
            expect(pack.actualWarnings.sort()).toEqual(expectedWarnings.sort());
            expect(pack.actualErrors.sort()).toEqual(expectedErrors.sort());
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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