# Copyright 2016- Game Server Services, Inc. or its affiliates. All Rights
# Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License").
# You may not use this file except in compliance with the License.
# A copy of the License is located at
#
#  http://www.apache.org/licenses/LICENSE-2.0
#
# or in the "license" file accompanying this file. This file is distributed
# on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
# express or implied. See the License for the specific language governing
# permissions and limitations under the License.
from __future__ import annotations
from typing import *

from ...core.model import CdkResource, Stack
from ...core.func import GetAttr
from ...core.model import NotificationSetting
from ...core.model import LogSetting

from ..ref.NamespaceRef import NamespaceRef
from .CurrentMasterData import CurrentMasterData
from .DistributorModel import DistributorModel

from .options.NamespaceOptions import NamespaceOptions


class Namespace(CdkResource):
    stack: Stack
    name: str
    description: Optional[str] = None
    assume_user_id: Optional[str] = None
    auto_run_stamp_sheet_notification: Optional[NotificationSetting] = None
    auto_run_transaction_notification: Optional[NotificationSetting] = None
    log_setting: Optional[LogSetting] = None

    def __init__(
        self,
        stack: Stack,
        name: str,
        options: Optional[NamespaceOptions] = NamespaceOptions(),
    ):
        super().__init__(
            "Distributor_Namespace_" + name
        )

        self.stack = stack
        self.name = name
        self.description = options.description if options.description else None
        self.assume_user_id = options.assume_user_id if options.assume_user_id else None
        self.auto_run_stamp_sheet_notification = options.auto_run_stamp_sheet_notification if options.auto_run_stamp_sheet_notification else None
        self.auto_run_transaction_notification = options.auto_run_transaction_notification if options.auto_run_transaction_notification else None
        self.log_setting = options.log_setting if options.log_setting else None
        stack.add_resource(
            self,
        )


    def alternate_keys(
        self,
    ):
        return "name"

    def resource_type(
        self,
    ) -> str:
        return "GS2::Distributor::Namespace"

    def properties(
        self,
    ) -> Dict[str, Any]:
        properties: Dict[str, Any] = {}

        if self.name is not None:
            properties["Name"] = self.name
        if self.description is not None:
            properties["Description"] = self.description
        if self.assume_user_id is not None:
            properties["AssumeUserId"] = self.assume_user_id
        if self.auto_run_stamp_sheet_notification is not None:
            properties["AutoRunStampSheetNotification"] = self.auto_run_stamp_sheet_notification.properties(
            )
        if self.auto_run_transaction_notification is not None:
            properties["AutoRunTransactionNotification"] = self.auto_run_transaction_notification.properties(
            )
        if self.log_setting is not None:
            properties["LogSetting"] = self.log_setting.properties(
            )

        return properties

    def ref(
        self,
    ) -> NamespaceRef:
        return NamespaceRef(
            self.name,
        )

    def get_attr_namespace_id(
        self,
    ) -> GetAttr:
        return GetAttr(
            self,
            "Item.NamespaceId",
            None,
        )

    def master_data(
        self,
        distributor_models: List[DistributorModel],
    ) -> Namespace:
        CurrentMasterData(
            self.stack,
            self.name,
            distributor_models,
        ).add_depends_on(
            self,
        )
        return self
