from notecoin.huobi.constant import *


class Trade:
    """
    The trade information with price and amount etc.

    :member
        price: The trading price in quote currency.
        amount: The trading volume in base currency.
        trade_id: The unique trade id of this trade.
        timestamp: The UNIX formatted timestamp generated by server in UTC.
        direction: The direction of the taker trade: 'buy' or 'sell'.
    """

    def __init__(self):
        self.price = 0.0
        self.amount = 0.0
        self.trade_id = 0
        self.ts = 0
        self.direction = ""

    def print_object(self, format_data=""):
        from huobi.utils.print_mix_object import PrintBasic
        PrintBasic.print_basic(self.trade_id, format_data + "Trade Id")
        PrintBasic.print_basic(self.ts, format_data + "Trade Time")
        PrintBasic.print_basic(self.price, format_data + "Price")
        PrintBasic.print_basic(self.amount, format_data + "Amount")
        PrintBasic.print_basic(self.direction, format_data + "Direction")


class TradeDetail:
    """
    The trade information with price and amount etc.

    :member
        price: The trading price in quote currency.
        amount: The trading volume in base currency.
        tradeId: The unique trade id of this trade.
        timestamp: The UNIX formatted timestamp generated by server in UTC.
        direction: The direction of the taker trade: 'buy' or 'sell'.
    """

    def __init__(self):
        self.price = 0.0
        self.amount = 0.0
        self.tradeId = 0
        self.ts = 0
        self.direction = ""

    def print_object(self, format_data=""):
        from huobi.utils.print_mix_object import PrintBasic
        PrintBasic.print_basic(self.tradeId, format_data + "Trade Id")
        PrintBasic.print_basic(self.ts, format_data + "Trade Time")
        PrintBasic.print_basic(self.price, format_data + "Price")
        PrintBasic.print_basic(self.amount, format_data + "Amount")
        PrintBasic.print_basic(self.direction, format_data + "Direction")


class TradeDetailEvent:
    """
    The trade received by subscription of trade.

    :member
        symbol: The symbol you subscribed.
        timestamp: The UNIX formatted timestamp generated by server in UTC.
        trade_list: The trade list. The content is Trade class.
    """

    def __init__(self):
        self.ch = ""
        self.id = 0
        self.ts = 0
        self.data = list()

    def print_object(self, format_data=""):
        from huobi.utils.print_mix_object import PrintBasic
        PrintBasic.print_basic(self.ch, format_data + "Channel")
        PrintBasic.print_basic(self.id, format_data + "ID")
        PrintBasic.print_basic(self.ts, format_data + "Unix Time")
        if len(self.data):
            for trade_detail in self.data:
                trade_detail.print_object()
                print()


class TradeDetailReq:
    """
    The trade received by subscription of trade.

    :member
        rep: The Channel you subscribed.
        trade_list: The trade list. The content is Trade class.
    """

    def __init__(self):
        self.rep = ""
        self.data = list()

    def print_object(self, format_data=""):
        from huobi.utils.print_mix_object import PrintBasic
        PrintBasic.print_basic(self.rep, format_data + "Channel")
        print()
        if len(self.data):
            for trade_detail in self.data:
                trade_detail.print_object()
                print()


class CandlestickEvent:
    """
    The candlestick/kline data received by subscription of candlestick/kline.

    :member
        ch: the topic you subscribed
        ts: the UNIX formatted timestamp generated by server in UTC.
        tick: the data of candlestick/kline.
    """

    def __init__(self):
        self.ch = ""
        self.ts = 0
        self.tick = Candlestick()

    def print_object(self, format_data=""):
        from huobi.utils.print_mix_object import PrintBasic
        PrintBasic.print_basic(self.ts, format_data + "Unix Time")
        PrintBasic.print_basic(self.ch, format_data + "Channel")
        if self.tick:
            self.tick.print_object()


class CandlestickReq:
    """
    The candlestick/kline data received by subscription of candlestick/kline.

    :member
        rep: the Channel or topic you subscribed.
        id: the UNIX formatted timestamp generated by server in UTC.
        data: the data of candlestick/kline.

    """

    def __init__(self):
        self.rep = ""
        self.id = 0
        self.data = list()

    def print_object(self, format_data=""):
        from huobi.utils.print_mix_object import PrintBasic
        PrintBasic.print_basic(self.rep, format_data + "Channel")
        PrintBasic.print_basic(self.id, format_data + "Unix Time")
        print()
        if len(self.data):
            for row in self.data:
                row.print_object()
                print()


class Candlestick:
    """
    The candlestick/kline data.

    :member
        id : keep the original timestamp
        timestamp: The UNIX formatted timestamp in UTC.
        high: The high price.
        low: The low price.
        open: The opening price.
        close: The closing price.
        amount: The aggregated trading volume in USDT.
        count: The number of completed trades. it returns 0 when get ETF candlestick
        vol: The trading volume in base currency.

    """

    def __init__(self):
        self.id = 0
        #self.timestamp = 0
        self.high = 0.0
        self.low = 0.0
        self.open = 0.0
        self.close = 0.0
        self.amount = 0.0
        self.count = 0
        self.vol = 0.0  # self.volume = 0.0

    def print_object(self, format_data=""):
        from huobi.utils.print_mix_object import PrintBasic
        PrintBasic.print_basic(self.id, format_data + "Id")
        #PrintBasic.print_basic(self.timestamp, format_data + "Unix Time")
        PrintBasic.print_basic(self.high, format_data + "High")
        PrintBasic.print_basic(self.low, format_data + "Low")
        PrintBasic.print_basic(self.open, format_data + "Open")
        PrintBasic.print_basic(self.close, format_data + "Close")
        PrintBasic.print_basic(self.count, format_data + "Count")
        PrintBasic.print_basic(self.amount, format_data + "Amount")
        PrintBasic.print_basic(self.vol, format_data + "Volume")


class DepthEntry:
    """
    An depth entry consisting of price and amount.

    :member
        price: The price of the depth.
        amount: The amount of the depth.
    """

    def __init__(self):
        self.price = 0.0
        self.amount = 0.0

    @staticmethod
    def json_parse(data_array):
        entry = DepthEntry()
        entry.price = data_array[0]
        entry.amount = data_array[1]
        return entry

    def print_object(self, format_data=""):
        from huobi.utils.print_mix_object import PrintBasic
        PrintBasic.print_basic(self.price, format_data + "Price")
        PrintBasic.print_basic(self.amount, format_data + "Amount")


class LastTradeAndBestQuote:
    """
    The last trade and best bid/ask.

    :member
        last_trade_price: The last trade price.
        last_trade_amount: The last trade amount.
        ask_price: The best ask price.
        ask_amount: The best ask amount.
        bid_price: The best bid price.
        bid_amount: The best bid amount.

    """

    def __init__(self):
        self.last_trade_price = 0.0
        self.last_trade_amount = 0.0
        self.ask_price = 0.0
        self.ask_amount = 0.0
        self.bid_price = 0.0
        self.bid_amount = 0.0


class MarketDetailEvent:
    """
     The 24H trade statistics received by subscription of trade statistics.

     :member
         Channel: topic you subscribe, include symbol.
         timestamp: The UNIX formatted timestamp generated by server in UTC.
         trade_statistics: The trade statistics.
     """

    def __init__(self):
        self.ch = ""
        self.ts = 0
        self.tick = MarketDetail()

    def print_object(self, format_data=""):
        from huobi.utils.print_mix_object import PrintBasic
        PrintBasic.print_basic(self.ch, format_data + "Channel")
        PrintBasic.print_basic(self.ts, format_data + "Timestamp")
        self.tick.print_object()


class MarketDetailMerged:
    """
    The best bid/ask consisting of price and amount.

    :member
        timestamp: The Unix formatted timestamp in UTC.
        bid_price: The best bid price.
        bid_amount: The best bid amount.
        ask_price: The best ask price.
        ask_amount: The best ask amount.

    """

    def __init__(self):
        self.amount = 0
        self.open = 0.0
        self.close = 0.0
        self.high = 0.0
        self.id = 0
        self.count = 0.0
        self.low = 0.0
        self.version = 0
        self.ask = []
        self.vol = 0.0
        self.bid = []

    def print_object(self, format_data=""):
        from huobi.utils.print_mix_object import PrintBasic

        # PrintBasic.print_basic(self.version, format_data + "Version")
        PrintBasic.print_basic(self.amount, format_data + "Amount")
        PrintBasic.print_basic(self.count, format_data + "Count")
        PrintBasic.print_basic(self.vol, format_data + "Volume")

        PrintBasic.print_basic(self.open, format_data + "Open")
        PrintBasic.print_basic(self.close, format_data + "Close")
        PrintBasic.print_basic(self.high, format_data + "High")
        PrintBasic.print_basic(self.low, format_data + "Low")

        print("Ask", self.ask)
        print("Bid", self.bid)


class MarketDetailReq:
    """
     The 24H trade statistics received by request of trade statistics only once.

     :member
         rep: The topic you subscribed.
         ts: The UNIX formatted timestamp generated by server in UTC.
         trade_statistics: The trade statistics.
     """

    def __init__(self):
        self.rep = 0
        self.ts = 0
        self.data = MarketDetail()

    def print_object(self, format_data=""):
        from huobi.utils.print_mix_object import PrintBasic
        PrintBasic.print_basic(self.ts, format_data + "Timestamp")
        PrintBasic.print_basic(self.rep, format_data + "Channel")
        self.data.print_object()


class MarketDetail:
    """
    The summary of trading in the market for the last 24 hours

    :member
        id: response ID
        open: The opening price of last 24 hours.
        close: The last price of last 24 hours.
        amount: The aggregated trading volume in USDT.
        high: The high price of last 24 hours.
        low: The low price of last 24 hours.
        count: The number of completed trades.
        volume: The trading volume in base currency of last 24 hours.
        version: inner data
    """

    def __init__(self):
        self.id = 0
        self.open = 0.0
        self.close = 0.0
        self.amount = 0.0
        self.high = 0.0
        self.low = 0.0
        self.count = 0
        self.vol = 0.0
        self.version = 0

    def print_object(self, format_data=""):
        from huobi.utils.print_mix_object import PrintBasic
        PrintBasic.print_basic(self.id, format_data + "ID")
        PrintBasic.print_basic(self.open, format_data + "Open")
        PrintBasic.print_basic(self.close, format_data + "Close")
        PrintBasic.print_basic(self.amount, format_data + "Amount")
        PrintBasic.print_basic(self.high, format_data + "High")
        PrintBasic.print_basic(self.low, format_data + "Low")
        PrintBasic.print_basic(self.count, format_data + "Count")
        PrintBasic.print_basic(self.vol, format_data + "Volume")
        # PrintBasic.print_basic(self.version, format_data + "Version")


class MarketTicker:
    """
    The ticker information.

    :member
        amount: The aggregated trading volume in last 24 hours (rotating 24h).
        count: The number of completed trades of last 24 hours (rotating 24h).
        open: The opening price of a nature day (Singapore time).
        close: The last price of a nature day (Singapore time).
        low: The low price of a nature day (Singapore time).
        high: The high price of a nature day (Singapore time).
        vol: The aggregated trading value in last 24 hours (rotating 24h).
        symbol: The trading symbol of this object, e.g. btcusdt, bccbtc.
        bid: Best bid price.
        bidSize: Best bid size.
        ask: Best ask price.
        askSize: Best ask size.
    """

    def __init__(self):
        self.amount = 0.0
        self.count = 0
        self.open = 0.0
        self.close = 0.0
        self.low = 0.0
        self.high = 0.0
        self.vol = 0.0
        self.symbol = ""
        self.bid = 0.0
        self.bidSize = 0.0
        self.ask = 0.0
        self.askSize = 0.0

    def print_object(self, format_data=""):
        from huobi.utils.print_mix_object import PrintBasic
        PrintBasic.print_basic(self.amount, format_data + "Amount")
        PrintBasic.print_basic(self.count, format_data + "Count")
        PrintBasic.print_basic(self.open, format_data + "Opening Price")
        PrintBasic.print_basic(self.close, format_data + "Last Price")
        PrintBasic.print_basic(self.low, format_data + "Low Price")
        PrintBasic.print_basic(self.high, format_data + "High Price")
        PrintBasic.print_basic(self.vol, format_data + "Vol")
        PrintBasic.print_basic(self.symbol, format_data + "Trading Symbol")
        PrintBasic.print_basic(self.bid, format_data + "Best Bid Price")
        PrintBasic.print_basic(self.bidSize, format_data + "Best Bid Size")
        PrintBasic.print_basic(self.ask, format_data + "Best Ask Price")
        PrintBasic.print_basic(self.askSize, format_data + "Best Ask Size")


class MbpFullEvent:
    """
    full price depth.

    :member
        ch: Topic of subscribed.
        timestamp: The UNIX formatted timestamp generated by server in UTC.
        data: The price depth.

    """

    def __init__(self):
        self.ch = ""
        self.ts = 0
        self.data = Mbp()

    @staticmethod
    def json_parse(json_data):
        mbp_event = MbpFullEvent()
        mbp_event.ts = json_data.get("ts")
        mbp_event.ch = json_data.get("ch")
        mbp = Mbp.json_parse(json_data.get("tick", {}))
        mbp_event.data = mbp
        return mbp_event

    def print_object(self, format_data=""):
        from huobi.utils.print_mix_object import PrintBasic
        PrintBasic.print_basic(self.ch, format_data + "Topic")
        PrintBasic.print_basic(self.ts, format_data + "Timestamp")
        self.data.print_object(format_data + "\t")


class MbpIncreaseEvent:
    """
    increasement of price depth.

    :member
        ch: Topic of subscribed.
        timestamp: The UNIX formatted timestamp generated by server in UTC.
        data: The price depth.

    """

    def __init__(self):
        self.ch = ""
        self.ts = 0
        self.data = Mbp()

    @staticmethod
    def json_parse(json_data):
        mbp_event = MbpIncreaseEvent()
        mbp_event.ts = json_data.get("ts")
        mbp_event.ch = json_data.get("ch")
        mbp = Mbp.json_parse(json_data.get("tick", {}))
        mbp_event.data = mbp
        return mbp_event

    def print_object(self, format_data=""):
        from huobi.utils.print_mix_object import PrintBasic
        PrintBasic.print_basic(self.ch, format_data + "Topic")
        PrintBasic.print_basic(self.ts, format_data + "Timestamp")

        self.data.print_object(format_data + "\t")


class MbpReq:
    """
    The market price depth.

    :member
        rep: request Topic
        id: The UNIX formatted timestamp generated by server in UTC.
        data: The price depth.
    """

    def __init__(self):
        self.rep = ""
        self.id = ""

        self.data = Mbp()

    @staticmethod
    def json_parse(data_json):
        mbp_event = MbpReq()
        mbp_event.id = data_json.get("id")
        mbp_event.rep = data_json.get("rep")
        data = data_json.get("data", {})
        mbp = Mbp.json_parse(data)
        mbp_event.data = mbp
        return mbp_event

    def print_object(self, format_data=""):
        from huobi.utils.print_mix_object import PrintBasic
        PrintBasic.print_basic(self.rep, format_data + "Topic")
        PrintBasic.print_basic(self.id, format_data + "Timestamp")

        self.data.print_object(format_data + "\t")


class Mbp:
    """
    Increasement of price depth information.

    :member
        seqNum: current seqNum.
        prevSeqNum: previous seqNum.
        bids: The list of the bid depth. The content is DepthEntry class.
        asks: The list of the ask depth. The content is DepthEntry class.

    """

    def __init__(self):
        self.seqNum = 0
        self.prevSeqNum = 0
        self.bids = list()
        self.asks = list()

    @staticmethod
    def json_parse(json_data):
        mbp = Mbp()
        bid_list = list()
        mbp.seqNum = json_data.get("seqNum", 0)
        # prevSeqNum only for increased subscribe, request doesn't have this value
        mbp.prevSeqNum = json_data.get("prevSeqNum", 0)
        for item in json_data.get("bids", []):
            depth_entry = DepthEntry()
            depth_entry.price = item[0]
            depth_entry.amount = item[1]
            bid_list.append(depth_entry)
        ask_list = list()
        for item in json_data.get("asks", []):
            depth_entry = DepthEntry()
            depth_entry.price = item[0]
            depth_entry.amount = item[1]
            ask_list.append(depth_entry)
        mbp.bids = bid_list
        mbp.asks = ask_list

        return mbp

    def print_object(self, format_data=""):
        from huobi.utils.print_mix_object import PrintBasic
        PrintBasic.print_basic(self.seqNum, format_data + "seqNum")
        PrintBasic.print_basic(self.prevSeqNum, format_data + "prevSeqNum")
        for entry in self.bids:
            PrintBasic.print_basic(str(entry.price) + ", amount: " + str(entry.amount), format_data + "Bids price: ")

        for entry in self.asks:
            PrintBasic.print_basic(str(entry.price) + ", amount: " + str(entry.amount), format_data + "Asks price: ")


class PriceDepthBboEvent:
    """
    The price depth received by subscription of price depth.

    :member
        symbol: The symbol you subscribed.
        timestamp: The UNIX formatted timestamp generated by server in UTC.
        data: The price depth.

    """

    def __init__(self):
        self.ts = 0
        self.ch = ""
        self.tick = PriceDepthBbo()

    def print_object(self, format_data=""):
        from huobi.utils.print_mix_object import PrintBasic
        PrintBasic.print_basic(self.ts, format_data + "Time")
        PrintBasic.print_basic(self.ch, format_data + "Channel")
        self.tick.print_object(format_data)


class PriceDepthBbo:
    """
    The price depth information.

    :member
        timestamp: The UNIX formatted timestamp in UTC.
        bid: the first bid near trade value.
        bidSize: the bid size.
        ask: The first ask near trade value.
        askSize: the ask size.
        quoteTime : quote time
        symbol : trade symbol


    """

    def __init__(self):
        self.seqId = 0
        self.ask = 0.0
        self.askSize = 0.0
        self.bid = 0.0
        self.bidSize = 0.0
        self.quoteTime = 0
        self.symbol = ""

    def print_object(self, format_data=""):
        from huobi.utils.print_mix_object import PrintBasic
        PrintBasic.print_basic(self.seqId, format_data + "Seq ID")
        PrintBasic.print_basic(self.ask, format_data + "Ask")
        PrintBasic.print_basic(self.askSize, format_data + "Ask Size")
        PrintBasic.print_basic(self.bid, format_data + "Bid")
        PrintBasic.print_basic(self.bidSize, format_data + "Bid Size")
        PrintBasic.print_basic(self.quoteTime, format_data + "Quote Time")
        PrintBasic.print_basic(self.symbol, format_data + "Symbol")


class PriceDepthEvent:
    """
    The price depth information.

    :member
        ts: The UNIX formatted timestamp in UTC.
        version:
        bids: The list of the bid depth. The content is DepthEntry class.
        asks: The list of the ask depth. The content is DepthEntry class.

    """

    def __init__(self):
        self.ch = ""
        self.tick = PriceDepth()

    def print_object(self, format_data=""):
        from huobi.utils.print_mix_object import PrintBasic
        PrintBasic.print_basic(self.ch, format_data + "Channel")
        self.tick.print_object("\t")


class PriceDepthReq:
    """
    The price depth information.

    :member
        ts: The UNIX formatted timestamp in UTC.
        version:
        bids: The list of the bid depth. The content is DepthEntry class.
        asks: The list of the ask depth. The content is DepthEntry class.

    """

    def __init__(self):
        self.rep = ""
        self.data = PriceDepth()

    def print_object(self, format_data=""):
        from huobi.utils.print_mix_object import PrintBasic
        PrintBasic.print_basic(self.rep, format_data + "Channel")
        self.data.print_object("\t")


class PriceDepth:
    """
    The price depth information.

    :member
        ts: The UNIX formatted timestamp in UTC.
        version:
        bids: The list of the bid depth. The content is DepthEntry class.
        asks: The list of the ask depth. The content is DepthEntry class.

    """

    def __init__(self):
        self.ts = 0
        self.version = 0
        self.bids = list()
        self.asks = list()

    @staticmethod
    def json_parse(dict_data):
        price_depth_obj = PriceDepth()
        price_depth_obj.ts = dict_data.get("ts")
        price_depth_obj.version = dict_data.get("version")
        bid_list = list()
        bids_array = dict_data.get("bids", [])
        for item in bids_array:
            depth_entry = DepthEntry.json_parse(item)
            bid_list.append(depth_entry)
        ask_list = list()
        asks_array = dict_data.get("asks", [])
        for item in asks_array:
            depth_entry = DepthEntry.json_parse(item)
            ask_list.append(depth_entry)
        price_depth_obj.bids = bid_list
        price_depth_obj.asks = ask_list

        return price_depth_obj

    def print_object(self, format_data=""):
        from huobi.utils.print_mix_object import PrintBasic
        PrintBasic.print_basic(self.ts, format_data + "UTC Time")
        PrintBasic.print_basic(self.version, format_data + "Version")
        if len(self.bids):
            i = 0
            print(format_data, "---- Top " + str(len(self.bids)) + " bids ----")
            for entry in self.bids:
                i = i + 1
                print(format_data, str(i) + ": price: " + str(entry.price) + ", amount: " + str(entry.amount))
                # print()

        if len(self.asks):
            i = 0
            print(format_data, "---- Top " + str(len(self.asks)) + " asks ----")
            for entry in self.asks:
                i = i + 1
                print(format_data, str(i) + ": price: " + str(entry.price) + ", amount: " + str(entry.amount))
                # print()
                #
