import * as ec2 from '@aws-cdk/aws-ec2';
import { Duration, Resource } from '@aws-cdk/core';
import { Construct } from 'constructs';
import { HostedZoneProviderProps } from './hosted-zone-provider';
import { HostedZoneAttributes, IHostedZone } from './hosted-zone-ref';
import { CfnHostedZone } from './route53.generated';
/**
 * Common properties to create a Route 53 hosted zone.
 *
 * @stability stable
 */
export interface CommonHostedZoneProps {
    /**
     * The name of the domain.
     *
     * For resource record types that include a domain
     * name, specify a fully qualified domain name.
     *
     * @stability stable
     */
    readonly zoneName: string;
    /**
     * Any comments that you want to include about the hosted zone.
     *
     * @default none
     * @stability stable
     */
    readonly comment?: string;
    /**
     * The Amazon Resource Name (ARN) for the log group that you want Amazon Route 53 to send query logs to.
     *
     * @default disabled
     * @stability stable
     */
    readonly queryLogsLogGroupArn?: string;
}
/**
 * Properties of a new hosted zone.
 *
 * @stability stable
 */
export interface HostedZoneProps extends CommonHostedZoneProps {
    /**
     * A VPC that you want to associate with this hosted zone.
     *
     * When you specify
     * this property, a private hosted zone will be created.
     *
     * You can associate additional VPCs to this private zone using `addVpc(vpc)`.
     *
     * @default public (no VPCs associated)
     * @stability stable
     */
    readonly vpcs?: ec2.IVpc[];
}
/**
 * Container for records, and records contain information about how to route traffic for a specific domain, such as example.com and its subdomains (acme.example.com, zenith.example.com).
 *
 * @stability stable
 */
export declare class HostedZone extends Resource implements IHostedZone {
    /**
     * ARN of this hosted zone, such as arn:${Partition}:route53:::hostedzone/${Id}.
     *
     * @stability stable
     */
    get hostedZoneArn(): string;
    /**
     * Import a Route 53 hosted zone defined either outside the CDK, or in a different CDK stack.
     *
     * Use when hosted zone ID is known. Hosted zone name becomes unavailable through this query.
     *
     * @param scope the parent Construct for this Construct.
     * @param id the logical name of this Construct.
     * @param hostedZoneId the ID of the hosted zone to import.
     * @stability stable
     */
    static fromHostedZoneId(scope: Construct, id: string, hostedZoneId: string): IHostedZone;
    /**
     * Imports a hosted zone from another stack.
     *
     * Use when both hosted zone ID and hosted zone name are known.
     *
     * @param scope the parent Construct for this Construct.
     * @param id the logical name of this Construct.
     * @param attrs the HostedZoneAttributes (hosted zone ID and hosted zone name).
     * @stability stable
     */
    static fromHostedZoneAttributes(scope: Construct, id: string, attrs: HostedZoneAttributes): IHostedZone;
    /**
     * Lookup a hosted zone in the current account/region based on query parameters.
     *
     * Requires environment, you must specify env for the stack.
     *
     * Use to easily query hosted zones.
     *
     * @see https://docs.aws.amazon.com/cdk/latest/guide/environments.html
     * @stability stable
     */
    static fromLookup(scope: Construct, id: string, query: HostedZoneProviderProps): IHostedZone;
    /**
     * ID of this hosted zone, such as "Z23ABC4XYZL05B".
     *
     * @stability stable
     */
    readonly hostedZoneId: string;
    /**
     * FQDN of this hosted zone.
     *
     * @stability stable
     */
    readonly zoneName: string;
    /**
     * Returns the set of name servers for the specific hosted zone. For example: ns1.example.com.
     *
     * This attribute will be undefined for private hosted zones or hosted zones imported from another stack.
     *
     * @stability stable
     */
    readonly hostedZoneNameServers?: string[];
    /**
     * VPCs to which this hosted zone will be added.
     *
     * @stability stable
     */
    protected readonly vpcs: CfnHostedZone.VPCProperty[];
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: HostedZoneProps);
    /**
     * Add another VPC to this private hosted zone.
     *
     * @param vpc the other VPC to add.
     * @stability stable
     */
    addVpc(vpc: ec2.IVpc): void;
}
/**
 * Construction properties for a PublicHostedZone.
 *
 * @stability stable
 */
export interface PublicHostedZoneProps extends CommonHostedZoneProps {
    /**
     * Whether to create a CAA record to restrict certificate authorities allowed to issue certificates for this domain to Amazon only.
     *
     * @default false
     * @stability stable
     */
    readonly caaAmazon?: boolean;
}
/**
 * Represents a Route 53 public hosted zone.
 *
 * @stability stable
 */
export interface IPublicHostedZone extends IHostedZone {
}
/**
 * Create a Route53 public hosted zone.
 *
 * @stability stable
 * @resource AWS::Route53::HostedZone
 */
export declare class PublicHostedZone extends HostedZone implements IPublicHostedZone {
    /**
     * Import a Route 53 public hosted zone defined either outside the CDK, or in a different CDK stack.
     *
     * @param scope the parent Construct for this Construct.
     * @param id the logical name of this Construct.
     * @param publicHostedZoneId the ID of the public hosted zone to import.
     * @stability stable
     */
    static fromPublicHostedZoneId(scope: Construct, id: string, publicHostedZoneId: string): IPublicHostedZone;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: PublicHostedZoneProps);
    /**
     * Add another VPC to this private hosted zone.
     *
     * @stability stable
     */
    addVpc(_vpc: ec2.IVpc): void;
    /**
     * Adds a delegation from this zone to a designated zone.
     *
     * @param delegate the zone being delegated to.
     * @param opts options for creating the DNS record, if any.
     * @stability stable
     */
    addDelegation(delegate: IPublicHostedZone, opts?: ZoneDelegationOptions): void;
}
/**
 * Options available when creating a delegation relationship from one PublicHostedZone to another.
 *
 * @stability stable
 */
export interface ZoneDelegationOptions {
    /**
     * A comment to add on the DNS record created to incorporate the delegation.
     *
     * @default none
     * @stability stable
     */
    readonly comment?: string;
    /**
     * The TTL (Time To Live) of the DNS delegation record in DNS caches.
     *
     * @default 172800
     * @stability stable
     */
    readonly ttl?: Duration;
}
/**
 * Properties to create a Route 53 private hosted zone.
 *
 * @stability stable
 */
export interface PrivateHostedZoneProps extends CommonHostedZoneProps {
    /**
     * A VPC that you want to associate with this hosted zone.
     *
     * Private hosted zones must be associated with at least one VPC. You can
     * associated additional VPCs using `addVpc(vpc)`.
     *
     * @stability stable
     */
    readonly vpc: ec2.IVpc;
}
/**
 * Represents a Route 53 private hosted zone.
 *
 * @stability stable
 */
export interface IPrivateHostedZone extends IHostedZone {
}
/**
 * Create a Route53 private hosted zone for use in one or more VPCs.
 *
 * Note that `enableDnsHostnames` and `enableDnsSupport` must have been enabled
 * for the VPC you're configuring for private hosted zones.
 *
 * @stability stable
 * @resource AWS::Route53::HostedZone
 */
export declare class PrivateHostedZone extends HostedZone implements IPrivateHostedZone {
    /**
     * Import a Route 53 private hosted zone defined either outside the CDK, or in a different CDK stack.
     *
     * @param scope the parent Construct for this Construct.
     * @param id the logical name of this Construct.
     * @param privateHostedZoneId the ID of the private hosted zone to import.
     * @stability stable
     */
    static fromPrivateHostedZoneId(scope: Construct, id: string, privateHostedZoneId: string): IPrivateHostedZone;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: PrivateHostedZoneProps);
}
