"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Logs = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [logs](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncloudwatchlogs.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Logs extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [logs](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncloudwatchlogs.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'logs';
        this.actionList = {
            "AssociateKmsKey": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_AssociateKmsKey.html",
                "description": "Associates the specified AWS Key Management Service (AWS KMS) customer master key (CMK) with the specified log group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "log-group": {
                        "required": true
                    }
                }
            },
            "CancelExportTask": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_CancelExportTask.html",
                "description": "Cancels an export task if it is in PENDING or RUNNING state",
                "accessLevel": "Write"
            },
            "CreateExportTask": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_CreateExportTask.html",
                "description": "Creates an ExportTask which allows you to efficiently export data from a Log Group to your Amazon S3 bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "log-group": {
                        "required": true
                    }
                }
            },
            "CreateLogDelivery": {
                "url": "",
                "description": "Creates the log delivery",
                "accessLevel": "Write"
            },
            "CreateLogGroup": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_CreateLogGroup.html",
                "description": "Creates a new log group with the specified name",
                "accessLevel": "Write"
            },
            "CreateLogStream": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_CreateLogStream.html",
                "description": "Creates a new log stream with the specified name",
                "accessLevel": "Write",
                "resourceTypes": {
                    "log-group": {
                        "required": true
                    }
                }
            },
            "DeleteDestination": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteDestination.html",
                "description": "Deletes the destination with the specified name and eventually disables all the subscription filters that publish to it",
                "accessLevel": "Write"
            },
            "DeleteLogDelivery": {
                "url": "",
                "description": "Deletes the log delivery information for specified log delivery",
                "accessLevel": "Write"
            },
            "DeleteLogGroup": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteLogGroup.html",
                "description": "Deletes the log group with the specified name and permanently deletes all the archived log events associated with it",
                "accessLevel": "Write",
                "resourceTypes": {
                    "log-group": {
                        "required": true
                    }
                }
            },
            "DeleteLogStream": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteLogStream.html",
                "description": "Deletes a log stream and permanently deletes all the archived log events associated with it",
                "accessLevel": "Write",
                "resourceTypes": {
                    "log-stream": {
                        "required": true
                    }
                }
            },
            "DeleteMetricFilter": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteMetricFilter.html",
                "description": "Deletes a metric filter associated with the specified log group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "log-group": {
                        "required": true
                    }
                }
            },
            "DeleteResourcePolicy": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteResourcePolicy.html",
                "description": "Deletes a resource policy from this account",
                "accessLevel": "Write"
            },
            "DeleteRetentionPolicy": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteRetentionPolicy.html",
                "description": "Deletes the retention policy of the specified log group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "log-group": {
                        "required": true
                    }
                }
            },
            "DeleteSubscriptionFilter": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteSubscriptionFilter.html",
                "description": "Deletes a subscription filter associated with the specified log group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "log-group": {
                        "required": true
                    }
                }
            },
            "DescribeDestinations": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeDestinations.html",
                "description": "Returns all the destinations that are associated with the AWS account making the request",
                "accessLevel": "List"
            },
            "DescribeExportTasks": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeExportTasks.html",
                "description": "Returns all the export tasks that are associated with the AWS account making the request",
                "accessLevel": "List"
            },
            "DescribeLogGroups": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeLogGroups.html",
                "description": "Returns all the log groups that are associated with the AWS account making the request",
                "accessLevel": "List",
                "resourceTypes": {
                    "log-group": {
                        "required": true
                    }
                }
            },
            "DescribeLogStreams": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeLogStreams.html",
                "description": "Returns all the log streams that are associated with the specified log group",
                "accessLevel": "List",
                "resourceTypes": {
                    "log-group": {
                        "required": true
                    }
                }
            },
            "DescribeMetricFilters": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeMetricFilters.html",
                "description": "Returns all the metrics filters associated with the specified log group",
                "accessLevel": "List",
                "resourceTypes": {
                    "log-group": {
                        "required": true
                    }
                }
            },
            "DescribeQueries": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeQueries.html",
                "description": "Returns a list of CloudWatch Logs Insights queries that are scheduled, executing, or have been executed recently in this account. You can request all queries, or limit it to queries of a specific log group or queries with a certain status.",
                "accessLevel": "List"
            },
            "DescribeResourcePolicies": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeResourcePolicies.html",
                "description": "Return all the resource policies in this account.",
                "accessLevel": "List"
            },
            "DescribeSubscriptionFilters": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeSubscriptionFilters.html",
                "description": "Returns all the subscription filters associated with the specified log group",
                "accessLevel": "List",
                "resourceTypes": {
                    "log-group": {
                        "required": true
                    }
                }
            },
            "DisassociateKmsKey": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DisassociateKmsKey.html",
                "description": "Disassociates the associated AWS Key Management Service (AWS KMS) customer master key (CMK) from the specified log group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "log-group": {
                        "required": true
                    }
                }
            },
            "FilterLogEvents": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_FilterLogEvents.html",
                "description": "Retrieves log events, optionally filtered by a filter pattern from the specified log group",
                "accessLevel": "Read",
                "resourceTypes": {
                    "log-group": {
                        "required": true
                    }
                }
            },
            "GetLogDelivery": {
                "url": "",
                "description": "Gets the log delivery information for specified log delivery",
                "accessLevel": "Read"
            },
            "GetLogEvents": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_GetLogEvents.html",
                "description": "Retrieves log events from the specified log stream",
                "accessLevel": "Read",
                "resourceTypes": {
                    "log-stream": {
                        "required": true
                    }
                }
            },
            "GetLogGroupFields": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_GetLogGroupFields.html",
                "description": "Returns a list of the fields that are included in log events in the specified log group, along with the percentage of log events that contain each field. The search is limited to a time period that you specify.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "log-group": {
                        "required": true
                    }
                }
            },
            "GetLogRecord": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_GetLogRecord.html",
                "description": "Retrieves all the fields and values of a single log event. All fields are retrieved, even if the original query that produced the logRecordPointer retrieved only a subset of fields. Fields are returned as field name/field value pairs.",
                "accessLevel": "Read"
            },
            "GetQueryResults": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_GetQueryResults.html",
                "description": "Returns the results from the specified query. If the query is in progress, partial results of that current execution are returned. Only the fields requested in the query are returned.",
                "accessLevel": "Read"
            },
            "ListLogDeliveries": {
                "url": "",
                "description": "Lists all the log deliveries for specified account and/or log source",
                "accessLevel": "List"
            },
            "ListTagsLogGroup": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_ListTagsLogGroup.html",
                "description": "Lists the tags for the specified log group",
                "accessLevel": "List",
                "resourceTypes": {
                    "log-group": {
                        "required": true
                    }
                }
            },
            "PutDestination": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutDestination.html",
                "description": "Creates or updates a Destination",
                "accessLevel": "Write"
            },
            "PutDestinationPolicy": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutDestinationPolicy.html",
                "description": "Creates or updates an access policy associated with an existing Destination",
                "accessLevel": "Write"
            },
            "PutLogEvents": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutLogEvents.html",
                "description": "Uploads a batch of log events to the specified log stream",
                "accessLevel": "Write",
                "resourceTypes": {
                    "log-stream": {
                        "required": true
                    }
                }
            },
            "PutMetricFilter": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutMetricFilter.html",
                "description": "Creates or updates a metric filter and associates it with the specified log group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "log-group": {
                        "required": true
                    }
                }
            },
            "PutResourcePolicy": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutResourcePolicy.html",
                "description": "Creates or updates a resource policy allowing other AWS services to put log events to this account",
                "accessLevel": "Write"
            },
            "PutRetentionPolicy": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutRetentionPolicy.html",
                "description": "Sets the retention of the specified log group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "log-group": {
                        "required": true
                    }
                }
            },
            "PutSubscriptionFilter": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutSubscriptionFilter.html",
                "description": "Creates or updates a subscription filter and associates it with the specified log group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "log-group": {
                        "required": true
                    }
                }
            },
            "StartQuery": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_StartQuery.html",
                "description": "Schedules a query of a log group using CloudWatch Logs Insights. You specify the log group and time range to query, and the query string to use.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "log-group": {
                        "required": true
                    }
                }
            },
            "StopQuery": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_StopQuery.html",
                "description": "Stops a CloudWatch Logs Insights query that is in progress. If the query has already ended, the operation returns an error indicating that the specified query is not running.",
                "accessLevel": "Read"
            },
            "TagLogGroup": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_TagLogGroup.html",
                "description": "Adds or updates the specified tags for the specified log group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "log-group": {
                        "required": true
                    }
                }
            },
            "TestMetricFilter": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_TestMetricFilter.html",
                "description": "Tests the filter pattern of a metric filter against a sample of log event messages",
                "accessLevel": "Read"
            },
            "UntagLogGroup": {
                "url": "https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_UntagLogGroup.html",
                "description": "Removes the specified tags from the specified log group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "log-group": {
                        "required": true
                    }
                }
            },
            "UpdateLogDelivery": {
                "url": "",
                "description": "Updates the log delivery information for specified log delivery",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "log-group": {
                "name": "log-group",
                "url": "",
                "arn": "arn:${Partition}:logs:${Region}:${Account}:log-group:${LogGroupName}",
                "conditionKeys": []
            },
            "log-stream": {
                "name": "log-stream",
                "url": "",
                "arn": "arn:${Partition}:logs:${Region}:${Account}:log-group:${LogGroupName}:log-stream:${LogStreamName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Associates the specified AWS Key Management Service (AWS KMS) customer master key (CMK) with the specified log group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_AssociateKmsKey.html
     */
    associateKmsKey() {
        this.add('logs:AssociateKmsKey');
        return this;
    }
    /**
     * Cancels an export task if it is in PENDING or RUNNING state
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_CancelExportTask.html
     */
    cancelExportTask() {
        this.add('logs:CancelExportTask');
        return this;
    }
    /**
     * Creates an ExportTask which allows you to efficiently export data from a Log Group to your Amazon S3 bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_CreateExportTask.html
     */
    createExportTask() {
        this.add('logs:CreateExportTask');
        return this;
    }
    /**
     * Creates the log delivery
     *
     * Access Level: Write
     */
    createLogDelivery() {
        this.add('logs:CreateLogDelivery');
        return this;
    }
    /**
     * Creates a new log group with the specified name
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_CreateLogGroup.html
     */
    createLogGroup() {
        this.add('logs:CreateLogGroup');
        return this;
    }
    /**
     * Creates a new log stream with the specified name
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_CreateLogStream.html
     */
    createLogStream() {
        this.add('logs:CreateLogStream');
        return this;
    }
    /**
     * Deletes the destination with the specified name and eventually disables all the subscription filters that publish to it
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteDestination.html
     */
    deleteDestination() {
        this.add('logs:DeleteDestination');
        return this;
    }
    /**
     * Deletes the log delivery information for specified log delivery
     *
     * Access Level: Write
     */
    deleteLogDelivery() {
        this.add('logs:DeleteLogDelivery');
        return this;
    }
    /**
     * Deletes the log group with the specified name and permanently deletes all the archived log events associated with it
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteLogGroup.html
     */
    deleteLogGroup() {
        this.add('logs:DeleteLogGroup');
        return this;
    }
    /**
     * Deletes a log stream and permanently deletes all the archived log events associated with it
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteLogStream.html
     */
    deleteLogStream() {
        this.add('logs:DeleteLogStream');
        return this;
    }
    /**
     * Deletes a metric filter associated with the specified log group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteMetricFilter.html
     */
    deleteMetricFilter() {
        this.add('logs:DeleteMetricFilter');
        return this;
    }
    /**
     * Deletes a resource policy from this account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteResourcePolicy.html
     */
    deleteResourcePolicy() {
        this.add('logs:DeleteResourcePolicy');
        return this;
    }
    /**
     * Deletes the retention policy of the specified log group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteRetentionPolicy.html
     */
    deleteRetentionPolicy() {
        this.add('logs:DeleteRetentionPolicy');
        return this;
    }
    /**
     * Deletes a subscription filter associated with the specified log group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteSubscriptionFilter.html
     */
    deleteSubscriptionFilter() {
        this.add('logs:DeleteSubscriptionFilter');
        return this;
    }
    /**
     * Returns all the destinations that are associated with the AWS account making the request
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeDestinations.html
     */
    describeDestinations() {
        this.add('logs:DescribeDestinations');
        return this;
    }
    /**
     * Returns all the export tasks that are associated with the AWS account making the request
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeExportTasks.html
     */
    describeExportTasks() {
        this.add('logs:DescribeExportTasks');
        return this;
    }
    /**
     * Returns all the log groups that are associated with the AWS account making the request
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeLogGroups.html
     */
    describeLogGroups() {
        this.add('logs:DescribeLogGroups');
        return this;
    }
    /**
     * Returns all the log streams that are associated with the specified log group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeLogStreams.html
     */
    describeLogStreams() {
        this.add('logs:DescribeLogStreams');
        return this;
    }
    /**
     * Returns all the metrics filters associated with the specified log group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeMetricFilters.html
     */
    describeMetricFilters() {
        this.add('logs:DescribeMetricFilters');
        return this;
    }
    /**
     * Returns a list of CloudWatch Logs Insights queries that are scheduled, executing, or have been executed recently in this account. You can request all queries, or limit it to queries of a specific log group or queries with a certain status.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeQueries.html
     */
    describeQueries() {
        this.add('logs:DescribeQueries');
        return this;
    }
    /**
     * Return all the resource policies in this account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeResourcePolicies.html
     */
    describeResourcePolicies() {
        this.add('logs:DescribeResourcePolicies');
        return this;
    }
    /**
     * Returns all the subscription filters associated with the specified log group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeSubscriptionFilters.html
     */
    describeSubscriptionFilters() {
        this.add('logs:DescribeSubscriptionFilters');
        return this;
    }
    /**
     * Disassociates the associated AWS Key Management Service (AWS KMS) customer master key (CMK) from the specified log group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DisassociateKmsKey.html
     */
    disassociateKmsKey() {
        this.add('logs:DisassociateKmsKey');
        return this;
    }
    /**
     * Retrieves log events, optionally filtered by a filter pattern from the specified log group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_FilterLogEvents.html
     */
    filterLogEvents() {
        this.add('logs:FilterLogEvents');
        return this;
    }
    /**
     * Gets the log delivery information for specified log delivery
     *
     * Access Level: Read
     */
    getLogDelivery() {
        this.add('logs:GetLogDelivery');
        return this;
    }
    /**
     * Retrieves log events from the specified log stream
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_GetLogEvents.html
     */
    getLogEvents() {
        this.add('logs:GetLogEvents');
        return this;
    }
    /**
     * Returns a list of the fields that are included in log events in the specified log group, along with the percentage of log events that contain each field. The search is limited to a time period that you specify.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_GetLogGroupFields.html
     */
    getLogGroupFields() {
        this.add('logs:GetLogGroupFields');
        return this;
    }
    /**
     * Retrieves all the fields and values of a single log event. All fields are retrieved, even if the original query that produced the logRecordPointer retrieved only a subset of fields. Fields are returned as field name/field value pairs.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_GetLogRecord.html
     */
    getLogRecord() {
        this.add('logs:GetLogRecord');
        return this;
    }
    /**
     * Returns the results from the specified query. If the query is in progress, partial results of that current execution are returned. Only the fields requested in the query are returned.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_GetQueryResults.html
     */
    getQueryResults() {
        this.add('logs:GetQueryResults');
        return this;
    }
    /**
     * Lists all the log deliveries for specified account and/or log source
     *
     * Access Level: List
     */
    listLogDeliveries() {
        this.add('logs:ListLogDeliveries');
        return this;
    }
    /**
     * Lists the tags for the specified log group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_ListTagsLogGroup.html
     */
    listTagsLogGroup() {
        this.add('logs:ListTagsLogGroup');
        return this;
    }
    /**
     * Creates or updates a Destination
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutDestination.html
     */
    putDestination() {
        this.add('logs:PutDestination');
        return this;
    }
    /**
     * Creates or updates an access policy associated with an existing Destination
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutDestinationPolicy.html
     */
    putDestinationPolicy() {
        this.add('logs:PutDestinationPolicy');
        return this;
    }
    /**
     * Uploads a batch of log events to the specified log stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutLogEvents.html
     */
    putLogEvents() {
        this.add('logs:PutLogEvents');
        return this;
    }
    /**
     * Creates or updates a metric filter and associates it with the specified log group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutMetricFilter.html
     */
    putMetricFilter() {
        this.add('logs:PutMetricFilter');
        return this;
    }
    /**
     * Creates or updates a resource policy allowing other AWS services to put log events to this account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutResourcePolicy.html
     */
    putResourcePolicy() {
        this.add('logs:PutResourcePolicy');
        return this;
    }
    /**
     * Sets the retention of the specified log group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutRetentionPolicy.html
     */
    putRetentionPolicy() {
        this.add('logs:PutRetentionPolicy');
        return this;
    }
    /**
     * Creates or updates a subscription filter and associates it with the specified log group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutSubscriptionFilter.html
     */
    putSubscriptionFilter() {
        this.add('logs:PutSubscriptionFilter');
        return this;
    }
    /**
     * Schedules a query of a log group using CloudWatch Logs Insights. You specify the log group and time range to query, and the query string to use.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_StartQuery.html
     */
    startQuery() {
        this.add('logs:StartQuery');
        return this;
    }
    /**
     * Stops a CloudWatch Logs Insights query that is in progress. If the query has already ended, the operation returns an error indicating that the specified query is not running.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_StopQuery.html
     */
    stopQuery() {
        this.add('logs:StopQuery');
        return this;
    }
    /**
     * Adds or updates the specified tags for the specified log group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_TagLogGroup.html
     */
    tagLogGroup() {
        this.add('logs:TagLogGroup');
        return this;
    }
    /**
     * Tests the filter pattern of a metric filter against a sample of log event messages
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_TestMetricFilter.html
     */
    testMetricFilter() {
        this.add('logs:TestMetricFilter');
        return this;
    }
    /**
     * Removes the specified tags from the specified log group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_UntagLogGroup.html
     */
    untagLogGroup() {
        this.add('logs:UntagLogGroup');
        return this;
    }
    /**
     * Updates the log delivery information for specified log delivery
     *
     * Access Level: Write
     */
    updateLogDelivery() {
        this.add('logs:UpdateLogDelivery');
        return this;
    }
    /**
     * Adds a resource of type log-group to the statement
     *
     * @param logGroupName - Identifier for the logGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onLogGroup(logGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:logs:${Region}:${Account}:log-group:${LogGroupName}';
        arn = arn.replace('${LogGroupName}', logGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type log-stream to the statement
     *
     * @param logGroupName - Identifier for the logGroupName.
     * @param logStreamName - Identifier for the logStreamName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onLogStream(logGroupName, logStreamName, account, region, partition) {
        var arn = 'arn:${Partition}:logs:${Region}:${Account}:log-group:${LogGroupName}:log-stream:${LogStreamName}';
        arn = arn.replace('${LogGroupName}', logGroupName);
        arn = arn.replace('${LogStreamName}', logStreamName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Logs = Logs;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYW1hem9uY2xvdWR3YXRjaGxvZ3MuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJhbWF6b25jbG91ZHdhdGNobG9ncy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSxzQ0FBb0U7QUFFcEU7Ozs7R0FJRztBQUNILE1BQWEsSUFBSyxTQUFRLHdCQUFlO0lBdVd2Qzs7OztPQUlHO0lBQ0gsWUFBYSxHQUFZO1FBQ3ZCLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQztRQTVXTixrQkFBYSxHQUFHLE1BQU0sQ0FBQztRQUNwQixlQUFVLEdBQVk7WUFDOUIsaUJBQWlCLEVBQUU7Z0JBQ2pCLEtBQUssRUFBRSwrRkFBK0Y7Z0JBQ3RHLGFBQWEsRUFBRSx1SEFBdUg7Z0JBQ3RJLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0Qsa0JBQWtCLEVBQUU7Z0JBQ2xCLEtBQUssRUFBRSxnR0FBZ0c7Z0JBQ3ZHLGFBQWEsRUFBRSw2REFBNkQ7Z0JBQzVFLGFBQWEsRUFBRSxPQUFPO2FBQ3ZCO1lBQ0Qsa0JBQWtCLEVBQUU7Z0JBQ2xCLEtBQUssRUFBRSxnR0FBZ0c7Z0JBQ3ZHLGFBQWEsRUFBRSw2R0FBNkc7Z0JBQzVILGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsbUJBQW1CLEVBQUU7Z0JBQ25CLEtBQUssRUFBRSxFQUFFO2dCQUNULGFBQWEsRUFBRSwwQkFBMEI7Z0JBQ3pDLGFBQWEsRUFBRSxPQUFPO2FBQ3ZCO1lBQ0QsZ0JBQWdCLEVBQUU7Z0JBQ2hCLEtBQUssRUFBRSw4RkFBOEY7Z0JBQ3JHLGFBQWEsRUFBRSxpREFBaUQ7Z0JBQ2hFLGFBQWEsRUFBRSxPQUFPO2FBQ3ZCO1lBQ0QsaUJBQWlCLEVBQUU7Z0JBQ2pCLEtBQUssRUFBRSwrRkFBK0Y7Z0JBQ3RHLGFBQWEsRUFBRSxrREFBa0Q7Z0JBQ2pFLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsbUJBQW1CLEVBQUU7Z0JBQ25CLEtBQUssRUFBRSxpR0FBaUc7Z0JBQ3hHLGFBQWEsRUFBRSx5SEFBeUg7Z0JBQ3hJLGFBQWEsRUFBRSxPQUFPO2FBQ3ZCO1lBQ0QsbUJBQW1CLEVBQUU7Z0JBQ25CLEtBQUssRUFBRSxFQUFFO2dCQUNULGFBQWEsRUFBRSxpRUFBaUU7Z0JBQ2hGLGFBQWEsRUFBRSxPQUFPO2FBQ3ZCO1lBQ0QsZ0JBQWdCLEVBQUU7Z0JBQ2hCLEtBQUssRUFBRSw4RkFBOEY7Z0JBQ3JHLGFBQWEsRUFBRSxzSEFBc0g7Z0JBQ3JJLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsaUJBQWlCLEVBQUU7Z0JBQ2pCLEtBQUssRUFBRSwrRkFBK0Y7Z0JBQ3RHLGFBQWEsRUFBRSw2RkFBNkY7Z0JBQzVHLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsWUFBWSxFQUFFO3dCQUNaLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0Qsb0JBQW9CLEVBQUU7Z0JBQ3BCLEtBQUssRUFBRSxrR0FBa0c7Z0JBQ3pHLGFBQWEsRUFBRSxpRUFBaUU7Z0JBQ2hGLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0Qsc0JBQXNCLEVBQUU7Z0JBQ3RCLEtBQUssRUFBRSxvR0FBb0c7Z0JBQzNHLGFBQWEsRUFBRSw2Q0FBNkM7Z0JBQzVELGFBQWEsRUFBRSxPQUFPO2FBQ3ZCO1lBQ0QsdUJBQXVCLEVBQUU7Z0JBQ3ZCLEtBQUssRUFBRSxxR0FBcUc7Z0JBQzVHLGFBQWEsRUFBRSx5REFBeUQ7Z0JBQ3hFLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsMEJBQTBCLEVBQUU7Z0JBQzFCLEtBQUssRUFBRSx3R0FBd0c7Z0JBQy9HLGFBQWEsRUFBRSx1RUFBdUU7Z0JBQ3RGLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0Qsc0JBQXNCLEVBQUU7Z0JBQ3RCLEtBQUssRUFBRSxvR0FBb0c7Z0JBQzNHLGFBQWEsRUFBRSwwRkFBMEY7Z0JBQ3pHLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0QscUJBQXFCLEVBQUU7Z0JBQ3JCLEtBQUssRUFBRSxtR0FBbUc7Z0JBQzFHLGFBQWEsRUFBRSwwRkFBMEY7Z0JBQ3pHLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0QsbUJBQW1CLEVBQUU7Z0JBQ25CLEtBQUssRUFBRSxpR0FBaUc7Z0JBQ3hHLGFBQWEsRUFBRSx3RkFBd0Y7Z0JBQ3ZHLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0Qsb0JBQW9CLEVBQUU7Z0JBQ3BCLEtBQUssRUFBRSxrR0FBa0c7Z0JBQ3pHLGFBQWEsRUFBRSw4RUFBOEU7Z0JBQzdGLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsdUJBQXVCLEVBQUU7Z0JBQ3ZCLEtBQUssRUFBRSxxR0FBcUc7Z0JBQzVHLGFBQWEsRUFBRSx5RUFBeUU7Z0JBQ3hGLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsaUJBQWlCLEVBQUU7Z0JBQ2pCLEtBQUssRUFBRSwrRkFBK0Y7Z0JBQ3RHLGFBQWEsRUFBRSxpUEFBaVA7Z0JBQ2hRLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0QsMEJBQTBCLEVBQUU7Z0JBQzFCLEtBQUssRUFBRSx3R0FBd0c7Z0JBQy9HLGFBQWEsRUFBRSxtREFBbUQ7Z0JBQ2xFLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0QsNkJBQTZCLEVBQUU7Z0JBQzdCLEtBQUssRUFBRSwyR0FBMkc7Z0JBQ2xILGFBQWEsRUFBRSw4RUFBOEU7Z0JBQzdGLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0Qsb0JBQW9CLEVBQUU7Z0JBQ3BCLEtBQUssRUFBRSxrR0FBa0c7Z0JBQ3pHLGFBQWEsRUFBRSwwSEFBMEg7Z0JBQ3pJLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsaUJBQWlCLEVBQUU7Z0JBQ2pCLEtBQUssRUFBRSwrRkFBK0Y7Z0JBQ3RHLGFBQWEsRUFBRSw0RkFBNEY7Z0JBQzNHLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsZ0JBQWdCLEVBQUU7Z0JBQ2hCLEtBQUssRUFBRSxFQUFFO2dCQUNULGFBQWEsRUFBRSw4REFBOEQ7Z0JBQzdFLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0QsY0FBYyxFQUFFO2dCQUNkLEtBQUssRUFBRSw0RkFBNEY7Z0JBQ25HLGFBQWEsRUFBRSxvREFBb0Q7Z0JBQ25FLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsWUFBWSxFQUFFO3dCQUNaLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsbUJBQW1CLEVBQUU7Z0JBQ25CLEtBQUssRUFBRSxpR0FBaUc7Z0JBQ3hHLGFBQWEsRUFBRSxvTkFBb047Z0JBQ25PLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsY0FBYyxFQUFFO2dCQUNkLEtBQUssRUFBRSw0RkFBNEY7Z0JBQ25HLGFBQWEsRUFBRSw0T0FBNE87Z0JBQzNQLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0QsaUJBQWlCLEVBQUU7Z0JBQ2pCLEtBQUssRUFBRSwrRkFBK0Y7Z0JBQ3RHLGFBQWEsRUFBRSx5TEFBeUw7Z0JBQ3hNLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0QsbUJBQW1CLEVBQUU7Z0JBQ25CLEtBQUssRUFBRSxFQUFFO2dCQUNULGFBQWEsRUFBRSxzRUFBc0U7Z0JBQ3JGLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0Qsa0JBQWtCLEVBQUU7Z0JBQ2xCLEtBQUssRUFBRSxnR0FBZ0c7Z0JBQ3ZHLGFBQWEsRUFBRSw0Q0FBNEM7Z0JBQzNELGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsZ0JBQWdCLEVBQUU7Z0JBQ2hCLEtBQUssRUFBRSw4RkFBOEY7Z0JBQ3JHLGFBQWEsRUFBRSxrQ0FBa0M7Z0JBQ2pELGFBQWEsRUFBRSxPQUFPO2FBQ3ZCO1lBQ0Qsc0JBQXNCLEVBQUU7Z0JBQ3RCLEtBQUssRUFBRSxvR0FBb0c7Z0JBQzNHLGFBQWEsRUFBRSw2RUFBNkU7Z0JBQzVGLGFBQWEsRUFBRSxPQUFPO2FBQ3ZCO1lBQ0QsY0FBYyxFQUFFO2dCQUNkLEtBQUssRUFBRSw0RkFBNEY7Z0JBQ25HLGFBQWEsRUFBRSwyREFBMkQ7Z0JBQzFFLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsWUFBWSxFQUFFO3dCQUNaLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsaUJBQWlCLEVBQUU7Z0JBQ2pCLEtBQUssRUFBRSwrRkFBK0Y7Z0JBQ3RHLGFBQWEsRUFBRSxtRkFBbUY7Z0JBQ2xHLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsbUJBQW1CLEVBQUU7Z0JBQ25CLEtBQUssRUFBRSxpR0FBaUc7Z0JBQ3hHLGFBQWEsRUFBRSxvR0FBb0c7Z0JBQ25ILGFBQWEsRUFBRSxPQUFPO2FBQ3ZCO1lBQ0Qsb0JBQW9CLEVBQUU7Z0JBQ3BCLEtBQUssRUFBRSxrR0FBa0c7Z0JBQ3pHLGFBQWEsRUFBRSwrQ0FBK0M7Z0JBQzlELGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsdUJBQXVCLEVBQUU7Z0JBQ3ZCLEtBQUssRUFBRSxxR0FBcUc7Z0JBQzVHLGFBQWEsRUFBRSx5RkFBeUY7Z0JBQ3hHLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsWUFBWSxFQUFFO2dCQUNaLEtBQUssRUFBRSwwRkFBMEY7Z0JBQ2pHLGFBQWEsRUFBRSxrSkFBa0o7Z0JBQ2pLLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsV0FBVyxFQUFFO2dCQUNYLEtBQUssRUFBRSx5RkFBeUY7Z0JBQ2hHLGFBQWEsRUFBRSxnTEFBZ0w7Z0JBQy9MLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0QsYUFBYSxFQUFFO2dCQUNiLEtBQUssRUFBRSwyRkFBMkY7Z0JBQ2xHLGFBQWEsRUFBRSxnRUFBZ0U7Z0JBQy9FLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0Qsa0JBQWtCLEVBQUU7Z0JBQ2xCLEtBQUssRUFBRSxnR0FBZ0c7Z0JBQ3ZHLGFBQWEsRUFBRSxvRkFBb0Y7Z0JBQ25HLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0QsZUFBZSxFQUFFO2dCQUNmLEtBQUssRUFBRSw2RkFBNkY7Z0JBQ3BHLGFBQWEsRUFBRSx5REFBeUQ7Z0JBQ3hFLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsbUJBQW1CLEVBQUU7Z0JBQ25CLEtBQUssRUFBRSxFQUFFO2dCQUNULGFBQWEsRUFBRSxpRUFBaUU7Z0JBQ2hGLGFBQWEsRUFBRSxPQUFPO2FBQ3ZCO1NBQ0YsQ0FBQztRQUNRLGtCQUFhLEdBQWtCO1lBQ3ZDLFdBQVcsRUFBRTtnQkFDWCxNQUFNLEVBQUUsV0FBVztnQkFDbkIsS0FBSyxFQUFFLEVBQUU7Z0JBQ1QsS0FBSyxFQUFFLHNFQUFzRTtnQkFDN0UsZUFBZSxFQUFFLEVBQUU7YUFDcEI7WUFDRCxZQUFZLEVBQUU7Z0JBQ1osTUFBTSxFQUFFLFlBQVk7Z0JBQ3BCLEtBQUssRUFBRSxFQUFFO2dCQUNULEtBQUssRUFBRSxrR0FBa0c7Z0JBQ3pHLGVBQWUsRUFBRSxFQUFFO2FBQ3BCO1NBQ0YsQ0FBQztJQVNGLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxlQUFlO1FBQ3BCLElBQUksQ0FBQyxHQUFHLENBQUMsc0JBQXNCLENBQUMsQ0FBQztRQUNqQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO1FBQ2xDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixJQUFJLENBQUMsR0FBRyxDQUFDLHVCQUF1QixDQUFDLENBQUM7UUFDbEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLGlCQUFpQjtRQUN0QixJQUFJLENBQUMsR0FBRyxDQUFDLHdCQUF3QixDQUFDLENBQUM7UUFDbkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksY0FBYztRQUNuQixJQUFJLENBQUMsR0FBRyxDQUFDLHFCQUFxQixDQUFDLENBQUM7UUFDaEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZUFBZTtRQUNwQixJQUFJLENBQUMsR0FBRyxDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFDakMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksaUJBQWlCO1FBQ3RCLElBQUksQ0FBQyxHQUFHLENBQUMsd0JBQXdCLENBQUMsQ0FBQztRQUNuQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksaUJBQWlCO1FBQ3RCLElBQUksQ0FBQyxHQUFHLENBQUMsd0JBQXdCLENBQUMsQ0FBQztRQUNuQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxjQUFjO1FBQ25CLElBQUksQ0FBQyxHQUFHLENBQUMscUJBQXFCLENBQUMsQ0FBQztRQUNoQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxlQUFlO1FBQ3BCLElBQUksQ0FBQyxHQUFHLENBQUMsc0JBQXNCLENBQUMsQ0FBQztRQUNqQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO1FBQ3BDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG9CQUFvQjtRQUN6QixJQUFJLENBQUMsR0FBRyxDQUFDLDJCQUEyQixDQUFDLENBQUM7UUFDdEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLElBQUksQ0FBQyxHQUFHLENBQUMsNEJBQTRCLENBQUMsQ0FBQztRQUN2QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx3QkFBd0I7UUFDN0IsSUFBSSxDQUFDLEdBQUcsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO1FBQzFDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG9CQUFvQjtRQUN6QixJQUFJLENBQUMsR0FBRyxDQUFDLDJCQUEyQixDQUFDLENBQUM7UUFDdEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLElBQUksQ0FBQyxHQUFHLENBQUMsMEJBQTBCLENBQUMsQ0FBQztRQUNyQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQkFBaUI7UUFDdEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO1FBQ25DLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixJQUFJLENBQUMsR0FBRyxDQUFDLHlCQUF5QixDQUFDLENBQUM7UUFDcEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLElBQUksQ0FBQyxHQUFHLENBQUMsNEJBQTRCLENBQUMsQ0FBQztRQUN2QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxlQUFlO1FBQ3BCLElBQUksQ0FBQyxHQUFHLENBQUMsc0JBQXNCLENBQUMsQ0FBQztRQUNqQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx3QkFBd0I7UUFDN0IsSUFBSSxDQUFDLEdBQUcsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO1FBQzFDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDJCQUEyQjtRQUNoQyxJQUFJLENBQUMsR0FBRyxDQUFDLGtDQUFrQyxDQUFDLENBQUM7UUFDN0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLElBQUksQ0FBQyxHQUFHLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxlQUFlO1FBQ3BCLElBQUksQ0FBQyxHQUFHLENBQUMsc0JBQXNCLENBQUMsQ0FBQztRQUNqQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksY0FBYztRQUNuQixJQUFJLENBQUMsR0FBRyxDQUFDLHFCQUFxQixDQUFDLENBQUM7UUFDaEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksWUFBWTtRQUNqQixJQUFJLENBQUMsR0FBRyxDQUFDLG1CQUFtQixDQUFDLENBQUM7UUFDOUIsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksaUJBQWlCO1FBQ3RCLElBQUksQ0FBQyxHQUFHLENBQUMsd0JBQXdCLENBQUMsQ0FBQztRQUNuQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxZQUFZO1FBQ2pCLElBQUksQ0FBQyxHQUFHLENBQUMsbUJBQW1CLENBQUMsQ0FBQztRQUM5QixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxlQUFlO1FBQ3BCLElBQUksQ0FBQyxHQUFHLENBQUMsc0JBQXNCLENBQUMsQ0FBQztRQUNqQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksaUJBQWlCO1FBQ3RCLElBQUksQ0FBQyxHQUFHLENBQUMsd0JBQXdCLENBQUMsQ0FBQztRQUNuQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO1FBQ2xDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGNBQWM7UUFDbkIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO1FBQ2hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG9CQUFvQjtRQUN6QixJQUFJLENBQUMsR0FBRyxDQUFDLDJCQUEyQixDQUFDLENBQUM7UUFDdEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksWUFBWTtRQUNqQixJQUFJLENBQUMsR0FBRyxDQUFDLG1CQUFtQixDQUFDLENBQUM7UUFDOUIsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZUFBZTtRQUNwQixJQUFJLENBQUMsR0FBRyxDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFDakMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksaUJBQWlCO1FBQ3RCLElBQUksQ0FBQyxHQUFHLENBQUMsd0JBQXdCLENBQUMsQ0FBQztRQUNuQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO1FBQ3BDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixJQUFJLENBQUMsR0FBRyxDQUFDLDRCQUE0QixDQUFDLENBQUM7UUFDdkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksVUFBVTtRQUNmLElBQUksQ0FBQyxHQUFHLENBQUMsaUJBQWlCLENBQUMsQ0FBQztRQUM1QixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxTQUFTO1FBQ2QsSUFBSSxDQUFDLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO1FBQzNCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFdBQVc7UUFDaEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO1FBQzdCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixJQUFJLENBQUMsR0FBRyxDQUFDLHVCQUF1QixDQUFDLENBQUM7UUFDbEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsR0FBRyxDQUFDLG9CQUFvQixDQUFDLENBQUM7UUFDL0IsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLGlCQUFpQjtRQUN0QixJQUFJLENBQUMsR0FBRyxDQUFDLHdCQUF3QixDQUFDLENBQUM7UUFDbkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7T0FPRztJQUNJLFVBQVUsQ0FBQyxZQUFvQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQzNGLElBQUksR0FBRyxHQUFHLHNFQUFzRSxDQUFDO1FBQ2pGLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGlCQUFpQixFQUFFLFlBQVksQ0FBQyxDQUFDO1FBQ25ELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7O09BUUc7SUFDSSxXQUFXLENBQUMsWUFBb0IsRUFBRSxhQUFxQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ25ILElBQUksR0FBRyxHQUFHLGtHQUFrRyxDQUFDO1FBQzdHLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGlCQUFpQixFQUFFLFlBQVksQ0FBQyxDQUFDO1FBQ25ELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGtCQUFrQixFQUFFLGFBQWEsQ0FBQyxDQUFDO1FBQ3JELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0NBQ0Y7QUF6NUJELG9CQXk1QkMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBBY3Rpb25zLCBQb2xpY3lTdGF0ZW1lbnQsIFJlc291cmNlVHlwZXMgfSBmcm9tIFwiLi4vc2hhcmVkXCI7XG5cbi8qKlxuICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFtsb2dzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvbGlzdF9hbWF6b25jbG91ZHdhdGNobG9ncy5odG1sKS5cbiAqXG4gKiBAcGFyYW0gc2lkIFtTSURdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfc2lkLmh0bWwpIG9mIHRoZSBzdGF0ZW1lbnRcbiAqL1xuZXhwb3J0IGNsYXNzIExvZ3MgZXh0ZW5kcyBQb2xpY3lTdGF0ZW1lbnQge1xuICBwdWJsaWMgc2VydmljZVByZWZpeCA9ICdsb2dzJztcbiAgcHJvdGVjdGVkIGFjdGlvbkxpc3Q6IEFjdGlvbnMgPSB7XG4gICAgXCJBc3NvY2lhdGVLbXNLZXlcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQXNzb2NpYXRlS21zS2V5Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJBc3NvY2lhdGVzIHRoZSBzcGVjaWZpZWQgQVdTIEtleSBNYW5hZ2VtZW50IFNlcnZpY2UgKEFXUyBLTVMpIGN1c3RvbWVyIG1hc3RlciBrZXkgKENNSykgd2l0aCB0aGUgc3BlY2lmaWVkIGxvZyBncm91cC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJsb2ctZ3JvdXBcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkNhbmNlbEV4cG9ydFRhc2tcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ2FuY2VsRXhwb3J0VGFzay5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQ2FuY2VscyBhbiBleHBvcnQgdGFzayBpZiBpdCBpcyBpbiBQRU5ESU5HIG9yIFJVTk5JTkcgc3RhdGVcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiXG4gICAgfSxcbiAgICBcIkNyZWF0ZUV4cG9ydFRhc2tcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlRXhwb3J0VGFzay5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQ3JlYXRlcyBhbiBFeHBvcnRUYXNrIHdoaWNoIGFsbG93cyB5b3UgdG8gZWZmaWNpZW50bHkgZXhwb3J0IGRhdGEgZnJvbSBhIExvZyBHcm91cCB0byB5b3VyIEFtYXpvbiBTMyBidWNrZXRcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJsb2ctZ3JvdXBcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkNyZWF0ZUxvZ0RlbGl2ZXJ5XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQ3JlYXRlcyB0aGUgbG9nIGRlbGl2ZXJ5XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIlxuICAgIH0sXG4gICAgXCJDcmVhdGVMb2dHcm91cFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVMb2dHcm91cC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQ3JlYXRlcyBhIG5ldyBsb2cgZ3JvdXAgd2l0aCB0aGUgc3BlY2lmaWVkIG5hbWVcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiXG4gICAgfSxcbiAgICBcIkNyZWF0ZUxvZ1N0cmVhbVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVMb2dTdHJlYW0uaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkNyZWF0ZXMgYSBuZXcgbG9nIHN0cmVhbSB3aXRoIHRoZSBzcGVjaWZpZWQgbmFtZVwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImxvZy1ncm91cFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVsZXRlRGVzdGluYXRpb25cIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlRGVzdGluYXRpb24uaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkRlbGV0ZXMgdGhlIGRlc3RpbmF0aW9uIHdpdGggdGhlIHNwZWNpZmllZCBuYW1lIGFuZCBldmVudHVhbGx5IGRpc2FibGVzIGFsbCB0aGUgc3Vic2NyaXB0aW9uIGZpbHRlcnMgdGhhdCBwdWJsaXNoIHRvIGl0XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIlxuICAgIH0sXG4gICAgXCJEZWxldGVMb2dEZWxpdmVyeVwiOiB7XG4gICAgICBcInVybFwiOiBcIlwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkRlbGV0ZXMgdGhlIGxvZyBkZWxpdmVyeSBpbmZvcm1hdGlvbiBmb3Igc3BlY2lmaWVkIGxvZyBkZWxpdmVyeVwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCJcbiAgICB9LFxuICAgIFwiRGVsZXRlTG9nR3JvdXBcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlTG9nR3JvdXAuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkRlbGV0ZXMgdGhlIGxvZyBncm91cCB3aXRoIHRoZSBzcGVjaWZpZWQgbmFtZSBhbmQgcGVybWFuZW50bHkgZGVsZXRlcyBhbGwgdGhlIGFyY2hpdmVkIGxvZyBldmVudHMgYXNzb2NpYXRlZCB3aXRoIGl0XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwibG9nLWdyb3VwXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZWxldGVMb2dTdHJlYW1cIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlTG9nU3RyZWFtLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJEZWxldGVzIGEgbG9nIHN0cmVhbSBhbmQgcGVybWFuZW50bHkgZGVsZXRlcyBhbGwgdGhlIGFyY2hpdmVkIGxvZyBldmVudHMgYXNzb2NpYXRlZCB3aXRoIGl0XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwibG9nLXN0cmVhbVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVsZXRlTWV0cmljRmlsdGVyXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZU1ldHJpY0ZpbHRlci5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVsZXRlcyBhIG1ldHJpYyBmaWx0ZXIgYXNzb2NpYXRlZCB3aXRoIHRoZSBzcGVjaWZpZWQgbG9nIGdyb3VwXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwibG9nLWdyb3VwXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZWxldGVSZXNvdXJjZVBvbGljeVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVSZXNvdXJjZVBvbGljeS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVsZXRlcyBhIHJlc291cmNlIHBvbGljeSBmcm9tIHRoaXMgYWNjb3VudFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCJcbiAgICB9LFxuICAgIFwiRGVsZXRlUmV0ZW50aW9uUG9saWN5XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZVJldGVudGlvblBvbGljeS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVsZXRlcyB0aGUgcmV0ZW50aW9uIHBvbGljeSBvZiB0aGUgc3BlY2lmaWVkIGxvZyBncm91cFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImxvZy1ncm91cFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVsZXRlU3Vic2NyaXB0aW9uRmlsdGVyXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZVN1YnNjcmlwdGlvbkZpbHRlci5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVsZXRlcyBhIHN1YnNjcmlwdGlvbiBmaWx0ZXIgYXNzb2NpYXRlZCB3aXRoIHRoZSBzcGVjaWZpZWQgbG9nIGdyb3VwXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwibG9nLWdyb3VwXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZXNjcmliZURlc3RpbmF0aW9uc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZURlc3RpbmF0aW9ucy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyBhbGwgdGhlIGRlc3RpbmF0aW9ucyB0aGF0IGFyZSBhc3NvY2lhdGVkIHdpdGggdGhlIEFXUyBhY2NvdW50IG1ha2luZyB0aGUgcmVxdWVzdFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIlxuICAgIH0sXG4gICAgXCJEZXNjcmliZUV4cG9ydFRhc2tzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlRXhwb3J0VGFza3MuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHVybnMgYWxsIHRoZSBleHBvcnQgdGFza3MgdGhhdCBhcmUgYXNzb2NpYXRlZCB3aXRoIHRoZSBBV1MgYWNjb3VudCBtYWtpbmcgdGhlIHJlcXVlc3RcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCJcbiAgICB9LFxuICAgIFwiRGVzY3JpYmVMb2dHcm91cHNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVMb2dHcm91cHMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHVybnMgYWxsIHRoZSBsb2cgZ3JvdXBzIHRoYXQgYXJlIGFzc29jaWF0ZWQgd2l0aCB0aGUgQVdTIGFjY291bnQgbWFraW5nIHRoZSByZXF1ZXN0XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJsb2ctZ3JvdXBcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlc2NyaWJlTG9nU3RyZWFtc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZUxvZ1N0cmVhbXMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHVybnMgYWxsIHRoZSBsb2cgc3RyZWFtcyB0aGF0IGFyZSBhc3NvY2lhdGVkIHdpdGggdGhlIHNwZWNpZmllZCBsb2cgZ3JvdXBcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImxvZy1ncm91cFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVzY3JpYmVNZXRyaWNGaWx0ZXJzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlTWV0cmljRmlsdGVycy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyBhbGwgdGhlIG1ldHJpY3MgZmlsdGVycyBhc3NvY2lhdGVkIHdpdGggdGhlIHNwZWNpZmllZCBsb2cgZ3JvdXBcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImxvZy1ncm91cFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVzY3JpYmVRdWVyaWVzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlUXVlcmllcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyBhIGxpc3Qgb2YgQ2xvdWRXYXRjaCBMb2dzIEluc2lnaHRzIHF1ZXJpZXMgdGhhdCBhcmUgc2NoZWR1bGVkLCBleGVjdXRpbmcsIG9yIGhhdmUgYmVlbiBleGVjdXRlZCByZWNlbnRseSBpbiB0aGlzIGFjY291bnQuIFlvdSBjYW4gcmVxdWVzdCBhbGwgcXVlcmllcywgb3IgbGltaXQgaXQgdG8gcXVlcmllcyBvZiBhIHNwZWNpZmljIGxvZyBncm91cCBvciBxdWVyaWVzIHdpdGggYSBjZXJ0YWluIHN0YXR1cy5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCJcbiAgICB9LFxuICAgIFwiRGVzY3JpYmVSZXNvdXJjZVBvbGljaWVzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlUmVzb3VyY2VQb2xpY2llcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJuIGFsbCB0aGUgcmVzb3VyY2UgcG9saWNpZXMgaW4gdGhpcyBhY2NvdW50LlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIlxuICAgIH0sXG4gICAgXCJEZXNjcmliZVN1YnNjcmlwdGlvbkZpbHRlcnNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVTdWJzY3JpcHRpb25GaWx0ZXJzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZXR1cm5zIGFsbCB0aGUgc3Vic2NyaXB0aW9uIGZpbHRlcnMgYXNzb2NpYXRlZCB3aXRoIHRoZSBzcGVjaWZpZWQgbG9nIGdyb3VwXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJsb2ctZ3JvdXBcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRpc2Fzc29jaWF0ZUttc0tleVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EaXNhc3NvY2lhdGVLbXNLZXkuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkRpc2Fzc29jaWF0ZXMgdGhlIGFzc29jaWF0ZWQgQVdTIEtleSBNYW5hZ2VtZW50IFNlcnZpY2UgKEFXUyBLTVMpIGN1c3RvbWVyIG1hc3RlciBrZXkgKENNSykgZnJvbSB0aGUgc3BlY2lmaWVkIGxvZyBncm91cFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImxvZy1ncm91cFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRmlsdGVyTG9nRXZlbnRzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0ZpbHRlckxvZ0V2ZW50cy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0cmlldmVzIGxvZyBldmVudHMsIG9wdGlvbmFsbHkgZmlsdGVyZWQgYnkgYSBmaWx0ZXIgcGF0dGVybiBmcm9tIHRoZSBzcGVjaWZpZWQgbG9nIGdyb3VwXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJsb2ctZ3JvdXBcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkdldExvZ0RlbGl2ZXJ5XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR2V0cyB0aGUgbG9nIGRlbGl2ZXJ5IGluZm9ybWF0aW9uIGZvciBzcGVjaWZpZWQgbG9nIGRlbGl2ZXJ5XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiXG4gICAgfSxcbiAgICBcIkdldExvZ0V2ZW50c1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9HZXRMb2dFdmVudHMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHJpZXZlcyBsb2cgZXZlbnRzIGZyb20gdGhlIHNwZWNpZmllZCBsb2cgc3RyZWFtXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJsb2ctc3RyZWFtXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJHZXRMb2dHcm91cEZpZWxkc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9HZXRMb2dHcm91cEZpZWxkcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyBhIGxpc3Qgb2YgdGhlIGZpZWxkcyB0aGF0IGFyZSBpbmNsdWRlZCBpbiBsb2cgZXZlbnRzIGluIHRoZSBzcGVjaWZpZWQgbG9nIGdyb3VwLCBhbG9uZyB3aXRoIHRoZSBwZXJjZW50YWdlIG9mIGxvZyBldmVudHMgdGhhdCBjb250YWluIGVhY2ggZmllbGQuIFRoZSBzZWFyY2ggaXMgbGltaXRlZCB0byBhIHRpbWUgcGVyaW9kIHRoYXQgeW91IHNwZWNpZnkuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJsb2ctZ3JvdXBcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkdldExvZ1JlY29yZFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9HZXRMb2dSZWNvcmQuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHJpZXZlcyBhbGwgdGhlIGZpZWxkcyBhbmQgdmFsdWVzIG9mIGEgc2luZ2xlIGxvZyBldmVudC4gQWxsIGZpZWxkcyBhcmUgcmV0cmlldmVkLCBldmVuIGlmIHRoZSBvcmlnaW5hbCBxdWVyeSB0aGF0IHByb2R1Y2VkIHRoZSBsb2dSZWNvcmRQb2ludGVyIHJldHJpZXZlZCBvbmx5IGEgc3Vic2V0IG9mIGZpZWxkcy4gRmllbGRzIGFyZSByZXR1cm5lZCBhcyBmaWVsZCBuYW1lL2ZpZWxkIHZhbHVlIHBhaXJzLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIlxuICAgIH0sXG4gICAgXCJHZXRRdWVyeVJlc3VsdHNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfR2V0UXVlcnlSZXN1bHRzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZXR1cm5zIHRoZSByZXN1bHRzIGZyb20gdGhlIHNwZWNpZmllZCBxdWVyeS4gSWYgdGhlIHF1ZXJ5IGlzIGluIHByb2dyZXNzLCBwYXJ0aWFsIHJlc3VsdHMgb2YgdGhhdCBjdXJyZW50IGV4ZWN1dGlvbiBhcmUgcmV0dXJuZWQuIE9ubHkgdGhlIGZpZWxkcyByZXF1ZXN0ZWQgaW4gdGhlIHF1ZXJ5IGFyZSByZXR1cm5lZC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCJcbiAgICB9LFxuICAgIFwiTGlzdExvZ0RlbGl2ZXJpZXNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJMaXN0cyBhbGwgdGhlIGxvZyBkZWxpdmVyaWVzIGZvciBzcGVjaWZpZWQgYWNjb3VudCBhbmQvb3IgbG9nIHNvdXJjZVwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIlxuICAgIH0sXG4gICAgXCJMaXN0VGFnc0xvZ0dyb3VwXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RUYWdzTG9nR3JvdXAuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkxpc3RzIHRoZSB0YWdzIGZvciB0aGUgc3BlY2lmaWVkIGxvZyBncm91cFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwibG9nLWdyb3VwXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJQdXREZXN0aW5hdGlvblwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9QdXREZXN0aW5hdGlvbi5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQ3JlYXRlcyBvciB1cGRhdGVzIGEgRGVzdGluYXRpb25cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiXG4gICAgfSxcbiAgICBcIlB1dERlc3RpbmF0aW9uUG9saWN5XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1B1dERlc3RpbmF0aW9uUG9saWN5Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJDcmVhdGVzIG9yIHVwZGF0ZXMgYW4gYWNjZXNzIHBvbGljeSBhc3NvY2lhdGVkIHdpdGggYW4gZXhpc3RpbmcgRGVzdGluYXRpb25cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiXG4gICAgfSxcbiAgICBcIlB1dExvZ0V2ZW50c1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9QdXRMb2dFdmVudHMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlVwbG9hZHMgYSBiYXRjaCBvZiBsb2cgZXZlbnRzIHRvIHRoZSBzcGVjaWZpZWQgbG9nIHN0cmVhbVwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImxvZy1zdHJlYW1cIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIlB1dE1ldHJpY0ZpbHRlclwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9QdXRNZXRyaWNGaWx0ZXIuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkNyZWF0ZXMgb3IgdXBkYXRlcyBhIG1ldHJpYyBmaWx0ZXIgYW5kIGFzc29jaWF0ZXMgaXQgd2l0aCB0aGUgc3BlY2lmaWVkIGxvZyBncm91cFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImxvZy1ncm91cFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiUHV0UmVzb3VyY2VQb2xpY3lcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUHV0UmVzb3VyY2VQb2xpY3kuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkNyZWF0ZXMgb3IgdXBkYXRlcyBhIHJlc291cmNlIHBvbGljeSBhbGxvd2luZyBvdGhlciBBV1Mgc2VydmljZXMgdG8gcHV0IGxvZyBldmVudHMgdG8gdGhpcyBhY2NvdW50XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIlxuICAgIH0sXG4gICAgXCJQdXRSZXRlbnRpb25Qb2xpY3lcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUHV0UmV0ZW50aW9uUG9saWN5Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJTZXRzIHRoZSByZXRlbnRpb24gb2YgdGhlIHNwZWNpZmllZCBsb2cgZ3JvdXBcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJsb2ctZ3JvdXBcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIlB1dFN1YnNjcmlwdGlvbkZpbHRlclwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9QdXRTdWJzY3JpcHRpb25GaWx0ZXIuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkNyZWF0ZXMgb3IgdXBkYXRlcyBhIHN1YnNjcmlwdGlvbiBmaWx0ZXIgYW5kIGFzc29jaWF0ZXMgaXQgd2l0aCB0aGUgc3BlY2lmaWVkIGxvZyBncm91cFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImxvZy1ncm91cFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiU3RhcnRRdWVyeVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9TdGFydFF1ZXJ5Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJTY2hlZHVsZXMgYSBxdWVyeSBvZiBhIGxvZyBncm91cCB1c2luZyBDbG91ZFdhdGNoIExvZ3MgSW5zaWdodHMuIFlvdSBzcGVjaWZ5IHRoZSBsb2cgZ3JvdXAgYW5kIHRpbWUgcmFuZ2UgdG8gcXVlcnksIGFuZCB0aGUgcXVlcnkgc3RyaW5nIHRvIHVzZS5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImxvZy1ncm91cFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiU3RvcFF1ZXJ5XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1N0b3BRdWVyeS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiU3RvcHMgYSBDbG91ZFdhdGNoIExvZ3MgSW5zaWdodHMgcXVlcnkgdGhhdCBpcyBpbiBwcm9ncmVzcy4gSWYgdGhlIHF1ZXJ5IGhhcyBhbHJlYWR5IGVuZGVkLCB0aGUgb3BlcmF0aW9uIHJldHVybnMgYW4gZXJyb3IgaW5kaWNhdGluZyB0aGF0IHRoZSBzcGVjaWZpZWQgcXVlcnkgaXMgbm90IHJ1bm5pbmcuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiXG4gICAgfSxcbiAgICBcIlRhZ0xvZ0dyb3VwXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1RhZ0xvZ0dyb3VwLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJBZGRzIG9yIHVwZGF0ZXMgdGhlIHNwZWNpZmllZCB0YWdzIGZvciB0aGUgc3BlY2lmaWVkIGxvZyBncm91cFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImxvZy1ncm91cFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiVGVzdE1ldHJpY0ZpbHRlclwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9UZXN0TWV0cmljRmlsdGVyLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJUZXN0cyB0aGUgZmlsdGVyIHBhdHRlcm4gb2YgYSBtZXRyaWMgZmlsdGVyIGFnYWluc3QgYSBzYW1wbGUgb2YgbG9nIGV2ZW50IG1lc3NhZ2VzXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiXG4gICAgfSxcbiAgICBcIlVudGFnTG9nR3JvdXBcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVW50YWdMb2dHcm91cC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmVtb3ZlcyB0aGUgc3BlY2lmaWVkIHRhZ3MgZnJvbSB0aGUgc3BlY2lmaWVkIGxvZyBncm91cFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImxvZy1ncm91cFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiVXBkYXRlTG9nRGVsaXZlcnlcIjoge1xuICAgICAgXCJ1cmxcIjogXCJcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJVcGRhdGVzIHRoZSBsb2cgZGVsaXZlcnkgaW5mb3JtYXRpb24gZm9yIHNwZWNpZmllZCBsb2cgZGVsaXZlcnlcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiXG4gICAgfVxuICB9O1xuICBwcm90ZWN0ZWQgcmVzb3VyY2VUeXBlczogUmVzb3VyY2VUeXBlcyA9IHtcbiAgICBcImxvZy1ncm91cFwiOiB7XG4gICAgICBcIm5hbWVcIjogXCJsb2ctZ3JvdXBcIixcbiAgICAgIFwidXJsXCI6IFwiXCIsXG4gICAgICBcImFyblwiOiBcImFybjoke1BhcnRpdGlvbn06bG9nczoke1JlZ2lvbn06JHtBY2NvdW50fTpsb2ctZ3JvdXA6JHtMb2dHcm91cE5hbWV9XCIsXG4gICAgICBcImNvbmRpdGlvbktleXNcIjogW11cbiAgICB9LFxuICAgIFwibG9nLXN0cmVhbVwiOiB7XG4gICAgICBcIm5hbWVcIjogXCJsb2ctc3RyZWFtXCIsXG4gICAgICBcInVybFwiOiBcIlwiLFxuICAgICAgXCJhcm5cIjogXCJhcm46JHtQYXJ0aXRpb259OmxvZ3M6JHtSZWdpb259OiR7QWNjb3VudH06bG9nLWdyb3VwOiR7TG9nR3JvdXBOYW1lfTpsb2ctc3RyZWFtOiR7TG9nU3RyZWFtTmFtZX1cIixcbiAgICAgIFwiY29uZGl0aW9uS2V5c1wiOiBbXVxuICAgIH1cbiAgfTtcblxuICAvKipcbiAgICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFtsb2dzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvbGlzdF9hbWF6b25jbG91ZHdhdGNobG9ncy5odG1sKS5cbiAgICpcbiAgICogQHBhcmFtIHNpZCBbU0lEXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX3NpZC5odG1sKSBvZiB0aGUgc3RhdGVtZW50XG4gICAqL1xuICBjb25zdHJ1Y3RvciAoc2lkPzogc3RyaW5nKSB7XG4gICAgc3VwZXIoc2lkKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBc3NvY2lhdGVzIHRoZSBzcGVjaWZpZWQgQVdTIEtleSBNYW5hZ2VtZW50IFNlcnZpY2UgKEFXUyBLTVMpIGN1c3RvbWVyIG1hc3RlciBrZXkgKENNSykgd2l0aCB0aGUgc3BlY2lmaWVkIGxvZyBncm91cC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQXNzb2NpYXRlS21zS2V5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyBhc3NvY2lhdGVLbXNLZXkoKSB7XG4gICAgdGhpcy5hZGQoJ2xvZ3M6QXNzb2NpYXRlS21zS2V5Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQ2FuY2VscyBhbiBleHBvcnQgdGFzayBpZiBpdCBpcyBpbiBQRU5ESU5HIG9yIFJVTk5JTkcgc3RhdGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ2FuY2VsRXhwb3J0VGFzay5odG1sXG4gICAqL1xuICBwdWJsaWMgY2FuY2VsRXhwb3J0VGFzaygpIHtcbiAgICB0aGlzLmFkZCgnbG9nczpDYW5jZWxFeHBvcnRUYXNrJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBhbiBFeHBvcnRUYXNrIHdoaWNoIGFsbG93cyB5b3UgdG8gZWZmaWNpZW50bHkgZXhwb3J0IGRhdGEgZnJvbSBhIExvZyBHcm91cCB0byB5b3VyIEFtYXpvbiBTMyBidWNrZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlRXhwb3J0VGFzay5odG1sXG4gICAqL1xuICBwdWJsaWMgY3JlYXRlRXhwb3J0VGFzaygpIHtcbiAgICB0aGlzLmFkZCgnbG9nczpDcmVhdGVFeHBvcnRUYXNrJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyB0aGUgbG9nIGRlbGl2ZXJ5XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICovXG4gIHB1YmxpYyBjcmVhdGVMb2dEZWxpdmVyeSgpIHtcbiAgICB0aGlzLmFkZCgnbG9nczpDcmVhdGVMb2dEZWxpdmVyeScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgYSBuZXcgbG9nIGdyb3VwIHdpdGggdGhlIHNwZWNpZmllZCBuYW1lXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZUxvZ0dyb3VwLmh0bWxcbiAgICovXG4gIHB1YmxpYyBjcmVhdGVMb2dHcm91cCgpIHtcbiAgICB0aGlzLmFkZCgnbG9nczpDcmVhdGVMb2dHcm91cCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgYSBuZXcgbG9nIHN0cmVhbSB3aXRoIHRoZSBzcGVjaWZpZWQgbmFtZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVMb2dTdHJlYW0uaHRtbFxuICAgKi9cbiAgcHVibGljIGNyZWF0ZUxvZ1N0cmVhbSgpIHtcbiAgICB0aGlzLmFkZCgnbG9nczpDcmVhdGVMb2dTdHJlYW0nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWxldGVzIHRoZSBkZXN0aW5hdGlvbiB3aXRoIHRoZSBzcGVjaWZpZWQgbmFtZSBhbmQgZXZlbnR1YWxseSBkaXNhYmxlcyBhbGwgdGhlIHN1YnNjcmlwdGlvbiBmaWx0ZXJzIHRoYXQgcHVibGlzaCB0byBpdFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVEZXN0aW5hdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgZGVsZXRlRGVzdGluYXRpb24oKSB7XG4gICAgdGhpcy5hZGQoJ2xvZ3M6RGVsZXRlRGVzdGluYXRpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWxldGVzIHRoZSBsb2cgZGVsaXZlcnkgaW5mb3JtYXRpb24gZm9yIHNwZWNpZmllZCBsb2cgZGVsaXZlcnlcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKi9cbiAgcHVibGljIGRlbGV0ZUxvZ0RlbGl2ZXJ5KCkge1xuICAgIHRoaXMuYWRkKCdsb2dzOkRlbGV0ZUxvZ0RlbGl2ZXJ5Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVsZXRlcyB0aGUgbG9nIGdyb3VwIHdpdGggdGhlIHNwZWNpZmllZCBuYW1lIGFuZCBwZXJtYW5lbnRseSBkZWxldGVzIGFsbCB0aGUgYXJjaGl2ZWQgbG9nIGV2ZW50cyBhc3NvY2lhdGVkIHdpdGggaXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlTG9nR3JvdXAuaHRtbFxuICAgKi9cbiAgcHVibGljIGRlbGV0ZUxvZ0dyb3VwKCkge1xuICAgIHRoaXMuYWRkKCdsb2dzOkRlbGV0ZUxvZ0dyb3VwJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVsZXRlcyBhIGxvZyBzdHJlYW0gYW5kIHBlcm1hbmVudGx5IGRlbGV0ZXMgYWxsIHRoZSBhcmNoaXZlZCBsb2cgZXZlbnRzIGFzc29jaWF0ZWQgd2l0aCBpdFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVMb2dTdHJlYW0uaHRtbFxuICAgKi9cbiAgcHVibGljIGRlbGV0ZUxvZ1N0cmVhbSgpIHtcbiAgICB0aGlzLmFkZCgnbG9nczpEZWxldGVMb2dTdHJlYW0nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWxldGVzIGEgbWV0cmljIGZpbHRlciBhc3NvY2lhdGVkIHdpdGggdGhlIHNwZWNpZmllZCBsb2cgZ3JvdXBcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlTWV0cmljRmlsdGVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyBkZWxldGVNZXRyaWNGaWx0ZXIoKSB7XG4gICAgdGhpcy5hZGQoJ2xvZ3M6RGVsZXRlTWV0cmljRmlsdGVyJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVsZXRlcyBhIHJlc291cmNlIHBvbGljeSBmcm9tIHRoaXMgYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVSZXNvdXJjZVBvbGljeS5odG1sXG4gICAqL1xuICBwdWJsaWMgZGVsZXRlUmVzb3VyY2VQb2xpY3koKSB7XG4gICAgdGhpcy5hZGQoJ2xvZ3M6RGVsZXRlUmVzb3VyY2VQb2xpY3knKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWxldGVzIHRoZSByZXRlbnRpb24gcG9saWN5IG9mIHRoZSBzcGVjaWZpZWQgbG9nIGdyb3VwXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZVJldGVudGlvblBvbGljeS5odG1sXG4gICAqL1xuICBwdWJsaWMgZGVsZXRlUmV0ZW50aW9uUG9saWN5KCkge1xuICAgIHRoaXMuYWRkKCdsb2dzOkRlbGV0ZVJldGVudGlvblBvbGljeScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlbGV0ZXMgYSBzdWJzY3JpcHRpb24gZmlsdGVyIGFzc29jaWF0ZWQgd2l0aCB0aGUgc3BlY2lmaWVkIGxvZyBncm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVTdWJzY3JpcHRpb25GaWx0ZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIGRlbGV0ZVN1YnNjcmlwdGlvbkZpbHRlcigpIHtcbiAgICB0aGlzLmFkZCgnbG9nczpEZWxldGVTdWJzY3JpcHRpb25GaWx0ZXInKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGFsbCB0aGUgZGVzdGluYXRpb25zIHRoYXQgYXJlIGFzc29jaWF0ZWQgd2l0aCB0aGUgQVdTIGFjY291bnQgbWFraW5nIHRoZSByZXF1ZXN0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVEZXN0aW5hdGlvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIGRlc2NyaWJlRGVzdGluYXRpb25zKCkge1xuICAgIHRoaXMuYWRkKCdsb2dzOkRlc2NyaWJlRGVzdGluYXRpb25zJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBhbGwgdGhlIGV4cG9ydCB0YXNrcyB0aGF0IGFyZSBhc3NvY2lhdGVkIHdpdGggdGhlIEFXUyBhY2NvdW50IG1ha2luZyB0aGUgcmVxdWVzdFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlRXhwb3J0VGFza3MuaHRtbFxuICAgKi9cbiAgcHVibGljIGRlc2NyaWJlRXhwb3J0VGFza3MoKSB7XG4gICAgdGhpcy5hZGQoJ2xvZ3M6RGVzY3JpYmVFeHBvcnRUYXNrcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgYWxsIHRoZSBsb2cgZ3JvdXBzIHRoYXQgYXJlIGFzc29jaWF0ZWQgd2l0aCB0aGUgQVdTIGFjY291bnQgbWFraW5nIHRoZSByZXF1ZXN0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVMb2dHcm91cHMuaHRtbFxuICAgKi9cbiAgcHVibGljIGRlc2NyaWJlTG9nR3JvdXBzKCkge1xuICAgIHRoaXMuYWRkKCdsb2dzOkRlc2NyaWJlTG9nR3JvdXBzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBhbGwgdGhlIGxvZyBzdHJlYW1zIHRoYXQgYXJlIGFzc29jaWF0ZWQgd2l0aCB0aGUgc3BlY2lmaWVkIGxvZyBncm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlTG9nU3RyZWFtcy5odG1sXG4gICAqL1xuICBwdWJsaWMgZGVzY3JpYmVMb2dTdHJlYW1zKCkge1xuICAgIHRoaXMuYWRkKCdsb2dzOkRlc2NyaWJlTG9nU3RyZWFtcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgYWxsIHRoZSBtZXRyaWNzIGZpbHRlcnMgYXNzb2NpYXRlZCB3aXRoIHRoZSBzcGVjaWZpZWQgbG9nIGdyb3VwXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVNZXRyaWNGaWx0ZXJzLmh0bWxcbiAgICovXG4gIHB1YmxpYyBkZXNjcmliZU1ldHJpY0ZpbHRlcnMoKSB7XG4gICAgdGhpcy5hZGQoJ2xvZ3M6RGVzY3JpYmVNZXRyaWNGaWx0ZXJzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBhIGxpc3Qgb2YgQ2xvdWRXYXRjaCBMb2dzIEluc2lnaHRzIHF1ZXJpZXMgdGhhdCBhcmUgc2NoZWR1bGVkLCBleGVjdXRpbmcsIG9yIGhhdmUgYmVlbiBleGVjdXRlZCByZWNlbnRseSBpbiB0aGlzIGFjY291bnQuIFlvdSBjYW4gcmVxdWVzdCBhbGwgcXVlcmllcywgb3IgbGltaXQgaXQgdG8gcXVlcmllcyBvZiBhIHNwZWNpZmljIGxvZyBncm91cCBvciBxdWVyaWVzIHdpdGggYSBjZXJ0YWluIHN0YXR1cy5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZVF1ZXJpZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIGRlc2NyaWJlUXVlcmllcygpIHtcbiAgICB0aGlzLmFkZCgnbG9nczpEZXNjcmliZVF1ZXJpZXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm4gYWxsIHRoZSByZXNvdXJjZSBwb2xpY2llcyBpbiB0aGlzIGFjY291bnQuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVSZXNvdXJjZVBvbGljaWVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyBkZXNjcmliZVJlc291cmNlUG9saWNpZXMoKSB7XG4gICAgdGhpcy5hZGQoJ2xvZ3M6RGVzY3JpYmVSZXNvdXJjZVBvbGljaWVzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBhbGwgdGhlIHN1YnNjcmlwdGlvbiBmaWx0ZXJzIGFzc29jaWF0ZWQgd2l0aCB0aGUgc3BlY2lmaWVkIGxvZyBncm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlU3Vic2NyaXB0aW9uRmlsdGVycy5odG1sXG4gICAqL1xuICBwdWJsaWMgZGVzY3JpYmVTdWJzY3JpcHRpb25GaWx0ZXJzKCkge1xuICAgIHRoaXMuYWRkKCdsb2dzOkRlc2NyaWJlU3Vic2NyaXB0aW9uRmlsdGVycycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERpc2Fzc29jaWF0ZXMgdGhlIGFzc29jaWF0ZWQgQVdTIEtleSBNYW5hZ2VtZW50IFNlcnZpY2UgKEFXUyBLTVMpIGN1c3RvbWVyIG1hc3RlciBrZXkgKENNSykgZnJvbSB0aGUgc3BlY2lmaWVkIGxvZyBncm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EaXNhc3NvY2lhdGVLbXNLZXkuaHRtbFxuICAgKi9cbiAgcHVibGljIGRpc2Fzc29jaWF0ZUttc0tleSgpIHtcbiAgICB0aGlzLmFkZCgnbG9nczpEaXNhc3NvY2lhdGVLbXNLZXknKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXRyaWV2ZXMgbG9nIGV2ZW50cywgb3B0aW9uYWxseSBmaWx0ZXJlZCBieSBhIGZpbHRlciBwYXR0ZXJuIGZyb20gdGhlIHNwZWNpZmllZCBsb2cgZ3JvdXBcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9GaWx0ZXJMb2dFdmVudHMuaHRtbFxuICAgKi9cbiAgcHVibGljIGZpbHRlckxvZ0V2ZW50cygpIHtcbiAgICB0aGlzLmFkZCgnbG9nczpGaWx0ZXJMb2dFdmVudHMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHZXRzIHRoZSBsb2cgZGVsaXZlcnkgaW5mb3JtYXRpb24gZm9yIHNwZWNpZmllZCBsb2cgZGVsaXZlcnlcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqL1xuICBwdWJsaWMgZ2V0TG9nRGVsaXZlcnkoKSB7XG4gICAgdGhpcy5hZGQoJ2xvZ3M6R2V0TG9nRGVsaXZlcnknKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXRyaWV2ZXMgbG9nIGV2ZW50cyBmcm9tIHRoZSBzcGVjaWZpZWQgbG9nIHN0cmVhbVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0dldExvZ0V2ZW50cy5odG1sXG4gICAqL1xuICBwdWJsaWMgZ2V0TG9nRXZlbnRzKCkge1xuICAgIHRoaXMuYWRkKCdsb2dzOkdldExvZ0V2ZW50cycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgYSBsaXN0IG9mIHRoZSBmaWVsZHMgdGhhdCBhcmUgaW5jbHVkZWQgaW4gbG9nIGV2ZW50cyBpbiB0aGUgc3BlY2lmaWVkIGxvZyBncm91cCwgYWxvbmcgd2l0aCB0aGUgcGVyY2VudGFnZSBvZiBsb2cgZXZlbnRzIHRoYXQgY29udGFpbiBlYWNoIGZpZWxkLiBUaGUgc2VhcmNoIGlzIGxpbWl0ZWQgdG8gYSB0aW1lIHBlcmlvZCB0aGF0IHlvdSBzcGVjaWZ5LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0dldExvZ0dyb3VwRmllbGRzLmh0bWxcbiAgICovXG4gIHB1YmxpYyBnZXRMb2dHcm91cEZpZWxkcygpIHtcbiAgICB0aGlzLmFkZCgnbG9nczpHZXRMb2dHcm91cEZpZWxkcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHJpZXZlcyBhbGwgdGhlIGZpZWxkcyBhbmQgdmFsdWVzIG9mIGEgc2luZ2xlIGxvZyBldmVudC4gQWxsIGZpZWxkcyBhcmUgcmV0cmlldmVkLCBldmVuIGlmIHRoZSBvcmlnaW5hbCBxdWVyeSB0aGF0IHByb2R1Y2VkIHRoZSBsb2dSZWNvcmRQb2ludGVyIHJldHJpZXZlZCBvbmx5IGEgc3Vic2V0IG9mIGZpZWxkcy4gRmllbGRzIGFyZSByZXR1cm5lZCBhcyBmaWVsZCBuYW1lL2ZpZWxkIHZhbHVlIHBhaXJzLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0dldExvZ1JlY29yZC5odG1sXG4gICAqL1xuICBwdWJsaWMgZ2V0TG9nUmVjb3JkKCkge1xuICAgIHRoaXMuYWRkKCdsb2dzOkdldExvZ1JlY29yZCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgdGhlIHJlc3VsdHMgZnJvbSB0aGUgc3BlY2lmaWVkIHF1ZXJ5LiBJZiB0aGUgcXVlcnkgaXMgaW4gcHJvZ3Jlc3MsIHBhcnRpYWwgcmVzdWx0cyBvZiB0aGF0IGN1cnJlbnQgZXhlY3V0aW9uIGFyZSByZXR1cm5lZC4gT25seSB0aGUgZmllbGRzIHJlcXVlc3RlZCBpbiB0aGUgcXVlcnkgYXJlIHJldHVybmVkLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0dldFF1ZXJ5UmVzdWx0cy5odG1sXG4gICAqL1xuICBwdWJsaWMgZ2V0UXVlcnlSZXN1bHRzKCkge1xuICAgIHRoaXMuYWRkKCdsb2dzOkdldFF1ZXJ5UmVzdWx0cycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIExpc3RzIGFsbCB0aGUgbG9nIGRlbGl2ZXJpZXMgZm9yIHNwZWNpZmllZCBhY2NvdW50IGFuZC9vciBsb2cgc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKi9cbiAgcHVibGljIGxpc3RMb2dEZWxpdmVyaWVzKCkge1xuICAgIHRoaXMuYWRkKCdsb2dzOkxpc3RMb2dEZWxpdmVyaWVzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogTGlzdHMgdGhlIHRhZ3MgZm9yIHRoZSBzcGVjaWZpZWQgbG9nIGdyb3VwXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdFRhZ3NMb2dHcm91cC5odG1sXG4gICAqL1xuICBwdWJsaWMgbGlzdFRhZ3NMb2dHcm91cCgpIHtcbiAgICB0aGlzLmFkZCgnbG9nczpMaXN0VGFnc0xvZ0dyb3VwJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBvciB1cGRhdGVzIGEgRGVzdGluYXRpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUHV0RGVzdGluYXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHB1dERlc3RpbmF0aW9uKCkge1xuICAgIHRoaXMuYWRkKCdsb2dzOlB1dERlc3RpbmF0aW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBvciB1cGRhdGVzIGFuIGFjY2VzcyBwb2xpY3kgYXNzb2NpYXRlZCB3aXRoIGFuIGV4aXN0aW5nIERlc3RpbmF0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1B1dERlc3RpbmF0aW9uUG9saWN5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyBwdXREZXN0aW5hdGlvblBvbGljeSgpIHtcbiAgICB0aGlzLmFkZCgnbG9nczpQdXREZXN0aW5hdGlvblBvbGljeScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFVwbG9hZHMgYSBiYXRjaCBvZiBsb2cgZXZlbnRzIHRvIHRoZSBzcGVjaWZpZWQgbG9nIHN0cmVhbVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9QdXRMb2dFdmVudHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHB1dExvZ0V2ZW50cygpIHtcbiAgICB0aGlzLmFkZCgnbG9nczpQdXRMb2dFdmVudHMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIG9yIHVwZGF0ZXMgYSBtZXRyaWMgZmlsdGVyIGFuZCBhc3NvY2lhdGVzIGl0IHdpdGggdGhlIHNwZWNpZmllZCBsb2cgZ3JvdXBcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUHV0TWV0cmljRmlsdGVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyBwdXRNZXRyaWNGaWx0ZXIoKSB7XG4gICAgdGhpcy5hZGQoJ2xvZ3M6UHV0TWV0cmljRmlsdGVyJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBvciB1cGRhdGVzIGEgcmVzb3VyY2UgcG9saWN5IGFsbG93aW5nIG90aGVyIEFXUyBzZXJ2aWNlcyB0byBwdXQgbG9nIGV2ZW50cyB0byB0aGlzIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUHV0UmVzb3VyY2VQb2xpY3kuaHRtbFxuICAgKi9cbiAgcHVibGljIHB1dFJlc291cmNlUG9saWN5KCkge1xuICAgIHRoaXMuYWRkKCdsb2dzOlB1dFJlc291cmNlUG9saWN5Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogU2V0cyB0aGUgcmV0ZW50aW9uIG9mIHRoZSBzcGVjaWZpZWQgbG9nIGdyb3VwXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2hMb2dzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1B1dFJldGVudGlvblBvbGljeS5odG1sXG4gICAqL1xuICBwdWJsaWMgcHV0UmV0ZW50aW9uUG9saWN5KCkge1xuICAgIHRoaXMuYWRkKCdsb2dzOlB1dFJldGVudGlvblBvbGljeScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgb3IgdXBkYXRlcyBhIHN1YnNjcmlwdGlvbiBmaWx0ZXIgYW5kIGFzc29jaWF0ZXMgaXQgd2l0aCB0aGUgc3BlY2lmaWVkIGxvZyBncm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9QdXRTdWJzY3JpcHRpb25GaWx0ZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHB1dFN1YnNjcmlwdGlvbkZpbHRlcigpIHtcbiAgICB0aGlzLmFkZCgnbG9nczpQdXRTdWJzY3JpcHRpb25GaWx0ZXInKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBTY2hlZHVsZXMgYSBxdWVyeSBvZiBhIGxvZyBncm91cCB1c2luZyBDbG91ZFdhdGNoIExvZ3MgSW5zaWdodHMuIFlvdSBzcGVjaWZ5IHRoZSBsb2cgZ3JvdXAgYW5kIHRpbWUgcmFuZ2UgdG8gcXVlcnksIGFuZCB0aGUgcXVlcnkgc3RyaW5nIHRvIHVzZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9TdGFydFF1ZXJ5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyBzdGFydFF1ZXJ5KCkge1xuICAgIHRoaXMuYWRkKCdsb2dzOlN0YXJ0UXVlcnknKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBTdG9wcyBhIENsb3VkV2F0Y2ggTG9ncyBJbnNpZ2h0cyBxdWVyeSB0aGF0IGlzIGluIHByb2dyZXNzLiBJZiB0aGUgcXVlcnkgaGFzIGFscmVhZHkgZW5kZWQsIHRoZSBvcGVyYXRpb24gcmV0dXJucyBhbiBlcnJvciBpbmRpY2F0aW5nIHRoYXQgdGhlIHNwZWNpZmllZCBxdWVyeSBpcyBub3QgcnVubmluZy5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9TdG9wUXVlcnkuaHRtbFxuICAgKi9cbiAgcHVibGljIHN0b3BRdWVyeSgpIHtcbiAgICB0aGlzLmFkZCgnbG9nczpTdG9wUXVlcnknKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIG9yIHVwZGF0ZXMgdGhlIHNwZWNpZmllZCB0YWdzIGZvciB0aGUgc3BlY2lmaWVkIGxvZyBncm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9UYWdMb2dHcm91cC5odG1sXG4gICAqL1xuICBwdWJsaWMgdGFnTG9nR3JvdXAoKSB7XG4gICAgdGhpcy5hZGQoJ2xvZ3M6VGFnTG9nR3JvdXAnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBUZXN0cyB0aGUgZmlsdGVyIHBhdHRlcm4gb2YgYSBtZXRyaWMgZmlsdGVyIGFnYWluc3QgYSBzYW1wbGUgb2YgbG9nIGV2ZW50IG1lc3NhZ2VzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaExvZ3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVGVzdE1ldHJpY0ZpbHRlci5odG1sXG4gICAqL1xuICBwdWJsaWMgdGVzdE1ldHJpY0ZpbHRlcigpIHtcbiAgICB0aGlzLmFkZCgnbG9nczpUZXN0TWV0cmljRmlsdGVyJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmVtb3ZlcyB0aGUgc3BlY2lmaWVkIHRhZ3MgZnJvbSB0aGUgc3BlY2lmaWVkIGxvZyBncm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoTG9ncy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VbnRhZ0xvZ0dyb3VwLmh0bWxcbiAgICovXG4gIHB1YmxpYyB1bnRhZ0xvZ0dyb3VwKCkge1xuICAgIHRoaXMuYWRkKCdsb2dzOlVudGFnTG9nR3JvdXAnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBVcGRhdGVzIHRoZSBsb2cgZGVsaXZlcnkgaW5mb3JtYXRpb24gZm9yIHNwZWNpZmllZCBsb2cgZGVsaXZlcnlcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKi9cbiAgcHVibGljIHVwZGF0ZUxvZ0RlbGl2ZXJ5KCkge1xuICAgIHRoaXMuYWRkKCdsb2dzOlVwZGF0ZUxvZ0RlbGl2ZXJ5Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgbG9nLWdyb3VwIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogQHBhcmFtIGxvZ0dyb3VwTmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBsb2dHcm91cE5hbWUuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqL1xuICBwdWJsaWMgb25Mb2dHcm91cChsb2dHcm91cE5hbWU6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06bG9nczoke1JlZ2lvbn06JHtBY2NvdW50fTpsb2ctZ3JvdXA6JHtMb2dHcm91cE5hbWV9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtMb2dHcm91cE5hbWV9JywgbG9nR3JvdXBOYW1lKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZWdpb259JywgcmVnaW9uIHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGFydGl0aW9ufScsIHBhcnRpdGlvbiB8fCAnYXdzJyk7XG4gICAgcmV0dXJuIHRoaXMub24oYXJuKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBsb2ctc3RyZWFtIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogQHBhcmFtIGxvZ0dyb3VwTmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBsb2dHcm91cE5hbWUuXG4gICAqIEBwYXJhbSBsb2dTdHJlYW1OYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIGxvZ1N0cmVhbU5hbWUuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqL1xuICBwdWJsaWMgb25Mb2dTdHJlYW0obG9nR3JvdXBOYW1lOiBzdHJpbmcsIGxvZ1N0cmVhbU5hbWU6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06bG9nczoke1JlZ2lvbn06JHtBY2NvdW50fTpsb2ctZ3JvdXA6JHtMb2dHcm91cE5hbWV9OmxvZy1zdHJlYW06JHtMb2dTdHJlYW1OYW1lfSc7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7TG9nR3JvdXBOYW1lfScsIGxvZ0dyb3VwTmFtZSk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7TG9nU3RyZWFtTmFtZX0nLCBsb2dTdHJlYW1OYW1lKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZWdpb259JywgcmVnaW9uIHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGFydGl0aW9ufScsIHBhcnRpdGlvbiB8fCAnYXdzJyk7XG4gICAgcmV0dXJuIHRoaXMub24oYXJuKTtcbiAgfVxufVxuIl19